<?php
use Blesta\Core\Util\Helpers\Helper;

/**
 * Settings Processor Helper
 *
 * Provides processing and validation for complex settings, particularly those
 * stored as JSON with multi-currency or range-based constraints.
 *
 * @package blesta
 * @subpackage helpers.settingsprocessor
 * @copyright Copyright (c) 2025, Phillips Data, Inc.
 * @license http://www.blesta.com/license/ The Blesta License Agreement
 * @link http://www.blesta.com/ Blesta
 */
class SettingsProcessor extends Helper
{
    /**
     * Validate currency-based settings with min/max constraints
     *
     * @param array $input_data Raw input data keyed by currency code,
     *   with each value containing 'min' and/or 'max' keys
     * @return array Empty array if valid, or errors keyed by currency containing error keys
     */
    public function validateCurrencyBasedSettings($input_data)
    {
        $errors = [];

        if (empty($input_data) || !is_array($input_data)) {
            return $errors;
        }

        foreach ($input_data as $currency => $limits) {
            if (!is_array($limits)) {
                continue;
            }

            $min = isset($limits['min']) && $limits['min'] !== '' ? $limits['min'] : null;
            $max = isset($limits['max']) && $limits['max'] !== '' ? $limits['max'] : null;

            // Validate minimum
            if ($min !== null && (!is_numeric($min) || (int)$min <= 0)) {
                $errors[$currency][] = 'min_amount';
            }

            // Validate maximum
            if ($max !== null && (!is_numeric($max) || (int)$max <= 0)) {
                $errors[$currency][] = 'max_amount';
            }

            // Validate that max > min
            if ($min !== null && $max !== null && (int)$max <= (int)$min) {
                $errors[$currency][] = 'max_less_than_min';
            }
        }

        return $errors;
    }

    /**
     * Process currency-based settings (format and structure)
     * No validation - use validateCurrencyBasedSettings() first if needed
     *
     * @param array $input_data Raw input data keyed by currency code,
     *   with each value containing 'min' and/or 'max' keys
     * @return array Processed settings array ready for JSON encoding
     */
    public function processCurrencyBasedSettings($input_data)
    {
        $processed_settings = [];

        if (empty($input_data) || !is_array($input_data)) {
            return $processed_settings;
        }

        foreach ($input_data as $currency => $limits) {
            if (!is_array($limits)) {
                continue;
            }

            $min = isset($limits['min']) && $limits['min'] !== '' ? $limits['min'] : null;
            $max = isset($limits['max']) && $limits['max'] !== '' ? $limits['max'] : null;

            // Skip if both are null
            if ($min === null && $max === null) {
                continue;
            }

            // Build the limit data structure
            $limit_data = [];
            $limit_data['min'] = $min !== null ? $this->formatDecimalValue($min) : null;
            $limit_data['max'] = $max !== null ? $this->formatDecimalValue($max) : null;

            $processed_settings[$currency] = $limit_data;
        }

        return $this->encodeJsonSetting($processed_settings);
    }

    /**
     * Format a decimal value with consistent precision
     *
     * @param mixed $value The value to format
     * @param int $precision Number of decimal places (default: 4)
     * @return string Formatted decimal string
     */
    public function formatDecimalValue($value, $precision = 4)
    {
        return number_format($value, $precision, '.', '');
    }

    /**
     * Encode data to JSON format with error handling
     *
     * @param mixed $data Data to encode
     * @return string JSON encoded string, or '{}' on error
     */
    public function encodeJsonSetting($data)
    {
        if (empty($data)) {
            return json_encode([]);
        }

        $json = json_encode($data);

        // Check for JSON encoding errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return json_encode([]);
        }

        return $json;
    }

    /**
     * Decode JSON setting with error handling
     *
     * @param string $json_string JSON string to decode
     * @return array Decoded array, or empty array on error
     */
    public function decodeJsonSetting($json_string)
    {
        if (empty($json_string)) {
            return [];
        }

        $decoded = json_decode($json_string, true);

        // Check for JSON decoding errors or non-array result
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded)) {
            return [];
        }

        return $decoded;
    }
}
