<?php
declare(strict_types=1);

/**
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link          https://cakephp.org CakePHP(tm) Project
 * @since         3.5.0
 * @license       https://opensource.org/licenses/mit-license.php MIT License
 */
namespace Cake\Test\TestCase\Console\Command;

use Cake\Console\CommandInterface;
use Cake\Console\TestSuite\ConsoleIntegrationTestTrait;
use Cake\TestSuite\TestCase;

/**
 * HelpCommand test.
 */
class HelpCommandTest extends TestCase
{
    use ConsoleIntegrationTestTrait;

    /**
     * setup method
     */
    protected function setUp(): void
    {
        parent::setUp();
        $this->setAppNamespace();
        $this->loadPlugins(['TestPlugin']);
    }

    /**
     * tearDown
     */
    protected function tearDown(): void
    {
        parent::tearDown();
        $this->clearPlugins();
    }

    /**
     * Test the verbose command listing
     */
    public function testMainVerbose(): void
    {
        $this->exec('help -v');
        $this->assertExitCode(CommandInterface::CODE_SUCCESS);
        $this->assertCommandListVerbose();
    }

    /**
     * Test the compact command listing (default)
     */
    public function testMainCompact(): void
    {
        $this->exec('help');
        $this->assertExitCode(CommandInterface::CODE_SUCCESS);
        $this->assertOutputContains('<info>Available Commands:</info>', 'single commands header');
        $this->assertOutputContains('<info>routes:</info>', 'routes group header');
        $this->assertOutputContains('<info>cache:</info>', 'cache group header');
        $this->assertOutputContains('clear', 'cache subcommand listed');
        $this->assertOutputContains('Clear all data in a single cache engine', 'inline description shown');
        $this->assertOutputNotContains('<info>app</info>:', 'no plugin group headers in compact mode');
        $this->assertOutputContains('To run a command', 'more info present');
    }

    /**
     * Assert the verbose help output.
     */
    protected function assertCommandListVerbose(): void
    {
        $this->assertOutputContains('<info>test_plugin</info>', 'plugin header should appear');
        $this->assertOutputContains('- sample', 'plugin command should appear');
        $this->assertOutputNotContains(
            '- test_plugin.sample',
            'only short alias for plugin command.',
        );
        $this->assertOutputNotContains(
            ' - abstract',
            'Abstract command classes should not appear.',
        );
        $this->assertOutputContains('<info>app</info>', 'app header should appear');
        $this->assertOutputContains('- sample', 'app shell');
        $this->assertOutputContains('<info>cakephp</info>', 'cakephp header should appear');
        $this->assertOutputContains('- routes', 'core shell');
        $this->assertOutputContains('- sample', 'short plugin name');
        $this->assertOutputContains('- abort', 'command object');
        $this->assertOutputContains('To run a command', 'more info present');
        $this->assertOutputContains('To get help', 'more info present');
        $this->assertOutputContains('This is a demo command', 'command description missing');
        $this->assertOutputContains('<info>custom_group</info>');
        $this->assertOutputContains('- grouped');
        $this->assertOutputNotContains(
            '- hidden',
            'Hidden commands should not appear in help output.',
        );
    }

    /**
     * Test filtering by command prefix (compact mode)
     */
    public function testFilterByPrefixCompact(): void
    {
        $this->exec('help cache');
        $this->assertExitCode(CommandInterface::CODE_SUCCESS);
        $this->assertOutputContains('<info>cache:</info>');
        $this->assertOutputContains('cache clear');
        $this->assertOutputContains('cache list');
        $this->assertOutputNotContains('routes');
        $this->assertOutputNotContains('sample');
    }

    /**
     * Test filtering by command prefix with verbose mode shows descriptions
     */
    public function testFilterByPrefixVerbose(): void
    {
        $this->exec('help cache -v');
        $this->assertExitCode(CommandInterface::CODE_SUCCESS);
        $this->assertOutputContains('Available Commands');
        $this->assertOutputContains('- cache clear');
        $this->assertOutputContains('Clear all data in a single cache engine');
        $this->assertOutputNotContains('- routes');
    }

    /**
     * Test help --xml
     */
    public function testMainAsXml(): void
    {
        $this->exec('help --xml');
        $this->assertExitCode(CommandInterface::CODE_SUCCESS);
        $this->assertOutputContains('<shells>');

        $find = '<shell name="sample" call_as="sample" provider="TestApp\Command\SampleCommand" help="sample -h"';
        $this->assertOutputContains($find);

        $find = '<shell name="schema_cache build" call_as="schema_cache build" ' .
            'provider="Cake\Command\SchemacacheBuildCommand" help="schema_cache build -h"';
        $this->assertOutputContains($find);

        $find = '<shell name="test_plugin.sample" call_as="test_plugin.sample" provider="TestPlugin\Command\SampleCommand" help="test_plugin.sample -h"';
        $this->assertOutputContains($find);

        $this->assertOutputNotContains(
            'HiddenCommand',
            'Hidden commands should not appear in XML output.',
        );
    }

    /**
     * Test that hidden commands are still executable
     */
    public function testHiddenCommandStillExecutable(): void
    {
        $this->exec('hidden');
        $this->assertExitCode(CommandInterface::CODE_SUCCESS);
        $this->assertOutputContains('Hidden Command Executed!');
    }
}
