<?php

declare (strict_types=1);
/*
 * This file is part of PHP CS Fixer.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *     Dariusz Rumiński <dariusz.ruminski@gmail.com>
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */
namespace PhpCsFixer\Fixer\Phpdoc;

use PhpCsFixer\AbstractFixer;
use PhpCsFixer\Fixer\ConfigurableFixerInterface;
use PhpCsFixer\Fixer\ConfigurableFixerTrait;
use PhpCsFixer\Fixer\DocBlockAnnotationTrait;
use PhpCsFixer\Fixer\WhitespacesAwareFixerInterface;
use PhpCsFixer\FixerConfiguration\FixerConfigurationResolver;
use PhpCsFixer\FixerConfiguration\FixerConfigurationResolverInterface;
use PhpCsFixer\FixerConfiguration\FixerOptionBuilder;
use PhpCsFixer\FixerDefinition\CodeSample;
use PhpCsFixer\FixerDefinition\FixerDefinition;
use PhpCsFixer\FixerDefinition\FixerDefinitionInterface;
use PhpCsFixer\Preg;
use PhpCsFixer\Tokenizer\Analyzer\Analysis\NamespaceUseAnalysis;
use PhpCsFixer\Tokenizer\Analyzer\AttributeAnalyzer;
use PhpCsFixer\Tokenizer\Analyzer\FullyQualifiedNameAnalyzer;
use PhpCsFixer\Tokenizer\CT;
use PhpCsFixer\Tokenizer\FCT;
use PhpCsFixer\Tokenizer\Token;
use PhpCsFixer\Tokenizer\Tokens;
/**
 * @phpstan-type _AutogeneratedInputConfiguration array{
 *  description?: string,
 *  fix_attribute?: bool,
 *  fix_internal?: bool,
 * }
 * @phpstan-type _AutogeneratedComputedConfiguration array{
 *  description: string,
 *  fix_attribute: bool,
 *  fix_internal: bool,
 * }
 *
 * @implements ConfigurableFixerInterface<_AutogeneratedInputConfiguration, _AutogeneratedComputedConfiguration>
 *
 * @no-named-arguments Parameter names are not covered by the backward compatibility promise.
 */
final class PhpdocTagNoNamedArgumentsFixer extends AbstractFixer implements ConfigurableFixerInterface, WhitespacesAwareFixerInterface
{
    /** @use ConfigurableFixerTrait<_AutogeneratedInputConfiguration, _AutogeneratedComputedConfiguration> */
    use ConfigurableFixerTrait;
    use DocBlockAnnotationTrait;
    public function getDefinition() : FixerDefinitionInterface
    {
        $code = <<<'PHP'
<?php

namespace ECSPrefix202510;

class Foo
{
    public function bar(string $s)
    {
    }
}
\class_alias('ECSPrefix202510\\Foo', 'Foo', \false);

PHP;
        return new FixerDefinition('There must be `@no-named-arguments` tag in PHPDoc of a class/enum/interface/trait.', [new CodeSample($code), new CodeSample($code, ['description' => 'Parameter names are not covered by the backward compatibility promise.'])]);
    }
    public function createConfigurationDefinition() : FixerConfigurationResolverInterface
    {
        return new FixerConfigurationResolver([(new FixerOptionBuilder('description', 'Description of the tag.'))->setAllowedTypes(['string'])->setDefault('')->getOption(), (new FixerOptionBuilder('fix_attribute', 'Whether to fix attribute classes.'))->setAllowedTypes(['bool'])->setDefault(\true)->getOption(), (new FixerOptionBuilder('fix_internal', 'Whether to fix internal elements (marked with `@internal`).'))->setAllowedTypes(['bool'])->setDefault(\true)->getOption()]);
    }
    /**
     * {@inheritdoc}
     *
     * Must run before PhpdocAlignFixer.
     * Must run after AlignMultilineCommentFixer, CommentToPhpdocFixer, PhpdocIndentFixer, PhpdocScalarFixer, PhpdocToCommentFixer, PhpdocTypesFixer.
     */
    public function getPriority() : int
    {
        return 0;
    }
    public function isCandidate(Tokens $tokens) : bool
    {
        return $tokens->isAnyTokenKindsFound(Token::getClassyTokenKinds());
    }
    protected function applyFix(\SplFileInfo $file, Tokens $tokens) : void
    {
        for ($index = $tokens->count() - 1; $index > 0; --$index) {
            if (!$tokens[$index]->isClassy()) {
                continue;
            }
            $prevIndex = $tokens->getPrevMeaningfulToken($index);
            \assert(\is_int($prevIndex));
            if ($tokens[$prevIndex]->isGivenKind(\T_NEW)) {
                continue;
            }
            if (!$this->configuration['fix_attribute'] && self::isAttributeClass($tokens, $prevIndex)) {
                continue;
            }
            $this->ensureIsDocBlockWithAnnotation($tokens, $index, 'no-named-arguments', $this->configuration['fix_internal'] ? ['no-named-arguments'] : ['internal', 'no-named-arguments'], []);
            $docBlockIndex = $tokens->getPrevTokenOfKind($index + 2, [[\T_DOC_COMMENT]]);
            \assert(\is_int($docBlockIndex));
            $content = $tokens[$docBlockIndex]->getContent();
            $newContent = Preg::replace('/@no-named-arguments.*\\R/', \rtrim('@no-named-arguments ' . $this->configuration['description']) . $this->whitespacesConfig->getLineEnding(), $content);
            if ($newContent !== $content) {
                $tokens[$docBlockIndex] = new Token([\T_DOC_COMMENT, $newContent]);
            }
        }
    }
    private static function isAttributeClass(Tokens $tokens, int $index) : bool
    {
        while ($tokens[$index]->isGivenKind([\T_FINAL, FCT::T_READONLY])) {
            $index = $tokens->getPrevMeaningfulToken($index);
            \assert(\is_int($index));
        }
        if (!$tokens[$index]->isGivenKind(CT::T_ATTRIBUTE_CLOSE)) {
            return \false;
        }
        $fullyQualifiedNameAnalyzer = new FullyQualifiedNameAnalyzer($tokens);
        foreach (AttributeAnalyzer::collect($tokens, $tokens->findBlockStart(Tokens::BLOCK_TYPE_ATTRIBUTE, $index)) as $attributeAnalysis) {
            foreach ($attributeAnalysis->getAttributes() as $attribute) {
                $attributeName = \strtolower($fullyQualifiedNameAnalyzer->getFullyQualifiedName($attribute['name'], $attribute['start'], NamespaceUseAnalysis::TYPE_CLASS));
                if ('attribute' === $attributeName) {
                    return \true;
                }
            }
        }
        return \false;
    }
}
