<?php
/**
 * @copyright 2008 - https://www.clicshopping.org
 * @Brand : ClicShoppingAI(TM) at Inpi all right Reserved
 * @Licence GPL 2 & MIT
 * @Info : https://www.clicshopping.org/forum/trademark/
 */

namespace ClicShopping\AI\Domain\Search;

use ClicShopping\OM\Registry;
use ClicShopping\OM\Hash;
use ClicShopping\OM\HTTP;

use ClicShopping\AI\Security\InputValidator;
use ClicShopping\AI\Security\SecurityLogger;
use ClicShopping\Apps\Configuration\Administrators\Classes\ClicShoppingAdmin\AdministratorAdmin;
use DateTimeImmutable;

/**
 * WebSearchLogger Class
 *
 * Gère la sauvegarde des requêtes de recherche web et de leurs résultats.
 * Sépare les données web_search des données RAG internes (table gpt).
 */
class WebSearchLogger
{
  private $db;
  private SecurityLogger $logger;
  private bool $debug;

  /**
   * Constructor
   */
  public function __construct()
  {
    $this->db = Registry::get('Db');
    $this->logger = new SecurityLogger();
    $this->debug = defined('CLICSHOPPING_APP_CHATGPT_RA_DEBUG_RAG_MANAGER') && CLICSHOPPING_APP_CHATGPT_RA_DEBUG_RAG_MANAGER === 'True';
  }

  /**
   * 🎯 Sauvegarde une requête de recherche web complète
   *
   * @param string $query Requête originale
   * @param string $translatedQuery Requête traduite en anglais
   * @param array $intentData Données d'intention (de Semantics::classifySearchIntent)
   * @param array $searchResults Résultats de WebSearchTool
   * @param string $responseSummary Synthèse générée par le LLM
   * @param array $metadata Métadonnées additionnelles
   * @return int|false ID de la requête sauvegardée ou false si échec
   */
  public function saveWebSearchRequest(
    string $query,
    string $translatedQuery,
    array $intentData,
    array $searchResults,
    string $responseSummary,
    array $metadata = []
  ): int|false {
    try {
      // Validation des données
      $validatedQuery = InputValidator::validateParameter(
        $query,
        'string',
        '',
        ['maxLength' => 2048, 'escape' => true]
      );

      $validatedTranslated = InputValidator::validateParameter(
        $translatedQuery,
        'string',
        '',
        ['maxLength' => 2048, 'escape' => true]
      );

      $validatedSummary = InputValidator::validateParameter(
        $responseSummary,
        'string',
        '',
        ['maxLength' => 8192, 'escape' => true]
      );

      $validatedUserAdmin = InputValidator::validateParameter(
        AdministratorAdmin::getUserAdmin(),
        'string',
        'system',
        [
          'maxLength' => 255,
          'pattern' => '/^[a-zA-Z0-9_\-\.\s]+$/'
        ]
      );

      // Calcul de la qualité
      $qualityScore = $this->calculateQualityScore($searchResults);

      // Audit trail
      $timestamp = (new DateTimeImmutable())->format('Y-m-d H:i:s');
      $auditPayload = [
        'session' => [
          'id' => session_id(),
          'ip' => HTTP::getIpAddress() ?? null,
          'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null,
        ],
        'intent' => $intentData,
        'search_metadata' => $searchResults['metadata'] ?? [],
        'execution_time' => $metadata['execution_time'] ?? 0,
        'api_cost' => $metadata['api_cost'] ?? 0,
        'cached' => $metadata['cached'] ?? false,
        'hash' => Hash::encryptDatatext($validatedUserAdmin . session_id() . $timestamp),
      ];

      // Préparer les données
      $array_sql = [
        'query' => $validatedQuery,
        'translated_query' => $validatedTranslated,
        'intent' => $intentData['intent'] ?? 'EXTERNAL_WEB',
        'intent_confidence' => $intentData['confidence'] ?? 0.0,
        'search_engine' => $searchResults['metadata']['search_engine'] ?? 'serpapi',
        'results_count' => count($searchResults['items'] ?? []),
        'response_summary' => $validatedSummary,
        'quality_score' => $qualityScore,
        'cached' => $metadata['cached'] ?? false ? 1 : 0,
        'cached_in_rag' => $metadata['cached_in_rag'] ?? false ? 1 : 0,
        'execution_time' => $metadata['execution_time'] ?? 0.0,
        'api_cost' => $metadata['api_cost'] ?? 0.0,
        'user_admin' => $validatedUserAdmin,
        'session_id' => session_id(),
        'ip_address' => HTTP::getIpAddress(),
        'date_added' => 'now()',
        'audit_data' => json_encode($auditPayload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
      ];

      // Insertion dans rag_web_search_requests
      $this->db->save('rag_web_search_requests', $array_sql);
      $requestId = $this->db->lastInsertId();

      if ($this->debug) {
        $this->logger->logSecurityEvent(
          "Web search request saved (ID: {$requestId})",
          'info'
        );
      }

      // Sauvegarder les résultats individuels
      if (!empty($searchResults['items'])) {
        $this->saveWebSearchResults($requestId, $searchResults['items']);
      }

      // Mettre à jour les statistiques
      $this->updateDailyStats($metadata);

      return $requestId;

    } catch (\Exception $e) {
      $this->logger->logSecurityEvent(
        "Error saving web search request: " . $e->getMessage(),
        'error'
      );

      // Ne pas propager l'erreur pour ne pas bloquer la réponse
      error_log("WebSearchLogger error: " . $e->getMessage());

      return false;
    }
  }

  /**
   * Sauvegarde les résultats individuels (items)
   *
   * @param int $requestId ID de la requête parente
   * @param array $items Items de résultats
   * @return int Nombre d'items sauvegardés
   */
  private function saveWebSearchResults(int $requestId, array $items): int
  {
    $savedCount = 0;

    try {
      foreach ($items as $item) {
        $array_sql = [
          'search_request_id' => $requestId,
          'position' => $item['position'] ?? 0,
          'title' => substr($item['title'] ?? '', 0, 500), // Limite
          'link' => substr($item['link'] ?? '', 0, 500),
          'snippet' => substr($item['snippet'] ?? '', 0, 1000),
          'source_domain' => $item['source'] ?? 'unknown',
          'relevance_score' => $item['relevance_score'] ?? 0.0,
          'date_added' => 'now()',
        ];

        $this->db->save('rag_web_search_results', $array_sql);
        $savedCount++;
      }

      if ($this->debug) {
        $this->logger->logSecurityEvent(
          "Saved {$savedCount} web search result items",
          'info'
        );
      }

    } catch (\Exception $e) {
      $this->logger->logSecurityEvent(
        "Error saving web search results: " . $e->getMessage(),
        'error'
      );
    }

    return $savedCount;
  }

  /**
   * Met à jour les statistiques quotidiennes
   *
   * @param array $metadata Métadonnées de la requête
   */
  private function updateDailyStats(array $metadata): void
  {
    try {
      $today = date('Y-m-d');

      // Récupérer les stats existantes
      $sql = "SELECT * FROM :table_rag_web_search_stats WHERE date = :date";
      $stmt = $this->db->prepare($sql);
      $stmt->bindValue(':date', $today);
      $stmt->execute();
      $stats = $stmt->fetch();

      if ($stats) {
        // Mettre à jour
        $sql = "UPDATE :table_rag_web_search_stats 
                SET total_searches = total_searches + 1,
                    cache_hits = cache_hits + :cached,
                    api_calls = api_calls + :api_call,
                    total_api_cost = total_api_cost + :cost,
                    avg_response_time = (avg_response_time * (total_searches - 1) + :exec_time) / total_searches
                WHERE date = :date";

        $stmt = $this->db->prepare($sql);
        $stmt->execute([
          'cached' => $metadata['cached'] ?? false ? 1 : 0,
          'api_call' => $metadata['cached'] ?? false ? 0 : 1,
          'cost' => $metadata['api_cost'] ?? 0.0,
          'exec_time' => $metadata['execution_time'] ?? 0.0,
          'date' => $today,
        ]);

      } else {
        // Créer nouvelle entrée
        $array_sql = [
          'date' => $today,
          'total_searches' => 1,
          'cache_hits' => $metadata['cached'] ?? false ? 1 : 0,
          'api_calls' => $metadata['cached'] ?? false ? 0 : 1,
          'failed_searches' => 0,
          'avg_response_time' => $metadata['execution_time'] ?? 0.0,
          'total_api_cost' => $metadata['api_cost'] ?? 0.0,
          'unique_queries' => 1,
          'price_comparisons' => 0,
        ];

        $this->db->save('rag_web_search_stats', $array_sql);
      }

    } catch (\Exception $e) {
      // Log mais ne pas bloquer
      if ($this->debug) {
        $this->logger->logSecurityEvent(
          "Error updating daily stats: " . $e->getMessage(),
          'warning'
        );
      }
    }
  }

  /**
   * Calcule le score de qualité des résultats
   *
   * @param array $results Résultats de recherche
   * @return float Score (0-1)
   */
  private function calculateQualityScore(array $results): float
  {
    $score = 0.5;

    // Featured snippet
    if (isset($results['featured_snippet']) && !empty($results['featured_snippet']['answer'])) {
      $score += 0.2;
    }

    // Résultats pertinents
    $relevantCount = 0;
    foreach ($results['items'] ?? [] as $item) {
      if (($item['relevance_score'] ?? 0) > 0.7) {
        $relevantCount++;
      }
    }
    $score += min($relevantCount * 0.1, 0.3);

    return min($score, 1.0);
  }

  /**
   * Supprime les anciennes entrées selon une politique de rétention
   *
   * @param int $daysToKeep Nombre de jours à conserver
   * @return int Nombre d'entrées supprimées
   */
  public function cleanOldSearches(int $daysToKeep = 90): int
  {
    try {
      // Supprimer les résultats individuels d'abord (pas de FK)
      $sql = "DELETE wsr FROM :table_rag_web_search_results wsr
              INNER JOIN :table_rag_web_search_requests wsreq ON wsr.search_request_id = wsreq.id
              WHERE wsreq.date_added < DATE_SUB(NOW(), INTERVAL :days DAY)
                AND wsreq.cached_in_rag = 0"; // Ne pas supprimer si stocké dans RAG

      $stmt = $this->db->prepare($sql);
      $stmt->bindInt('days', $daysToKeep);
      $stmt->execute();
      $deletedResults = $stmt->rowCount();

      // Supprimer les requêtes principales
      $sql = "DELETE FROM :table_rag_web_search_requests 
              WHERE date_added < DATE_SUB(NOW(), INTERVAL :days DAY)
                AND cached_in_rag = 0";

      $stmt = $this->db->prepare($sql);
      $stmt->bindInt('days', $daysToKeep);
      $stmt->execute();
      $deletedRequests = $stmt->rowCount();

      if ($this->debug) {
        $this->logger->logSecurityEvent(
          "Cleaned {$deletedRequests} old web search requests and {$deletedResults} results",
          'info'
        );
      }

      return $deletedRequests;

    } catch (\Exception $e) {
      $this->logger->logSecurityEvent(
        "Error cleaning old searches: " . $e->getMessage(),
        'error'
      );
      return 0;
    }
  }

  /**
   * Obtient les statistiques globales
   *
   * @param int $days Nombre de jours à analyser
   * @return array Statistiques
   */
  public function getStats(int $days = 30): array
  {
    try {
      $sql = "SELECT 
                COUNT(*) as total_searches,
                SUM(cached) as cached_hits,
                SUM(CASE WHEN cached = 0 THEN 1 ELSE 0 END) as api_calls,
                AVG(execution_time) as avg_execution_time,
                AVG(quality_score) as avg_quality_score,
                SUM(api_cost) as total_api_cost,
                SUM(cached_in_rag) as rag_stored_count
              FROM :table_rag_web_search_requests
              WHERE date_added >= DATE_SUB(NOW(), INTERVAL :days DAY)";

      $stmt = $this->db->prepare($sql);
      $stmt->execute(['days' => $days]);
      $stats = $stmt->fetch();

      $totalSearches = (int)$stats['total_searches'];
      $cachedHits = (int)$stats['cached_hits'];
      $cacheHitRate = $totalSearches > 0 ? ($cachedHits / $totalSearches) * 100 : 0;

      return [
        'period_days' => $days,
        'total_searches' => $totalSearches,
        'cached_hits' => $cachedHits,
        'api_calls' => (int)$stats['api_calls'],
        'cache_hit_rate' => round($cacheHitRate, 2) . '%',
        'avg_execution_time' => round((float)$stats['avg_execution_time'], 3) . 's',
        'avg_quality_score' => round((float)$stats['avg_quality_score'], 2),
        'total_api_cost' => round((float)$stats['total_api_cost'], 4) . ' USD',
        'rag_stored_count' => (int)$stats['rag_stored_count'],
      ];

    } catch (\Exception $e) {
      $this->logger->logSecurityEvent(
        "Error getting stats: " . $e->getMessage(),
        'error'
      );

      return [];
    }
  }
}