<?php
/**
 *
 * @copyright 2008 - https://www.clicshopping.org
 * @Brand : ClicShoppingAI(TM) at Inpi all right Reserved
 * @Licence GPL 2 & MIT
 * @Info : https://www.clicshopping.org/forum/trademark/
 *
 */

namespace ClicShopping\AI\Helper\Formatter\SubResultFormatters;

use ClicShopping\OM\Hash;
use ClicShopping\OM\Registry;
use ClicShopping\AI\Security\LlmGuardrails;
use ClicShopping\Apps\Configuration\ChatGpt\Classes\ClicShoppingAdmin\Gpt;

/**
 * AnalyticsFormatter - Formats analytics query results
 */
class AnalyticsFormatter extends AbstractFormatter
{
  /**
   * @var \ClicShopping\OM\Language Language instance for translations
   */
  private $language;
  
  /**
   * @var string Current language code
   */
  private string $languageCode;
  
  /**
   * Constructor
   * 
   * @param bool $debug Enable debug mode
   * @param bool $displaySql Display SQL queries
   */
  public function __construct(bool $debug = false, bool $displaySql = false)
  {
    parent::__construct($debug, $displaySql);
    
    // Initialize language
    $this->language = Registry::get('Language');
    $this->languageCode = $this->language->get('code');
    $this->language->loadDefinitions('rag_formatters', $this->languageCode, null, 'ClicShoppingAdmin');
  }
  
  public function canHandle(array $results): bool
  {
    $type = $results['type'] ?? '';
    return in_array($type, ['analytics_results', 'analytics_response']);
  }

  public function format(array $results): array
  {
    $question = $results['question'] ?? $results['query'] ?? 'Unknown request';

    // DEBUG: Log what we receive
    if ($this->debug) {
      error_log("=== ANALYTICS FORMATTER DEBUG ===");
      error_log("Results keys: " . implode(', ', array_keys($results)));
      error_log("Has 'results': " . (isset($results['results']) ? 'YES' : 'NO'));
      error_log("Has 'data_results': " . (isset($results['data_results']) ? 'YES' : 'NO'));
      
      $dataRows = $results['results'] ?? $results['data_results'] ?? [];
      if (!empty($dataRows)) {
        error_log("Data rows is array: " . (is_array($dataRows) ? 'YES' : 'NO'));
        error_log("Data rows count: " . count($dataRows));
        if (is_array($dataRows) && !empty($dataRows)) {
          error_log("First row keys: " . implode(', ', array_keys($dataRows[0])));
        }
      } else {
        error_log("No data rows found");
      }
    }

    // TASK 6.2: Check if this is a multi-query result
    if (isset($results['multiple_results']) && is_array($results['multiple_results'])) {
      return $this->formatMultipleResults($results);
    }

    $output = "<div class='analytics-results'>";
    $output .= "<h4>" . $this->language->getDef('text_rag_analytics_results_for') . " " . htmlspecialchars($question) . "</h4>";

    // Display source attribution
    if (isset($results['source_attribution'])) {
      $output .= $this->formatSourceAttribution($results['source_attribution']);
    }

    // SQL query display
    if ($this->displaySql && isset($results['sql_query'])) {
      $output .= $this->formatSqlQuery($results['sql_query']);
    }

    // Interpretation
    $interpretationText = '';
    $isHtmlContent = false;
    
    if (isset($results['text_response']) && !empty($results['text_response'])) {
      $interpretationText = $results['text_response'];
      // Check if text_response contains HTML
      $isHtmlContent = (strpos($interpretationText, '<div') !== false || strpos($interpretationText, '<p>') !== false);
    } elseif (isset($results['interpretation']) && $results['interpretation'] !== 'Array') {
      $interpretationText = $results['interpretation'];
    }

    if (!empty($interpretationText)) {
      // ✅ TASK 5.3.2.1: Don't double-encode HTML content from text_response
      if ($isHtmlContent) {
        // text_response already contains formatted HTML - use as-is
        $output .= "<div class='interpretation'>" . $interpretationText . "</div>";
      } else {
        // Plain text - apply HTML encoding
        $output .= "<div class='interpretation'><strong>" . $this->language->getDef('text_rag_analytics_interpretation') . "</strong> " 
                . Hash::displayDecryptedDataText($interpretationText) . "</div>";
      }
    }

    // Guardrails
    $output .= "<div class='mt-2'></div>";
    $lmGuardrails = LlmGuardrails::checkGuardrails($question, Hash::displayDecryptedDataText($interpretationText));

    if (is_array($lmGuardrails)) {
      $output .= $this->formatGuardrailsMetrics($lmGuardrails);
    } else {
      $output .= "<div class='alert alert-warning'>" . htmlspecialchars($lmGuardrails) . "</div>";
    }

    $output .= "<div class='mt-2'></div>";

    // Data table - Support both 'results' and 'data_results' keys
    $dataRows = $results['results'] ?? $results['data_results'] ?? [];
    
    if (is_array($dataRows) && !empty($dataRows)) {
      $output .= $this->formatDataTable($dataRows);
    } else {
      $output .= "<div class='alert alert-info'>";
      $output .= "<strong>" . $this->language->getDef('text_rag_analytics_note') . "</strong> " . $this->language->getDef('text_rag_analytics_detailed_data_available');
      $output .= "</div>";
    }

    $output .= "</div>";

    // Save audit data
    $auditExtra = [
      'embeddings_context' => $results['embeddings_context'] ?? [],
      'similarity_scores'  => $results['similarity_scores'] ?? [],
      'processing_chain'   => $results['processing_chain'] ?? []
    ];
    Gpt::saveData($question, $output, $auditExtra);

    return [
      'type' => 'formatted_results',
      'content' => $output
    ];
  }

  /**
   * Format multiple query results with clear separation
   * 
   * TASK 6.2: Display results for multiple queries with clear labels
   * Each sub-query gets its own section, even if one fails
   *
   * @param array $results Results containing multiple_results array
   * @return array Formatted output
   */
  private function formatMultipleResults(array $results): array
  {
    $originalQuery = $results['question'] ?? $results['query'] ?? 'Unknown request';
    $multipleResults = $results['multiple_results'] ?? [];
    $queryCount = count($multipleResults);

    $output = "<div class='analytics-results multiple-queries'>";
    $output .= "<h4>" . $this->language->getDef('text_rag_analytics_results_for') . " " . htmlspecialchars($originalQuery) . "</h4>";
    $output .= "<div class='alert alert-info'>";
    $output .= "<strong>" . $this->language->getDef('text_rag_analytics_note') . "</strong> " . str_replace('{count}', $queryCount, $this->language->getDef('text_rag_analytics_note_multiple_queries'));
    $output .= "</div>";

    // Display source attribution
    if (isset($results['source_attribution'])) {
      $output .= $this->formatSourceAttribution($results['source_attribution']);
    }

    // Process each sub-query result
    foreach ($multipleResults as $index => $subResult) {
      $subQueryNum = $index + 1;
      $subQuery = $subResult['query'] ?? $this->language->getDef('text_rag_analytics_sub_query') . " {$subQueryNum}";
      
      $output .= "<div class='sub-query-result' style='margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 5px;'>";
      $output .= "<h5 style='color: #0066cc;'>📊 " . $this->language->getDef('text_rag_analytics_query') . " {$subQueryNum} : " . htmlspecialchars($subQuery) . "</h5>";

      // Check if this sub-query failed
      if (isset($subResult['error']) || (isset($subResult['success']) && $subResult['success'] === false)) {
        $errorMsg = $subResult['error'] ?? $this->language->getDef('text_rag_analytics_unknown_error');
        $output .= "<div class='alert alert-warning'>";
        $output .= "<strong>⚠️ " . $this->language->getDef('text_rag_analytics_error') . "</strong> " . htmlspecialchars($errorMsg);
        $output .= "</div>";
        $output .= "</div>"; // Close sub-query-result
        continue;
      }

      // SQL query display
      if ($this->displaySql && isset($subResult['sql'])) {
        $output .= $this->formatSqlQuery($subResult['sql']);
      }

      // Interpretation
      if (isset($subResult['interpretation']) && !empty($subResult['interpretation'])) {
        $output .= "<div class='interpretation'><strong>" . $this->language->getDef('text_rag_analytics_interpretation') . "</strong> " 
                . Hash::displayDecryptedDataText($subResult['interpretation']) . "</div>";
      }

      // Data table
      if (isset($subResult['rows']) && is_array($subResult['rows']) && !empty($subResult['rows'])) {
        $output .= "<div class='results-table'>";
        $output .= "<h6>" . $this->language->getDef('text_rag_analytics_data') . "</h6>";
        $output .= $this->generateTable($subResult['rows'], 'table table-bordered table-striped');
        $output .= "<div class='result-count'><em>" . $this->language->getDef('text_rag_analytics_result_count') . " " . $subResult['row_count'] . "</em></div>";
        $output .= "</div>";
      } else {
        $output .= "<div class='alert alert-info'>";
        $output .= "<strong>" . $this->language->getDef('text_rag_analytics_note') . "</strong> " . $this->language->getDef('text_rag_analytics_no_data_found');
        $output .= "</div>";
      }

      $output .= "</div>"; // Close sub-query-result
    }

    // Display all SQL queries if requested
    if ($this->displaySql && isset($results['sql_queries']) && is_array($results['sql_queries'])) {
      $output .= "<div class='all-sql-queries' style='margin-top: 20px;'>";
      $output .= "<h5>" . $this->language->getDef('text_rag_analytics_all_sql_queries') . "</h5>";
      foreach ($results['sql_queries'] as $index => $sql) {
        $output .= "<div class='sql-query-item'>";
        $output .= "<strong>" . $this->language->getDef('text_rag_analytics_query') . " " . ($index + 1) . " :</strong>";
        $output .= $this->formatSqlQuery($sql);
        $output .= "</div>";
      }
      $output .= "</div>";
    }

    $output .= "</div>"; // Close analytics-results

    // Save audit data
    $auditExtra = [
      'multiple_queries' => true,
      'query_count' => $queryCount,
      'embeddings_context' => $results['embeddings_context'] ?? [],
      'similarity_scores'  => $results['similarity_scores'] ?? [],
      'processing_chain'   => $results['processing_chain'] ?? []
    ];
    Gpt::saveData($originalQuery, $output, $auditExtra);

    return [
      'type' => 'formatted_results',
      'content' => $output
    ];
  }

  private function formatSqlQuery(string $sql): string
  {
    $formatted = $this->prettySql($sql);
    $escaped = htmlspecialchars($formatted, ENT_NOQUOTES | ENT_SUBSTITUTE, 'UTF-8');

    return "<div class='col-md-12 row sql-query'>
            <strong>" . $this->language->getDef('text_rag_analytics_sql_query') . "</strong>
            <pre>{$escaped}</pre>
          </div>";
  }

  private function formatDataTable(array $data): string
  {
    if (empty($data)) {
      return '';
    }

    $output = "<div class='results-table'>";
    $output .= "<h5>" . $this->language->getDef('text_rag_analytics_data') . "</h5>";
    
    // Use inherited method from AbstractFormatter
    $output .= $this->generateTable($data, 'table table-bordered table-striped');
    
    $output .= "</div>";

    return $output;
  }

  private function formatGuardrailsMetrics(array $guardrails): string
  {
    // Implementation similar to original ResultFormatter
    $output = "<div class='guardrails-metrics'>";
    // Add guardrails display logic here
    $output .= "</div>";
    return $output;
  }
}
