<?php

declare(strict_types=1);

namespace Drupal\Tests\Core\Database;

use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Event\DatabaseEvent;
use Drupal\Core\Database\Event\StatementEvent;
use Drupal\Core\Database\Event\StatementExecutionEndEvent;
use Drupal\Core\Database\Event\StatementExecutionFailureEvent;
use Drupal\Core\Database\Event\StatementExecutionStartEvent;
use Drupal\Core\Database\Exception\EventException;
use Drupal\Tests\Core\Database\Stub\StubConnection;
use Drupal\Tests\Core\Database\Stub\StubPDO;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests Drupal\Core\Database\Connection.
 */
#[CoversClass(Connection::class)]
#[Group('Database')]
class DatabaseEventsTest extends UnitTestCase {

  /**
   * A database connection.
   */
  protected Connection $connection;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->connection = new StubConnection($this->createMock(StubPDO::class), []);
  }

  /**
   * Tests event enabling and disabling.
   *
   * @legacy-covers ::isEventEnabled
   * @legacy-covers ::enableEvents
   * @legacy-covers ::disableEvents
   */
  public function testEventEnablingAndDisabling(): void {
    $this->connection->enableEvents(StatementEvent::all());
    $this->assertTrue($this->connection->isEventEnabled(StatementExecutionStartEvent::class));
    $this->assertTrue($this->connection->isEventEnabled(StatementExecutionEndEvent::class));
    $this->assertTrue($this->connection->isEventEnabled(StatementExecutionFailureEvent::class));
    $this->connection->disableEvents([
      StatementExecutionEndEvent::class,
    ]);
    $this->assertTrue($this->connection->isEventEnabled(StatementExecutionStartEvent::class));
    $this->assertFalse($this->connection->isEventEnabled(StatementExecutionEndEvent::class));
    $this->assertTrue($this->connection->isEventEnabled(StatementExecutionFailureEvent::class));
    $this->connection->disableEvents(StatementEvent::all());
    $this->assertFalse($this->connection->isEventEnabled(StatementExecutionStartEvent::class));
    $this->assertFalse($this->connection->isEventEnabled(StatementExecutionEndEvent::class));
    $this->assertFalse($this->connection->isEventEnabled(StatementExecutionFailureEvent::class));
  }

  /**
   * Tests enable invalid event.
   *
   * @legacy-covers ::enableEvents
   */
  public function testEnableInvalidEvent(): void {
    $this->expectException(\AssertionError::class);
    $this->expectExceptionMessage('Event class foo does not exist');
    $this->connection->enableEvents(['foo']);
  }

  /**
   * Tests disable invalid event.
   *
   * @legacy-covers ::disableEvents
   */
  public function testDisableInvalidEvent(): void {
    $this->expectException(\AssertionError::class);
    $this->expectExceptionMessage('Event class bar does not exist');
    $this->connection->disableEvents(['bar']);
  }

  /**
   * Tests event dispatching when no container available.
   *
   * @legacy-covers ::dispatchEvent
   */
  public function testEventDispatchingWhenNoContainerAvailable(): void {
    $this->expectException(EventException::class);
    $this->expectExceptionMessage('The event dispatcher service is not available. Database API events can only be fired if the container is initialized');
    $this->connection->dispatchEvent($this->createMock(DatabaseEvent::class));
  }

}
