<?php

namespace Elgg\Plugins;

/**
 * Test the functionality of the group tool based container logic
 *
 * @since 3.3
 */
abstract class GroupToolContainerLogicIntegrationTestCase extends IntegrationTestCase {

	/**
	 * @var \ElggGroup
	 */
	protected $group;
	
	/**
	 * @var \ElggUser
	 */
	protected $owner;
	
	/**
	 * {@inheritDoc}
	 * @see \Elgg\BaseTestCase::up()
	 */
	public function up() {
		$this->owner = $this->createUser();
		elgg()->session_manager->setLoggedInUser($this->owner);
		
		$this->group = $this->createGroup();
	}
	
	/**
	 * Get the type of the content being gated behind the group tool option
	 *
	 * @return string
	 */
	abstract public function getContentType(): string;
	
	/**
	 * Get the subtype of the content being gated behind the group tool option
	 *
	 * @return string
	 */
	abstract public function getContentSubtype(): string;
	
	/**
	 * Get the name of the group tool option
	 *
	 * @return string
	 */
	abstract public function getGroupToolOption(): string;
	
	public function testGroupOwnerCanWriteContentWithToolEnabled() {
		
		$this->assertTrue($this->group->enableTool($this->getGroupToolOption()));
		
		$this->assertTrue($this->group->canWriteToContainer($this->owner->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testGroupOwnerCantWriteContentWithToolDisabled() {
		$this->assertTrue($this->group->disableTool($this->getGroupToolOption()));
		
		$this->assertFalse($this->group->canWriteToContainer($this->owner->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testUserCantWriteContentWithToolEnabled() {
		$user = $this->createUser();
		
		$this->assertTrue($this->group->enableTool($this->getGroupToolOption()));
		
		$this->assertFalse($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testUserCantWriteContentWithToolDisabled() {
		$user = $this->createUser();
		
		$this->assertTrue($this->group->disableTool($this->getGroupToolOption()));
		
		$this->assertFalse($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testGroupMemberCanWriteContentWithToolEnabled() {
		$user = $this->createUser();
		$this->group->join($user);
		
		$this->assertTrue($this->group->enableTool($this->getGroupToolOption()));
		
		$this->assertTrue($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testGroupMemberCantWriteContentWithToolDisabled() {
		$user = $this->createUser();
		$this->group->join($user);
		
		$this->assertTrue($this->group->disableTool($this->getGroupToolOption()));
		
		$this->assertFalse($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testAdminCanWriteContentWithToolEnabled() {
		$user = $this->getAdmin();
		
		$this->assertTrue($this->group->enableTool($this->getGroupToolOption()));
		
		$this->assertTrue($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testAdminCantWriteContentWithToolDisabled() {
		$user = $this->getAdmin();
		
		$this->assertTrue($this->group->disableTool($this->getGroupToolOption()));
		
		$this->assertFalse($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
	
	public function testAdminMemberCantWriteContentWithToolDisabled() {
		$user = $this->getAdmin();
		$this->group->join($user);
		
		$this->assertTrue($this->group->disableTool($this->getGroupToolOption()));
		
		$this->assertFalse($this->group->canWriteToContainer($user->guid, $this->getContentType(), $this->getContentSubtype()));
	}
}
