<?php
/*
Gibbon: the flexible, open school platform
Founded by Ross Parker at ICHK Secondary. Built by Ross Parker, Sandra Kuipers and the Gibbon community (https://gibbonedu.org/about/)
Copyright © 2010, Gibbon Foundation
Gibbon™, Gibbon Education Ltd. (Hong Kong)

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

namespace Gibbon\Tables\Renderer;

use Gibbon\Domain\DataSet;
use Gibbon\Tables\DataTable;
use Gibbon\Forms\Layout\Element;
use Gibbon\Tables\Columns\ActionColumn;
use Gibbon\Tables\Columns\ExpandableColumn;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

/**
 * SpreadsheetRenderer
 *
 * @version v16
 * @since   v16
 */
class SpreadsheetRenderer implements RendererInterface
{
    protected $path;
    protected $excel;
    protected $rowCount;
    protected $sheet;

    protected $headersAdded = false;

    public function __construct()
    {
        // Create new PHPSpreadsheet object
        $this->excel = new Spreadsheet();

        // Header row count
        $this->rowCount = 1;

        // Create Sheet
        $this->excel->setActiveSheetIndex(0) ;
        $this->sheet = $this->excel->getActiveSheet();
    }

    /**
     * Render the table to a spreadsheet file: xlsx, xls, csv
     *
     * @param DataTable $table
     * @param DataSet $dataSet
     * @return string
     */
    public function renderTable(DataTable $table, DataSet $dataSet)
    {
        // Create styles
        $headerStyle = [
            'fill' => [
                'type' => Fill::FILL_SOLID,
                'color' => ['rgb' => 'eeeeee'],
            ],
            'borders' => [
                'allborders' => ['style' => Border::BORDER_THIN, 'color' => ['argb' => '444444'], ],
            ],
            'font' => [
                'size' => 12,
                'bold' => true,
            ],
            'alignment' => [
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
        ];
        $rowStyle = [
            'font' => [
                'size' => 12,
            ],
        ];

        $currentStyle = ['fill' => [ 'type' => Fill::FILL_SOLID, 'color' => ['rgb' => 'B3EFC2'],]];
        $errorStyle = ['fill' => [ 'type' => Fill::FILL_SOLID, 'color' => ['rgb' => 'F6CECB'],]];
        $warningStyle = ['fill' => [ 'type' => Fill::FILL_SOLID, 'color' => ['rgb' => 'FFD2A9'],]];
        $rowStripeStyle = ['fill' => [ 'type' => Fill::FILL_SOLID, 'color' => ['rgb' => 'FAFAFA'],]];

        // Set document properties
        $this->excel->getProperties()->setCreator($table->getMetaData('creator'))
            ->setLastModifiedBy($table->getMetaData('creator'))
            ->setTitle($table->getMetaData('name', $table->getTitle()))
            ->setDescription(__('This information is confidential. Generated by Gibbon (https://gibbonedu.org).')) ;

        if ($dataSet->count() == 0 && $table->getMetaData('tableCount') <= 1) {
            $this->sheet->setCellValue('A1', __('Your query has returned 0 rows.', 'Query Builder'));
            $this->sheet->getStyle('A1')->applyFromArray($rowStyle);
        } else {
            $totalColumnDepth = $table->getTotalColumnDepth();

            if (!$this->headersAdded) { 
                for ($i = 0; $i < $totalColumnDepth; $i++) {
                    $this->sheet->getRowDimension($this->rowCount)->setRowHeight(24);

                    $cellCount = 0;
                    foreach ($table->getColumns($i) as $columnName => $column) {
                        if ($column instanceof ActionColumn || $column instanceof ExpandableColumn) continue;
                        
                        if ($column->getDepth() < $i) {
                            $cellCount++;
                            continue;
                        }

                        $alpha = $this->num2alpha($cellCount);
                        $range = $alpha.$this->rowCount;

                        $this->sheet->setCellValue($alpha.$this->rowCount, $column->getLabel());

                        $colSpan = $column->getTotalSpan();
                        if ($colSpan > 1) {
                            $range = $alpha.$this->rowCount.':'.$this->num2alpha($cellCount+$colSpan-1).$this->rowCount;
                            $this->sheet->mergeCells($range);
                        }

                        $rowspan = ($column->getTotalDepth() > 1) ? 1 : ($totalColumnDepth - $column->getDepth());
                        if ($rowspan > 1) {
                            $this->sheet->calculateColumnWidths();
                            $this->sheet->getColumnDimension($alpha)->setAutoSize(false);

                            $range = $alpha.$this->rowCount.':'.$alpha.($this->rowCount+$rowspan-1);
                            $this->sheet->mergeCells($range);
                        }

                        if ($column->getWidth() == 'auto') {
                            $this->sheet->getColumnDimension($alpha)->setAutoSize(true);
                        } else {
                            $this->sheet->getColumnDimension($alpha)->setWidth(intval($column->getWidth()));
                        }

                        $this->sheet->getStyle($range)->applyFromArray($headerStyle);

                        $cellCount += $colSpan;

                        // Handle details as a separate spreadsheet column
                        if ($column->hasDetailsFormatter()) {
                            $alpha = $this->num2alpha($cellCount);
                            $this->sheet->setCellValue($alpha.$this->rowCount, $column->getDescription());
                            $this->sheet->getStyle($alpha.$this->rowCount)->applyFromArray($headerStyle);
                            $this->sheet->getColumnDimension($alpha)->setAutoSize(true);

                            $cellCount++;
                        }
                    }
                    $this->rowCount++;
                }
                $this->headersAdded = true;
            }

            // TABLE ROWS
            foreach ($dataSet as $data) {
                $this->sheet->getRowDimension($this->rowCount)->setRowHeight(-1);

                $row = $this->createTableRow($data, $table);

                // CELLS
                $cellCount = 0;
                foreach ($table->getColumns() as $columnName => $column) {
                    if ($column instanceof ActionColumn || $column instanceof ExpandableColumn) continue;

                    $alpha = $this->num2alpha($cellCount);

                    $cellContent = $this->stripTags($column->getOutput($data, false));

                    if (is_numeric($cellContent) && strpos($cellContent, ".") === false) {
                        $this->sheet->setCellValueExplicit( $alpha.$this->rowCount, $cellContent, DataType::TYPE_NUMERIC);
                    } else {
                        $this->sheet->setCellValueExplicit( $alpha.$this->rowCount, $cellContent, DataType::TYPE_STRING);
                    }
                    $this->sheet->getStyle($alpha.$this->rowCount)->applyFromArray($rowStyle);

                    $cellStyle = null;
                    if (stripos($row->getClass(), 'error') !== false) $cellStyle = $errorStyle;
                    else if (stripos($row->getClass(), 'warning') !== false) $cellStyle = $warningStyle;
                    else if (stripos($row->getClass(), 'current') !== false) $cellStyle = $currentStyle;
                    else if ($this->rowCount % 2 != 0) $cellStyle = $rowStripeStyle;

                    if (!empty($cellStyle)) $this->sheet->getStyle($alpha.$this->rowCount)->applyFromArray($cellStyle);

                    $cellCount++;

                    // Handle details as a separate spreadsheet column
                    if ($column->hasDetailsFormatter()) {
                        $alpha = $this->num2alpha($cellCount);
                        $cellContent = $this->stripTags($column->getDetailsOutput($data));

                        $this->sheet->setCellValueExplicit( $alpha.$this->rowCount, $cellContent, DataType::TYPE_STRING);
                        $this->sheet->getStyle($alpha.$this->rowCount)->applyFromArray($rowStyle);

                        $cellCount++;
                    }
                }
                $this->rowCount++;
            }
        }

        if ($table->getMetaData('tableCount') <= 1) {
            $this->save($table);
        }
        
        return;
    }

    protected function createTableRow(array $data, DataTable $table)
    {
        $row = new Element();

        foreach ($table->getRowModifiers() as $callable) {
            $row = $callable($data, $row, $table->getColumnCount());
        }

        return $row;
    }

    protected function stripTags($content)
    {
        // Collapses consecutive spaces, removes newlines, then converts <br/> to newlines.
        $content = preg_replace(['/\s\s+/', '/[\r\n]+/', '/\<br(\s*)?\/?\>/i'], ['', '', PHP_EOL], $content);
        return strip_tags($content);
    }

    protected function num2alpha($n)
    {
        for ($r = ''; $n >= 0; $n = intval($n / 26) - 1) {
            $r = chr($n % 26 + 0x41).$r;
        }

        return $r;
    }

    public function save(DataTable $table) 
    {
        $filename = $table->getMetaData('filename');
        $exportFileType = $table->getMetaData('filetype');
        if (empty($exportFileType)) $exportFileType = 'Excel2007';

        // FINALIZE THE DOCUMENT SO IT IS READY FOR DOWNLOAD
        // Set active sheet index to the first sheet, so Excel opens this as the first sheet
        $this->excel->setActiveSheetIndex(0);

        switch($exportFileType) {
            case 'Excel2007':
                $filename .= '.xlsx';
                $mimetype = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
                $objWriter = IOFactory::createWriter($this->excel, 'Xlsx');
                break;
            case 'Excel5':
                $filename .= '.xls';
                $mimetype = 'application/vnd.ms-excel';
                $objWriter = IOFactory::createWriter($this->excel, 'Xls');
                break;
            case 'OpenDocument':
                $filename .= '.ods';
                $mimetype = 'application/vnd.oasis.opendocument.spreadsheet';
                $objWriter = IOFactory::createWriter($this->excel, 'Ods');
                break;
            case 'CSV':
                $filename .= '.csv';
                $mimetype = 'text/csv';
                $objWriter = IOFactory::createWriter($this->excel, 'Csv');
                break;
        }

        // Fix mimetype so that spreadsheet files can be viewed on iOS devices
        if(stristr($_SERVER['HTTP_USER_AGENT'], 'ipad') !== false or stristr($_SERVER['HTTP_USER_AGENT'], 'iphone') !== false or stristr($_SERVER['HTTP_USER_AGENT'], 'ipod') !== false) {
            $mimetype = 'application/octet-stream';
            header('Content-type: application/octet-stream');
        }

        // Redirect output to a client’s web browser (Excel2007)
        header('Content-Type: '.$mimetype);
        header('Content-Disposition: attachment;filename="'.$filename.'"');
        header('Cache-Control: max-age=1');

        // If you're serving to IE over SSL, then the following may be needed
        header ('Expires: Mon, 26 Jul 1997 05:00:00 GMT'); // Date in the past
        header ('Last-Modified: '.gmdate('D, d M Y H:i:s').' GMT'); // always modified
        header ('Cache-Control: cache, must-revalidate'); // HTTP/1.1
        header ('Pragma: public'); // HTTP/1.0

        $objWriter->save('php://output');
    }
}
