/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { ContainerField, ContainerFieldValue } from "./ContainerField.js";
import { Config, Listener, ObservableListenerOpts } from "../Observable.js";
import { Component } from "../Component.js";
import { FieldEventMap } from "./Field.js";
export type FormHandler<ValueType extends ContainerFieldValue = ContainerFieldValue> = ((form: Form<ValueType>) => any | Promise<any>) | undefined;
export interface FormEventMap<Type, ValueType extends ContainerFieldValue = ContainerFieldValue> extends FieldEventMap<Type> {
    /**
     * Fires when the form is valid and submitted. The event is fired after calling the handler.
     *
     * @param form
     */
    submit: (form: Type, handlerResponse: any) => any;
    /**
     * Not fired by the framework. But comes in handy when you extend this form and add a cancel button
     *
     * @param form
     */
    cancel: (form: Type) => any;
}
export interface Form<ValueType extends ContainerFieldValue = ContainerFieldValue> extends ContainerField<ValueType> {
    on<K extends keyof FormEventMap<this, ValueType>, L extends Listener>(eventName: K, listener: Partial<FormEventMap<this, ValueType>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof FormEventMap<this, ValueType>>(eventName: K, listener: Partial<FormEventMap<this, ValueType>>[K]): boolean;
    fire<K extends keyof FormEventMap<this, ValueType>>(eventName: K, ...args: Parameters<FormEventMap<any, ValueType>[K]>): boolean;
    get el(): HTMLFormElement;
}
/**
 * Form component
 *
 * Forms can be used to submit or present data.
 *
 * @example Password validation
 *
 * ```
 * 	textfield({
 * 		type: "password",
 * 		label: "Password",
 * 		name: "password",
 * 		listeners: {
 * 			validate: (field) => {
 * 				const form = field.findAncestorByInstanceType(Form)!;
 * 				if(field.getValue() != form.findField("confirm")!.getValue()) {
 * 					field.setInvalid("The passwords don't match");
 * 				}
 * 			}
 * 		},
 * 	}),
 *
 * 	textfield({
 * 		itemId: "confirm",//item ID used instead of name so this field won't be submitted
 * 		type: "password",
 * 		label: "Confirm password"
 * 	}),
 *
 * ```
 *
 */
export declare class Form<ValueType extends ContainerFieldValue = ContainerFieldValue> extends ContainerField<ValueType> {
    /**
     * When this is set to true, the field will use the values set as their original value, used for resetting and
     * determining if the field was modified.
     */
    static TRACK_RESET_VALUES: boolean;
    protected baseCls: string;
    hideLabel: boolean;
    constructor();
    /**
     * Executed when form is submitted.
     *
     * If a promise is returned the "submit" event will fire after it has been resolved.
     *
     * @param form
     */
    handler: FormHandler<ValueType>;
    protected internalRender(): HTMLElement;
    /**
     * @inheritDoc
     */
    reset(): void;
    set value(v: Partial<ValueType>);
    get value(): ValueType;
    /**
     * Get the modified field values since the form was:
     *
     * - rendered OR
     * - value was set (usually through a load) OR
     * - submitted
     */
    get modified(): Partial<ValueType>;
    /**
     * Validates the form and submits it using the handler function passed with the config.
     */
    submit(): Promise<boolean>;
    setInvalid(msg: string): void;
}
export type FormConfig<ValueType extends ContainerFieldValue = ContainerFieldValue> = Config<Form<ValueType>, FormEventMap<Form<ValueType>>> & {
    /**
     * Executed when form is submitted.
     *
     * If a promise is returned the "submit" event will fire after it has been resolved.
     *
     * @param form
     */
    handler?: FormHandler<ValueType>;
};
/**
 * Shorthand function to create {@see Form}
 *
 * @param config
 * @param items
 */
export declare const form: <ValueType extends ContainerFieldValue = ContainerFieldValue>(config?: FormConfig<ValueType>, ...items: Component[]) => Form<ValueType>;
//# sourceMappingURL=Form.d.ts.map