<?php

namespace go\modules\business\multisafepay;

use go\core;
use go\core\exception\Forbidden;
use go\core\http\Request;
use go\core\model\Link;
use go\core\orm\exception\SaveException;
use go\core\orm\Mapping;
use go\modules\business\business\model\Business;
use go\modules\business\finance\model\Debtor;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\FinanceBusiness;
use go\modules\business\finance\model\FinanceDocumentPayment;
use go\modules\business\finance\model\FinanceDocumentStatus;
use go\modules\business\finance\model\Payment;
use go\modules\business\finance\model\PaymentProviderInterface;
use go\modules\community\addressbook\model\Contact;
use go\modules\community\addressbook\model\EmailAddress;
use MultiSafepayAPI\Client;


/**
 *
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author System Administrat <admin@intermesh.localhost>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module implements PaymentProviderInterface
{
	use core\event\EventEmitterTrait;

	/**
	 * @var Client
	 */
	private $msp;

	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }

	public function getDependencies() : array
	{
		return ['business/finance'];
	}

	public function requiredLicense(): ?string
	{
		return "billing";
	}


	public function getSettings()
	{
	 return model\Settings::get();
	}


	private function getMsp() : Client {
		if(!isset($this->msp)) {
			//autoload composer libs for this module
			require_once __DIR__ . '/vendor/autoload.php';

			$this->msp = new Client;

			$apiUrl = Module::get()->getSettings()->test ?
				'https://testapi.multisafepay.com/v1/json/' :
				'https://api.multisafepay.com/v1/json/';

			$this->msp->setApiKey(Module::get()->getSettings()->apiKey);
			$this->msp->setApiUrl($apiUrl);
		}

		return $this->msp;
	}


	/**
	 * @throws Forbidden
	 */
	private function getDocument(int $documentId, string $token) : FinanceDocument {
		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}

		return $document;
	}

	public function pagePayment(int $documentId, string $token) {

		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new core\exception\Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}

		$org = Contact::findById($document->getOrganizationId());

		$emailAdr = $org->findEmailByType(EmailAddress::TYPE_BILLING, true);

		if(!$emailAdr) {
			$email = go()->getAuthState()->getUser(['email'])->email;
		} else{
			$email = $emailAdr->email;
		}

		try {
			$order = [
				"type" => "paymentlink",
				"order_id" => uniqid($document->id),
				"currency" => "EUR",
				"amount" => $document->getTotalPrice() * 100,
				"description" => $document->number,
				"gateway" => "",
				"payment_options" => array(
					"notification_url" => go()->getAuthState()->getPageUrl() .	"/business/multisafepay/webhook/". $document->id . "/" . $document->token,
					"redirect_url" => go()->getAuthState()->getPageUrl() .	"/business/multisafepay/return/". $document->id . "/" . $document->token,
					"cancel_url" =>  go()->getAuthState()->getPageUrl() .	"/business/multisafepay/cancel/". $document->id . "/" . $document->token,
					"close_window" => "true"
				),
				"customer" => array(
					"locale" => $org->language,
					"ip_address" => Request::get()->getRemoteIpAddress(),

					"first_name" => $document->getOrganization(),
					"last_name" => "",
					"address1" => "",
					"address2" => "",
					"zip_code" => "",
					"city" => "",
					"state" => "",
					"country" => "",
					"phone" => "",
					"email" => $email
				),

				"plugin" => array(
					"shop" => "Group-Office groupware and CRM",
					"shop_version" => go()->getMajorVersion(),
					"plugin_version" => go()->getMajorVersion(),
					"partner" => "Intermesh",
					"shop_root_url" => "https://www.group-office.com",
				)
			];
//echo "<pre>";
//			var_dump($order);
//			exit();

			$this->getMsp()->orders->post($order);

			$url = $this->getMsp()->orders->getPaymentLink();

			//Make sure this page is not cached by the browser
			header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
			header("Cache-Control: post-check=0, pre-check=0", false);
			header("Pragma: no-cache");
			header("Location: " . $url);
			exit();
		} catch (Exception $e) {
			echo "Error " . htmlspecialchars($e->getMessage());
		}

	
	}

	public function pageReturn(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);
		$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();
		try {
			$this->pageWebhook($documentId, $token);
		}catch(\Exception $e) {
			$error = $e->getMessage();
			core\ErrorHandler::logException($e);
		}

		require(__DIR__ . '/views/web/return.php');
	}

	/**
	 * @throws SaveException
	 */
	public function pageWebhook(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);

		if(!$document->isPaid()) {

			$status = $this->getMsp()->orders->get($endpoint = 'orders', $_GET['transactionid'], $body = array(), $query_string = false);

			go()->debug("MSP Webhook " . $documentId);
			go()->debug($status);

			if ($status->status == 'completed') {
				$payment = (new Payment())->setValues([
					'businessId' => $document->findBook()->businessId,
					'documentId' => $document->id,
					'organizationId' => $document->getOrganizationId(),
					'reference' => 'multisafepay-' . $status->transaction_id,
					'date' => new \DateTime(),
					'amount' => $status->amount / 100,
					'description' => "MultiSafePay transaction: " . $status->transaction_id
				]);

				if (!$payment->save()) {
					throw new SaveException($payment);
				}
			}
		}
	}

	public function pageCancel(int $documentId, string $token) {
		$document = $this->getDocument($documentId, $token);
		$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();

		require(__DIR__ . '/views/web/cancel.php');
	}


	public function getPaymentProviderTitle(): string
	{
		return go()->t("Pay online via MultiSafePay");
	}
}
