<?php /** @noinspection PhpUnused */

namespace go\modules\business\support;

use Exception;
use Faker\Generator;
use go\core;
use go\core\fs\Blob;
use go\core\model;
use go\core\model\User;
use go\core\orm\Mapping;
use go\core\orm\Property;
use go\core\TemplateParser;
use GO\Email\Model\Account;
use go\modules\business\support\cron\AutoExpire;
use go\modules\business\support\cron\ImapImport;
use go\modules\business\support\model\Settings;
use go\modules\business\support\model\SupportList;
use go\modules\business\support\model\SupportTicket;
use go\modules\business\support\model\UserSettings;
use go\modules\community\comments\model\Comment;
use go\modules\community\tasks\model\Task;
use go\modules\community\tasks\model\TaskList;

/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author System Administrat <admin@intermesh.localhost>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module
{
	public function getAuthor(): string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

	public function getDependencies(): array
	{
		return ['community/tasks', 'community/addressbook', 'community/comments'];
	}

	public function autoInstall(): bool
	{
		return true;
	}

	public function requiredLicense(): ?string
	{
		return "groupoffice-pro";
	}

	/**
	 * The development status of this module
	 * @return string
	 */
	public function getStatus(): string
	{
		return self::STATUS_STABLE;
	}

	protected function beforeUninstall(): bool
	{

		SupportList::delete(["role" => TaskList::Support]);


		return parent::beforeUninstall();
	}

	/**
	 * @throws Exception
	 */
	protected function afterInstall(model\Module $model): bool
	{
		ImapImport::install('*/5 * * * *');
		AutoExpire::install('0 2 * * *');

		$taskList = new TaskList();
		$taskList->name = go()->t('Support', 'business', 'support');
		$taskList->setRole('support');
		$taskList->save();

		return parent::afterInstall($model);
	}

	public function defineListeners()
	{
		Comment::on(Comment::EVENT_BEFORE_SAVE, static::class, 'onCommentBeforeSave');
		User::on(Property::EVENT_MAPPING, static::class, 'onUserMap');
	}

	/**
	 * @throws Exception
	 */
	public static function onUserMap(Mapping $mapping)
	{
		$mapping->addHasOne('supportSettings', UserSettings::class, ['id' => 'userId'], true);
	}


	public static $sendMailOnComment = true;

	/**
	 * Besides this hook also consider {@see Task::onCommentAdded()}
	 * @throws Exception
	 */
	public static function onCommentBeforeSave(Comment $comment)
	{
		// if comment has mimeMessageId set it was imported from an e-mail
		if ($comment->section == "private" || $comment->mimeMessageId || !self::$sendMailOnComment || $comment->entityTypeId != SupportTicket::entityType()->getId()) {
			return;
		}

		/** @var SupportTicket $supportTicket */
		$supportTicket = $comment->findEntity();

		if ($supportTicket->createdBy == go()->getUserId()) {
			// only send email from admin to customer
			return;
		}

		$accountId = self::get()->getSettings()->outgoingEmailAccountId;

		if (!$accountId) {
			$accountId = go()->getDbConnection()
				->selectSingleValue('accountId')
				->from('business_support_email_account')
				->where('tasklistId', '=', $supportTicket->tasklistId)
				->single();
		}

		if (!$accountId) {
			$accountId = go()->getDbConnection()
				->selectSingleValue('accountId')
				->from('business_support_email_account')
				->single();
		}

		if (!$accountId) {
			if (model\Alert::$enabled) {
				$supportTicket->createAlert(new core\util\DateTime())
					->setData([
						'title' => "Error sending comment email",
						'body' => "No account configured for sending support e-mail. Go to System Settings -> Support"
					])
					->save();
			}
			return;
		}

		$account = Account::model()->findByPk($accountId, false, true);
		$user = User::findById($supportTicket->createdBy, ['id', 'displayName', 'email', 'timezone', 'dateFormat', 'timeFormat']);

		$commentCreator = go()->getAuthState()->getUser(['id', 'displayName', 'email']);

		$template = self::get()->getSettings()->getEmailTemplate();

		$templateParser = new TemplateParser();
		$templateParser->addFilter('fullThread', function ($supportTicket) {
			return self::_commentsToHtmlThread($supportTicket);
		});
		$templateParser->addModel("url", $supportTicket->url());
		$templateParser->addModel("ticket", $supportTicket);
		$templateParser->addModel("ticketCreator", $user);
		$templateParser->addModel('comment', $comment);
		$templateParser->addModel('commentCreator', $commentCreator);

		try {
			$message = $template->toMessage($templateParser);

			$message->getMailer()->setEmailAccount($account);

			$alias = $account->getDefaultAlias();
			$message->setFrom($alias->email, $alias->name);

			$comment->mimeMessageId = $message->getId();

			$refs = array_reverse(
				Comment::findFor($supportTicket)
					->selectSingleValue('mimeMessageId')
					->where('mimeMessageId', '!=', null)
					->all());

			if (!empty($refs)) {
				$message->setInReplyTo($refs[0]);
				$message->setReferences(...$refs);
			}

			foreach ($comment->attachments as $a) {
				$message->addBlob(Blob::findById($a->blobId), $a->name);
			}

			$body = $message->getBody();

			// inline images
			$blobIds = Blob::parseFromHtml($body);
			foreach ($blobIds as $blobId) {
				$blob = Blob::findById($blobId);

				if ($blob) {
					$img = core\mail\Attachment::fromBlob($blob);
					$contentId = $message->embed($img);
					$body = Blob::replaceSrcInHtml($body, $blobId, $contentId);
				}
			}

			if (!empty($supportTicket->cc)) {
				$cc = new core\mail\AddressList($supportTicket->cc);
				$message->addCc(...$cc->toArray());
			}

			$message
				->setBody($body, 'text/html')
				->setTo(new core\mail\Address($user->email, $user->displayName))
//				->setBcc(go()->getSettings()->systemEmail)
				->send();

			//The ID changes after send in Swift!. We set it here again to avoid confusion.
			$message->setId($comment->mimeMessageId);

		} catch (Exception $e) {
			if (model\Alert::$enabled) {
				$supportTicket->createAlert(new core\util\DateTime())
					->setData([
						'title' => "Error sending comment email",
						'body' => $e->getMessage()
					])
					->save();
			}
		}

	}

	/**
	 * @param SupportTicket $ticket
	 * @return string HTML formatted thread
	 * @throws Exception
	 */
	private static function _commentsToHtmlThread(SupportTicket $ticket): string
	{
		$comments = Comment::findFor($ticket)
			->where('section', '=', null);

		$rows = [];
		foreach ($comments as $comment) {
			$creator = model\UserDisplay::findById($comment->createdBy);
			$date = new core\util\DateTime($comment->date);

			$row = '<tr><th style="border-top:1px solid #ccc; padding-top:10px;">' . $creator->displayName . '</th><th style="border-top:1px solid #ccc;text-align:right"><small>' . $date->toUserFormat(true) . '</small></th></tr>' .
				'<tr><td colspan="2">' . $comment->text . '</td></tr>';

			if ($comment->attachments) {
				$row .= '<tr><td style="padding-top:10px;">';
				foreach ($comment->attachments as $att) {
					$row .= '<div style="border:1px solid #aaa; border-radius:4px; padding:3px; float:left">' . $att->name . '</div>';
				}
				$row .= '</td></tr>';
			}
			$rows[] = $row;
		}
		return '<table id="__go-fullthread" style="width: 500px;">' . implode(array_reverse($rows)) . '</table><br id="__go-fullthread">';
	}

	protected function rights(): array
	{
		return [
			'mayChangeTasklists', // allows Tasklist/set (hide ui elements that use this)
			'mayChangeCategories', // allows creating global  categories for everyone. Personal cats can always be created.
		];
	}


	public function getSettings()
	{
		return Settings::get();
	}

	public function demo(Generator $faker)
	{
		$tasklists = TaskList::find()->where(['role' => TaskList::Support]);

		static::$sendMailOnComment = false;

		$titles = [
			"I can't create tasks",
			"My iPhone won't connect",
			"The printer is not working",
			"The meeting room beamer is broken",
			"Fix issue with automatic problem solver",
			"How can I setup two factor authentication?",
			"How can I connect my phone?",
			"Setup MacOS",
			"Setup Windows",
			"I want a faster laptop",
			"I want a bigger screen",
			"I deleted all my important data! Can you restore it?",
			"Where is the upload button?",
			"Does Group-Office have super cow powers?",
			"There's no more coffee!",
			"Can you setup a chat account?",
			"Where can I setup video meetings?",
			"Oops, I deleted my documents",
			"How many backups do you make?",
		];

		foreach ($tasklists as $tasklist) {
			\go\modules\community\tasks\Module::get()->demoTasks($faker, $tasklist, true, $titles, 20);
		}
	}
}
