import {CommentEditor, commenteditor} from "./CommentEditor.js";
import {
	avatar,
	browser,
	btn,
	checkbox,
	checkboxselectcolumn,
	column,
	comp,
	Component,
	DataSourceStore,
	form,
	Form,
	mstbar,
	Notifier,
	paginator,
	router,
	searchbtn,
	splitter,
	t,
	Table,
	tbar
} from "@intermesh/goui";
import {authManager, client, FilterCondition, img, jmapds, User} from "@intermesh/groupoffice-core";
import {TaskTable} from "./TaskTable.js";
import {TasklistTable, tasklisttable} from "./TasklistTable.js";
import {CommentList} from "./CommentList.js";


export class MainPanel extends Component {
	private taskTable: TaskTable;


	private user?: User;
	private taskListTable!: TasklistTable;
	private west: Component;
	private center: Component;
	private east: Component;
	private commentsList: CommentList;
	private commentEditor: CommentEditor;
	private commentForm: Form;


	constructor() {
		super("section");

		this.id = "support";
		this.cls = "vbox fit";
		this.on("render", async () => {
			try {
				this.user = await authManager.requireLogin();
			} catch (e) {
				console.warn(e);
				Notifier.error(t("Login is required on this page"));
			}

			await this.taskListTable.store.load();
			await this.taskTable.store.load();
		});


		this.taskTable = new TaskTable();

		this.taskTable.on("navigate", async (table, rowIndex) => {

			await router.goto("supportclient/" + table.store.get(rowIndex)!.id);
		});


		this.west = this.createWest();

		this.commentEditor = commenteditor({
			flex: 1
		});

		this.items.add(

				comp({
						flex: 1, cls: "hbox mobile-cards"
					},

					this.west,

					splitter({
						stateId: "support-splitter-west",
						resizeComponentPredicate: this.west
					}),

					this.center = comp({
							cls: 'active vbox',
							itemId: 'table-container',
							width: 550,
							style: {
								minWidth: "365px", //for the resizer's boundaries
								maxWidth: "850px"
							}
						},

						tbar({},
							btn({
								cls: "for-small-device",
								title: t("Menu"),
								icon: "menu",
								handler: (button, ev) => {
									this.activatePanel(this.west);
								}
							}),

							'->',

							searchbtn({
								listeners: {
									input: (sender, text) => {

										(this.taskTable.store.queryParams.filter as FilterCondition).text = text;
										this.taskTable.store.load();

									}
								}
							}),

							mstbar({table: this.taskTable}),

							btn({
								itemId: "add",
								icon: "add",
								text: t("New request"),
								cls: "filled primary",
								handler: async () => {

									this.mask();

									// Dynamic import to delay loading of required modules
									const mods = await import("./SupportCreateWindow.js");
									const win = new mods.SupportCreateWindow;

									// const queryCallId =  JmapDataSource.store("Module").nextCallId;
									const queryResponse = await jmapds("Module").query({
										filter: {
											name: "supportclient",
											package: "business"
										}
									});

									const modules = await jmapds("Module").get(queryResponse.ids);

									const form = win.findChildByType(Form)!;

									form.value = {
										tasklistId: modules.list[0].settings.defaultTaskListId
									};

									this.unmask();

									win.show();
								}
							})
						),

						comp({
								flex: 1,
								stateId: "support",
								cls: "scroll border-top main"
							},
							this.taskTable
						),


						paginator({
							store: this.taskTable.store
						})
					),


					splitter({
						stateId: "support-splitter",
						resizeComponentPredicate: "table-container"
					}),

					this.east = comp({
							itemId: 'scroll-component',
							tabIndex: -1,
							flex: 1,
							cls: 'scroll'
						},
						tbar({},
							btn({
								itemId: "back",
								cls: "for-small-device",
								icon: "chevron_left",
								text: t("Back"),
								handler: () => {
									router.goto("supportclient");
								}
							})
						),

						this.commentsList = new CommentList(),


						this.commentForm = form({
								hidden: true,
								handler: (form) => {
									jmapds("Comment").create(
										Object.assign({
												entity: "SupportTicket",
												entityId: (this.commentsList.store.queryParams.filter as FilterCondition).entityId,
											}, form.value
										)
									).then(() => {
										form.reset();
									});
								}
							},

							comp({
									cls: "hbox pad"
								},
								comp({
									itemId: "avatar-container"
								}),

								this.commentEditor
							),


							tbar({},
								btn({
									type: "button",
									text: t("Attach files"),
									icon: "attach_file",
									handler: async () => {

										const files = await browser.pickLocalFiles(true);
										this.mask();
										const blobs = await client.uploadMultiple(files);
										this.unmask();


										const atts = this.commentEditor.attachments.value;
										for (let blob of blobs) {
											atts.push({
												name: blob.name,
												blobId: blob.id
											});
										}
										this.commentEditor.attachments.value = atts;

									}
								}),

								'->',

								btn({
									type: "submit",
									text: t("Send"),
									icon: "send"
								})
							)
						)
					)
				)
			)

	}

	private activatePanel(active:Component) {
		this.center.el.classList.remove("active");
		this.east.el.classList.remove("active");
		this.west.el.classList.remove("active");

		active.el.classList.add("active");
	}


	private createWest() {


		return comp({
				cls: "vbox scroll",
				width: 300
			},
			tbar({

				},
				comp({
					tagName: "h3",
					text: t("Help"),
					flex: 1
				}),
				'->',
				btn({
					cls: "for-small-device",
					title: t("Close"),
					icon: "close",
					handler: (button, ev) => {
						this.activatePanel(this.center);
					}
				})
			),
			tbar({},
				checkbox({
					type: "switch",
					label: t("Show completed"),
					listeners: {
						change: async (field, newValue, oldValue) => {
							if(newValue) {
								delete (this.taskTable.store.queryParams.filter as FilterCondition).complete;
							} else
							{
								(this.taskTable.store.queryParams.filter as FilterCondition).complete = false;
							}
							await this.taskTable.store.load();
						}
					}
				})
			),

			comp({tagName: "hr"}),

			this.taskListTable = tasklisttable({
				fitParent: true,
				cls: "no-row-lines",
				// headers: false,
				rowSelectionConfig: {
					multiSelect: true,
					listeners: {
						selectionchange: (tableRowSelect) => {

							const taskListIds = tableRowSelect.selected.map((index) => tableRowSelect.list.store.get(index)!.id);

							(this.taskTable.store.queryParams.filter as FilterCondition).taskListId = taskListIds;

							this.taskTable.store.load();
						}
					}
				},
				columns: [
					checkboxselectcolumn(),
					column({
						id: "name",
						header: t("Name"),
						sortable: true,
						resizable: false
					})
				]
			})
		);
	}

	async setTaskId(taskId?: string) {

		const table = this.findChildByType(Table<DataSourceStore>)!,
			form = this.findChildByType(Form)!,
			avatarContainer = this.findChild("avatar-container");

		try {

			if (!this.user) {
				this.user = await authManager.requireLogin();

				avatarContainer?.items.add(this.user.avatarId ?
					img({
						cls: "goui-avatar",
						blobId: this.user.avatarId,
						title: this.user.displayName
					}) :
					avatar({
						displayName: this.user.displayName!
					})
				);
			}

			if (taskId) {
				(this.commentsList.store.queryParams.filter as FilterCondition).entityId = taskId;
				await this.commentsList.store.load();

				const rowIndex = table.store.findIndex((r, index, records) => r.id == taskId);
				if (rowIndex > -1) {
					table.rowSelection!.selected = [rowIndex];
				}

				form.reset();
				form.show();

				this.activatePanel(this.east);

				//scroll down when images finish loading
				Promise.all(Array.from(document.images).filter(img => !img.complete).map(img => new Promise(resolve => { img.onload = img.onerror = resolve; }))).then(() => {
					this.commentForm.el.scrollIntoView({
						behavior: "smooth",
						block: "end"
					})
				});

			} else {
				this.commentsList.items.clear();
				form.hide();

				this.activatePanel(this.center);
			}
		} catch (e) {
			Notifier.error(e as Error);
		}
	}
}
