/** 
 * Copyright Intermesh
 * 
 * This file is part of Group-Office. You should have received a copy of the
 * Group-Office license along with Group-Office. See the file /LICENSE.TXT
 * 
 * If you have questions write an e-mail to info@intermesh.nl
 * 
 * @copyright Copyright Intermesh
 * @version $Id: ProjectDialog.js 23462 2018-03-06 11:37:47Z mschering $
 * @author Merijn Schering <mschering@intermesh.nl>
 */
GO.projects2.ProjectDialog = function(config){
	config = config || {};

	config.goDialogId='project';
	
	this.buildForm();
	
	var focusName = function(){
		this.nameField.focus();		
	};

	config.stateId='pm-project-dialog';
	config.collapsible=true;
	config.maximizable=true;
	config.layout='fit';
	config.modal=false;
	config.resizable=true;
	config.width=dp(920);
	config.minWidth = dp(800);
	config.height=dp(800);
	
	if(!config.closeAction) {
		config.closeAction='hide';
	}
	config.title= t("Project", "projects2");					
	config.items= this.formPanel;
	config.focus= focusName.createDelegate(this);
	config.bbar= [
		this.createLinkButton = new go.links.CreateLinkButton(),
		'->',
			this.applyBtn = new Ext.Button({
			text: t("Apply"),
			handler: function () {
				this.submitForm();
			},
			scope: this
		}),
		this.saveBtn = new Ext.Button({
			text: t("Save"),
			handler: function () {
				this.submitForm(true);
			},
			scope: this
		})
	];

	//Add a hidden submit button so the form will submit on enter
	this.formPanel.add(new Ext.Button({
		hidden: true,
		hideMode: "offsets",
		type: "submit",
		handler: function() {
			this.submitForm(true);
		},
		scope: this
	}));

	GO.projects2.ProjectDialog.superclass.constructor.call(this, config);

	this.addEvents({
		'save' : true,
		'hide': function() {
			this.createLinkButton.reset();
		}.bind(this)
	});
}

Ext.extend(GO.projects2.ProjectDialog, GO.Window,{

	permissionsPanel: null,
	
	/**
	 * If the projects needs to duplicate another
	 * set this field to the id to duplicate from
	 */
	duplicate_id: false,

	init : function(){
				
	},
	
	setLinkEntity : function(config) {		
		this.on('load', function() {
			if(config.entity === 'Contact') {
				if (config.data.isOrganization) {
					this.customerField.setValue(config.data.id);
				} else {
					this.contactField.setValue(config.data.id);
					if (config.data.organizationIds.length) {
						this.customerField.setValue(config.data.organizationIds[0]);
					}
				}
			}
		}, this, {single: true});
	},
	
	
	show : function (config) {

		config = config || {};

		if(!this.rendered)
			this.render(Ext.getBody());

		if(!config.initialization) {
			//other wise link button will be cleared on second call
			this.setProjectId(config.project_id);
		}
		
		if(!this.inititalized) {

			GO.request({
				url: 'core/multiRequest',
				maskEl:this.getEl(),
				params:{
					requests:Ext.encode({
						types:{
							r:'projects2/type/store'
						},
						statuses:{
							r:'projects2/status/store',
							forEditing:true,
							limit: 400
						}
					})
				},
				success: function(options, response, result)
				{
					GO.projects2.typesStore.loadData(result.types);
					GO.projects2.statusesStore.loadData(result.statuses);

					this.inititalized=true;
                    
					if(GO.util.empty(this.selectType.getValue())) {
						this.selectType.selectFirst();
					}
					config.initialization = true;
					this.show(config);
				},
				scope:this
			});
			return false;
		}

		GO.projects2.statusesStore.reload(); // removes phanton records inserted by remoteComboText

		this.duplicate_id = config.duplicate_id || false;
		if(!config.project_id)
		{
			config.project_id=0;
			

			if(typeof(config.template_id) == "undefined")
			{
				if(!this.selectTemplateWindow){
					this.selectTemplateWindow =new GO.projects2.SelectTemplateWindow();
					this.selectTemplateWindow.projectDialog = this;
				}
				this.selectTemplateWindow.show(config);
				return false;
			} else {
				if(!GO.settings.modules.projects2.write_permission)
					this.formPanel.baseParams.template_id=config.template_id;

				config.values = config.values || {};
				config.values.template_id=config.template_id;
			}
		}

		this.formPanel.form.reset();
		this.selectType.setDisabled(!GO.settings.modules.projects2.write_permission);
		this.templateSelect.setDisabled(!GO.settings.modules.projects2.write_permission);

		this.parent_project_id=this.formPanel.baseParams.parent_project_id=config.parent_project_id;
		
		this.propertiesPanel.show();

		this.formPanel.load({
			url: GO.url('projects2/project/load'),
			params: {
				template_id: !config.project_id ? config.values.template_id : 0
			},
			success: function (form, action) {
				if (this.permissionsPanel) {
					this.permissionsPanel.setAcl(action.result.data[this.permissionsPanel.fieldName]);
				}
				this.setWritePermission(action.result.data.write_permission);

				if (config.project_id && config.project_id > 0) {
					this.formPanel.baseParams.company_id = action.result.data.company_id;
					this.formPanel.baseParams.contact_id = action.result.data.contact_id;
				}
				this.parent_project_id = this.formPanel.baseParams.parent_project_id = action.result.data.parent_project_id;

				this.selectType.setDisabled(action.result.data.permission_level < GO.permissionLevels.writeAndDelete);
				this.templateSelect.setDisabled(action.result.data.permission_level < GO.permissionLevels.writeAndDelete);

				this.applyTemplate(action.result.data.template);
				this.setTitle(this.title + ": " + action.result.data.name);

				const tbarItems = this.timeEntryGrid.getTopToolbar().items;
				if (action.result.data.completed) {
					tbarItems.get('add').disable();
					tbarItems.get('delete').disable();
				} else {
					tbarItems.get('add').enable();
					tbarItems.get('delete').enable();
				}

				//applytemplate might have rendered new form fields
				var v = Ext.apply(action.result.data, config.values);
				this.setValues(v);

				var fields = action.result.data.template.fields.split(',');
				if (fields.indexOf('status') > -1) {
					//if(!go.util.empty(action.result.remoteComboTexts.status_id)) {
					this.statusField.setRemoteText(action.result.remoteComboTexts.status_id);
					// simulated selecting status
					this.statusField.fireEvent('select', this.statusField, {json: {concept: action.result.data.concept}});
				}
				this.selectType.setRemoteText(action.result.remoteComboTexts.type_id);
				this.selectParentProject.setRemoteText(action.result.data.parent_project_name);

				//Remove the disabled customer field tabs

				GO.projects2.ProjectDialog.superclass.show.call(this);

				this.fireEvent('load', this);
			},
			failure: function (form, action) {
				GO.errorDialog.show(action.result.feedback)
			},
			scope: this

		});
	},
	
	setWritePermission : function(writePermission) {
		this.applyBtn.setDisabled(!writePermission);
		this.saveBtn.setDisabled(!writePermission);
	},
	
	setValues : function(values) {
		if(values) {
			for(var key in values) {
				var field = this.formPanel.form.findField(key);
				if(field) {
					field.setValue(values[key]);
				}
			}
		}
		
	},
	setProjectId : function(project_id) {

		const isNew = project_id == 0;
		this.formPanel.form.baseParams['id']=project_id;
		this.project_id=project_id;

		if(this.createType){
			this.createType.getEl().up('.x-form-item').setDisplayed(isNew);

			this.newTypeName.getEl().up('.x-form-item').setDisplayed(false);
			this.newTypeName.allowBlank=true;
		}

		if(this.feesPanel){
			this.feesPanel.setProjectId(project_id);
			if(go.util.empty(project_id)) {
				this.feesPanel.setDisabled(true);
			} else {
				this.feesPanel.setDisabled(false);
			}
		}
		
		this.expensesPanel.setProjectId(project_id);
		if (this.incomePanel) {
			this.incomePanel.setProjectId(project_id);
		}
		this.expenseBudgetsGrid.setProjectId(project_id);
	
		this.timeEntryGrid.setProjectId(project_id);
		this.createLinkButton.setEntity("Project", project_id);

		// Prevent earlier choices for contact / custemer field from being remembered in case of new project
		if(isNew) {
			this.contactField.setValue(null);
			this.customerField.setValue(null);
		}

		//button
		this.maxBudgetedUnitsField.items.get(1).setVisible(!isNew);
		this.maxBudgetedCostsField.items.get(1).setVisible(!isNew);
		//field
		this.maxBudgetedUnitsField.items.get(0).setReadOnly(!isNew);
		this.maxBudgetedCostsField.items.get(0).setReadOnly(!isNew)

	},
	
	submitForm : function(hide){
		var params={};
		if(this.expensesPanel.store.loaded){
			params['expenses']=Ext.encode(this.expensesPanel.getGridData(false));
		}
		
		
		if(this.selectType.disabled) {
			params.type_id=this.selectType.getValue();
		}
		
		if(this.templateSelect.disabled) {
			params.template_id=this.templateSelect.getValue();
		}

		var submitUrl = (!this.duplicate_id) ? 
			GO.url('projects2/project/submit') : 
			GO.url('projects2/project/duplicate');
		this.formPanel.form.submit(
		{
			url:submitUrl,
			params: params,
			waitMsg:t("Saving..."),
			success:function(form, action){
				if(action.result.parent_project_id) {
					this.parent_project_id = action.result.parent_project_id;
				}
				if (this.permissionsPanel && action.result[this.permissionsPanel.fieldName]) {
					this.permissionsPanel.setAcl(action.result[this.permissionsPanel.fieldName]);
				}
				if(action.result.id) {
					this.nameField.setValue(this.nameField.getValue().replace('{autoid}',action.result.id));

					this.setProjectId(action.result.id, true);
					this.createLinkButton.save();
				}

				if(action.result.type_id) {
					this.selectType.store.reload({
						callback:function(){
							this.selectType.setValue(action.result.type_id);
						},
						scope:this
					});
				}

				if(this.feesPanel) {
					this.feesPanel.store.commitChanges();
				}

				if (this.link_config && this.link_config.callback) {
					this.link_config.callback.call(this);
				}

				this.fireEvent('save', this, this.project_id, this.parent_project_id);	
				
				GO.dialog.TabbedFormDialog.prototype.refreshActiveDisplayPanels.call(this);
				
				if(hide) {
					this.hide();
				}
			},		
			failure: function(form, action) {
				if(action.failureType == 'client') {
					Ext.MessageBox.alert(t("Error"), t("You have errors in your form. The invalid fields are marked."));			
				} else {
					Ext.MessageBox.alert(t("Error"), action.result.feedback);
				}
				go.form.Dialog.prototype.showFirstInvalidField.call(this);
			},
			scope: this
		});		
	},

	applyTemplate : function(templateData) {
		var showCustomerField = true;
		var showContactField = true;
		var showResUserIdField = true;
		var showStatus=true;
		var showDate=true;
		var showBudgetFees=true;
		var showExpenses=true;
		var showDefaultDistance=false;
		var showTravelCosts=false;
		var showIncome=true;
		var showReferenceNo=true;


		if(!templateData){
			var record = GO.projects2.templatesStore.getById(this.templateSelect.getValue());
			if(record) {
				templateData = record.data;
			}
		}else {
			this.templateSelect.setRemoteText(templateData.name);
		}
		if(templateData)
			this.loadedTemplateID=templateData.id;


		if(this.templatePanels) {

			for(var i=0,count=this.templatePanels.length;i<count;i++) {
				//remove will destoy tab. Without destorying form elements are note
				//removed from the Ext.form.BasicForm which causes problems.
				this.tabPanel.remove(this.templatePanels[i], true);
			}
		}
			
		if(templateData) {
			var defaultStatusId = templateData.default_status_id;
			if(defaultStatusId>0 && go.util.empty(this.project_id)){
				this.formPanel.form.findField('status_id').setValue(defaultStatusId);
			}

			var defaultTypeId = templateData.default_type_id;
			if(defaultTypeId>0 && go.util.empty(this.project_id)){
				this.selectType.setValue(defaultTypeId);

				if(templateData.default_type_name) {
					this.selectType.setRemoteText(templateData.default_type_name);
				}
			}

			this.setTitle(templateData.name);
			var fieldString = templateData.fields;
			var fields = fieldString.split(',');
			showCustomerField=fields.indexOf('customer')>-1;
			showContactField=fields.indexOf('contact')>-1;
			showResUserIdField=fields.indexOf('responsible_user_id')>-1;
			showStatus=fields.indexOf('status')>-1;
			showDate=fields.indexOf('date')>-1;
			showBudgetFees=fields.indexOf('budget_fees')>-1 && GO.timeregistration2;
			showExpenses=fields.indexOf('expenses')>-1;
			showDefaultDistance=fields.indexOf('default_distance')>-1;
			showTravelCosts=fields.indexOf('travel_costs')>-1;
			showIncome=fields.indexOf('income')>-1;
			showReferenceNo=fields.indexOf('reference_no')>-1;

			if(GO.projects2.templates && GO.projects2.templates[templateData.id]) {
				//get the values because panels that are destroyed might contain fields
				//that will be added empty again.
				var orgValues = this.formPanel.form.getValues();					

				this.tabPanel.doLayout();

				this.formPanel.form.setValues(orgValues);
			}
			this.feesPanel.setTemplate(templateData);

		} else {
			this.setTitle(t("Project", "projects2"));
			this.tabPanel.items.each(function(p){
				if(p.category_id)
					this.tabPanel.unhideTabStripItem(p.id);
			}, this);

		}
			
		if(this.customerField) {
			this.customerField.getEl().up('.x-form-item').setDisplayed(showCustomerField);
		}
		if(this.contactField) {
			this.contactField.getEl().up('.x-form-item').setDisplayed(showContactField);
		}
		this.selectUser.getEl().up('.x-form-item').setDisplayed(showResUserIdField);

		if(this.feesPanel){
			if(showBudgetFees){
				this.tabPanel.unhideTabStripItem(this.timeEntryGrid);
				this.tabPanel.unhideTabStripItem(this.feesPanel);
				this.maxBudgetedUnitsField.setVisible(true);
				this.maxBudgetedUnitsField.setDisabled(false);
			} else {
				this.tabPanel.hideTabStripItem(this.timeEntryGrid);
				this.tabPanel.hideTabStripItem(this.feesPanel);
				this.maxBudgetedUnitsField.setVisible(false);
				this.maxBudgetedUnitsField.setDisabled(true);
			}
		}

		if(showExpenses){
			this.tabPanel.unhideTabStripItem(this.expenseBudgetsGrid);
			this.tabPanel.unhideTabStripItem(this.expensesPanel);
		} else {
			this.tabPanel.hideTabStripItem(this.expenseBudgetsGrid);
			this.tabPanel.hideTabStripItem(this.expensesPanel);
		}
			
		if (showIncome) {
			this.tabPanel.unhideTabStripItem(this.incomePanel);
			this.incomeField.setVisible(true);
			this.incomeField.setDisabled(false);
		} else {
			this.tabPanel.hideTabStripItem(this.incomePanel);
			this.incomeField.setVisible(false);
			this.incomeField.setDisabled(true);
		}
			
		this.statusField.setVisible(showStatus);
		this.dateFields.setVisible(showDate);
		
		this.mileageDistance.setVisible(showDefaultDistance);
		this.mileageDistance.setDisabled(!showDefaultDistance);
		this.travelCosts.setVisible(showTravelCosts);
		this.travelCosts.setDisabled(!showTravelCosts);
		this.referenceNoField.setVisible(showReferenceNo);
		this.referenceNoField.setDisabled(!showReferenceNo);
	},

	changeMaxBudget: function(type) {
		if(!this.project_id) return;

		const form = new Ext.form.FormPanel({
			url: GO.url('projects2/project'),
			cls: 'go-form-panel',
			items: [
				{xtype:'textfield', fieldLabel: t('New max. budget', 'projects2'), name: type, allowBlank: false, anchor: '100%', value: this.formPanel.form.findField(type).getValue()},
				{xtype:'textarea', fieldLabel: t('Reason', 'projects2'), name: 'reason', allowBlank: false, anchor: '100%', height: 270}
			]
		}), dlg = new Ext.Window({
			title: t('Change maximum budget', 'projects2'),
			width: 500,
			height:400,
			layout: 'fit',
			defaultButton: form.form.findField(type),
			items: [form],
			buttons: ['->', {text: t('Save'), handler: () => {

					const v = form.form.getValues();
					form.form.submit(
						{
							url:GO.url('projects2/project/submit'),
							params: {id: this.project_id},
							waitMsg:t("Saving..."),
							success:(form, action) =>{
								// made change/. set comment
								go.Db.store('Comment').set({create:
									{'#maxBudgetComment':{
										text: t(type === 'max_budgeted_units' ? 'Max. hour budget changed' : 'Max. cost budget changed', 'projects2') + ' '+
											t('from','projects2')+' '+
											this.formPanel.form.findField(type).getValue()+ ' '+t('to','projects2')+' '+ v[type] +
											': ' + v['reason'],
										entity: 'Project',
										entityId: this.project_id
									}}
								});
								this.formPanel.form.findField(type).setValue(v[type]);
								dlg.hide();

							},
							failure: function(form, action) {
								Ext.MessageBox.alert(t("Error"), action.failureType == 'client' ? t("You have errors in your form. The invalid fields are marked.") : action.result.feedback);
							}
						});
				}}]
		});
		dlg.show();

	},

	buildForm : function () {
		
		var items = [];

		var leftCol = new Ext.form.FieldSet({
			itemId:'leftCol',
			columnWidth: .6,
			mobile: {
				columnWidth: 1
			},
			layout:'form',
			border:false
		});

		
		leftCol.add(this.selectParentProject = new GO.projects2.SelectProject({
			fieldLabel: t("Parent project", "projects2"),
			emptyText:t("Search project", "projects2"),
			hiddenName: 'parent_project_id',
			anchor:'100%',
			minListWidth:600,
			region: 'northdele',
			store:new GO.data.JsonStore({
				url: GO.url('projects2/project/store'),
				fields:['id', 'path','use_tasks_panel'],
				remoteSort: true
			})
			
		}));
		
		leftCol.add(this.nameField = new Ext.form.TextField({
			name: 'name',
			anchor: '100%',
			allowBlank:false,
			fieldLabel: t("Name", "projects2")
		}));
		

		leftCol.add(this.templateSelect = new GO.projects2.SelectTemplate({
			anchor:'100%',
			fieldLabel: t("Template", "projects2"),
			hiddenName:'template_id',
			store:GO.projects2.templatesStore,
			valueField:'id',
			displayField:'name',
			mode: 'local',
			triggerAction: 'all',
			editable: false,
			selectOnFocus:true,
			disabled:false, //!GO.settings.modules.projects2.write_permission,
			listeners:{
				change:function(){
					this.applyTemplate();
					
					go.customfields.CustomFields.filterFieldSets(this.formPanel);
					
					
				},
				scope:this
			}
		}));

		if(GO.settings.modules.projects2.write_permission){
			leftCol.add(this.createType = new Ext.form.Checkbox({
				name:'create_type',
				hideLabel:true,
				boxLabel:t("Create new permission type", "projects2"),
				listeners:{
					check:function(cb, checked){
						this.selectType.getEl().up('.x-form-item').setDisplayed(!checked);
						this.selectType.setDisabled(checked);
						this.newTypeName.getEl().up('.x-form-item').setDisplayed(checked);
						this.newTypeName.allowBlank=!checked;
						this.newTypeName.setValue(this.nameField.getValue());
					},
					scope:this
				}
			}));
		

			leftCol.add(this.newTypeName = new Ext.form.TextField({
				name:'new_type_name',
				fieldLabel:t("Name"),
				anchor:'100%'
			}));
		}

		leftCol.add(this.selectType = new GO.projects2.SelectType({    		
			anchor:'100%',
			allowBlank:false
		}));



		leftCol.add(new GO.form.HtmlComponent({
			html:'<br />',
			hideLabel:true
		}));

		leftCol.add(this.selectUser = new go.modules.business.business.EmployeeCombo({
			hiddenName:'responsible_user_id',
			fieldLabel:t("Manager", "projects2"),
			anchor:'100%',
			allowBlank:true
		}));		

		if(go.Modules.isAvailable("community", "addressbook"))
		{  
			this.contactField = new go.modules.community.addressbook.ContactCombo({
				anchor:'100%',
				fieldLabel: t("Contact", "projects2"),
				hiddenName:'contact_id',
				isOrganization: false
			});
			
			leftCol.add(this.contactField);
			this.customerField = new go.modules.community.addressbook.ContactCombo({
				anchor:'100%',
				fieldLabel: t("Customer", "projects2"),
				hiddenName: 'company_id',
				isOrganization: true
			});
			
			leftCol.add(this.customerField);
			
			this.contactField.on('select', function(combo, record){
				if(record.data.organizations.length) {
					this.customerField.setValue(record.data.organizations[0].id);
				}
			}, this);			
			
		}


		var rightCol = new Ext.form.FieldSet({
			itemId:'rightCol',
			columnWidth: .4,
			mobile: {
				columnWidth: 1,
			},
			layout:'form',
			border:false
		});

		this.statusField = new GO.form.ComboBox({
			//xtype:'combo',
			anchor:'100%',
			fieldLabel: t("Status", "projects2"),
			hiddenName:'status_id',
			store:GO.projects2.statusesStore,
			valueField:'id',
			displayField:'name',
			mode: 'local',
			triggerAction: 'all',
			editable: false,
			selectOnFocus:true,
			forceSelection: true,
			allowBlank:false,
			listeners: {
				'select': function(me,record, index) {
					[this.timeEntryGrid,
						this.feesPanel,
						this.expensesPanel,
						this.expenseBudgetsGrid,
						this.incomePanel
					].forEach(p => p.setDisabled(record.json.concept || !this.project_id))
				},scope:this
			}
		});
		
		rightCol.add(this.statusField);

		var now = new Date();
		

		this.dateFields = new Ext.Panel({
			border:false,
			layout:'form',
			forceLayout:true,
			bodyStyle:'padding:0px',
			items:[{
				xtype:'datefield',
				name: 'start_time',
				format: GO.settings['date_format'],
				fieldLabel: t("Start date", "projects2"),
				value: now.format(GO.settings.date_format)
			},{
				xtype:'datefield',
				format: GO.settings['date_format'],
				name: 'due_time',
				fieldLabel: t("Due at", "projects2")//,
			}]
		});

		rightCol.add(this.dateFields);

		this.maxBudgetedUnitsField = new Ext.Container({
			layout: "form",
			cls: 'go-hbox',
			items:[{
				fieldLabel: t('Max. hour budget', 'projects2'),
				xtype:'numberfield',
				name: 'max_budgeted_units',
				width: dp(140),
				//suffix: t('Hours'),
				decimalPrecision: 0
			},{
				xtype:'button',
				text: t('Change'),
				handler: () => { this.changeMaxBudget('max_budgeted_units'); }
			}]
		});
		this.maxBudgetedCostsField = new Ext.Container({

			layout: "form",
			cls: 'go-hbox',
			items:[
				{
					fieldLabel: t('Max. cost budget', 'projects2'),
					xtype:'numberfield',
					name: 'max_budgeted_costs',
					width: dp(140),
					//prefix: GO.settings.currency,
					decimalPrecision: 0
				},{
					xtype:'button',
					text: t('Change'),
					handler: () => { this.changeMaxBudget('max_budgeted_costs'); }
				}
			]
		});

		let c = go.User.capabilities['go:legacy:projects2'] || {};
		if(c.mayBudget) {
			rightCol.add(this.maxBudgetedUnitsField);
			rightCol.add(this.maxBudgetedCostsField);
		}
		rightCol.add(this.mileageDistance = new GO.form.NumberField({
			name:'default_distance',
			fieldLabel:t("Default distance", "projects2"),
			anchor:'100%'
		}));
			
		rightCol.add(this.travelCosts = new GO.form.NumberField({
			name:'travel_costs',
			fieldLabel:t("Travel costs", "projects2"),
			anchor:'100%'
		}));

		rightCol.add(this.incomeField = new Ext.form.ComboBox({
			anchor:'100%',
			fieldLabel : t("Income type", "projects2"),
			hiddenName : 'income_type',
			store : new Ext.data.ArrayStore({
				fields : ['value', 'text'],
				data : [
				[1, t("Contract Price", "projects2")],
				[2, t("Post calculation", "projects2")],
				[3, t("Not billable", "projects2")]
				]
			}),
			value : 'income_type',
			valueField : 'value',
			displayField : 'text',
			mode : 'local',
			triggerAction : 'all',
			editable : false,
			selectOnFocus : true,
			forceSelection : true
		}));

		rightCol.add(this.referenceNoField = new Ext.form.TextField({
			anchor: '-20',
			fieldLabel : t("Reference no.", "projects2"),
			name: 'reference_no',
			maxLength: 64
		}));

		var description = new Ext.form.TextArea({
			name: 'description',
			anchor: '100%',			
			grow: true,
			allowBlank:true,
			fieldLabel: t("Description")
		});

		this.propertiesPanel = new Ext.Panel({
			autoScroll: true,
			url: GO.url('projects2/project/load'),
			border: false,
			baseParams: {},
			title:t("Properties"),			
			layout: "form",
			defaults: {
				anchor: "100%"
			},
			items:[{
					autoHeight:true,
					layout:'column',			
					items:[leftCol, rightCol]
			},{
				xtype:"fieldset",
				border: false,
				items:[description]
			}]
		});
		
		var items  = [this.propertiesPanel];

		this.feesPanel = new GO.projects2.ResourceGrid();
		items.push(this.feesPanel);
          

		this.timeEntryGrid = new GO.projects2.TimeEntryGrid({
			stateId: 'pr2-dialog-time-entries'
		});
		items.push(this.timeEntryGrid);

    	this.expenseBudgetsGrid =  new GO.projects2.ExpenseBudgetsGrid();
		items.push(this.expenseBudgetsGrid);
		
		this.incomePanel =  new GO.projects2.IncomeGrid({
			stateId: 'project-income-grid'
		});
		items.push(this.incomePanel);

		this.expensesPanel = new GO.projects2.ExpensesGrid();
		items.push(this.expensesPanel);

		//add
		this.permissionsPanel = new GO.grid.PermissionsPanel({
			isOverwritable: true,
			levels: [
				GO.permissionLevels.read,
				GO.permissionLevels.create,
				GO.permissionLevels.write,
				GO.permissionLevels.writeAndDelete,
				GO.projects2.permissionLevelFinance, //finance
				GO.permissionLevels.manage
			],
			levelLabels : {
				45: "Finance"
			}
		});
		items.push(this.permissionsPanel);
		
		this.tabPanel = new Ext.TabPanel({
			activeTab: 0,
			deferredRender: false,
			border: false,
			items: items,
			anchor: '100% 100%',
			enableTabScroll:true
		});
		
		
		go.customfields.CustomFields.getFormFieldSets("Project").forEach(function(fs) {
			//console.log(fs);
			if(fs.fieldSet.isTab) {
				fs.title = null;
				fs.collapsible = false;
				var pnl = new Ext.Panel({
					autoScroll: true,
					hideMode: 'offsets', //Other wise some form elements like date pickers render incorrectly.
					title: fs.fieldSet.name,
					items: [fs]
				});
				this.tabPanel.add(pnl);
			}else
			{			
				this.propertiesPanel.add(fs);
			}
		}, this);
		
		this.formPanel = new Ext.form.FormPanel({
			waitMsgTarget:true,
			url: GO.url('projects2/project/load'),
			border: false,
			baseParams: {
				parent_project_id:0
			},
			items:this.tabPanel,
			labelWidth:125
		});    
	}
});
