/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, ComponentEventMap } from "./Component.js";
import { Menu } from "./menu/Menu.js";
import { Config, Listener, ObservableListenerOpts } from "./Observable.js";
import { MaterialIcon } from "./MaterialIcon.js";
type ButtonType = "button" | "submit" | "reset";
/**
 * @inheritDoc
 */
export interface ButtonEventMap<Type> extends ComponentEventMap<Type> {
    /**
     * Fires before showing the button menu. Return false to abort.
     *
     * @param container
     * @param item
     * @param index
     */
    beforeshowmenu: (button: Type, menu: Menu) => false | void;
    /**
     * Fires when the button menu is shown
     *
     * @param button
     * @param menu
     * @param ev
     */
    showmenu: (button: Type, menu: Menu) => false | void;
    /**
     * Fires when the button is clicked.
     *
     * You can also pass a handler function to the button config
     *
     * @see ButtonConfig.handler
     * @param button
     * @param ev
     */
    click: (button: Type, ev: MouseEvent) => void;
}
export interface Button extends Component {
    on<K extends keyof ButtonEventMap<this>, L extends Listener>(eventName: K, listener: Partial<ButtonEventMap<this>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof ButtonEventMap<this>>(eventName: K, listener: Partial<ButtonEventMap<this>>[K]): boolean;
    fire<K extends keyof ButtonEventMap<this>>(eventName: K, ...args: Parameters<ButtonEventMap<any>[K]>): boolean;
    get el(): HTMLButtonElement;
}
/**
 * Button component
 *
 * @example
 *
 * ```typescript
 * btn({
 *   icon: "star",
 *   text: "Test 1"
 *   handler: (e) => alert("Hi!")
 * })
 * ```
 *
 */
export declare class Button extends Component {
    private _iconEl?;
    private _textEl?;
    protected baseCls: string;
    /**
     * If set a handler will be generated with router.goto(this.route);
     */
    route?: string;
    /**
     * Function to be executed on click (added to el.onclick)
     *
     * The handler only fires on the primary mouse button and when the button is duoble clicked it will
     * only fire once!
     */
    handler?: (button: this, ev?: MouseEvent) => any;
    private _menu?;
    private _icon?;
    private _text?;
    /**
     * Turn on if you want this button to be clickable fast.
     * We disable this by default because some users tend to double click on all buttons and we don't
     * want to double submit.
     */
    allowFastClick: boolean;
    constructor();
    /**
     * Find the first menu in the tree of submenu's
     */
    private findTopMenu;
    /**
     * Button type. "button" or "submit", defaults to "button".
     */
    set type(type: ButtonType);
    get type(): ButtonType;
    protected internalRender(): HTMLElement;
    private onMenuButtonClick;
    private onMenuMouseEnter;
    /**
     * Add menu to this button
     */
    set menu(menu: Menu | undefined);
    get menu(): Menu | undefined;
    showMenu(): void;
    protected internalRemove(): void;
    /**
     * Button text
     *
     * Set's the button text and adds a "text" css class
     *
     * This overrides the "html" property. Use html if you want something custom.
     */
    set text(text: string);
    get text(): string;
    /**
     * Set's the button icon and adds a "icon" css class
     */
    set icon(icon: MaterialIcon | "" | undefined);
    get icon(): MaterialIcon | "" | undefined;
    private get iconEl();
    private get textEl();
}
/**
 * Shorthand function to create {@see Button}
 *
 * @param config
 */
export declare const btn: (config?: Config<Button, ButtonEventMap<Button>>) => Button;
export {};
//# sourceMappingURL=Button.d.ts.map