/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, ComponentEventMap } from "./Component.js";
import { Config, Listener, ObservableListenerOpts } from "./Observable.js";
export interface CardContainerEventMap<Type> extends ComponentEventMap<Type> {
    /**
     * Fires before adding an item. Return false to abort.
     *
     * @param container
     * @param item
     * @param index
     */
    cardchange: (container: Type, index: number | undefined, oldIndex: number | undefined) => false | void;
}
export interface CardContainer extends Component {
    on<K extends keyof CardContainerEventMap<this>, L extends Listener>(eventName: K, listener: Partial<CardContainerEventMap<this>>[K], options?: ObservableListenerOpts): L;
    un<K extends keyof CardContainerEventMap<this>>(eventName: K, listener: Partial<CardContainerEventMap<this>>[K]): boolean;
    fire<K extends keyof CardContainerEventMap<this>>(eventName: K, ...args: Parameters<CardContainerEventMap<any>[K]>): boolean;
}
/**
 * Card container
 *
 * Holds multiple components but only shows one.
 *
 *
 * @example
 * ```
 * const cards = CardContainer.create({
 * 	tagName: "main",
 * 	items: [
 * 		Component.create({
 * 			cls: "pad",
 * 			html: "<h1>Tab 1</h1><p>Tab 1 content</p>",
 * 			id: "tab1"
 * 		}),
 * 		Component.create({
 * 			cls: "pad",
 * 			html: "<h1>Tab 2</h1><p>Tab2 content</p>",
 * 			id: "tab2"
 * 		})
 * 	]
 * });
 * ```
 *
 */
export declare class CardContainer extends Component {
    private _activeItem?;
    protected baseCls: string;
    constructor();
    protected internalRender(): HTMLElement;
    protected renderItems(): void;
    private setCardVisibilities;
    /**
     * The active card index. Defaults to 0 if not given.
     */
    set activeItem(ref: number | Component);
    /**
     * The active card index. Defaults to 0 if not given.
     */
    get activeItem(): number;
    focus(o?: FocusOptions): void;
}
/**
 * Shorthand function to create {@see CardContainer}
 *
 * @param config
 * @param items
 */
export declare const cards: (config?: Config<CardContainer, CardContainerEventMap<CardContainer>>, ...items: Component[]) => CardContainer;
//# sourceMappingURL=CardContainer.d.ts.map