/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Component, createComponent } from "./Component.js";
import { FunctionUtil } from "../util/FunctionUtil.js";
export class DraggableComponent extends Component {
    constructor(tagName = "div") {
        super(tagName);
        this._dragConstrainTo = window;
        /**
         * Enable dragging
         */
        this._draggable = true;
        this.onDragHandleClick = (e) => {
            //prevent click events under draggable items
            //needed for table header resize that triggered a sort on click too
            e.stopPropagation();
        };
        this.onDragHandleMouseDown = (e) => {
            //stop if clicked on button inside drag handle. to prevent window dragging on buttons.
            const target = e.target;
            if (target != this.el && (target.tagName == "BUTTON" || target.closest("BUTTON"))) {
                return;
            }
            if (e.button != 0) {
                //only drag with left click
                return;
            }
            e.preventDefault();
            //e.stopPropagation();
            this.focus();
            const el = this.el, rect = el.getBoundingClientRect();
            if (this.setPosition === undefined) {
                const cmpStyle = getComputedStyle(el);
                this.setPosition = cmpStyle.position == 'absolute' || cmpStyle.position == 'fixed';
            }
            this.dragData = {
                startOffsetLeft: el.offsetLeft,
                startOffsetTop: el.offsetTop,
                grabOffsetLeft: e.clientX - rect.x,
                grabOffsetTop: e.clientY - rect.y,
                x: e.clientX,
                y: e.clientY,
                startX: e.clientX,
                startY: e.clientY,
                data: {}
            };
            if (this.fire('dragstart', this, this.dragData, e) !== false) {
                this.onDragStart(e);
            }
        };
        if (this.baseCls) {
            this.baseCls += " draggable";
        }
        else {
            this.baseCls = "draggable";
        }
        this.on("render", () => {
            // invoke draggable setter once
            if (this._draggable) {
                this.toggleDraggable(this._draggable);
            }
        }, { once: true });
    }
    /**
     * Enable or disable dragging
     */
    set draggable(draggable) {
        this._draggable = draggable;
        if (this.rendered) {
            this.toggleDraggable(draggable);
        }
    }
    toggleDraggable(draggable) {
        const dragHandle = this.getDragHandle();
        if (!dragHandle) {
            return;
        }
        if (draggable) {
            dragHandle.classList.add("goui-draghandle");
            dragHandle.addEventListener('click', this.onDragHandleClick);
            dragHandle.addEventListener('mousedown', this.onDragHandleMouseDown);
        }
        else {
            dragHandle.classList.remove("goui-draghandle");
            dragHandle.removeEventListener('click', this.onDragHandleClick);
            dragHandle.removeEventListener('mousedown', this.onDragHandleMouseDown);
        }
    }
    get draggable() {
        return this._draggable;
    }
    /**
     * Returns the DOM element that can be grabbed to drag the component
     * @protected
     */
    getDragHandle() {
        return this.el;
    }
    /**
     * Constrain dragging to this element
     * @param el
     * @param pad Supply paddings
     */
    dragConstrainTo(el, pad) {
        this._dragConstrainTo = el;
        this._dragConstrainPad = pad;
    }
    calcConstrainBox(el, pad) {
        let box = {
            left: 0,
            right: 0,
            bottom: 0,
            top: 0
        };
        if (el instanceof Window) {
            //window is a special case. The page might be scrolled and we want to constrain to the viewport then.
            box.right = window.innerWidth;
            box.bottom = window.innerHeight;
        }
        else {
            const rect = el.getBoundingClientRect();
            box.left = rect.left;
            box.right = rect.right;
            box.bottom = rect.bottom;
            box.top = rect.top;
        }
        if (pad) {
            if (pad.left)
                box.left += pad.left;
            if (pad.right)
                box.right -= pad.right;
            if (pad.top)
                box.top += pad.top;
            if (pad.bottom)
                box.bottom -= pad.bottom;
        }
        return box;
    }
    onDragStart(e) {
        e.preventDefault();
        this.constrainBox = this.calcConstrainBox(this._dragConstrainTo, this._dragConstrainPad);
        const onDrag = FunctionUtil.onRepaint((e) => {
            this.onDrag(e);
        });
        document.addEventListener('mousemove', onDrag);
        document.addEventListener('mouseup', (e) => {
            document.removeEventListener('mousemove', onDrag);
            this.fire("drop", this, this.dragData, e);
        }, { once: true });
    }
    onDrag(e) {
        const d = this.dragData;
        d.x = e.clientX;
        d.y = e.clientY;
        this.constrainCoords();
        if (this.setPosition) {
            this.el.style.top = (d.startOffsetTop + d.y - d.startY) + "px";
            this.el.style.left = (d.startOffsetLeft + d.x - d.startX) + "px";
        }
        this.fire("drag", this, this.dragData, e);
    }
    constrainCoords() {
        if (!this.constrainBox) {
            return;
        }
        const maxTop = this.constrainBox.bottom - this.el.offsetHeight + this.dragData.grabOffsetTop;
        const maxLeft = this.constrainBox.right - this.el.offsetWidth + this.dragData.grabOffsetLeft;
        this.dragData.y = Math.max(this.constrainBox.top + this.dragData.grabOffsetTop, Math.min(this.dragData.y, maxTop));
        this.dragData.x = Math.max(this.constrainBox.left + this.dragData.grabOffsetLeft, Math.min(this.dragData.x, maxLeft));
        return;
    }
    /**
     * Constrain the component to the given element
     *
     * @param el
     * @param pad
     */
    constrainTo(el, pad) {
        const constraints = this.calcConstrainBox(el, pad);
        let maxTop = constraints.bottom - this.el.offsetHeight, maxLeft = constraints.right - this.el.offsetWidth, minTop = 0, minLeft = 0;
        if (this.el.offsetTop > maxTop) {
            console.warn("Contraining to top " + maxTop);
            this.el.style.top = maxTop + "px";
        }
        else if (this.el.offsetTop < minTop) {
            console.warn("Contraining to top " + minTop);
            this.el.style.top = minTop + "px";
        }
        if (this.el.offsetLeft > maxLeft) {
            console.warn("Contraining to left " + maxLeft);
            this.el.style.left = maxLeft + "px";
        }
        else if (this.el.offsetLeft < minLeft) {
            console.warn("Contraining to left " + minLeft);
            this.el.style.left = minLeft + "px";
        }
    }
}
/**
 * Shorthand function to create {@see DraggableComponent}
 *
 * @param config
 * @param items
 */
export const draggable = (config, ...items) => createComponent(new DraggableComponent(config === null || config === void 0 ? void 0 : config.tagName), config, items);
//# sourceMappingURL=DraggableComponent.js.map