import { ChipsField } from "./ChipsField";
import { menu } from "../menu";
import { btn } from "../Button";
import { createComponent } from "../Component";
import { FunctionUtil } from "../../util";
/**
 * Chips component that auto completes user input
 */
export class AutocompleteChips extends ChipsField {
    /**
     * Constructor
     *
     * @param list The drop down list or {@see Table}.
     * @param buffer
     */
    constructor(list, buffer = 300) {
        super();
        this.list = list;
        this.buffer = buffer;
        //disable create
        this.textInputToValue = async (text) => {
            return false;
        };
        this.menu = menu({
            cls: "goui-dropdown scroll",
            removeOnClose: false,
            height: 300,
            listeners: {
                hide: this.onMenuHide.bind(this)
            }
        }, list);
        this.menuButton = btn({
            icon: "expand_more",
            type: "button",
            handler: (button, ev) => {
                this.fire("autocomplete", this, "");
            },
            menu: this.menu
        });
        this.initList();
    }
    initList() {
        this.list.el.addEventListener('keydown', (ev) => {
            switch (ev.key) {
                case "Enter":
                    ev.preventDefault();
                    this.menu.hide();
                    break;
            }
        });
        if (!this.list.rowSelection) {
            this.list.rowSelectionConfig = { multiSelect: false };
        }
        if (!this.list.rowSelection.multiSelect) {
            this.list.on("rowclick", () => {
                this.menu.hide();
            });
            this.list.store.on("datachanged", () => {
                this.list.rowSelection.selected = [0];
            }, { buffer: 0 });
        }
        else {
            this.list.store.on("datachanged", () => {
                this.list.rowSelection.clear();
                this.list.store.data.forEach((record, index) => {
                    if (this.isPickerRecordInValue(record)) {
                        this.list.rowSelection.add(index);
                    }
                });
            }, { buffer: 0 });
        }
    }
    isPickerRecordInValue(record) {
        if (!this.value || !this.value.length) {
            return false;
        }
        if (!this.valuesToCompare) {
            this.valuesToCompare = this.value.map((i) => JSON.stringify(i));
            // clear cache after event loop
            setTimeout(() => {
                this.valuesToCompare = undefined;
            });
        }
        const v = JSON.stringify(this.pickerRecordToValue(this, record));
        return this.valuesToCompare.indexOf(v) > -1;
    }
    /**
     * Method that transforms a record from the TablePicker store to a value for this field.
     * This is not necessarily a text value. In conjunction with {@see valueToTextField()} this
     * could also be an ID of an object for example.
     *
     * @param field
     * @param record
     */
    pickerRecordToValue(field, record) {
        return record.id;
    }
    internalRender() {
        this.buttons = this.buttons || [];
        this.buttons.push(this.menuButton);
        const el = super.internalRender();
        this.menu.alignTo = this.wrap;
        this.menu.alignToInheritWidth = true;
        this.editor.el.addEventListener('input', FunctionUtil.buffer(this.buffer, this.onInput.bind(this)));
        this.editor.el.addEventListener('keydown', (ev) => {
            switch (ev.key) {
                case 'Enter':
                    if (!this.menu.hidden) {
                        ev.preventDefault();
                        this.menu.hide();
                    }
                    break;
                case 'ArrowDown':
                    ev.preventDefault();
                    this.menuButton.showMenu();
                    this.list.focus();
                    break;
                case 'Escape':
                    if (!this.menu.hidden) {
                        this.menu.hide();
                        ev.preventDefault();
                        ev.stopPropagation();
                        this.focus();
                    }
                    break;
            }
        });
        return el;
    }
    onInput(ev) {
        this.menuButton.showMenu();
        this.fire("autocomplete", this, this.editor.el.innerText);
    }
    onMenuHide() {
        if (!this.menu.rendered) {
            return;
        }
        const newValues = this.list.rowSelection.selected.map((index) => {
            const record = this.list.store.get(index);
            return this.pickerRecordToValue(this, record);
        });
        this.list.rowSelection.clear();
        this.editor.el.innerText = "";
        this.focus();
        // set value after focus as this will start tracking for the change event
        if (this.list.rowSelection.multiSelect) {
            this.value = newValues;
        }
        else {
            if (!this.value) {
                this.value = [];
            }
            this.value = this.value.concat(newValues);
        }
    }
}
/**
 * Shorthand function to create {@see AutocompleteChips}
 *
 * @param config
 */
export const autocompletechips = (config) => createComponent(new AutocompleteChips(config.list), config);
//# sourceMappingURL=AutocompleteChips.js.map