/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Observable } from "../component/Observable.js";
/**
 * Collection of items
 *
 * @category Utility
 */
export class Collection extends Observable {
    constructor(items = []) {
        super();
        this.items = items;
    }
    [Symbol.iterator]() {
        return this.items[Symbol.iterator]();
    }
    /**
     * Add items at the end
     *
     * @returns the index of the last added item
     */
    add(...items) {
        let index = -1;
        items.forEach((item) => {
            index = this.items.length;
            if (!this.fire("beforeadd", this, item, index)) {
                return -1;
            }
            this.items.push(item);
            this.fire("add", this, item, index);
            this.fire("datachanged", this);
        });
        return index;
    }
    /**
     * Insert items at the given index
     *
     * @param index Use negative indexes to insert from the end. For example -1 inserts before the last item.
     */
    insert(index, ...items) {
        if (index < 0) {
            index = this.count() + index;
        }
        items.forEach((item) => {
            if (!this.fire("beforeadd", this, item, index)) {
                return -1;
            }
            this.items.splice(index, 0, item);
            this.fire("add", this, item, index);
            this.fire("datachanged", this);
            index++;
        });
        return index;
    }
    /**
     * Get an item at the given index
     * @param index
     */
    get(index) {
        return this.items[index];
    }
    /**
     * Return first item
     */
    first() {
        return this.get(0);
    }
    /**
     * return the last item
     */
    last() {
        return this.get(this.count() - 1);
    }
    /**
     * Find the index of an item. Returns -1 if not found.
     * @param item
     */
    indexOf(item) {
        return this.items.indexOf(item);
    }
    /**
     * Remove items
     */
    remove(...items) {
        items.forEach((item) => {
            const index = this.indexOf(item);
            if (index == -1) {
                return false;
            }
            else {
                return this.removeAt(index);
            }
        });
    }
    /**
     * Remove an item
     *
     * @param index Item index
     */
    removeAt(index) {
        const item = this.get(index);
        if (!item) {
            return false;
        }
        if (!this.fire("beforeremove", this, item, index)) {
            return false;
        }
        this.items.splice(index, 1);
        this.fire("remove", this, item, index);
        this.fire("datachanged", this);
        return true;
    }
    /**
     * Count the number of items
     */
    count() {
        return this.items.length;
    }
    /**
     * Clears all items
     */
    clear() {
        while (this.count()) {
            this.removeAt(0);
        }
        return this;
    }
    /**
     * Replace the collection with an array
     *
     * @param items
     */
    replace(...items) {
        this.clear().add(...items);
    }
    /**
     * Performs the specified action for each element in an array.
     * @param callbackfn  A function that accepts up to three arguments. forEach calls the callbackfn function one time for each element in the array.
     * @param thisArg  An object to which the this keyword can refer in the callbackfn function. If thisArg is omitted, undefined is used as the this value.
     */
    forEach(callbackfn, thisArg) {
        return this.items.forEach(callbackfn, thisArg);
    }
    getArray() {
        return this.items;
    }
    /**
     * Return all items
     */
    all() {
        return this.items;
    }
    /**
     * Returns the value of the first element in the array where predicate is true, and undefined
     * otherwise.
     * @param predicate find calls predicate once for each element of the array, in ascending
     * order, until it finds one where predicate returns true. If such an element is found, find
     * immediately returns that element value. Otherwise, find returns undefined.
     *
     * @returns CollectionItem | undefined
     */
    find(predicate) {
        return this.items.find(predicate);
    }
    /**
     * The filter() method creates a new array with all elements that pass the test implemented by the provided function.
     *
     * @param predicate find calls predicate once for each element of the array, in ascending
     * order, until it finds one where predicate returns true. If such an element is found, find
     * immediately returns that element value. Otherwise, find returns undefined.
     *
     * @returns CollectionItem | undefined
     */
    filter(predicate) {
        return this.items.filter(predicate);
    }
    /**
     * Check if the given item is present
     *
     * @param item
     */
    has(item) {
        return this.findIndex((v) => v == item) > -1;
    }
    /**
     * Returns the index of the first element in the array where predicate is true, and -1
     * otherwise.
     * @param predicate find calls predicate once for each element of the array, in ascending
     * order, until it finds one where predicate returns true. If such an element is found,
     * findIndex immediately returns that element index. Otherwise, findIndex returns -1.
     */
    findIndex(predicate) {
        return this.items.findIndex(predicate);
    }
}
//# sourceMappingURL=Collection.js.map