/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
/**
 * Utilities to operate on objects
 * @category Utility
 */
export class ObjectUtil {
    /**
     * Simple JSON path function
     *
     * eg.
     * const obj = {
     *   foo : {
     *     bar: {
     *       test: 1
     *     }
     *   }
     * }
     *
     * Object.path(obj, 'foo.bar.test'); // 1
     *
     * @param obj
     * @param path
     * @return The value from the path or undefined if not found
     */
    static path(obj, path) {
        const dotPos = path.indexOf(".");
        const prop = dotPos > -1 ? path.substr(0, dotPos) : path;
        const next = obj[prop];
        if (!next) {
            return next;
        }
        if (dotPos == -1) {
            return next;
        }
        if (!ObjectUtil.isObject(next)) {
            return undefined;
        }
        return this.path(next, path.substr(dotPos + 1));
    }
    /**
     * Deep merge two key value objects
     * @param o1
     * @param o2
     */
    static merge(o1, o2) {
        if (!this.isObject(o2)) {
            return o2;
        }
        for (let key in o2) {
            if (key in o1 && this.isObject(o1[key])) {
                o1[key] = this.merge(o1[key], o2[key]);
            }
            else {
                o1[key] = o2[key];
            }
        }
        return o1;
    }
    /**
     * Clone an object
     *
     * @param source
     */
    static clone(source) {
        return structuredClone(source);
    }
}
ObjectUtil.isObject = (obj) => {
    return Object.prototype.toString.call(obj) === '[object Object]';
};
//# sourceMappingURL=ObjectUtil.js.map