<?php

namespace go\modules\business\paypal;

use DateTime;
use Exception;
use go\core;
use go\core\exception\Forbidden;
use go\core\orm\exception\SaveException;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\finance\model\Payment;
use go\modules\business\finance\model\PaymentProviderInterface;
use Mollie\Api\Exceptions\ApiException;
use Mollie\Api\MollieApiClient;


/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author Merijn Schering <mschering@intermesh.nl>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module implements PaymentProviderInterface
{
	use core\event\EventEmitterTrait;

	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

	public function getDependencies() : array
	{
		return ['business/finance'];
	}

	public function requiredLicense(): ?string
	{
		return "billing";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }



	public function getSettings()
	{
	 return model\Settings::get();
	}

	private $paypal;


	/**
	 * @throws Forbidden
	 */
	private function getDocument(int $documentId, string $token) : FinanceDocument {
		$document = FinanceDocument::findById($documentId);
		if ($document->token != $token) {
			throw new Forbidden();
		} else{
			go()->setAuthState(new core\auth\TemporaryState($document->createdBy));
		}

		return $document;
	}

	private $client;

	private function getClient() : core\http\Client {
		if(!isset($this->client)) {
			$this->client = new core\http\Client();
			$this->getAccessToken();
		}

		return $this->client;
	}

	private function getApiUrl() : string {
		return $this->getSettings()->sandbox ? "https://api-m.sandbox.paypal.com" : "https://api-m.paypal.com" ;
	}

	public function getAccessToken() : string {
		$client = $this->getClient();

		$client->setHeader("Authorization", "Basic " . base64_encode($this->getSettings()->clientId . ":" . $this->getSettings()->secret));

		// https://developer.paypal.com/reference/get-an-access-token/

		$client->setHeader('Accept', 'application/json');
		$tokenResponse = $client->post($this->getApiUrl() . "/v1/oauth2/token", "grant_type=client_credentials");

		$tokenBody = core\util\JSON::decode($tokenResponse['body']);

		$client->setHeader("Authorization", "Bearer " . $tokenBody->access_token);
		$client->unsetHeader('Accept');

		return $tokenBody->access_token;
	}

	/**
	 * @param int $documentId
	 * @param string $token
	 * @throws ApiException
	 * @throws Forbidden
	 */
	public function pagePayment(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);

		$order = [
			"intent" => "CAPTURE",
			"purchase_units" => [
				[
					"reference_id" => $document->id,
		      "amount" => [
		        "currency_code" => "EUR",
		        "value" => number_format($document->getTotalPrice(), 2, ".", "")
		      ]
				]
			],
			"application_context" => [
				//"brand_name" =>
				"cancel_url" => go()->getAuthState()->getPageUrl() .	"/business/paypal/cancel/". $document->id . "/" . $document->token,
				"locale" => "en-US",
				"return_url" => go()->getAuthState()->getPageUrl() .	"/business/paypal/return/". $document->id . "/" . $document->token,
			]
		];

		$orderResponse = $this->getClient()->postJson($this->getApiUrl() . "/v2/checkout/orders", $order);

		header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
		header("Cache-Control: post-check=0, pre-check=0", false);
		header("Pragma: no-cache");

//echo "<pre>";
//var_dump($orderResponse);

		foreach($orderResponse['body']['links'] as $link) {
			if($link['rel'] == "approve") {
				header("Location: " . $link['href']);
				exit();
			}
		}


//		require(__DIR__ . '/views/web/payment.php');
	}

	/**
	 * @throws Forbidden
	 */
	public function pageReturn(int $documentId, string $token) {

		$document = $this->getDocument($documentId, $token);

		//paypal sends
		//token=7FM192069G979013E&PayerID=RDRYKQWDMXD9Y
		//token is the order ID

		try {
			if(!$document->isPaid()) {
				$client = $this->getClient();
				$client->setHeader('Content-Type', 'application/json');
				$orderId = $_GET['token'];

				$response = $client->get($this->getApiUrl(). "/v2/checkout/orders/" . $orderId);

				$order = core\util\JSON::decode($response['body'], true);
//		echo "<pre>";
//		var_dump($order);

				//purchase_units reference_id is the document ID
				if ($order['status'] == 'APPROVED') {
					$p = (new Payment())->setValues([
						'businessId' => $document->findBook()->businessId,
						'documentId' => $document->id,
						'customerId' => $document->getCustomerId(),
						'reference' => 'paypal-' . $order['id'],
						'date' => new DateTime(),
						'amount' => $order['purchase_units'][0]['amount']['value'],
						'description' => "Paypal order ID: " . $order['id']
					]);

					if (!$p->save()) {
						throw new SaveException($p);
					}
				}
			}
		}
		catch(Exception $e) {
			$error = $e->getMessage();
		}

		$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();

		require(__DIR__ . '/views/web/return.php');
	}

	/**
	 * @param int $documentId
	 * @param string $token
	 * @throws Forbidden
	 * @throws SaveException
	 * @throws Exception
	 */
	public function pageWebhook(int $documentId, string $token) {

		//todo purcahse_units reference_id is the document ID
   /**
    * array(8) {
   ["id"]=>
   string(17) "6VL66263PD4470328"
   ["intent"]=>
   string(7) "CAPTURE"
   ["status"]=>
   string(8) "APPROVED"
   ["payment_source"]=>
   array(1) {
   ["paypal"]=>
   array(4) {
   ["email_address"]=>
   string(28) "mschering-buyer@intermesh.nl"
   ["account_id"]=>
   string(13) "RDRYKQWDMXD9Y"
   ["name"]=>
   array(2) {
   ["given_name"]=>
   string(4) "test"
   ["surname"]=>
   string(5) "buyer"
   }
   ["address"]=>
   array(1) {
   ["country_code"]=>
   string(2) "NL"
   }
   }
   }
   ["purchase_units"]=>
   array(1) {
   [0]=>
   array(4) {
   ["reference_id"]=>
   string(2) "29"
   ["amount"]=>
   array(2) {
   ["currency_code"]=>
   string(3) "EUR"
   ["value"]=>
   string(6) "968.00"
   }
   ["payee"]=>
   array(2) {
   ["email_address"]=>
   string(34) "mschering-facilitator@intermesh.nl"
   ["merchant_id"]=>
   string(13) "DPSEXT29ZXSGW"
   }
   ["shipping"]=>
   array(2) {
   ["name"]=>
   array(1) {
   ["full_name"]=>
   string(10) "test buyer"
   }
   ["address"]=>
   array(5) {
   ["address_line_1"]=>
   string(21) "25513540 River N343 W"
   ["admin_area_2"]=>
   string(8) "Den Haag"
   ["admin_area_1"]=>
   string(4) "2585"
   ["postal_code"]=>
   string(2) "GJ"
   ["country_code"]=>
   string(2) "NL"
   }
   }
   }
   }
   ["payer"]=>
   array(4) {
   ["name"]=>
   array(2) {
   ["given_name"]=>
   string(4) "test"
   ["surname"]=>
   string(5) "buyer"
   }
   ["email_address"]=>
   string(28) "mschering-buyer@intermesh.nl"
   ["payer_id"]=>
   string(13) "RDRYKQWDMXD9Y"
   ["address"]=>
   array(1) {
   ["country_code"]=>
   string(2) "NL"
   }
   }
   ["create_time"]=>
   string(20) "2022-10-07T08:04:44Z"
   ["links"]=>
   array(3) {
   [0]=>
   array(3) {
   ["href"]=>
   string(67) "https://api.sandbox.paypal.com/v2/checkout/orders/6VL66263PD4470328"
   ["rel"]=>
   string(4) "self"
   ["method"]=>
   string(3) "GET"
   }
   [1]=>
   array(3) {
   ["href"]=>
   string(67) "https://api.sandbox.paypal.com/v2/checkout/orders/6VL66263PD4470328"
   ["rel"]=>
   string(6) "update"
   ["method"]=>
   string(5) "PATCH"
   }
   [2]=>
   array(3) {
   ["href"]=>
   string(75) "https://api.sandbox.paypal.com/v2/checkout/orders/6VL66263PD4470328/capture"
   ["rel"]=>
   string(7) "capture"
   ["method"]=>
   string(4) "POST"
   }
   }
   }
    */
	}

	public function pageNotify() {
		$body = core\http\Request::get()->getBody();

		go()->debug($body);
	}


	public function pageCancel(int $documentId, string $token) {
		$document = $this->getDocument($documentId, $token);
		$returnUrl = $_GET['returnUrl'] ?? $document->getCustomerUrl();

		require(__DIR__ . '/views/web/cancel.php');
	}

	public function getPaymentProviderTitle(): string
	{
		return go()->t("Pay with PayPal");
	}
}
