<?php /** @noinspection PhpUnused */

namespace go\modules\business\support;

use Faker\Generator;
use go\core;
use go\core\db\Criteria;
use go\core\fs\Blob;
use go\core\orm\Filters;
use go\core\orm\Query;
use go\core\model;
use go\core\model\Acl;
use go\core\model\User;
use go\core\orm\Mapping;
use go\core\orm\Property;
use go\core\util\ArrayObject;
use GO\Email\Model\Account;
use go\modules\business\business\model\Business;
use go\modules\business\contracts\model\Contract;
use go\modules\business\finance\model\FinanceDocument;
use go\modules\business\support\cron\ImapImport;
use go\modules\business\support\model\Settings;
use go\modules\business\support\model\SupportTasklist;
use go\modules\business\support\model\UserSettings;
use go\modules\community\comments\model\Comment;
use go\modules\community\tasks\controller\Task as TaskCtlr;
use go\modules\community\tasks\controller\TaskList as TaskListCtlr;
use go\modules\community\tasks\controller\TaskListGrouping;
use go\modules\community\tasks\model\Task;
use Exception;
use go\modules\community\tasks\model\TaskList;
use Swift_EmbeddedFile;

/**
 * @copyright (c) 2020, Intermesh BV https://www.intermesh.nl
 * @author System Administrat <admin@intermesh.localhost>
 * @license http://www.gnu.org/licenses/agpl-3.0.html AGPLv3
 */
class Module extends core\Module
{
	public function getAuthor() : string
	{
		return "Intermesh BV <info@intermesh.nl>";
	}

	public function getDependencies() : array
	{
		return ['community/tasks', 'community/addressbook', 'community/comments'];
	}

	public function autoInstall(): bool
	{
		return true;
	}

	public function requiredLicense(): ?string
	{
		return "groupoffice-pro";
	}

    /**
     * The development status of this module
     * @return string
     */
    public function getStatus() : string{
        return self::STATUS_STABLE;
    }


	/**
	 * @throws Exception
	 */
	protected function afterInstall(model\Module $model): bool
	{
		ImapImport::install("*/5 * * * *");


		$taskList = new TaskList();
		$taskList->name = go()->t('Support', 'business', 'support');
		$taskList->setRole('support');
		$taskList->save();

		return parent::afterInstall($model);
	}

	public function defineListeners()
	{
		Task::on(Task::EVENT_PERMISSION_LEVEL, static::class, 'onTaskPermissionLevel');
		Task::on(Task::EVENT_FILTER_PERMISSION_LEVEL, static::class, 'onTaskFilterPermissionLevel');

		TaskCtlr::on(TaskCtlr::EVENT_BEFORE_SET, static::class, "onBeforeTaskSet");
		Comment::on(Comment::EVENT_BEFORE_SAVE, static::class, 'onCommentBeforeSave');

		Task::on(Task::EVENT_URL, static::class, 'onTaskURL');

		User::on(Property::EVENT_MAPPING, static::class, 'onUserMap');

		//redefine permissions for tasks in tasklists with role = support
		TaskList::on(TaskList::EVENT_PERMISSION_LEVEL, static::class, 'onTaskListPermissionLevel');
		TaskList::on(core\orm\Property::EVENT_MAPPING, static::class, 'onTaskListMap');
		TaskList::on(TaskList::EVENT_FILTER_PERMISSION_LEVEL, static::class, 'onTaskListFilterPermissionLevel');
		TaskList::on(TaskList::EVENT_FILTER, static::class, 'onTaskListFilter');
		TaskList::on(TaskList::EVENT_BEFORE_SAVE, static::class, 'onTaskListBeforeSave');


		//Allow task controller access for users with access to supportclient
		TaskListCtlr::on(TaskListCtlr::EVENT_CHECK_PERMISSION, static::class, 'onControllerCheckPermission');
		TaskListGrouping::on(TaskListGrouping::EVENT_CHECK_PERMISSION, static::class, 'onControllerCheckPermission');
		TaskCtlr::on(TaskCtlr::EVENT_CHECK_PERMISSION, static::class, 'onControllerCheckPermission');
	}

	public static function onControllerCheckPermission() {
		if(\go\core\model\Module::isAvailableFor("business", "supportclient")) {
			return true;
		}
	}

	public static function onTaskListBeforeSave(TaskList $taskList) {
		if(empty($taskList->support) && $taskList->getRole() == "support") {
			$taskList->support = new SupportTasklist($taskList);
		}
	}

	/**
	 * @throws Exception
	 */
	public static function onUserMap(Mapping $mapping) {
		$mapping->addHasOne('supportSettings', UserSettings::class, ['id' => 'userId'], true);
	}

	/**
	 * @throws Exception
	 */
	public static function onTaskListMap(Mapping $mapping) {
		$mapping->addHasOne('support', SupportTasklist::class, ['id' => 'taskListId'], false);
	}
	/**
	 * @throws Exception
	 */
	public static function onTaskListFilter(Filters $filters) {
		$filters->add('forSupport', function(Criteria $c, $value, Query $query) {
			$query->join("business_support_tasklist", "st", "st.taskListId = tasklist.id");
			Acl::applyToQuery($query, "st.customerAclId");
		});
	}

	public static function onTaskListFilterPermissionLevel(Criteria $criteria, $value, Query $query, array $filterCondition, Filters $filters) {
		if($query->isFilterUsed('forSupport')) {
			//stop normal permissions when "forSupport" filter is used. in onTaskListFilter above we use the customerAclId instead
			return false;
		}
	}

	public static function onTaskListPermissionLevel(TaskList $taskList) {
		if($taskList->getRole() != "support") {
			return;
		}

		if(go()->getAuthState() && go()->getAuthState()->isAdmin()) {
			return Acl::LEVEL_MANAGE;
		}

		if($taskList->isNew()) {
			return \go\core\model\Module::findByName('community', 'tasks')
				->getUserRights()->mayChangeTasklists ? Acl::LEVEL_CREATE : false;
		}

		// For agents, we need the normal permissions. If it's not allowed then check the customerAclId
		$normalAclLevel = Acl::getUserPermissionLevel($taskList->aclId, go()->getAuthState()->getUserId());
		if($normalAclLevel >= Acl::LEVEL_CREATE) {
			return $normalAclLevel;
		} else {
			if(!$taskList->support) {
				throw new \LogicException($taskList->name .' has no support property!');
			}
			return Acl::getUserPermissionLevel($taskList->support->customerAclId, go()->getAuthState()->getUserId()) > 0 ? Acl::LEVEL_CREATE : false;
		}
	}

	public static $sendMailOnComment = true;

	/**
	 * Besides this hook also consider {@see Task::onCommentAdded()}
	 * @throws Exception
	 */
	public static function onCommentBeforeSave(Comment $comment) {
		// if comment has mimeMessageId set it was imported from an e-mail
		if($comment->section == "private" || $comment->mimeMessageId || !self::$sendMailOnComment || $comment->entityTypeId != Task::entityType()->getId()) {
			return;
		}

		/** @var Task $task */
		$task = $comment->findEntity();

		if($task->createdBy == go()->getUserId()) {
			// only send email from admin to customer
			return;
		}

		// check if this is a support task
		$tasklist = TaskList::findById($task->tasklistId, ['role']);
		if($tasklist->getRole() != 'support') {
			return;
		}

		$accountId = go()->getDbConnection()
			->selectSingleValue('accountId')
			->from('business_support_email_account')
			->where('tasklistId', '=', $task->tasklistId)
			->single();

		// TODO, perhaps a global account should be configurable?
		if(!$accountId) {
			$accountId = go()->getDbConnection()
				->selectSingleValue('accountId')
				->from('business_support_email_account')
				->single();
		}

		if(!$accountId) {
			return;
		}

		$account = Account::model()->findByPk($accountId, false,true);

		$subject = "Re: ". $task->title();

		$user = User::findById($task->createdBy, ['id', 'displayName', 'email', 'timezone', 'dateFormat', 'timeFormat']);

		$body = $comment->text ."<br /><br/>";
		$url = $task->url();

		$body .= '<hr>'. go()->t("This message was sent from our support system. You can view details about this issue here:", "business", "support") . '<br /><br />';
		$body .= '<a href="' . $url . '">' . $url . '</a>';

		try {
			$message = go()
				->getMailer()
				->setEmailAccount($account)
				->compose();

			$comment->mimeMessageId = $message->getId();

			$refs = array_reverse(Comment::findFor($task)->selectSingleValue('mimeMessageId')->all());

			if(!empty($refs)) {
				$headers = $message->getHeaders();
				$headers->addTextHeader("References", implode(" ", $refs));
				$headers->addTextHeader("In-Reply-To", $refs[0]);
			}

			foreach($comment->attachments as $a) {
				$message->addBlob(Blob::findById($a->blobId), $a->name);
			}

			// inline images
			$blobIds = Blob::parseFromHtml($body);
			foreach ($blobIds as $blobId) {
				$blob = Blob::findById($blobId);

				if($blob) {
					$img = Swift_EmbeddedFile::fromPath($blob->getFile()->getPath());
					$img->setContentType($blob->type);
					$contentId = $message->embed($img);
					$body = Blob::replaceSrcInHtml($body, $blobId, $contentId);
				}
			}

			$message->setSubject($subject)
				->setBody($body, 'text/html')
				->setTo($user->email, $user->displayName)
//				->setBcc(go()->getSettings()->systemEmail)
				->send();

			//The ID changes after send in Swift!. We set it here again to avoid confusion.
			$message->setId($comment->mimeMessageId);

		}catch(Exception $e) {
			if(model\Alert::$enabled) {
				$task->createAlert(new core\util\DateTime())
					->setData([
						'title' => "Error sending comment email",
						'body' => $e->getMessage()
					])
					->save();
			}
		}

	}

	/**
	 * Adds "forSupport" set param so the server can set the support tasklist ID
	 *
	 * @param core\Controller $ctrl
	 * @param ArrayObject $params
	 * @return bool|void
	 * @noinspection PhpUnusedParameterInspection
	 */
	public static function onBeforeTaskSet(core\Controller $ctrl, ArrayObject $params) {
		if(empty($params['forSupport'])) {
			return true;
		}

		if(isset($params['create'])) {
			foreach($params['create'] as &$e) {
				if(empty($e['tasklistId'])) {
					$e['tasklistId'] = self::get()->getSettings()->getDefaultTaskListId();
				}
			}
		}
	}

	public static function onTaskFilterPermissionLevel(Criteria $criteria, $value, Query $query, array $filterCondition, Filters $filters) {
		if($query->isFilterUsed('createdByMe')) {
			//stop normal permissions when "tickets" filter is used
			return false;
		}
	}

	private static function isSupportTask(Task $task) : bool {
		return TaskList::find()->selectSingleValue('role')->where('id', '=', $task->tasklistId)->single() === TaskList::Support;
	}

	/**
	 * Alters the link to the task for support customers
	 *
	 * @param Task $task
	 * @return string|void
	 */
	public static function onTaskURL(Task $task) {
		if(!self::isSupportTask($task) || empty(Settings::get()->taskUrlTemplate)) {
			//continue normal url
			return;
		}

		return str_replace("{id}", $task->id, Settings::get()->taskUrlTemplate);
	}

	/**
	 * Changes the permission level for support customers. It will be writable if it's created by the user.
	 *
	 * @param Task $task
	 * @return int|void
	 */
	public static function onTaskPermissionLevel(Task $task) {

		if(go()->getAuthState() && go()->getAuthState()->isAdmin()) {
			return Acl::LEVEL_MANAGE;
		}
		if(!self::isSupportTask($task)) {
			//continue normal level
			return;
		}

		if($task->isNew() || $task->createdBy == go()->getUserId()) {
			return Acl::LEVEL_WRITE;
		}
	}

	public function getSettings()
	{
		return Settings::get();
	}

	public function demo(Generator $faker)
	{
		$tasklists = TaskList::find()->where(['role' => TaskList::Support]);

		static::$sendMailOnComment = false;

		$titles = [
			"I can't create tasks",
			"My iPhone won't connect",
			"The printer is not working",
			"The meeting room beamer is broken",
			"Fix issue with automatic problem solver",
			"How can I setup two factor authentication?",
			"How can I connect my phone?",
			"Setup MacOS",
			"Setup Windows",
			"I want a faster laptop",
			"I want a bigger screen",
			"I deleted all my important data! Can you restore it?",
			"Where is the upload button?",
			"Does Group-Office have super cow powers?",
			"There's no more coffee!",
			"Can you setup a chat account?",
			"Where can I setup video meetings?",
			"Oops, I deleted my documents",
			"How many backups do you make?",
		];

		foreach($tasklists as $tasklist) {
			\go\modules\community\tasks\Module::get()->demoTasks($faker, $tasklist, true, $titles, 20);
		}
	}

	public function checkAcls()
	{
		return parent::checkAcls();
	}
}
