<?php

//namespace GO\Projects2\Report;
namespace GOFS\Projects2\Report;

use GO;
use GO\Base\Exception\AccessDenied;
use GO\Base\Fs\File;
use GO\Base\Util\Date;
use GO\Base\Db\FindParams;
use GO\Base\Util\Number;
use go\core\http\Exception;
use go\core\orm\Query;
use go\core\util\DateTime;
use go\modules\business\business\model\Activity;
use GO\Projects2\Model\Expense;
use GO\Projects2\Model\Project;
use GO\Projects2\Model\Template;
use GO\Projects2\Model\TimeEntry;
use GO\Projects2\Report\AbstractReport;


final class SustainReport extends AbstractReport
{
	const SUSTAINABLE_BG_COLOR_GREEN = "0092D050";
	const SUSTAINABLE_BG_COLOR_BLUE = "0000B0F0";
	const SUSTAINABLE_BG_COLOR_YELLOW = "00FFC000";
	const SUSTAINABLE_BG_COLOR_ORANGE = "00ED7D31";

	/**
	 * @var int[]
	 */
	private $projectIds = [];

	/**
	 * @var string[]
	 */
	private $prjHeaders = [];

	/**
	 * @var DateTime | null
	 */
	private $periodStart;

	/**
	 * @var DateTime | null
	 */
	private $periodEnd;

	/**
	 * @var DateTime
	 */
	private $nowDate;

	/**
	 * @var int[]
	 */
	private $statusIds = [];

	public function name()
	{
		return "Sustainable projects";
	}

	public function fileExtension()
	{
		return 'xls';
	}

	public function supportedProjectTypes()
	{
		return array(Template::PROJECT_TYPE_PROJECT);
	}


	public function supportsStatusFilter()
	{
		return true;
	}

	public function supportsSelectedProject()
	{
		return false;
	}

	public function supportsBatchReport()
	{
		return true;
	}


	public function supportsDateRange()
	{
		return true;
	}

	public function render($return = false)
	{
		GO::setMaxExecutionTime(300);
		GO::setMemoryLimit(0);

		$this->setQueryParams();

		// File peparatation
		$filePath = \GO::config()->tmpdir . $this->nowDate->format('Y-m-d') . ' ' . $this->name();
		if (isset($this->periodStart)) {
			$filePath .= '_' . $this->periodStart->format("Y-m-d");
		}
		if (isset($this->periodEnd)) {
			$filePath .= '_' . $this->periodEnd->format("Y-m-d");
		}
		$filePath .= '.' . $this->fileExtension();

		$file = new File($filePath);
		if ($file->exists()) {
			$file->delete();
			$file = new File($filePath);
		}

		$this->_xlsObject = new \GO\Base\Fs\XlsFile($filePath);
		$currSheet = $this->_xlsObject->getCurrentSheet();
		$currSheet->setTitle('Projects');

		// Sheet 1: All project data
		$this->projectIds = $this->getProjectIds();
		$this->addProjectHeaders($currSheet);

		foreach ($this->projectIds as $projectId) {
			$this->addProjectRow($projectId);
		}

		// Sheet 2: Time tracking
		$currSheet = $this->_xlsObject->getExcelObject()->createSheet();
		$currSheet->setTitle('Time tracking');
		$this->_xlsObject->setSheetNumber(1);

		// Headers
		$this->_xlsObject->putRecord(['user.name', 'date','comments', 'project.path', 'internal_fee', 'external_fee', 'units',' travel_distance', 'travel_costs']);

		// Get relevant user Ids. We do this by getting user_id fields from pr2_resources records
		$this->addTimeTrackingRecords($currSheet);

		// Wrap it up
		$this->_xlsObject->writeToFile();

		if ($return) {
			return $file->getContents();
		} else {
			$file->touch();
			$file->putContents($file->getContents());
			\GO\Base\Util\Http::outputDownloadHeaders($file);
			echo $file->getContents();
		}

	}

	/**
	 * Set local variables based on Request parameters
	 *
	 * @return void
	 * @throws \Exception
	 */
	private function setQueryParams()
	{
		$tz = new \DateTimeZone('Europe/London');

		$dateFormat = go()->getAuthState()->getUser()->dateFormat ?? 'd/m/Y';
		if (isset($_REQUEST['startdate'])) {
			$this->periodStart = DateTime::createFromFormat($dateFormat, $_REQUEST['startdate'], $tz);
		}
		if (isset($_REQUEST['enddate'])) {
			$this->periodEnd = DateTime::createFromFormat($dateFormat, $_REQUEST['enddate'], $tz);
		}

		if (isset($_REQUEST['status_id']) && is_array($_REQUEST['status_id'])) {
			$this->statusIds = $_REQUEST['status_id'];
		}

		$this->nowDate = new DateTime();
		$this->nowDate->setTimezone($tz);
	}

	/**
	 * Retrieve projectIds using  the parameters set in the setQueryParams method
	 *
	 * @return array
	 * @throws \Exception
	 */
	private function getProjectIds(): array
	{
		// Get parameters start date, end date, status if applicable
		$ids = [];
		$fp = FindParams::newInstance()
			->ignoreAcl()
			->joinRelation('template')
			->select('t.id');
		$fp->getCriteria()
			->addCondition('project_type', Template::PROJECT_TYPE_PROJECT, '=', 'template');
		if (count($this->statusIds)) {
			$fp->getCriteria()->addRawCondition('t.status_id IN (' . implode(',', $this->statusIds) . ')');
		}

		if (isset($this->startDate)) {
			$fp->getCriteria()->addCondition('start_time', $this->startDate, '>=');
		}

		if (isset($this->endDate)) {
			$fp->getCriteria()->addCondition('start_time', $this->endDate, '<');
		}

		return Project::model()->find($fp)->fetchAll(\PDO::FETCH_COLUMN);
	}


	/**
	 * Get all project column names
	 *
	 * The attributes of the full project list are basically a merge of three reports and the hours booked and budgeted by
	 * resource. The three pre-existing reports are:
	 *  - "Export all"
	 * - "Finance Report"
	 * - "Project Cost Report"
	 *
	 * @return array
	 * @throws Exception
	 */
	private function getFullProjectsColumnNames(): array
	{
		$attributes = [
			// "All Projects"
			'id',
			'user_id',
			'name',
			'customer',
			'description',
			'ctime',
			'mtime',
			'path',
			'type_id',
			'parent_project_id',

			// "Finance report"
			"Expenses (Period)",
			"Expenses (Total)",
			"Expense budget (Total)",
			"Time spent (Period)",
			"Time spent (Total)",
			"Time budget (Total)",

			// "Project Cost Report"
			"manager_full_name",
			"start_time",
			"due_time",
			"project_status",
			"budgeted_income",
			"budgeted_internal_fees",
			"budgeted_expenses",
			"budgeted_travel_costs",
			"budgeted_total",
			"actual_income",
			"actual_internal_fees",
			"actual_expenses",
			"actual_travel_costs",
			"actual_total",
			"subproject_budgeted_income",
			"subproject_budgeted_internal_fees",
			"subproject_budgeted_expenses",
			"subproject_budgeted_travel_costs",
			"subproject_budgeted_total",
			"subproject_actual_income",
			"subproject_actual_internal_fees",
			"subproject_actual_expenses",
			"subproject_actual_travel_costs",
			"subproject_actual_total",
			"3 - Business Development (Tenders and Quotations)",
			"14 - Feasibility - Concept Design",
			"9 - Feasibility - Energy Centre Assessment",
			"8 - Feasibility - Energy Demand Assessment",
			"15 - Feasibility - GIS Outputs",
			"11 - Feasibility - Heat Supply Options",
			"10 - Feasibility - Network Route Assessment",
			"16 - Feasibility - Reports",
			"12 - Feasibility - Technology Sizing",
			"13 - Feasibility - TEM",
			"17 - Training - Attendance and Self-Training",
			"1 - Training - Preparation and Delivery",
			"2 - Travelling",
		];

		// Get all resources for projects, generate more attributes
		$fp = FindParams::newInstance()
			->ignoreAcl()
			->joinRelation('user')
			->select('user.displayName')
			->order('user.displayName', 'ASC')
			->distinct(true);
		$fp->getCriteria()
			->addRawCondition('t.project_id IN (' . implode(',', $this->projectIds) . ')');

		$stmt = GO\Projects2\Model\Resource::model()->find($fp);
		while ($rec = $stmt->fetch()) {
			$initials = '';
			foreach (explode(' ', $rec->displayName) as $item) {
				$initials .= strtoupper(substr($item, 0, 1));
			}

			$attributes[] = $initials . ' - hours booked';
			$attributes[] = $initials . ' - hours budgeted';
		}
		$this->prjHeaders = $attributes;
		return $attributes;
	}


	/**
	 * @throws Exception
	 */
	private function addProjectHeaders(\PHPExcel_Worksheet $sheet)
	{
		$this->_xlsObject->putRecord($this->getFullProjectsColumnNames());
		$this->fillRange($sheet, 'A1:J1', self::SUSTAINABLE_BG_COLOR_GREEN);
		$this->fillRange($sheet, 'K1:P1', self::SUSTAINABLE_BG_COLOR_BLUE);
		$this->fillRange($sheet, 'Q1:BA1', self::SUSTAINABLE_BG_COLOR_YELLOW);
//		$this->fillRange($sheet, 'BB1:', self::SUSTAINABLE_BG_COLOR_ORANGE); // TODO
	}


	/**
	 * For each retrieved project, add a project row as per the specification document.
	 *
	 * @param int $projectId
	 * @return void
	 * @throws Exception
	 * @throws AccessDenied
	 */
	private function addProjectRow(int $projectId)
	{
		$values = [(string)$projectId];

		/*
		 * Part 1: "All Projects"
		*/
		$allProjectsFlds = ['user_id', 'name', 'customer', 'description', 'ctime', 'mtime', 'path', 'type_id', 'parent_project_id'];
		$Project = (new Project())->findByPk($projectId, false, true);
		foreach ($allProjectsFlds as $fld) {
			if($fld == 'ctime' || $fld == 'mtime') {
				$values[] = DateTime::createFromFormat('U', $Project->{$fld})->format("d/m/Y H:i:s");
			} else {
				$values[] = (string)$Project->{$fld};
			}
		}

		/*
		 * Part 2:  "Finance report"
  		 */
		$fp2 = FindParams::newInstance()
			->ignoreAcl()
			->select('t.nett,t.date')
			->order('date');
		$fp2->getCriteria()->addCondition('project_id', $projectId);
		$st2 = Expense::model()->find($fp2);

		$total_expenses = 0;
		$total_period = 0;
		foreach ($st2 as $expEntry) {
			// 2b. Sum all expenses
			$total_expenses += $expEntry->nett;
			if ($expEntry->date >= $this->startDate && $expEntry->date < $this->endDate) {
				// 2c. Filter out current period
				$total_period += $expEntry->nett;
			}
		}
		$values[] = Number::localize($total_period);
		$values[] = Number::localize($total_expenses);

		$bpq = GO::getDbConnection()->prepare(
			'SELECT SUM(nett) AS total_budget FROM pr2_expense_budgets WHERE project_id = :project_id;'
		);
		$bpq->execute(['project_id' => $projectId]);
		if ($bprow = $bpq->fetch(\PDO::FETCH_ASSOC)) {
			$values[] = Number::localize($bprow['total_budget']);
		} else {
			$values[] = Number::localize(0);
		};

		$numWorkedHours = 0;
		$numPeriodicalHours = 0;
		$fp2 = FindParams::newInstance()
			->ignoreAcl()
			->select('t.units,t.date')
			->order('t.date');
		$fp2->getCriteria()->addCondition('project_id', $projectId);
		$st2 = TimeEntry::model()->find($fp2);
		foreach ($st2 as $entry) {
			$numWorkedHours += $entry->units;
			if ($entry->date >= $this->startDate && $entry->date < $this->endDate) {
				$numPeriodicalHours += $entry->units;
			}
		}
		$values[] = Number::localize($numPeriodicalHours);
		$values[] = Number::localize($numWorkedHours);

		$tbuq = GO::getDbConnection()->prepare(
			"SELECT SUM(`budgeted_units`) AS `total_budgeted_units` FROM `pr2_resources` WHERE `project_id` = :project_id;"
		);
		$tbuq->execute(['project_id' => $projectId]);
		if ($bprow = $bpq->fetch(\PDO::FETCH_ASSOC)) {
			$values[] = Number::localize($bprow['total_budgeted_units']);
		} else {
			$values[] = Number::localize(0);
		}

		/*
		 * Part 3: "Project Cost Report"
		 */
		$stmt = Project::model()->findByAttribute('parent_project_id', $projectId, FindParams::newInstance()->select('t.*'));
		$PCRdata = $this->itterProjects($stmt);
		$this->formatPCRData($Project, $PCRdata, $values);

		/*
		 * Part 4: Hours booked / budgeted by employee
		 */
		for($idx = count($this->prjHeaders) - count($values); $idx < count($this->prjHeaders) ; $idx++) {
			$values[$idx] = '';
		}
		foreach($Project->resources as $resource) {
			$initials = '';
			foreach(explode(" ", $resource->user->displayName) as $item) {
				$initials .= strtoupper(substr($item, 0,1));
			}
			if($bdgIdx = array_search($initials . " - hours budgeted", $this->prjHeaders)) {
				$values[$bdgIdx] = Number::localize($resource->budgeted_units);
			};

			// Total booked by selected period
			$fp2 = FindParams::newInstance()
				->ignoreAcl()
				->select('COALESCE(SUM(t.units),0) as totalBooked');
			$fp2->getCriteria()->addCondition('project_id', $projectId)
				->addCondition('user_id', $resource->user_id);
			if($this->periodStart) {
				$fp2->getCriteria()->addCondition('date',$this->periodStart->format("U"), '>=');
			}
			if($this->periodEnd) {
				$fp2->getCriteria()->addCondition('date',$this->periodEnd->format("U"), '<=');
			}

			$bkdIdx = array_search($initials . " - hours booked", $this->prjHeaders);
			$st2 = TimeEntry::model()->findSingle($fp2);
			$values[$bkdIdx] = Number::localize($st2->totalBooked);
		}

		// We done. Attach the record, move on
		$this->_xlsObject->putRecord($values);
	}


	/**
	 * Recursively retrieve Project Costs etc.
	 *
	 * Shamelessly stolen from ProjectCostBatchReport, data rendering moved to formatPCRData method
	 *
	 * @param $stmt
	 * @param $MathValues
	 * @return array|false|mixed
	 */
	private function itterProjects($stmt, $MathValues = false)
	{
		if (!$MathValues) {
			$MathValues = array(
				'budgeted_income' => 0,
				'budgeted_internal_fees' => 0,
				'budgeted_expenses' => 0,
				'budgeted_travel_costs' => 0,
				'budgeted_total' => 0,
				'actual_income' => 0,
				'actual_internal_fees' => 0,
				'actual_expenses' => 0,
				'actual_travel_costs' => 0,
				'actual_total' => 0,
				'standard_task' => array()
			);
		}

		foreach ($stmt as $projectModel) {

			$newStmt = Project::model()->findByAttribute('parent_project_id', $projectModel->id, FindParams::newInstance()->select('t.*'));

			if ($newStmt) {
				$MathSubValues = $this->itterProjects($newStmt);

			} else {
				$MathSubValues = array(
					'budgeted_income' => 0,
					'budgeted_internal_fees' => 0,
					'budgeted_expenses' => 0,
					'budgeted_travel_costs' => 0,
					'budgeted_total' => 0,
					'actual_income' => 0,
					'actual_internal_fees' => 0,
					'actual_expenses' => 0,
					'actual_travel_costs' => 0,
					'actual_total' => 0,
					'standard_task' => array()
				);
			}

			/**
			 * Sum the data
			 */
			$amountSums = $projectModel->getBudgetCalculationArrays();

			$budget_sum = $amountSums['budget_sum'];
			$real_sum = $amountSums['real_sum'];

			$MathValues['budgeted_income'] += $budget_sum['budget'];
			$MathValues['budgeted_internal_fees'] += $budget_sum['internalFee'];
			$MathValues['budgeted_expenses'] += $budget_sum['expenses'];
			$MathValues['budgeted_travel_costs'] += $budget_sum['mileage'];
			$MathValues['budgeted_total'] += $budget_sum['sum'];


			$MathValues['actual_income'] += $real_sum['budget'];
			$MathValues['actual_internal_fees'] += $real_sum['internalFee'];
			$MathValues['actual_expenses'] += $real_sum['expenses'];
			$MathValues['actual_travel_costs'] += $real_sum['mileage'];
			$MathValues['actual_total'] += $real_sum['sum'];

			$MathValues['budgeted_income'] += $MathSubValues['budgeted_income'];
			$MathValues['budgeted_internal_fees'] += $MathSubValues['budgeted_internal_fees'];
			$MathValues['budgeted_expenses'] += $MathSubValues['budgeted_expenses'];
			$MathValues['budgeted_travel_costs'] += $MathSubValues['budgeted_travel_costs'];
			$MathValues['budgeted_total'] += $MathSubValues['budgeted_total'];


			$MathValues['actual_income'] += $MathSubValues['actual_income'];
			$MathValues['actual_internal_fees'] += $MathSubValues['actual_internal_fees'];
			$MathValues['actual_expenses'] += $MathSubValues['actual_expenses'];
			$MathValues['actual_travel_costs'] += $MathSubValues['actual_travel_costs'];
			$MathValues['actual_total'] += $MathSubValues['actual_total'];

			$MathValues['subproject_budgeted_income'] = $MathSubValues['budgeted_income'];
			$MathValues['subproject_budgeted_internal_fees'] = $MathSubValues['budgeted_internal_fees'];
			$MathValues['subproject_budgeted_expenses'] = $MathSubValues['budgeted_expenses'];
			$MathValues['subproject_budgeted_travel_costs'] = $MathSubValues['budgeted_travel_costs'];
			$MathValues['subproject_budgeted_total'] = $MathSubValues['budgeted_total'];
			$MathValues['subproject_actual_income'] = $MathSubValues['actual_income'];
			$MathValues['subproject_actual_internal_fees'] = $MathSubValues['actual_internal_fees'];
			$MathValues['subproject_actual_expenses'] = $MathSubValues['actual_expenses'];
			$MathValues['subproject_actual_travel_costs'] = $MathSubValues['actual_travel_costs'];
			$MathValues['subproject_actual_total'] = $MathSubValues['actual_total'];


			$timeStms = (new Query())
				->select('t.id, t.code, t.name, SUM(te.duration) AS total')
				->from('business_activity', 't')
				->groupBy(['t.id'])
				->join((new TimeEntry())->tableName(), 'te', 'te.standard_task_id = t.id')
				->where(['te.project_id' => $projectModel->id])
				->all();

			foreach ($timeStms as $standardTaskTime) {
				$standardTaskId = intval($standardTaskTime['id']);
				if (!array_key_exists($standardTaskId, $MathSubValues['standard_task'])) {
					$MathSubValues['standard_task'][$standardTaskId] = array(
						'name' => $standardTaskTime['code'] . "  - " . $standardTaskTime['name'],
						'duration' => 0
					);
				}
				$MathSubValues['standard_task'][$standardTaskId]['duration'] += floatval($standardTaskTime['total']);
			}

			foreach ($this->getStandardTask() as $standardTaskModel) {
				if (empty($MathSubValues['standard_task'][$standardTaskModel->id]['duration']) || $MathSubValues['standard_task'][$standardTaskModel->id]['duration'] == '') {
					$value = 0;
				} else {
					$value = $MathSubValues['standard_task'][$standardTaskModel->id]['duration'];
				}

				if (!array_key_exists((int)$standardTaskModel->id, $MathValues['standard_task'])) {
					$MathValues['standard_task'][$standardTaskModel->id] = array(
						'name' => $standardTaskModel->code . "  - " . $standardTaskModel->name,
						'duration' => 0
					);
				}

				$MathValues['standard_task'][$standardTaskModel->id]['duration'] += $value;

			}
		}

		return $MathValues;
	}


	/**
	 * Prepare Project Costs data for rendering into Excel rows
	 *
	 * @param array $data Project Costs array per project
	 * @param array $values values of the row to be rendered
	 * @return void
	 */
	private function formatPCRData(Project $projectModel, array $data, array &$values)
	{
		$values[] = $projectModel->responsibleUser ? $projectModel->responsibleUser->name : ' ';
		$values[] = Date::get_timestamp($projectModel->start_time);
		$values[] = Date::get_timestamp($projectModel->due_time);
		$values[] = $projectModel->status->name;
		foreach (['budgeted_income', 'budgeted_internal_fees', 'budgeted_expenses', 'budgeted_travel_costs',
			         'budgeted_total', 'actual_income', 'actual_internal_fees', 'actual_expenses', 'actual_travel_costs',
			         'actual_total', 'subproject_budgeted_income', 'subproject_budgeted_internal_fees', 'subproject_budgeted_expenses',
			         'subproject_budgeted_travel_costs', 'subproject_budgeted_total', 'subproject_actual_income',
			         'subproject_actual_internal_fees', 'subproject_actual_expenses', 'subproject_actual_travel_costs',
			         'subproject_actual_total'] as $key) {
			$values[] = isset($data[$key]) ? Number::localize($data[$key]) : "0";
		}

		// We do not need all the standard tasks, so we filter them out
		$standardTaskKeys = ['3 - Business Development (Tenders and Quotations)', '14 - Feasibility - Concept Design',
			'9 - Feasibility - Energy Centre Assessment', '8 - Feasibility - Energy Demand Assessment',
			'15 - Feasibility - GIS Outputs', '11 - Feasibility - Heat Supply Options', '10 - Feasibility - Network Route Assessment',
			'16 - Feasibility - Reports', '12 - Feasibility - Technology Sizing', '13 - Feasibility - TEM',
			'17 - Training - Attendance and Self-Training', '1 - Training - Preparation and Delivery', '2 - Travelling'
		];

		foreach ($standardTaskKeys as $key) {
			$value = 0;
			foreach ($data['standard_task'] as $stdTask) {
				if ($stdTask['name'] === $key) {
					$value = $stdTask['duration'];
					break;
				}
			}
			$values[] = Number::localize($value);
		}
	}


	/**
	 * Shamelessly stolen from ProjectCostBatchReport
	 *
	 * @return array
	 */
	private function getStandardTask()
	{
		if (empty($this->standardTaskArray)) {
			$this->standardTaskArray = Activity::find()->select('id, code, name')->orderBy(['name' => 'ASC'])->all();
		}
		return $this->standardTaskArray;
	}


	/**
	 * Fill the data on the 'time tracking' tab.
	 *
	 * IN this tab, time tracking is being merged with leave day data. It does not make much sense, but that is what
	 * Sustainable wishes
	 *
	 * @todo merge with leave days
	 * @return void
	 */
	private function addTimeTrackingRecords()
	{
		$sql = <<<SQL
SELECT `user`.`displayName`, FROM_UNIXTIME( `t`.`date`, '%d/%m/%Y %H:%i') AS `date`, `t`.`comments`, `project`.`path`, `t`.`internal_fee`, `t`.`external_fee`, `t`.`units`, `t`.`travel_distance`, `t`.`travel_costs`
FROM `pr2_hours` `t`
INNER JOIN `core_user` `user` ON `t`.`user_id` = `user`.`id`
INNER JOIN `pr2_projects` `project` ON `t`.`project_id` = `project`.`id` 
WHERE `t`.`status` < 2
AND `t`.`date` >= :start1
AND `t`.`date` < :end1
UNION 
SELECT `user`.`displayName`, FROM_UNIXTIME( `t`.`first_date`, '%d/%m/%Y %H:%i') AS `date`, `t`.`description` AS `comments`, `ba`.`name` AS `path`, `emp`.`hourlyCosts` AS `internal_fee`, `emp`.`hourlyCosts` AS `external_fee`, `t`.`n_hours` AS `units`, 0 AS `travel_distance`, 0 AS `travel_costs`
FROM `ld_leave_days` `t`
INNER JOIN `core_user` `user` ON `t`.`user_id` = `user`.`id`
INNER JOIN `business_employee` `emp` ON `user`.`id` = `emp`.`id`
INNER JOIN `business_activity` `ba` ON `t`.`ld_credit_type_id` = `ba`.`id`
WHERE `t`.`status` < 2
AND `t`.`first_date` >= :start2
AND `t`.`first_date` < :end2
ORDER BY `displayName` ASC, `date` ASC;
SQL;

		$q = GO::getDbConnection()->prepare($sql);
		$q->execute(['start1' => $this->startDate, 'end1' => $this->endDate, 'start2' => $this->startDate, 'end2' => $this->endDate]);
		foreach ($q->fetchAll(\PDO::FETCH_NUM) as $row) {
			foreach ($row as $key => $value) {
				if (is_numeric($value)) {
					$row[$key] = Number::localize($value);
				} elseif (!is_string($value)) {
					$row[$key] = (string) $value;
				} else {
					$row[$key] = $value;
				}
			}
			$this->_xlsObject->putRecord($row);
		}
	}


	/**
	 * Get a cell range, give pretty background color
	 *
	 * @param $currSheet
	 * @param string $range
	 * @param string $color
	 * @return void
	 */
	public function fillRange(&$currSheet, string $range, string $color)
	{
		$currSheet->getStyle($range)->getFill()->setFillType(\PHPExcel_Style_Fill::FILL_SOLID)
			->getStartColor()->setARGB($color);

	}

}