/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Config, Listener, ObservableListenerOpts } from "../component/Observable.js";
import { Collection, CollectionEventMap } from "../util/Collection.js";
/**
 * Comparator interface for sorting data
 */
export interface Comparator {
    /**
     * The property to sort on
     */
    property: string;
    /**
     * Sort ascending or not
     */
    isAscending?: boolean;
}
export type StoreRecord = Record<string, any>;
/**
 * @inheritDoc
 */
export interface StoreEventMap<Type, RecordType> extends CollectionEventMap<Type, RecordType> {
    /**
     * Fires when data is loaded into the store
     *
     * @param store
     * @param append Whether the records were added to the store.
     */
    beforeload: (store: Type, append: boolean) => void;
    /**
     * Fires when data is loaded into the store
     *
     * @param store
     * @param records
     * @param append Whether the records were added to the store.
     */
    load: (store: Type, records: RecordType[], append: boolean) => void;
    /**
     * Fires when data load failed
     *
     * @param store
     * @param reason
     */
    loadexception: (store: Type, reason: any) => void;
}
export interface Store<RecordType = StoreRecord> {
    on<K extends keyof StoreEventMap<this, RecordType>, L extends Listener>(eventName: K, listener: Partial<StoreEventMap<this, RecordType>>[K], options?: ObservableListenerOpts): L;
    fire<K extends keyof StoreEventMap<this, RecordType>>(eventName: K, ...args: Parameters<StoreEventMap<any, RecordType>[K]>): boolean;
}
export type storeRecordType<StoreType> = StoreType extends Store<infer RecordType> ? RecordType : never;
/**
 * Generic data store used by components
 *
 * @category Data
 */
export declare class Store<RecordType = StoreRecord> extends Collection<RecordType> {
    private _loading;
    private _loaded;
    /**
     * Sort the data on field and direction
     */
    sort: Comparator[];
    /**
     * True when the store is loading
     */
    get loading(): boolean;
    /**
     * True when data has been loaded at least once
     */
    get loaded(): boolean;
    /**
     * Load a set of records
     *
     * @param records
     * @param append
     */
    loadData(records: RecordType[], append?: boolean): void;
    set data(records: RecordType[]);
    get data(): RecordType[];
    /**
     * Reload the data from the source
     */
    reload(): Promise<RecordType[]>;
    clear(): this;
    /**
     * Returns the loaded records. If append is true it only returns the new records.
     * Override this function for new store types.
     *
     * @param append
     * @protected
     */
    protected internalLoad(append: boolean): Promise<RecordType[]>;
    /**
     * Loads the data from source into the store
     *
     * @param append
     */
    load(append?: boolean): Promise<RecordType[]>;
    /**
     * Load the next set of records when paging.
     * Doesn't do anything in the array store but can be implemented in async stores.
     */
    loadNext(append?: boolean): Promise<RecordType[]>;
    /**
     * Load the next set of records when paging.
     * Doesn't do anything in the array store but can be implemented in async stores.
     */
    loadPrevious(): Promise<RecordType[]>;
    hasNext(): boolean;
    hasPrevious(): boolean;
}
export type StoreConfig<RecordType extends StoreRecord = StoreRecord> = Omit<Config<Store<RecordType>, StoreEventMap<Store<RecordType>, RecordType>>, "hasNext" | "loadPrevious" | "hasPrevious" | "loadNext" | "load" | "reload" | "clear" | "loadData" | "[SymbolConstructor.iterator]" | // does not work :(
"add" | "count" | "find" | "findIndex" | "first" | "forEach" | "get" | "getArray" | "has" | "indexOf" | "removeAt" | "remove" | "last" | "replace" | "insert">;
/**
 * Shorthand function to create a {@see Store}
 *
 * @param config
 */
export declare const store: <RecordType extends StoreRecord = StoreRecord>(config?: StoreConfig<RecordType> | undefined) => Store<RecordType>;
//# sourceMappingURL=Store.d.ts.map