/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Michael de Hart <mdhart@intermesh.nl>
 */
import { comp, Component, createComponent } from "./Component.js";
import { t } from "../Translate.js";
import { E } from "../util/Element.js";
import { rowselect } from "./table/RowSelect.js";
import { dragData } from "../DragData";
import { root } from "./Root";
import { Window } from "./Window";
const dropPin = comp({
    cls: "drop-pin",
    hidden: true
});
root.items.add(dropPin);
/**
 * List component
 *
 * Create a list with a custom item renderer. Also capable of selecting rows.
 */
export class List extends Component {
    // protected fragment?: DocumentFragment;
    /**
     * Row selection object
     * @param rowSelectionConfig
     */
    set rowSelectionConfig(rowSelectionConfig) {
        if (typeof rowSelectionConfig != "boolean") {
            rowSelectionConfig.list = this;
            this.rowSelect = rowselect(rowSelectionConfig);
        }
        else {
            this.rowSelect = rowselect({ list: this });
        }
    }
    set sortable(sortable) {
        this.draggable = true;
        this.dropBetween = true;
        this.dropOn = false;
        const ref = this.on("drop", (list, e, dropRow, dropIndex, position, dragData) => {
            const store = dragData.cmp.store;
            // remove the dragged record from the store
            store.removeAt(dragData.storeIndex);
            if (dragData.storeIndex < dropIndex) {
                // if inserting in the same store we need to substract 1 from the index as we took one off.
                dropIndex--;
            }
            //add the record to the new position
            switch (position) {
                case "before":
                    // reorder in the tree where it's dropped
                    store.insert(dropIndex, dragData.record);
                    break;
                case "after":
                    store.insert(dropIndex + 1, dragData.record);
                    break;
            }
        });
    }
    constructor(store, renderer, tagName = "ul") {
        super(tagName);
        this.store = store;
        this.renderer = renderer;
        /**
         * Shown when the list is empty.
         */
        this.emptyStateHtml = `<div class="goui-empty-state"><i class="icon">article</i><p>${t("Nothing to show")}</p></div>`;
        this.emptyStateTag = 'li';
        /**
         * Allow items to be dragged
         */
        this.draggable = false;
        /**
         * Allow to drop between items
         */
        this.dropBetween = false;
        /**
         * Allow to drop on items
         */
        this.dropOn = false;
        this.itemTag = 'li';
        this.tabIndex = 0;
        store.on("beforeload", () => {
            this.mask();
        });
        store.on("load", () => {
            this.unmask();
            if (this.emptyEl) {
                this.emptyEl.hidden = this.store.count() > 0;
            }
        });
        store.on("loadexception", (store, reason) => {
            Window.error(reason);
            this.unmask();
        });
    }
    get rowSelection() {
        return this.rowSelect;
    }
    get el() {
        return super.el;
    }
    internalRender() {
        const el = super.internalRender();
        this.initNavigateEvent();
        el.on("mousedown", (e) => {
            this.onMouseEvent(e, "rowmousedown");
        }).on("dblclick", (e) => {
            this.onMouseEvent(e, "rowdblclick");
        }).on("click", (e) => {
            this.onMouseEvent(e, "rowclick");
        }).on("contextmenu", (e) => {
            this.onMouseEvent(e, "rowcontextmenu");
        }).on("keydown", (e) => {
            this.onKeyDown(e);
        });
        this.renderEmptyState();
        this.renderBody();
        this.initStore();
        return el;
    }
    onKeyDown(e) {
        if (e.key == "Delete" || e.metaKey && e.key == "Backspace") {
            this.fire("delete", this);
        }
    }
    initStore() {
        // handling remove and add per items allows a drag and drop action via store.remove and store.add
        this.store.on("remove", this.onRecordRemove.bind(this));
        this.store.on("add", this.onRecordAdd.bind(this));
    }
    onRecordRemove(collection, item, index) {
        var _a;
        const rows = this.getRowElements();
        (_a = rows[index]) === null || _a === void 0 ? void 0 : _a.remove();
        if (this.rowSelection) {
            this.rowSelection.remove(index, true);
        }
    }
    //todo inserting doesn't work with groups yet. It can only append to the last
    onRecordAdd(collection, item, index) {
        const container = this.renderGroup(item);
        const row = this.renderRow(item, index);
        if (index == collection.count() - 1) {
            container.append(row);
        }
        else {
            const before = container.children[index];
            container.insertBefore(row, before);
        }
        this.onRowAppend(row, item, index);
    }
    getRowElements() {
        return Array.from(this.el.querySelectorAll(":scope > .data"));
    }
    initNavigateEvent() {
        this.on('rowmousedown', (list, storeIndex, row, ev) => {
            if (!ev.shiftKey && !ev.ctrlKey) {
                this.fire("navigate", this, storeIndex);
            }
        });
        if (this.rowSelection) {
            this.el.addEventListener('keydown', (ev) => {
                if (!ev.shiftKey && !ev.ctrlKey && (ev.key == "ArrowDown" || ev.key == "ArrowUp")) {
                    const selected = this.rowSelect.selected;
                    if (selected.length) {
                        const storeIndex = selected[0];
                        this.fire("navigate", this, storeIndex);
                    }
                }
            });
        }
    }
    renderEmptyState() {
        this.emptyEl = E(this.emptyStateTag).css({ 'captionSide': 'bottom', height: '100%' });
        this.emptyEl.hidden = this.store.count() > 0;
        this.emptyEl.innerHTML = this.emptyStateHtml;
        this.el.appendChild(this.emptyEl);
    }
    renderBody() {
        this.renderRows(this.store.items);
        if (this.rowSelect) {
            this.rowSelect.on('rowselect', (rowSelect, storeIndex) => {
                const tr = this.getRowElements()[storeIndex];
                if (!tr) {
                    //row not rendered (yet?). selected class will also be addded on render
                    return;
                }
                tr.classList.add('selected');
                // focus so it scrolls in view
                //tr.focus();
            });
            this.rowSelect.on('rowdeselect', (rowSelect, storeIndex) => {
                const tr = this.getRowElements()[storeIndex];
                if (!tr) {
                    console.error("No row found for selected index: " + storeIndex + ". Maybe it's not rendered yet?");
                    return;
                }
                tr.classList.remove('selected');
            });
        }
    }
    focusRow(index) {
        const tr = this.getRowElements()[index];
        if (tr) {
            tr.focus();
        }
    }
    renderRows(records) {
        for (let i = 0, l = records.length; i < l; i++) {
            const container = this.renderGroup(records[i]), row = this.renderRow(records[i], i);
            if (this.rowSelection && this.rowSelection.selected.indexOf(i) > -1) {
                row.cls("+selected");
            }
            container.append(row);
            this.onRowAppend(row, records[i], i);
        }
        this.fire("renderrows", this, records);
    }
    onRowAppend(row, record, index) {
    }
    renderGroup(record) {
        // return this.fragment!;
        return this.el;
    }
    renderRow(record, storeIndex) {
        const row = E(this.itemTag)
            .cls('+data')
            .attr('tabindex', '0');
        if (this.draggable) {
            row.draggable = true;
            row.ondragstart = this.onNodeDragStart.bind(this);
        }
        this.bindDropEvents(row);
        const r = this.renderer(record, row, this, storeIndex);
        if (typeof r === "string") {
            row.innerHTML = r;
        }
        else if (Array.isArray(r)) {
            r.forEach(c => {
                c.parent = this;
                c.render(row);
            });
        } // else NO-OP renderder will be responsible for appending html to the row @see Table
        if (this.rowSelection && this.rowSelection.selected.indexOf(storeIndex) > -1) {
            row.classList.add("selected");
        }
        return row;
    }
    // private onScroll() {
    // 	const el = this.el;
    // 	const pixelsLeft = el.scrollHeight - el.scrollTop - el.offsetHeight;
    //
    // 	if (pixelsLeft < 100) {
    // 		if (!this.store.loading && this.store.hasNext()) {
    // 			this.store.loadNext(true).finally(() => {
    // 				this.fire("scrolleddown", this);
    // 			});
    // 		}
    // 	}
    // }
    onMouseEvent(e, type) {
        const row = this.findRowByEvent(e), index = row ? this.getRowElements().indexOf(row) : -1;
        if (index !== -1) {
            this.fire(type, this, index, row, e);
        }
    }
    findRowByEvent(e) {
        return e.target.closest(".data");
    }
    bindDropEvents(row) {
        row.ondrop = this.onNodeDrop.bind(this);
        row.ondragend = this.onNodeDragEnd.bind(this);
        row.ondragover = this.onNodeDragOver.bind(this);
        row.ondragenter = this.onNodeDragEnter.bind(this);
        row.ondragleave = this.onNodeDragLeave.bind(this);
    }
    onNodeDragStart(e) {
        e.stopPropagation();
        const row = e.target;
        //needed for iOS
        e.dataTransfer.setData('text/plain', 'goui');
        dragData.row = row;
        dragData.cmp = this;
        dragData.storeIndex = this.getRowElements().indexOf(row);
        dragData.record = this.store.get(dragData.storeIndex);
        // had to add this class because otherwise dragleave fires immediately on child nodes: https://stackoverflow.com/questions/7110353/html5-dragleave-fired-when-hovering-a-child-element
        root.el.cls("+dragging");
    }
    onNodeDragEnd(e) {
        root.el.cls("-dragging");
    }
    onNodeDragEnter(e) {
        const dropRow = this.findDropRow(e);
        if (this.dropAllowed(e, dropRow)) {
            e.stopPropagation();
            e.preventDefault();
            dropRow.cls("+drag-over");
            this.onNodeDragEnterAllowed(e, dropRow);
        }
    }
    onNodeDragEnterAllowed(e, dropRow) {
    }
    onNodeDragLeave(e) {
        const dropRow = this.findDropRow(e);
        if (this.dropAllowed(e, dropRow)) {
            e.stopPropagation();
            e.preventDefault();
            this.clearOverClasses(dropRow);
            this.onNodeDragLeaveAllowed(e, dropRow);
        }
    }
    onNodeDragLeaveAllowed(e, dropRow) {
    }
    findDropRow(e) {
        return e.target.closest("LI");
    }
    clearOverClasses(dropRow) {
        dropRow.cls("-drag-over");
        dropRow.classList.remove("before");
        dropRow.classList.remove("after");
        dropRow.classList.remove("on");
        dropPin.hidden = true;
    }
    onNodeDragOver(e) {
        if (!this.dropOn && !this.dropBetween) {
            return;
        }
        const dropRow = this.findDropRow(e);
        if (this.dropAllowed(e, dropRow)) {
            e.stopPropagation();
            e.preventDefault();
            const pos = this.getDropPosition(e);
            dropRow.classList.toggle("before", "before" == pos);
            dropRow.classList.toggle("after", "after" == pos);
            dropRow.classList.toggle("on", "on" == pos);
            dropPin.hidden = pos == "on" || pos == undefined;
            const dropRowRect = dropRow.getBoundingClientRect();
            dropPin.el.style.top = (pos == "before" ? dropRowRect.y : dropRowRect.y + dropRowRect.height - 1) + "px";
            dropPin.el.style.left = dropRowRect.x + "px";
            dropPin.el.style.width = dropRowRect.width + "px";
        }
    }
    dropAllowed(e, dropRow) {
        return this.fire("dropallowed", this, e, dropRow, dragData);
    }
    getDropPosition(e) {
        if (!this.dropBetween) {
            return this.dropOn ? "on" : undefined;
        }
        const betweenZone = 6;
        if (e.offsetY < betweenZone) {
            return "before";
        }
        else if (e.offsetY > e.target.offsetHeight - betweenZone) {
            return "after";
        }
        else {
            return this.dropOn ? "on" : undefined;
        }
    }
    onNodeDrop(e) {
        const dropPos = this.getDropPosition(e);
        if (!dropPos) {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        const dropRow = this.findDropRow(e), dropIndex = this.getRowElements().indexOf(dropRow);
        this.clearOverClasses(dropRow);
        this.fire("drop", this, e, dropRow, dropIndex, dropPos, dragData);
    }
}
/**
 * Shorthand function to create {@see Table}
 *
 * @param config
 */
export const list = (config) => createComponent(new List(config.store, config.renderer), config);
//# sourceMappingURL=List.js.map