/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { ContainerField } from "./ContainerField.js";
import { Notifier } from "../../Notifier.js";
import { createComponent } from "../Component.js";
import { Field } from "./Field.js";
import { t } from "../../Translate.js";
/**
 * Form component
 *
 * Forms can be used to submit or present data.
 *
 * @example Password validation
 *
 * ```
 * 	textfield({
 * 		type: "password",
 * 		label: "Password",
 * 		name: "password",
 * 		listeners: {
 * 			validate: (field) => {
 * 				const form = field.findAncestorByInstanceType(Form)!;
 * 				if(field.getValue() != form.findField("confirm")!.getValue()) {
 * 					field.setInvalid("The passwords don't match");
 * 				}
 * 			}
 * 		},
 * 	}),
 *
 * 	textfield({
 * 		itemId: "confirm",//item ID used instead of name so this field won't be submitted
 * 		type: "password",
 * 		label: "Confirm password"
 * 	}),
 *
 * ```
 *
 */
export class Form extends ContainerField {
    constructor() {
        super("form");
        this.baseCls = "goui-form";
        this.hideLabel = true;
    }
    internalRender() {
        const el = super.internalRender();
        // disable browser validation
        this.el.noValidate = true;
        el.addEventListener("submit", (event) => {
            event.preventDefault();
            this.submit();
        });
        el.addEventListener("reset", (e) => {
            e.preventDefault();
            this.reset();
        });
        //Submit forms on CTRL + ENTER. (For html area or text area's)
        el.addEventListener("keydown", (e) => {
            if ((e.ctrlKey || e.metaKey) && e.key == "Enter") {
                e.preventDefault();
                if (document.activeElement && "blur" in document.activeElement) {
                    document.activeElement.blur();
                }
                this.submit();
            }
        });
        return el;
    }
    /**
     * @inheritDoc
     */
    reset() {
        this.findFields().forEach((field) => {
            field.reset();
        });
    }
    set value(v) {
        super.value = v;
        this.trackReset();
    }
    get value() {
        return super.value;
    }
    /**
     * Get the modified field values since the form was:
     *
     * - rendered OR
     * - value was set (usually through a load) OR
     * - submitted
     */
    get modified() {
        const v = {};
        this.findFields().forEach((field) => {
            if (field instanceof Field) {
                if (field.name && !field.disabled && field.isModified()) {
                    v[field.name] = field.value;
                }
            }
            else {
                //for Extjs compat try .getName() and .getValue()
                const fieldName = field.getName();
                if (fieldName && !field.disabled && field.isDirty()) {
                    v[fieldName] = field.getValue();
                }
            }
        });
        return v;
    }
    /**
     * Validates the form and submits it using the handler function passed with the config.
     */
    async submit() {
        const el = this.el;
        this.clearInvalid();
        if (this.isValid()) {
            el.cls(['+valid', '-invalid']);
            let handlerResponse = undefined;
            if (this.handler) {
                try {
                    handlerResponse = await this.handler(this);
                }
                catch (e) {
                    el.cls(['-valid', '+invalid']);
                    const msg = typeof (e) == "string" ? e : e.message;
                    Notifier.error(msg);
                    return;
                }
            }
            this.fire("submit", this, handlerResponse);
        }
        else {
            el.cls(['-valid', '+invalid']);
            const invalid = this.findFirstInvalid();
            if (invalid) {
                invalid.focus();
            }
            this.setInvalid(t('You have errors in your form. The invalid fields are marked.'));
        }
    }
    setInvalid(msg) {
        super.setInvalid(msg);
        Notifier.error(msg);
    }
}
/**
 * When this is set to true, the field will use the values set as their original value, used for resetting and
 * determining if the field was modified.
 */
Form.TRACK_RESET_VALUES = false;
/**
 * Shorthand function to create {@see Form}
 *
 * @param config
 * @param items
 */
export const form = (config, ...items) => createComponent(new Form, config, items);
//# sourceMappingURL=Form.js.map