/**
 * @license https://github.com/Intermesh/goui/blob/main/LICENSE MIT License
 * @copyright Copyright 2023 Intermesh BV
 * @author Merijn Schering <mschering@intermesh.nl>
 */
import { Observable, ObservableEventMap, ObservableListenerOpts } from "../component/index.js";
import { Comparator } from "./Store.js";
import { RouterEventMap } from "../Router";
/**
 * The response of the {@see AbstractDataSource.get()} method
 * @category Data
 */
export interface GetResponse<EntityType extends BaseEntity> {
    /**
     * The list of entities in the order they were requested
     */
    list: EntityType[];
    /**
     * If an ID is not found on the server it will be in this list
     */
    notFound?: EntityID[];
    /**
     * The state of the server
     */
    state?: string;
}
/**
 * @category Data
 */
export interface SetRequest<EntityType> {
    [key: string]: any;
    create: Record<EntityID, Partial<EntityType>>;
    update: Record<EntityID, Partial<EntityType>>;
    destroy: EntityID[];
    ifInstate?: string;
}
/**
 * @category Data
 */
export declare enum CommitErrorType {
    'forbidden' = 0,
    'overQuota' = 1,
    'tooLarge' = 2,
    'rateLimit' = 3,
    'notFound' = 4,
    'invalidPatch' = 5,
    'willDestroy' = 6,
    'invalidProperties' = 7,
    'singleton' = 8,
    'requestTooLarge' = 9,
    'stateMismatch' = 10
}
export type CommitEntityError = Record<EntityID, CommitError>;
/**
 * The base of an entity. It should at lease have an "id" property.
 * @category Data
 */
export interface BaseEntity {
    id: EntityID;
}
/**
 * Default entity
 *
 * Allows any property.
 * @category Data
 */
export interface DefaultEntity extends BaseEntity {
    [key: string]: any;
}
/**
 * @category Data
 */
export interface CommitError {
    type: CommitErrorType;
    description?: string;
}
/**
 * @category Data
 */
export interface Changes {
    created?: EntityID[];
    updated?: EntityID[];
    destroyed?: EntityID[];
    newState?: string;
    oldState?: string;
    hasMoreChanges?: boolean;
}
/**
 * @category Data
 */
export interface CommitResponse<EntityType extends BaseEntity> {
    created?: Record<EntityID, EntityType>;
    updated?: Record<EntityID, EntityType>;
    destroyed?: EntityID[];
    notCreated?: CommitEntityError;
    notUpdated?: CommitEntityError;
    notDestroyed?: CommitEntityError;
    newState?: string;
    oldState?: string;
}
/**
 * @category Data
 */
export type EntityID = string;
/**
 * @category Data
 */
export type QueryFilter = Record<string, any>;
/**
 * @category Data
 */
export interface QueryParams {
    /**
     * The maximum number of ID's to return
     */
    limit?: number;
    /**
     * Start at this position
     */
    position?: number;
    /**
     * Return a "total" number of entities in the response.
     */
    calculateTotal?: boolean;
    /**
     * Sort the results
     */
    sort?: Comparator[];
    filter?: QueryFilter;
}
export interface QueryResponse {
    /**
     * The entity ID's in the correct order
     */
    ids: EntityID[];
    /**
     * If calculateTotal was set to true this will show the total number of results
     */
    total?: number;
    /**
     * The state of the query on the server
     */
    queryState?: string;
}
/**
 * @category Data
 */
export interface DataSourceEventMap<Type extends Observable> extends ObservableEventMap<Type> {
    /**
     * Fires when data changed in the store
     */
    change: (dataSource: Type, changes: Changes) => void;
}
export type dataSourceEntityType<DS> = DS extends AbstractDataSource<infer EntityType> ? EntityType : never;
export interface AbstractDataSource<EntityType extends BaseEntity = DefaultEntity> extends Observable {
    on<K extends keyof DataSourceEventMap<this>>(eventName: K, listener: DataSourceEventMap<this>[K], options?: ObservableListenerOpts): DataSourceEventMap<this>[K];
    un<K extends keyof DataSourceEventMap<this>>(eventName: K, listener: RouterEventMap<this>[K]): boolean;
    fire<K extends keyof DataSourceEventMap<this>>(eventName: K, ...args: Parameters<DataSourceEventMap<this>[K]>): boolean;
}
type SaveData<EntityType extends BaseEntity> = {
    data: Partial<EntityType>;
    resolve: (value: any) => void;
    reject: (reason?: any) => void;
};
interface DestroyData {
    resolve: (value: EntityID) => void;
    reject: (reason?: any) => void;
}
type GetData = {
    resolves: ((value: any | undefined) => void)[];
    rejects: ((reason?: any) => void)[];
};
/**
 * Abstract DataSource class
 *
 * A DataSource collection is a single source of truth for all types of data.
 * When the DataSource changes it fires an event. All components and stores listen to the
 * 'change' event to update themselves. This approach reduces the amount of code that has
 * to be written and maintained.
 *
 * Use a {@see DataSourceStore} in components to list data from datasources.
 * The {@see Form} component can also load from a datasource.
 *
 * @category Data
 */
export declare abstract class AbstractDataSource<EntityType extends BaseEntity = DefaultEntity> extends Observable {
    readonly id: string;
    /**
     * JMAP state
     *
     * @private
     */
    private _state?;
    private readonly delayedCommit;
    private readonly delayedGet;
    private _browserStore?;
    /**
     * Store data in the browser storage so it will persist across sessions
     */
    persist: boolean;
    /**
     * Extra parameters to send to the Foo/set
     */
    commitBaseParams: {};
    /**
     * Extra /set parameters that will reset after commit
     */
    setParams: {
        [key: string]: any;
    };
    /**
     * Get the local server state ID of the store
     * @protected
     */
    getState(): Promise<string | undefined>;
    /**
     * Set's the local server state ID
     *
     * Setting it to undefined will reset the store.
     *
     * @param state
     * @protected
     */
    protected setState(state: string | undefined): Promise<boolean | undefined>;
    clearCache(): Promise<void | null>;
    /**
     * Get the browser storage object to save state to the browser
     * @private
     */
    private get browserStore();
    constructor(id: string);
    protected data: Record<EntityID, EntityType>;
    protected creates: Record<EntityID, SaveData<EntityType>>;
    protected updates: Record<EntityID, SaveData<EntityType>>;
    protected destroys: Record<EntityID, DestroyData>;
    protected getIds: Record<EntityID, GetData>;
    /**
     * Get entities from the store
     *
     * It will return a list of entities ordered by the requested ID's
     *
     * @param ids
     */
    get(ids?: EntityID[]): Promise<GetResponse<EntityType>>;
    protected add(data: EntityType): Promise<EntityType>;
    protected remove(id: EntityID): Promise<string>;
    /**
     * Get a single entity.
     *
     * Multiple calls will be buffered and returned together on the next event loop. This way multiple calls can
     * be joined together in a single HTTP request to the server.
     *
     * @param id
     */
    single(id: EntityID): Promise<EntityType | undefined>;
    private returnGet;
    /**
     * Does the actual getting of entities. First checks if present in this onbject, otherwise it will be requested
     * from the remote source.
     *
     * @protected
     */
    protected doGet(): Promise<void>;
    /**
     * Implements getting entities from a remote source
     *
     * @param ids
     * @protected
     */
    protected abstract internalGet(ids: EntityID[]): Promise<GetResponse<EntityType>>;
    /**
     * Create entity
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param data
     * @param createId The create ID to use when committing this entity to the server
     */
    create(data: Partial<EntityType>, createId?: EntityID): Promise<EntityType>;
    /**
     * Reset the data source.
     *
     * Clears all data and will resync
     */
    reset(): Promise<boolean | undefined>;
    /**
     * Update an entity
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param id
     * @param data
     */
    update(id: EntityID, data: Partial<EntityType>): Promise<EntityType>;
    private _createId;
    private createID;
    /**
     * Destroy an entity
     *
     * Multiple calls will be joined together in a single call on the next event loop
     *
     * @param id
     */
    destroy(id: EntityID): Promise<unknown>;
    /**
     * Ask for confirmation and delete entities by ID
     *
     * @example
     * ```
     * const tbl = this.projectTable!,
     * 	ids = tbl.rowSelection!.selected.map(index => tbl.store.get(index)!.id);
     *
     * const result = await jmapds("Project3")
     * 	.confirmDestroy(ids);
     *
     * if(result != false) {
     * 	btn.parent!.hide();
     * }
     * ```
     * @param ids The ID's to delete
     */
    confirmDestroy(ids: EntityID[]): Promise<false | unknown[]>;
    /**
     * Fetch updates from remote
     */
    updateFromServer(): Promise<void>;
    /**
     * Implements fetching updates from remote
     *
     * @protected
     */
    protected abstract internalRemoteChanges(state: string | undefined): Promise<Changes>;
    /**
     * Commit pending changes to remote
     */
    private commit;
    /**
     * Implements commit (save and destroy) to the remote source
     * @protected
     */
    protected abstract internalCommit(params: SetRequest<EntityType>): Promise<CommitResponse<EntityType>>;
    /**
     * Query the server for a list of entity ID's
     *
     * It takes filters and sort parameters.
     *
     * @link https://jmap.io/spec-core.html#query
     */
    query(params?: QueryParams): Promise<QueryResponse>;
    /**
     * Check's if we are up-to-date with the server and fetches updates if needed.
     *
     * If no state is returned by the data source this function will ignore states and the data source should then
     * always refresh data.
     *
     * @param serverState
     * @param retVal
     * @private
     */
    protected checkState<T>(serverState: string | undefined, retVal: T): Promise<T>;
    /**
     * Handle the query to the remote
     * @param params
     */
    protected abstract internalQuery(params: QueryParams): Promise<QueryResponse>;
}
export {};
//# sourceMappingURL=AbstractDataSource.d.ts.map