<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 ********************************************************************
 */
declare(strict_types=1);

namespace ILIAS\UI;

/**
 * URLBuilderToken
 *
 * This is a token used by Data/URLBuilder to control usage of
 * URL parameters. See URLBuilder interface for more details.
 */
class URLBuilderToken
{
    public const TOKEN_LENGTH = 24;
    private array $namespace;
    private string $name;
    private string $token;

    public function __construct(array $namespace, string $name)
    {
        $this->namespace = $namespace;
        $this->name = $name;
        $this->token = $this->createToken();
    }

    /**
     * Get the token hash value
     */
    public function getToken(): string
    {
        return $this->token;
    }

    /**
     * Get the full name of the token
     * including its namespace
     */
    public function getName(): string
    {
        return implode(URLBuilder::SEPARATOR, $this->namespace) . URLBuilder::SEPARATOR . $this->name;
    }

    public function __toString(): string
    {
        return $this->getName();
    }

    /**
     * Creates a randomized string of length URLBuilderToken::TOKEN_LENGTH
     * generated by the random_bytes() function
     * md5() is used as a fallback on systems that don't support random_bytes()
     */
    private function createToken(): string
    {
        try {
            $token = bin2hex(random_bytes(self::TOKEN_LENGTH / 2));
        } catch (\Throwable $t) {
            $token = md5(uniqid((string) time(), true));
            $token = substr($token, 0, self::TOKEN_LENGTH);
        }
        return $token;
    }

    /**
     * Output the JS equivalent of the token
     * as a string. Used by the URLBuilder renderer.
     */
    public function render(): string
    {
        $namespace = [];
        foreach ($this->namespace as $name) {
            $namespace[] = '"' . $name . '"';
        }
        $output = 'new il.UI.core.URLBuilderToken([' . implode(',', $namespace) . '], "' . $this->name . '", "' . $this->token . '")';
        return $output;
    }
}
