<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

require_once(__DIR__ . "/../../../../../../../vendor/composer/vendor/autoload.php");
require_once(__DIR__ . "/../../../Base.php");
require_once(__DIR__ . "/InputTest.php");
require_once(__DIR__ . "/CommonFieldRendering.php");

use ILIAS\UI\Implementation\Component as I;
use ILIAS\UI\Implementation\Component\SignalGenerator;
use ILIAS\UI\Implementation\Component\Input\InputData;
use ILIAS\Data\Password as PWD;
use ILIAS\UI\Component\Input\Field;
use ILIAS\Data;
use ILIAS\Refinery\Factory as Refinery;

class _PWDInputData implements InputData
{
    /**
     * @ineritdoc
     */
    public function get(string $name): string
    {
        return 'some value';
    }

    /**
     * @inheritcoc
     */
    public function getOr(string $name, $default): string
    {
        return 'some alternative value';
    }

    /**
     * @inheritcoc
     */
    public function has($name): bool
    {
        return true;
    }
}

class PasswordInputTest extends ILIAS_UI_TestBase
{
    use CommonFieldRendering;

    protected DefNamesource $name_source;

    public function setUp(): void
    {
        $this->name_source = new DefNamesource();
    }

    public function testImplementsFactoryInterface(): void
    {
        $f = $this->getFieldFactory();
        $pwd = $f->password("label", "byline");
        $this->assertInstanceOf(\ILIAS\UI\Component\Input\Container\Form\FormInput::class, $pwd);
        $this->assertInstanceOf(Field\Password::class, $pwd);
    }

    public function testRender(): void
    {
        $f = $this->getFieldFactory();
        $label = "label";
        $byline = "byline";
        $name = "name_0";
        $pwd = $f->password($label, $byline)->withNameFrom($this->name_source);
        $expected = $this->getFormWrappedHtml(
            'password-field-input',
            $label,
            '
            <div class="c-field-password">
                <input id="id_1" type="password" name="' . $name . '" autocomplete="off" />
            </div>
            ',
            $byline,
            'id_1'
        );
        $this->assertEquals($expected, $this->render($pwd));
    }

    public function testCommonRendering(): void
    {
        $f = $this->getFieldFactory();
        $label = "label";
        $pwd = $f->password($label)->withNameFrom($this->name_source);

        $this->testWithError($pwd);
        $this->testWithNoByline($pwd);
        $this->testWithRequired($pwd);
        $this->testWithDisabled($pwd);
        $this->testWithAdditionalOnloadCodeRendersId($pwd);
    }

    public function testRenderValue(): void
    {
        $f = $this->getFieldFactory();
        $label = "label";
        $value = "value_0";
        $name = "name_0";
        $pwd = $f->password($label)->withValue($value)->withNameFrom($this->name_source);
        $expected = $this->getFormWrappedHtml(
            'password-field-input',
            $label,
            '
            <div class="c-field-password">
                <input id="id_1" type="password" name="' . $name . '" value="' . $value . '" autocomplete="off" />
            </div>
            ',
            null,
            'id_1'
        );
        $this->assertEquals($expected, $this->render($pwd));
    }

    public function testValueRequired(): void
    {
        $f = $this->getFieldFactory();
        $label = "label";
        $name = "name_0";
        $pwd = $f->password($label)->withNameFrom($this->name_source)->withRequired(true);

        $pwd1 = $pwd->withInput(new DefInputData([$name => "0"]));
        $value1 = $pwd1->getContent();
        $this->assertTrue($value1->isOk());

        $pwd2 = $pwd->withInput(new DefInputData([$name => ""]));
        $value2 = $pwd2->getContent();
        $this->assertTrue($value2->isError());
    }

    public function testValueType(): void
    {
        $f = $this->getFieldFactory();
        $label = "label";
        $pwd = $f->password($label)->withNameFrom($this->name_source);
        $this->assertNull($pwd->getValue());

        $post = new _PWDInputData();
        $pwd = $pwd->withInput($post);
        $this->assertEquals($post->getOr('', ''), $pwd->getValue());
        $this->assertInstanceOf(PWD::class, $pwd->getContent()->value());
    }
}
