<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/
declare(strict_types=1);

class ilDclTextFieldModel extends ilDclBaseFieldModel
{
    /**
     * @param string|int $filter_value
     */
    public function getRecordQueryFilterObject(
        $filter_value = "",
        ?ilDclBaseFieldModel $sort_field = null
    ): ?ilDclRecordQueryObject {
        $join_str
            = "INNER JOIN il_dcl_record_field AS filter_record_field_{$this->getId()} ON (filter_record_field_{$this->getId()}.record_id = record.id AND filter_record_field_{$this->getId()}.field_id = "
            . $this->db->quote($this->getId(), 'integer') . ") ";
        $join_str .= "INNER JOIN il_dcl_stloc{$this->getStorageLocation()}_value AS filter_stloc_{$this->getId()} ON (filter_stloc_{$this->getId()}.record_field_id = filter_record_field_{$this->getId()}.id AND filter_stloc_{$this->getId()}.value LIKE "
            . $this->db->quote("%$filter_value%", 'text') . ") ";

        $sql_obj = new ilDclRecordQueryObject();
        $sql_obj->setJoinStatement($join_str);

        return $sql_obj;
    }

    /**
     * @throws ilDclInputException
     */
    public function checkValidityFromForm(ilPropertyFormGUI &$form, ?int $record_id = null): void
    {
        if ($this->getProperty(ilDclBaseFieldModel::PROP_URL)) {
            $value = [
                'link' => $form->getInput("field_" . $this->getId()),
                'title' => $form->getInput("field_" . $this->getId() . "_title"),
            ];
        } else {
            $value = $form->getInput('field_' . $this->getId());
        }
        $this->checkValidity($value, $record_id);
    }

    public function checkValidity($value, ?int $record_id = null): bool
    {
        if (isset($value['link'])) {
            $value = $value['link'];
            $link = (substr($value, 0, 3) === 'www') ? 'https://' . $value : $value;
            if (!filter_var($link, FILTER_VALIDATE_URL) && !filter_var($link, FILTER_VALIDATE_EMAIL) && $link != '') {
                throw new ilDclInputException(ilDclInputException::NOT_URL);
            }
        }
        $this->checkRegexAndLength($value);
        return parent::checkValidity($value, $record_id);
    }

    protected function areEqual($value_1, $value_2): bool
    {
        return parent::areEqual($value_1['link'] ?? $value_1, $value_2['link'] ?? $value_2);
    }

    /**
     * @inheritDoc
     */
    public function getValidFieldProperties(): array
    {
        return [
            ilDclBaseFieldModel::PROP_LENGTH,
            ilDclBaseFieldModel::PROP_REGEX,
            ilDclBaseFieldModel::PROP_URL,
            ilDclBaseFieldModel::PROP_LINK_DETAIL_PAGE_TEXT,
        ];
    }

    protected function checkRegexAndLength(string $value): void
    {
        if ($this->getProperty(ilDclBaseFieldModel::PROP_LENGTH) < $this->strlen($value)) {
            throw new ilDclInputException(ilDclInputException::LENGTH_EXCEPTION);
        }

        if ($this->getProperty(ilDclBaseFieldModel::PROP_REGEX) != null) {
            $regex = $this->getProperty(ilDclBaseFieldModel::PROP_REGEX);
            if (substr($regex, 0, 1) != "/") {
                $regex = "/" . $regex;
            }
            if (substr($regex, -1) != "/") {
                $regex .= "/";
            }

            try {
                $preg_match = preg_match($regex, $value);
            } catch (ErrorException) {
                throw new ilDclInputException(ilDclInputException::REGEX_CONFIG_EXCEPTION);
            }

            if ($preg_match === false) {
                throw new ilDclInputException(ilDclInputException::REGEX_EXCEPTION);
            }
        }
    }

    public function strlen(string $value, string $encoding = 'UTF-8'): int
    {
        switch (true) {
            case function_exists('mb_strlen'):
                return mb_strlen($value, $encoding);
            case function_exists('iconv_strlen'):
                return iconv_strlen($value, $encoding);
            default:
                return strlen($value);
        }
    }

    public function fillHeaderExcel(ilExcel $worksheet, int &$row, int &$col): void
    {
        parent::fillHeaderExcel($worksheet, $row, $col);
        if ($this->getProperty(ilDclBaseFieldModel::PROP_URL)) {
            $worksheet->setCell($row, $col, $this->getTitle() . '_title');
            $col++;
        }
    }

    public function checkTitlesForImport(array &$titles, array &$import_fields): void
    {
        foreach ($titles as $k => $title) {
            if (!mb_detect_encoding($title, "UTF-8", true) == "UTF-8") {
                $title = mb_convert_encoding($title, 'UTF-8', 'ISO-8859-1');
            }
            if ($title == $this->getTitle()) {
                $import_fields[$k] = $this;
                if ($this->hasProperty(ilDclBaseFieldModel::PROP_URL) && $titles[$k + 1] == $this->getTitle() . '_title') {
                    unset($titles[$k + 1]);
                }
            }
        }
    }
}
