<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

/**
 * Builds the links to join/(re-)start the LearningSequence.
 *
 * @author   Nils Haagen <nils.haagen@concepts-and-training.de>
 */
class ilLSLaunchlinksBuilder
{
    public const PERM_PARTICIPATE = 'participate';
    public const PERM_UNPARTICIPATE = 'unparticipate';

    public const CMD_STANDARD = ilObjLearningSequenceLearnerGUI::CMD_STANDARD;
    public const CMD_EXTRO = ilObjLearningSequenceLearnerGUI::CMD_EXTRO;
    public const CMD_START = ilObjLearningSequenceLearnerGUI::CMD_START;
    public const CMD_VIEW = ilObjLearningSequenceLearnerGUI::CMD_VIEW;
    public const CMD_UNSUBSCRIBE = ilObjLearningSequenceLearnerGUI::CMD_UNSUBSCRIBE;

    public function __construct(
        protected ilLanguage $lng,
        protected ilAccess $access,
        protected ilCtrl $ctrl,
        protected ILIAS\UI\Factory $ui_factory,
        protected int $lso_ref_id,
        protected int $usr_id,
        protected $first_access,
        protected ilLearningSequenceRoles $roles,
        protected ilLSLearnerItemsQueries $ls_items
    ) {
    }

    protected function mayJoin(): bool
    {
        return $this->access->checkAccess(self::PERM_PARTICIPATE, '', $this->lso_ref_id);
    }

    public function currentUserMayUnparticipate(): bool
    {
        return $this->mayUnparticipate();
    }

    protected function mayUnparticipate(): bool
    {
        return $this->access->checkAccess(self::PERM_UNPARTICIPATE, '', $this->lso_ref_id);
    }

    protected function isMember(): bool
    {
        return $this->roles->isMember($this->usr_id);
    }

    protected function hasCompleted(): bool
    {
        return $this->roles->isCompletedByUser($this->usr_id);
    }

    protected function getLink(string $cmd): string
    {
        return $this->ctrl->getLinkTargetByClass('ilObjLearningSequenceLearnerGUI', $cmd);
    }

    public function getLinks(): array
    {
        $cmd = $this->ctrl->getCmd();
        $links = [];

        if (!$this->isMember() && $this->mayJoin()) {
            if (!$this->ls_items->hasItems()) {
                return [];
            }
            return [[
                $this->lng->txt("lso_player_start"),
                $this->getLink(self::CMD_START),
                true
            ]];
        }

        if (!$this->hasCompleted() && $this->ls_items->hasItems()) {
            $label = "lso_player_resume";
            if ($this->first_access === -1) {
                $label = "lso_player_start";
            }
            $links[] = [
                $this->lng->txt($label),
                $this->getLink(self::CMD_VIEW),
                true
            ];
        } else {
            if ($this->ls_items->hasItems()) {
                $links[] = [
                    $this->lng->txt("lso_player_review"),
                    $this->getLink(self::CMD_VIEW),
                    true
                ];
            }

            if ($cmd === self::CMD_STANDARD) {
                $links[] = [
                    $this->lng->txt("lso_player_extro"),
                    $this->getLink(self::CMD_EXTRO),
                    false
                ];
            }
            if ($cmd === self::CMD_EXTRO) {
                $links[] = [
                    $this->lng->txt("lso_player_abstract"),
                    $this->getLink(self::CMD_STANDARD),
                    false
                ];
            }
        }

        if ($this->mayUnparticipate()) {
            $links[] = [
                $this->lng->txt("unparticipate"),
                $this->getLink(self::CMD_UNSUBSCRIBE),
                false
            ];
        }

        return $links;
    }

    public function getLaunchbuttonsComponent(): array
    {
        $buttons = [];
        foreach ($this->getLinks() as $idx => $entry) {
            list($label, $link, $primary) = $entry;
            if ($primary) {
                $buttons[] = $this->ui_factory->button()->primary($label, $link);
            } else {
                $buttons[] = $this->ui_factory->button()->standard($label, $link);
            }
        }
        return $buttons;
    }
}
