<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

namespace ILIAS\UI\Component\Dropzone\File;

use ILIAS\UI\Component\Input\Field\File as FileInput;
use ILIAS\UI\Component\Component;
use ILIAS\UI\Component\Input\Container\Form\FormInput;

/**
 * Describes a factory for file dropzones.
 * @author  nmaerchy <nm@studer-raimann.ch>
 * @author  Stefan Wanzenried <sw@studer-raimann.ch>
 * @author  Thibeau Fuhrer <thibeau@sr.solutions>
 */
interface Factory
{
    /**
     * ---
     * description:
     *   purpose: >
     *      The standard dropzone is used to drop files dragged from outside
     *      the browser window. The dropped files are presented to the user and
     *      can be uploaded to the server.
     *   composition: >
     *      Standard dropzones consist of a visible area where files can
     *      be dropped. They MUST contain a message explaining that it is possible to
     *      drop files inside. The dropped files are presented to the user in a roundtrip
     *      modal, which contains a file input.
     *   effect: >
     *      A standard dropzone is highlighted when the user is dragging files
     *      over the dropzone. After dropping, the dropped files are presented
     *      to the user with some meta information of the files such the file name
     *      and file size.
     *   rivals:
     *      Rival 1: >
     *          A wrapper dropzone can hold other ILIAS UI components instead of
     *          a message.
     *      Rival 2: >
     *          A file-input can be used instead of this component if other values
     *          have to be submitted at the same time.
     * rules:
     *   accessibility:
     *     1: >
     *        Standard dropzones MUST offer the possibility to select files
     *        manually from the computer.
     * ---
     * @param string    $title
     * @param string    $message
     * @param string    $post_url
     * @param FileInput $file_input
     * @param FormInput $additional_input
     * @return \ILIAS\UI\Component\Dropzone\File\Standard
     */
    public function standard(
        string $title,
        string $message,
        string $post_url,
        FileInput $file_input,
        ?FormInput $additional_input = null,
    ): Standard;

    /**
     * ---
     * description:
     *   purpose: >
     *      A wrapper dropzone is used to display other ILIAS UI components
     *      inside it. In contrast to the standard dropzone, the wrapper
     *      dropzone is not visible by default. Only the wrapped components are
     *      visible. Any wrapper dropzone gets highlighted once the user is dragging
     *      files over the browser window. Thus, a user needs to have the knowledge
     *      that there are wrapper dropzones present. They can be introduced to offer
     *      additional approaches to complete some workflow more conveniently.
     *      Especially in situation where space is scarce such as appointments
     *      in the calendar.
     *   composition: >
     *      A wrapper dropzone contains one or multiple ILIAS UI components.
     *      A roundtrip modal is used to present the dropped files and to initialize
     *      the upload process with a file input.
     *   effect: >
     *      All wrapper dropzones on the page are highlighted when the user
     *      dragging files over the browser window. After dropping the files, the
     *      roundtrip modal is opened showing all files. The modal contains a button
     *      to start the upload process.
     *   rivals:
     *      Rival 1: >
     *          A standard dropzone displays a message instead of other
     *          ILIAS UI components.
     * rules:
     *   usage:
     *     1: >
     *        Most pages SHOULD NOT contain a wrapper dropzone. Whenever you want to introduce a
     *        new usage of the Wrapper-Dropzone, propose it to the Jour Fixe.
     *     2: Wrapper dropzones MUST contain one or more ILIAS UI components.
     *     3: Wrapper dropzones MUST NOT contain any other file dropzones.
     *     4: Wrapper dropzones MUST NOT be used in modals.
     * ---
     * @param string                $title
     * @param string                $post_url
     * @param Component|Component[] $content
     * @param FileInput             $file_input
     * @param FormInput             $additional_input
     * @return \ILIAS\UI\Component\Dropzone\File\Wrapper
     */
    public function wrapper(
        string $title,
        string $post_url,
        $content,
        FileInput $file_input,
        ?FormInput $additional_input = null,
    ): Wrapper;
}
