<?php
/**
 * Kkart Template
 *
 * Functions for the templating system.
 *
 * @package  Kkart\Functions
 * @version  2.5.0
 */

use Automattic\Jetpack\Constants;

defined( 'ABSPATH' ) || exit;

/**
 * Handle redirects before content is output - hooked into template_redirect so is_page works.
 */
function kkart_template_redirect() {
	global $wp_query, $wp;

	// phpcs:disable WordPress.Security.NonceVerification.Recommended
	// When default permalinks are enabled, redirect shop page to post type archive url.
	if ( ! empty( $_GET['page_id'] ) && '' === get_option( 'permalink_structure' ) && kkart_get_page_id( 'shop' ) === absint( $_GET['page_id'] ) && get_post_type_archive_link( 'product' ) ) {
		wp_safe_redirect( get_post_type_archive_link( 'product' ) );
		exit;
	}
	// phpcs:enable WordPress.Security.NonceVerification.Recommended

	// When on the checkout with an empty cart, redirect to cart page.
	if ( is_page( kkart_get_page_id( 'checkout' ) ) && kkart_get_page_id( 'checkout' ) !== kkart_get_page_id( 'cart' ) && KKART()->cart->is_empty() && empty( $wp->query_vars['order-pay'] ) && ! isset( $wp->query_vars['order-received'] ) && ! is_customize_preview() && apply_filters( 'kkart_checkout_redirect_empty_cart', true )  && !pagelayer_is_live()) {
		kkart_add_notice( __( 'Checkout is not available whilst your cart is empty.', 'kkart' ), 'notice' );
		wp_safe_redirect( kkart_get_cart_url() );
		exit;

	}
	
	// When on the order pay page without id, redirect to cart page.
	if ( is_page( kkart_get_page_id( 'order_pay' ) ) && !isset( $_GET['key'], $_GET['order_id'])  && ! is_customize_preview() && apply_filters( 'kkart_order_pay_redirect', true )  && !pagelayer_is_live()) {
		wp_safe_redirect( kkart_get_cart_url() );
		exit;
	}
	
	// When on the order received page without id, redirect to cart page.
	if ( is_page( kkart_get_page_id( 'order_received' ) ) && !isset( $_GET['key'], $_GET['order_id'])  && ! is_customize_preview() && apply_filters( 'kkart_order_received_redirect', true )  && !pagelayer_is_live()) {
		wp_safe_redirect( kkart_get_cart_url() );
		exit;
	}

	// Logout.
	if ( isset( $wp->query_vars['customer-logout'] ) && ! empty( $_REQUEST['_wpnonce'] ) && wp_verify_nonce( sanitize_key( $_REQUEST['_wpnonce'] ), 'customer-logout' ) ) {
		wp_safe_redirect( str_replace( '&amp;', '&', wp_logout_url( kkart_get_page_permalink( 'myaccount' ) ) ) );
		exit;
	}

	// Redirect to the correct logout endpoint.
	if ( isset( $wp->query_vars['customer-logout'] ) && 'true' === $wp->query_vars['customer-logout'] ) {
		wp_safe_redirect( esc_url_raw( kkart_get_account_endpoint_url( 'customer-logout' ) ) );
		exit;
	}

	// Trigger 404 if trying to access an endpoint on wrong page.
	if ( is_kkart_endpoint_url() && ! is_account_page() && ! is_checkout() && apply_filters( 'kkart_account_endpoint_page_not_found', true ) ) {
		$wp_query->set_404();
		status_header( 404 );
		include get_query_template( '404' );
		exit;
	}

	// Redirect to the product page if we have a single product.
	if ( is_search() && is_post_type_archive( 'product' ) && apply_filters( 'kkart_redirect_single_search_result', true ) && 1 === absint( $wp_query->found_posts ) ) {
		$product = kkart_get_product( $wp_query->post );

		if ( $product && $product->is_visible() ) {
			wp_safe_redirect( get_permalink( $product->get_id() ), 302 );
			exit;
		}
	}

	// Ensure gateways and shipping methods are loaded early.
	if ( is_add_payment_method_page() || is_checkout() ) {
		// Buffer the checkout page.
		ob_start();

		// Ensure gateways and shipping methods are loaded early.
		KKART()->payment_gateways();
		KKART()->shipping();
	}
}
add_action( 'template_redirect', 'kkart_template_redirect' );

/**
 * When loading sensitive checkout or account pages, send a HTTP header to limit rendering of pages to same origin iframes for security reasons.
 *
 * Can be disabled with: remove_action( 'template_redirect', 'kkart_send_frame_options_header' );
 *
 * @since  2.3.10
 */
function kkart_send_frame_options_header() {

	if ( ( is_checkout() || is_account_page() ) && ! is_customize_preview() ) {
		send_frame_options_header();
	}
}
add_action( 'template_redirect', 'kkart_send_frame_options_header' );

/**
 * No index our endpoints.
 * Prevent indexing pages like order-received.
 *
 * @since 2.5.3
 */
function kkart_prevent_endpoint_indexing() {
	// phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.PHP.NoSilencedErrors.Discouraged
	if ( is_kkart_endpoint_url() || isset( $_GET['download_file'] ) ) {
		@header( 'X-Robots-Tag: noindex' );
	}
	// phpcs:enable WordPress.Security.NonceVerification.Recommended, WordPress.PHP.NoSilencedErrors.Discouraged
}
add_action( 'template_redirect', 'kkart_prevent_endpoint_indexing' );

/**
 * Remove adjacent_posts_rel_link_wp_head - pointless for products.
 *
 * @since 3.0.0
 */
function kkart_prevent_adjacent_posts_rel_link_wp_head() {
	if ( is_singular( 'product' ) ) {
		remove_action( 'wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0 );
	}
}
add_action( 'template_redirect', 'kkart_prevent_adjacent_posts_rel_link_wp_head' );

/**
 * Show the gallery if JS is disabled.
 *
 * @since 3.0.6
 */
function kkart_gallery_noscript() {
	?>
	<noscript><style>.kkart-product-gallery{ opacity: 1 !important; }</style></noscript>
	<?php
}
add_action( 'wp_head', 'kkart_gallery_noscript' );

/**
 * When the_post is called, put product data into a global.
 *
 * @param mixed $post Post Object.
 * @return KKART_Product
 */
function kkart_setup_product_data( $post ) {
	unset( $GLOBALS['product'] );

	if ( is_int( $post ) ) {
		$post = get_post( $post );
	}

	if ( empty( $post->post_type ) || ! in_array( $post->post_type, array( 'product', 'product_variation' ), true ) ) {
		return;
	}

	$GLOBALS['product'] = kkart_get_product( $post );

	return $GLOBALS['product'];
}
add_action( 'the_post', 'kkart_setup_product_data' );

/**
 * Sets up the kkart_loop global from the passed args or from the main query.
 *
 * @since 3.3.0
 * @param array $args Args to pass into the global.
 */
function kkart_setup_loop( $args = array() ) {
	$default_args = array(
		'loop'         => 0,
		'columns'      => kkart_get_default_products_per_row(),
		'name'         => '',
		'is_shortcode' => false,
		'is_paginated' => true,
		'is_search'    => false,
		'is_filtered'  => false,
		'total'        => 0,
		'total_pages'  => 0,
		'per_page'     => 0,
		'current_page' => 1,
	);

	// If this is a main KKART query, use global args as defaults.
	if ( $GLOBALS['wp_query']->get( 'kkart_query' ) ) {
		$default_args = array_merge(
			$default_args,
			array(
				'is_search'    => $GLOBALS['wp_query']->is_search(),
				'is_filtered'  => is_filtered(),
				'total'        => $GLOBALS['wp_query']->found_posts,
				'total_pages'  => $GLOBALS['wp_query']->max_num_pages,
				'per_page'     => $GLOBALS['wp_query']->get( 'posts_per_page' ),
				'current_page' => max( 1, $GLOBALS['wp_query']->get( 'paged', 1 ) ),
			)
		);
	}

	// Merge any existing values.
	if ( isset( $GLOBALS['kkart_loop'] ) ) {
		$default_args = array_merge( $default_args, $GLOBALS['kkart_loop'] );
	}

	$GLOBALS['kkart_loop'] = wp_parse_args( $args, $default_args );
}
add_action( 'kkart_before_shop_loop', 'kkart_setup_loop' );

/**
 * Resets the kkart_loop global.
 *
 * @since 3.3.0
 */
function kkart_reset_loop() {
	unset( $GLOBALS['kkart_loop'] );
}
add_action( 'kkart_after_shop_loop', 'kkart_reset_loop', 999 );

/**
 * Gets a property from the kkart_loop global.
 *
 * @since 3.3.0
 * @param string $prop Prop to get.
 * @param string $default Default if the prop does not exist.
 * @return mixed
 */
function kkart_get_loop_prop( $prop, $default = '' ) {
	kkart_setup_loop(); // Ensure shop loop is setup.

	return isset( $GLOBALS['kkart_loop'], $GLOBALS['kkart_loop'][ $prop ] ) ? $GLOBALS['kkart_loop'][ $prop ] : $default;
}

/**
 * Sets a property in the kkart_loop global.
 *
 * @since 3.3.0
 * @param string $prop Prop to set.
 * @param string $value Value to set.
 */
function kkart_set_loop_prop( $prop, $value = '' ) {
	if ( ! isset( $GLOBALS['kkart_loop'] ) ) {
		kkart_setup_loop();
	}
	$GLOBALS['kkart_loop'][ $prop ] = $value;
}

/**
 * Set the current visbility for a product in the kkart_loop global.
 *
 * @since 4.4.0
 * @param int  $product_id Product it to cache visbiility for.
 * @param bool $value The poduct visibility value to cache.
 */
function kkart_set_loop_product_visibility( $product_id, $value ) {
	kkart_set_loop_prop( "product_visibility_$product_id", $value );
}

/**
 * Gets the cached current visibility for a product from the kkart_loop global.
 *
 * @since 4.4.0
 * @param int $product_id Product id to get the cached visibility for.
 *
 * @return bool|null The cached product visibility, or null if on visibility has been cached for that product.
 */
function kkart_get_loop_product_visibility( $product_id ) {
	return kkart_get_loop_prop( "product_visibility_$product_id", null );
}

/**
 * Should the Kkart loop be displayed?
 *
 * This will return true if we have posts (products) or if we have subcats to display.
 *
 * @since 3.4.0
 * @return bool
 */
function kkart_product_loop() {
	return have_posts() || 'products' !== kkart_get_loop_display_mode();
}

/**
 * Output generator tag to aid debugging.
 *
 * @param string $gen Generator.
 * @param string $type Type.
 * @return string
 */
function kkart_generator_tag( $gen, $type ) {
	$version = Constants::get_constant( 'KKART_VERSION' );

	switch ( $type ) {
		case 'html':
			$gen .= "\n" . '<meta name="generator" content="Kkart ' . esc_attr( $version ) . '">';
			break;
		case 'xhtml':
			$gen .= "\n" . '<meta name="generator" content="Kkart ' . esc_attr( $version ) . '" />';
			break;
	}
	return $gen;
}

/**
 * Add body classes for KKART pages.
 *
 * @param  array $classes Body Classes.
 * @return array
 */
function kkart_body_class( $classes ) {
	$classes = (array) $classes;

	if ( is_kkart() ) {

		$classes[] = 'kkart';
		$classes[] = 'kkart-page';

	} elseif ( is_checkout() ) {

		$classes[] = 'kkart-checkout';
		$classes[] = 'kkart-page';

	} elseif ( is_cart() ) {

		$classes[] = 'kkart-cart';
		$classes[] = 'kkart-page';

	} elseif ( is_account_page() ) {

		$classes[] = 'kkart-account';
		$classes[] = 'kkart-page';

	}

	if ( is_store_notice_showing() ) {
		$classes[] = 'kkart-demo-store';
	}

	foreach ( KKART()->query->get_query_vars() as $key => $value ) {
		if ( is_kkart_endpoint_url( $key ) ) {
			$classes[] = 'kkart-' . sanitize_html_class( $key );
		}
	}

	$classes[] = 'kkart-no-js';

	add_action( 'wp_footer', 'kkart_no_js' );

	return array_unique( $classes );
}

/**
 * NO JS handling.
 *
 * @since 3.4.0
 */
function kkart_no_js() {
	?>
	<script type="text/javascript">
		(function () {
			var c = document.body.className;
			c = c.replace(/kkart-no-js/, 'kkart-js');
			document.body.className = c;
		})()
	</script>
	<?php
}

/**
 * Display the classes for the product cat div.
 *
 * @since 2.4.0
 * @param string|array $class One or more classes to add to the class list.
 * @param object       $category object Optional.
 */
function kkart_product_cat_class( $class = '', $category = null ) {
	// Separates classes with a single space, collates classes for post DIV.
	echo 'class="' . esc_attr( join( ' ', kkart_get_product_cat_class( $class, $category ) ) ) . '"';
}

/**
 * Get the default columns setting - this is how many products will be shown per row in loops.
 *
 * @since 3.3.0
 * @return int
 */
function kkart_get_default_products_per_row() {
	$columns      = get_option( 'kkart_catalog_columns', 4 );
	$product_grid = kkart_get_theme_support( 'product_grid' );
	$min_columns  = isset( $product_grid['min_columns'] ) ? absint( $product_grid['min_columns'] ) : 0;
	$max_columns  = isset( $product_grid['max_columns'] ) ? absint( $product_grid['max_columns'] ) : 0;

	if ( $min_columns && $columns < $min_columns ) {
		$columns = $min_columns;
		update_option( 'kkart_catalog_columns', $columns );
	} elseif ( $max_columns && $columns > $max_columns ) {
		$columns = $max_columns;
		update_option( 'kkart_catalog_columns', $columns );
	}

	if ( has_filter( 'loop_shop_columns' ) ) { // Legacy filter handling.
		$columns = apply_filters( 'loop_shop_columns', $columns );
	}

	$columns = absint( $columns );

	return max( 1, $columns );
}

/**
 * Get the default rows setting - this is how many product rows will be shown in loops.
 *
 * @since 3.3.0
 * @return int
 */
function kkart_get_default_product_rows_per_page() {
	$rows         = absint( get_option( 'kkart_catalog_rows', 4 ) );
	$product_grid = kkart_get_theme_support( 'product_grid' );
	$min_rows     = isset( $product_grid['min_rows'] ) ? absint( $product_grid['min_rows'] ) : 0;
	$max_rows     = isset( $product_grid['max_rows'] ) ? absint( $product_grid['max_rows'] ) : 0;

	if ( $min_rows && $rows < $min_rows ) {
		$rows = $min_rows;
		update_option( 'kkart_catalog_rows', $rows );
	} elseif ( $max_rows && $rows > $max_rows ) {
		$rows = $max_rows;
		update_option( 'kkart_catalog_rows', $rows );
	}

	return $rows;
}

/**
 * Reset the product grid settings when a new theme is activated.
 *
 * @since 3.3.0
 */
function kkart_reset_product_grid_settings() {
	$product_grid = kkart_get_theme_support( 'product_grid' );

	if ( ! empty( $product_grid['default_rows'] ) ) {
		update_option( 'kkart_catalog_rows', absint( $product_grid['default_rows'] ) );
	}

	if ( ! empty( $product_grid['default_columns'] ) ) {
		update_option( 'kkart_catalog_columns', absint( $product_grid['default_columns'] ) );
	}

	wp_cache_flush(); // Flush any caches which could impact settings or templates.
}
add_action( 'after_switch_theme', 'kkart_reset_product_grid_settings' );

/**
 * Get classname for kkart loops.
 *
 * @since 2.6.0
 * @return string
 */
function kkart_get_loop_class() {
	$loop_index = kkart_get_loop_prop( 'loop', 0 );
	$columns    = absint( max( 1, kkart_get_loop_prop( 'columns', kkart_get_default_products_per_row() ) ) );

	$loop_index ++;
	kkart_set_loop_prop( 'loop', $loop_index );

	if ( 0 === ( $loop_index - 1 ) % $columns || 1 === $columns ) {
		return 'first';
	}

	if ( 0 === $loop_index % $columns ) {
		return 'last';
	}

	return '';
}


/**
 * Get the classes for the product cat div.
 *
 * @since 2.4.0
 *
 * @param string|array $class One or more classes to add to the class list.
 * @param object       $category object Optional.
 *
 * @return array
 */
function kkart_get_product_cat_class( $class = '', $category = null ) {
	$classes   = is_array( $class ) ? $class : array_map( 'trim', explode( ' ', $class ) );
	$classes[] = 'product-category';
	$classes[] = 'product';
	$classes[] = kkart_get_loop_class();
	$classes   = apply_filters( 'product_cat_class', $classes, $class, $category );

	return array_unique( array_filter( $classes ) );
}

/**
 * Adds extra post classes for products via the WordPress post_class hook, if used.
 *
 * Note: For performance reasons we instead recommend using kkart_product_class/kkart_get_product_class instead.
 *
 * @since 2.1.0
 * @param array        $classes Current classes.
 * @param string|array $class Additional class.
 * @param int          $post_id Post ID.
 * @return array
 */
function kkart_product_post_class( $classes, $class = '', $post_id = 0 ) {
	if ( ! $post_id || ! in_array( get_post_type( $post_id ), array( 'product', 'product_variation' ), true ) ) {
		return $classes;
	}

	$product = kkart_get_product( $post_id );

	if ( ! $product ) {
		return $classes;
	}

	$classes[] = 'product';
	$classes[] = kkart_get_loop_class();
	$classes[] = $product->get_stock_status();

	if ( $product->is_on_sale() ) {
		$classes[] = 'sale';
	}
	if ( $product->is_featured() ) {
		$classes[] = 'featured';
	}
	if ( $product->is_downloadable() ) {
		$classes[] = 'downloadable';
	}
	if ( $product->is_virtual() ) {
		$classes[] = 'virtual';
	}
	if ( $product->is_sold_individually() ) {
		$classes[] = 'sold-individually';
	}
	if ( $product->is_taxable() ) {
		$classes[] = 'taxable';
	}
	if ( $product->is_shipping_taxable() ) {
		$classes[] = 'shipping-taxable';
	}
	if ( $product->is_purchasable() ) {
		$classes[] = 'purchasable';
	}
	if ( $product->get_type() ) {
		$classes[] = 'product-type-' . $product->get_type();
	}
	if ( $product->is_type( 'variable' ) && $product->get_default_attributes() ) {
		$classes[] = 'has-default-attributes';
	}

	$key = array_search( 'hentry', $classes, true );
	if ( false !== $key ) {
		unset( $classes[ $key ] );
	}

	return $classes;
}

/**
 * Get product taxonomy HTML classes.
 *
 * @since 3.4.0
 * @param array  $term_ids Array of terms IDs or objects.
 * @param string $taxonomy Taxonomy.
 * @return array
 */
function kkart_get_product_taxonomy_class( $term_ids, $taxonomy ) {
	$classes = array();

	foreach ( $term_ids as $term_id ) {
		$term = get_term( $term_id, $taxonomy );

		if ( empty( $term->slug ) ) {
			continue;
		}

		$term_class = sanitize_html_class( $term->slug, $term->term_id );
		if ( is_numeric( $term_class ) || ! trim( $term_class, '-' ) ) {
			$term_class = $term->term_id;
		}

		// 'post_tag' uses the 'tag' prefix for backward compatibility.
		if ( 'post_tag' === $taxonomy ) {
			$classes[] = 'tag-' . $term_class;
		} else {
			$classes[] = sanitize_html_class( $taxonomy . '-' . $term_class, $taxonomy . '-' . $term->term_id );
		}
	}

	return $classes;
}

/**
 * Retrieves the classes for the post div as an array.
 *
 * This method was modified from WordPress's get_post_class() to allow the removal of taxonomies
 * (for performance reasons). Previously kkart_product_post_class was hooked into post_class. @since 3.6.0
 *
 * @since 3.4.0
 * @param string|array           $class      One or more classes to add to the class list.
 * @param int|WP_Post|KKART_Product $product Product ID or product object.
 * @return array
 */
function kkart_get_product_class( $class = '', $product = null ) {
	if ( is_null( $product ) && ! empty( $GLOBALS['product'] ) ) {
		// Product was null so pull from global.
		$product = $GLOBALS['product'];
	}

	if ( $product && ! is_a( $product, 'KKART_Product' ) ) {
		// Make sure we have a valid product, or set to false.
		$product = kkart_get_product( $product );
	}

	if ( $class ) {
		if ( ! is_array( $class ) ) {
			$class = preg_split( '#\s+#', $class );
		}
	} else {
		$class = array();
	}

	$post_classes = array_map( 'esc_attr', $class );

	if ( ! $product ) {
		return $post_classes;
	}

	// Run through the post_class hook so 3rd parties using this previously can still append classes.
	// Note, to change classes you will need to use the newer kkart_post_class filter.
	// @internal This removes the kkart_product_post_class filter so classes are not duplicated.
	$filtered = has_filter( 'post_class', 'kkart_product_post_class' );

	if ( $filtered ) {
		remove_filter( 'post_class', 'kkart_product_post_class', 20 );
	}

	$post_classes = apply_filters( 'post_class', $post_classes, $class, $product->get_id() );

	if ( $filtered ) {
		add_filter( 'post_class', 'kkart_product_post_class', 20, 3 );
	}

	$classes = array_merge(
		$post_classes,
		array(
			'product',
			'type-product',
			'post-' . $product->get_id(),
			'status-' . $product->get_status(),
			kkart_get_loop_class(),
			$product->get_stock_status(),
		),
		kkart_get_product_taxonomy_class( $product->get_category_ids(), 'product_cat' ),
		kkart_get_product_taxonomy_class( $product->get_tag_ids(), 'product_tag' )
	);

	if ( $product->get_image_id() ) {
		$classes[] = 'has-post-thumbnail';
	}
	if ( $product->get_post_password() ) {
		$classes[] = post_password_required( $product->get_id() ) ? 'post-password-required' : 'post-password-protected';
	}
	if ( $product->is_on_sale() ) {
		$classes[] = 'sale';
	}
	if ( $product->is_featured() ) {
		$classes[] = 'featured';
	}
	if ( $product->is_downloadable() ) {
		$classes[] = 'downloadable';
	}
	if ( $product->is_virtual() ) {
		$classes[] = 'virtual';
	}
	if ( $product->is_sold_individually() ) {
		$classes[] = 'sold-individually';
	}
	if ( $product->is_taxable() ) {
		$classes[] = 'taxable';
	}
	if ( $product->is_shipping_taxable() ) {
		$classes[] = 'shipping-taxable';
	}
	if ( $product->is_purchasable() ) {
		$classes[] = 'purchasable';
	}
	if ( $product->get_type() ) {
		$classes[] = 'product-type-' . $product->get_type();
	}
	if ( $product->is_type( 'variable' ) && $product->get_default_attributes() ) {
		$classes[] = 'has-default-attributes';
	}

	// Include attributes and any extra taxonomies only if enabled via the hook - this is a performance issue.
	if ( apply_filters( 'kkart_get_product_class_include_taxonomies', false ) ) {
		$taxonomies = get_taxonomies( array( 'public' => true ) );
		$type       = 'variation' === $product->get_type() ? 'product_variation' : 'product';
		foreach ( (array) $taxonomies as $taxonomy ) {
			if ( is_object_in_taxonomy( $type, $taxonomy ) && ! in_array( $taxonomy, array( 'product_cat', 'product_tag' ), true ) ) {
				$classes = array_merge( $classes, kkart_get_product_taxonomy_class( (array) get_the_terms( $product->get_id(), $taxonomy ), $taxonomy ) );
			}
		}
	}

	/**
	 * Kkart Post Class filter.
	 *
	 * @since 3.6.2
	 * @param array      $classes Array of CSS classes.
	 * @param KKART_Product $product Product object.
	 */
	$classes = apply_filters( 'kkart_post_class', $classes, $product );

	return array_map( 'esc_attr', array_unique( array_filter( $classes ) ) );
}

/**
 * Display the classes for the product div.
 *
 * @since 3.4.0
 * @param string|array           $class      One or more classes to add to the class list.
 * @param int|WP_Post|KKART_Product $product_id Product ID or product object.
 */
function kkart_product_class( $class = '', $product_id = null ) {
	echo 'class="' . esc_attr( implode( ' ', kkart_get_product_class( $class, $product_id ) ) ) . '"';
}

/**
 * Outputs hidden form inputs for each query string variable.
 *
 * @since 3.0.0
 * @param string|array $values Name value pairs, or a URL to parse.
 * @param array        $exclude Keys to exclude.
 * @param string       $current_key Current key we are outputting.
 * @param bool         $return Whether to return.
 * @return string
 */
function kkart_query_string_form_fields( $values = null, $exclude = array(), $current_key = '', $return = false ) {
	if ( is_null( $values ) ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$values = $_GET;
	} elseif ( is_string( $values ) ) {
		$url_parts = wp_parse_url( $values );
		$values    = array();

		if ( ! empty( $url_parts['query'] ) ) {
			// This is to preserve full-stops, pluses and spaces in the query string when ran through parse_str.
			$replace_chars = array(
				'.' => '{dot}',
				'+' => '{plus}',
			);

			$query_string = str_replace( array_keys( $replace_chars ), array_values( $replace_chars ), $url_parts['query'] );

			// Parse the string.
			parse_str( $query_string, $parsed_query_string );

			// Convert the full-stops, pluses and spaces back and add to values array.
			foreach ( $parsed_query_string as $key => $value ) {
				$new_key            = str_replace( array_values( $replace_chars ), array_keys( $replace_chars ), $key );
				$new_value          = str_replace( array_values( $replace_chars ), array_keys( $replace_chars ), $value );
				$values[ $new_key ] = $new_value;
			}
		}
	}
	$html = '';

	foreach ( $values as $key => $value ) {
		if ( in_array( $key, $exclude, true ) ) {
			continue;
		}
		if ( $current_key ) {
			$key = $current_key . '[' . $key . ']';
		}
		if ( is_array( $value ) ) {
			$html .= kkart_query_string_form_fields( $value, $exclude, $key, true );
		} else {
			$html .= '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( wp_unslash( $value ) ) . '" />';
		}
	}

	if ( $return ) {
		return $html;
	}

	echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}

/**
 * Get the terms and conditons page ID.
 *
 * @since 3.4.0
 * @return int
 */
function kkart_terms_and_conditions_page_id() {
	$page_id = kkart_get_page_id( 'terms' );
	return apply_filters( 'kkart_terms_and_conditions_page_id', 0 < $page_id ? absint( $page_id ) : 0 );
}

/**
 * Get the privacy policy page ID.
 *
 * @since 3.4.0
 * @return int
 */
function kkart_privacy_policy_page_id() {
	$page_id = get_option( 'wp_page_for_privacy_policy', 0 );
	return apply_filters( 'kkart_privacy_policy_page_id', 0 < $page_id ? absint( $page_id ) : 0 );
}

/**
 * See if the checkbox is enabled or not based on the existance of the terms page and checkbox text.
 *
 * @since 3.4.0
 * @return bool
 */
function kkart_terms_and_conditions_checkbox_enabled() {
	$page_id = kkart_terms_and_conditions_page_id();
	$page    = $page_id ? get_post( $page_id ) : false;
	return $page && kkart_get_terms_and_conditions_checkbox_text();
}

/**
 * Get the terms and conditons checkbox text, if set.
 *
 * @since 3.4.0
 * @return string
 */
function kkart_get_terms_and_conditions_checkbox_text() {
	/* translators: %s terms and conditions page name and link */
	return trim( apply_filters( 'kkart_get_terms_and_conditions_checkbox_text', get_option( 'kkart_checkout_terms_and_conditions_checkbox_text', sprintf( __( 'I have read and agree to the website %s', 'kkart' ), '[terms]' ) ) ) );
}

/**
 * Get the privacy policy text, if set.
 *
 * @since 3.4.0
 * @param string $type Type of policy to load. Valid values include registration and checkout.
 * @return string
 */
function kkart_get_privacy_policy_text( $type = '' ) {
	$text = '';

	switch ( $type ) {
		case 'checkout':
			/* translators: %s privacy policy page name and link */
			$text = get_option( 'kkart_checkout_privacy_policy_text', sprintf( __( 'Your personal data will be used to process your order, support your experience throughout this website, and for other purposes described in our %s.', 'kkart' ), '[privacy_policy]' ) );
			break;
		case 'registration':
			/* translators: %s privacy policy page name and link */
			$text = get_option( 'kkart_registration_privacy_policy_text', sprintf( __( 'Your personal data will be used to support your experience throughout this website, to manage access to your account, and for other purposes described in our %s.', 'kkart' ), '[privacy_policy]' ) );
			break;
	}

	return trim( apply_filters( 'kkart_get_privacy_policy_text', $text, $type ) );
}

/**
 * Output t&c checkbox text.
 *
 * @since 3.4.0
 */
function kkart_terms_and_conditions_checkbox_text() {
	$text = kkart_get_terms_and_conditions_checkbox_text();

	if ( ! $text ) {
		return;
	}

	echo wp_kses_post( kkart_replace_policy_page_link_placeholders( $text ) );
}

/**
 * Output t&c page's content (if set). The page can be set from checkout settings.
 *
 * @since 3.4.0
 */
function kkart_terms_and_conditions_page_content() {
	$terms_page_id = kkart_terms_and_conditions_page_id();

	if ( ! $terms_page_id ) {
		return;
	}

	$page = get_post( $terms_page_id );

	if ( $page && 'publish' === $page->post_status && $page->post_content && ! has_shortcode( $page->post_content, 'kkart_checkout' ) ) {
		echo '<div class="kkart-terms-and-conditions" style="display: none; max-height: 200px; overflow: auto;">' . wp_kses_post( kkart_format_content( $page->post_content ) ) . '</div>';
	}
}

/**
 * Render privacy policy text on the checkout.
 *
 * @since 3.4.0
 */
function kkart_checkout_privacy_policy_text() {
	echo '<div class="kkart-privacy-policy-text">';
	kkart_privacy_policy_text( 'checkout' );
	echo '</div>';
}

/**
 * Render privacy policy text on the register forms.
 *
 * @since 3.4.0
 */
function kkart_registration_privacy_policy_text() {
	echo '<div class="kkart-privacy-policy-text">';
	kkart_privacy_policy_text( 'registration' );
	echo '</div>';
}

/**
 * Output privacy policy text. This is custom text which can be added via the customizer/privacy settings section.
 *
 * Loads the relevant policy for the current page unless a specific policy text is required.
 *
 * @since 3.4.0
 * @param string $type Type of policy to load. Valid values include registration and checkout.
 */
function kkart_privacy_policy_text( $type = 'checkout' ) {
	if ( ! kkart_privacy_policy_page_id() ) {
		return;
	}
	echo wp_kses_post( wpautop( kkart_replace_policy_page_link_placeholders( kkart_get_privacy_policy_text( $type ) ) ) );
}

/**
 * Replaces placeholders with links to Kkart policy pages.
 *
 * @since 3.4.0
 * @param string $text Text to find/replace within.
 * @return string
 */
function kkart_replace_policy_page_link_placeholders( $text ) {
	$privacy_page_id = kkart_privacy_policy_page_id();
	$terms_page_id   = kkart_terms_and_conditions_page_id();
	$privacy_link    = $privacy_page_id ? '<a href="' . esc_url( get_permalink( $privacy_page_id ) ) . '" class="kkart-privacy-policy-link" target="_blank">' . __( 'privacy policy', 'kkart' ) . '</a>' : __( 'privacy policy', 'kkart' );
	$terms_link      = $terms_page_id ? '<a href="' . esc_url( get_permalink( $terms_page_id ) ) . '" class="kkart-terms-and-conditions-link" target="_blank">' . __( 'terms and conditions', 'kkart' ) . '</a>' : __( 'terms and conditions', 'kkart' );

	$find_replace = array(
		'[terms]'          => $terms_link,
		'[privacy_policy]' => $privacy_link,
	);

	return str_replace( array_keys( $find_replace ), array_values( $find_replace ), $text );
}

/**
 * Template pages
 */

if ( ! function_exists( 'kkart_content' ) ) {

	/**
	 * Output Kkart content.
	 *
	 * This function is only used in the optional 'kkart.php' template.
	 * which people can add to their themes to add basic kkart support.
	 * without hooks or modifying core templates.
	 */
	function kkart_content() {

		if ( is_singular( 'product' ) ) {

			while ( have_posts() ) :
				the_post();
				kkart_get_template_part( 'content', 'single-product' );
			endwhile;

		} else {
			?>

			<?php if ( apply_filters( 'kkart_show_page_title', true ) ) : ?>

				<h1 class="page-title"><?php kkart_page_title(); ?></h1>

			<?php endif; ?>

			<?php do_action( 'kkart_archive_description' ); ?>

			<?php if ( kkart_product_loop() ) : ?>

				<?php do_action( 'kkart_before_shop_loop' ); ?>

				<?php kkart_product_loop_start(); ?>

				<?php if ( kkart_get_loop_prop( 'total' ) ) : ?>
					<?php while ( have_posts() ) : ?>
						<?php the_post(); ?>
						<?php kkart_get_template_part( 'content', 'product' ); ?>
					<?php endwhile; ?>
				<?php endif; ?>

				<?php kkart_product_loop_end(); ?>

				<?php do_action( 'kkart_after_shop_loop' ); ?>

				<?php
			else :
				do_action( 'kkart_no_products_found' );
			endif;
		}
	}
}

/**
 * Global
 */

if ( ! function_exists( 'kkart_output_content_wrapper' ) ) {

	/**
	 * Output the start of the page wrapper.
	 */
	function kkart_output_content_wrapper() {
		kkart_get_template( 'global/wrapper-start.php' );
	}
}
if ( ! function_exists( 'kkart_output_content_wrapper_end' ) ) {

	/**
	 * Output the end of the page wrapper.
	 */
	function kkart_output_content_wrapper_end() {
		kkart_get_template( 'global/wrapper-end.php' );
	}
}

if ( ! function_exists( 'kkart_get_sidebar' ) ) {

	/**
	 * Get the shop sidebar template.
	 */
	function kkart_get_sidebar() {
		kkart_get_template( 'global/sidebar.php' );
	}
}

if ( ! function_exists( 'kkart_demo_store' ) ) {

	/**
	 * Adds a demo store banner to the site if enabled.
	 */
	function kkart_demo_store() {
		if ( ! is_store_notice_showing() ) {
			return;
		}

		$notice = get_option( 'kkart_demo_store_notice' );

		if ( empty( $notice ) ) {
			$notice = __( 'This is a demo store for testing purposes &mdash; no orders shall be fulfilled.', 'kkart' );
		}

		$notice_id = md5( $notice );

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo apply_filters( 'kkart_demo_store', '<p class="kkart-store-notice demo_store" data-notice-id="' . esc_attr( $notice_id ) . '" style="display:none;">' . wp_kses_post( $notice ) . ' <a href="#" class="kkart-store-notice__dismiss-link">' . esc_html__( 'Dismiss', 'kkart' ) . '</a></p>', $notice );
	}
}

/**
 * Loop
 */

if ( ! function_exists( 'kkart_page_title' ) ) {

	/**
	 * Page Title function.
	 *
	 * @param  bool $echo Should echo title.
	 * @return string
	 */
	function kkart_page_title( $echo = true ) {

		if ( is_search() ) {
			/* translators: %s: search query */
			$page_title = sprintf( __( 'Search results: &ldquo;%s&rdquo;', 'kkart' ), get_search_query() );

			if ( get_query_var( 'paged' ) ) {
				/* translators: %s: page number */
				$page_title .= sprintf( __( '&nbsp;&ndash; Page %s', 'kkart' ), get_query_var( 'paged' ) );
			}
		} elseif ( is_tax() ) {

			$page_title = single_term_title( '', false );

		} else {

			$shop_page_id = kkart_get_page_id( 'shop' );
			$page_title   = get_the_title( $shop_page_id );

		}

		$page_title = apply_filters( 'kkart_page_title', $page_title );

		if ( $echo ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $page_title;
		} else {
			return $page_title;
		}
	}
}

if ( ! function_exists( 'kkart_product_loop_start' ) ) {

	/**
	 * Output the start of a product loop. By default this is a UL.
	 *
	 * @param bool $echo Should echo?.
	 * @return string
	 */
	function kkart_product_loop_start( $echo = true ) {
		ob_start();

		kkart_set_loop_prop( 'loop', 0 );

		echo '<ul class="products columns-'. esc_attr( kkart_get_loop_prop( 'columns' ) ) .'">';

		$loop_start = apply_filters( 'kkart_product_loop_start', ob_get_clean() );

		if ( $echo ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $loop_start;
		} else {
			return $loop_start;
		}
	}
}

if ( ! function_exists( 'kkart_product_loop_end' ) ) {

	/**
	 * Output the end of a product loop. By default this is a UL.
	 *
	 * @param bool $echo Should echo?.
	 * @return string
	 */
	function kkart_product_loop_end( $echo = true ) {
		ob_start();

		echo '</ul>';

		$loop_end = apply_filters( 'kkart_product_loop_end', ob_get_clean() );

		if ( $echo ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $loop_end;
		} else {
			return $loop_end;
		}
	}
}
if ( ! function_exists( 'kkart_template_loop_product_title' ) ) {

	/**
	 * Show the product title in the product loop. By default this is an H2.
	 */
	function kkart_template_loop_product_title() {
		echo '<h2 class="' . esc_attr( apply_filters( 'kkart_product_loop_title_classes', 'kkart-loop-product__title' ) ) . '">' . get_the_title() . '</h2>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}
if ( ! function_exists( 'kkart_template_loop_category_title' ) ) {

	/**
	 * Show the subcategory title in the product loop.
	 *
	 * @param object $category Category object.
	 */
	function kkart_template_loop_category_title( $category ) {
		?>
		<h2 class="kkart-loop-category__title">
			<?php
			echo esc_html( $category->name );

			if ( $category->count > 0 ) {
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters( 'kkart_subcategory_count_html', ' <mark class="count">(' . esc_html( $category->count ) . ')</mark>', $category );
			}
			?>
		</h2>
		<?php
	}
}

if ( ! function_exists( 'kkart_template_loop_product_link_open' ) ) {
	/**
	 * Insert the opening anchor tag for products in the loop.
	 */
	function kkart_template_loop_product_link_open() {
		global $product;

		$link = apply_filters( 'kkart_loop_product_link', get_the_permalink(), $product );

		echo '<a href="' . esc_url( $link ) . '" class="kkart-LoopProduct-link kkart-loop-product__link">';
	}
}

if ( ! function_exists( 'kkart_template_loop_product_link_close' ) ) {
	/**
	 * Insert the closing anchor tag for products in the loop.
	 */
	function kkart_template_loop_product_link_close() {
		echo '</a>';
	}
}

if ( ! function_exists( 'kkart_template_loop_category_link_open' ) ) {
	/**
	 * Insert the opening anchor tag for categories in the loop.
	 *
	 * @param int|object|string $category Category ID, Object or String.
	 */
	function kkart_template_loop_category_link_open( $category ) {
		echo '<a href="' . esc_url( get_term_link( $category, 'product_cat' ) ) . '">';
	}
}

if ( ! function_exists( 'kkart_template_loop_category_link_close' ) ) {
	/**
	 * Insert the closing anchor tag for categories in the loop.
	 */
	function kkart_template_loop_category_link_close() {
		echo '</a>';
	}
}

if ( ! function_exists( 'kkart_taxonomy_archive_description' ) ) {

	/**
	 * Show an archive description on taxonomy archives.
	 */
	function kkart_taxonomy_archive_description() {
		if ( is_product_taxonomy() && 0 === absint( get_query_var( 'paged' ) ) ) {
			$term = get_queried_object();

			if ( $term && ! empty( $term->description ) ) {
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo '<div class="term-description">' . kkart_format_content( $term->description ) . '</div>';
			}
		}
	}
}
if ( ! function_exists( 'kkart_product_archive_description' ) ) {

	/**
	 * Show a shop page description on product archives.
	 */
	function kkart_product_archive_description() {
		// Don't display the description on search results page.
		if ( is_search() ) {
			return;
		}

		if ( is_post_type_archive( 'product' ) && in_array( absint( get_query_var( 'paged' ) ), array( 0, 1 ), true ) ) {
			$shop_page = get_post( kkart_get_page_id( 'shop' ) );
			if ( $shop_page ) {
				$description = kkart_format_content( $shop_page->post_content );
				if ( $description ) {
					// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					echo '<div class="page-description">' . $description . '</div>';
				}
			}
		}
	}
}

if ( ! function_exists( 'kkart_template_loop_add_to_cart' ) ) {

	/**
	 * Get the add to cart template for the loop.
	 *
	 * @param array $args Arguments.
	 */
	function kkart_template_loop_add_to_cart( $args = array() ) {
		global $product;

		if ( $product ) {
			$defaults = array(
				'quantity'   => 1,
				'class'      => implode(
					' ',
					array_filter(
						array(
							'button',
							'product_type_' . $product->get_type(),
							$product->is_purchasable() && $product->is_in_stock() ? 'add_to_cart_button' : '',
							$product->supports( 'ajax_add_to_cart' ) && $product->is_purchasable() && $product->is_in_stock() ? 'ajax_add_to_cart' : '',
						)
					)
				),
				'attributes' => array(
					'data-product_id'  => $product->get_id(),
					'data-product_sku' => $product->get_sku(),
					'aria-label'       => $product->add_to_cart_description(),
					'rel'              => 'nofollow',
				),
			);

			$args = apply_filters( 'kkart_loop_add_to_cart_args', wp_parse_args( $args, $defaults ), $product );

			if ( isset( $args['attributes']['aria-label'] ) ) {
				$args['attributes']['aria-label'] = wp_strip_all_tags( $args['attributes']['aria-label'] );
			}
				
			extract($args);
			
			echo apply_filters(
				'kkart_loop_add_to_cart_link', // WPCS: XSS ok.
				sprintf(
					'<a href="%s" data-quantity="%s" class="%s" %s>%s</a>',
					esc_url( $product->add_to_cart_url() ),
					esc_attr( isset( $args['quantity'] ) ? $args['quantity'] : 1 ),
					esc_attr( isset( $args['class'] ) ? $args['class'] : 'button' ),
					isset( $args['attributes'] ) ? kkart_implode_html_attributes( $args['attributes'] ) : '',
					esc_html( $product->add_to_cart_text() )
				),
				$product,
				$args
			);
		}
	}
}

if ( ! function_exists( 'kkart_template_loop_product_thumbnail' ) ) {

	/**
	 * Get the product thumbnail for the loop.
	 */
	function kkart_template_loop_product_thumbnail() {
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo kkart_get_product_thumbnail();
	}
}
if ( ! function_exists( 'kkart_template_loop_price' ) ) {

	/**
	 * Get the product price for the loop.
	 */
	function kkart_template_loop_price() {
		global $product;
		if ( $price_html = $product->get_price_html() ) : ?>
			<span class="price"><?php echo $price_html; ?></span>
		<?php endif;
	}
}
if ( ! function_exists( 'kkart_template_loop_rating' ) ) {

	/**
	 * Display the average rating in the loop.
	 */
	function kkart_template_loop_rating() {
		
		global $product;

		if ( ! kkart_review_ratings_enabled() ) {
			return;
		}

		echo kkart_get_rating_html( $product->get_average_rating() ); // WordPress.XSS.EscapeOutput.OutputNotEscaped.
	}
}
if ( ! function_exists( 'kkart_show_product_loop_sale_flash' ) ) {

	/**
	 * Get the sale flash for the loop.
	 */
	function kkart_show_product_loop_sale_flash() {
		global $post, $product;
		
		if ( $product->is_on_sale() ){
			echo apply_filters( 'kkart_sale_flash', '<span class="onsale">' . esc_html__( 'Sale!', 'kkart' ) . '</span>', $post, $product ); 
		}
	}
}

if ( ! function_exists( 'kkart_get_product_thumbnail' ) ) {

	/**
	 * Get the product thumbnail, or the placeholder if not set.
	 *
	 * @param string $size (default: 'kkart_thumbnail').
	 * @param int    $deprecated1 Deprecated since Kkart 2.0 (default: 0).
	 * @param int    $deprecated2 Deprecated since Kkart 2.0 (default: 0).
	 * @return string
	 */
	function kkart_get_product_thumbnail( $size = 'kkart_thumbnail', $deprecated1 = 0, $deprecated2 = 0 ) {
		global $product;

		$image_size = apply_filters( 'single_product_archive_thumbnail_size', $size );

		return $product ? $product->get_image( $image_size ) : '';
	}
}

if ( ! function_exists( 'kkart_result_count' ) ) {

	/**
	 * Output the result count text (Showing x - x of x results).
	 */
	function kkart_result_count() {
		if ( ! kkart_get_loop_prop( 'is_paginated' ) || ! kkart_products_will_display() ) {
			return;
		}
		$args = array(
			'total'    => kkart_get_loop_prop( 'total' ),
			'per_page' => kkart_get_loop_prop( 'per_page' ),
			'current'  => kkart_get_loop_prop( 'current_page' ),
		);
	
		extract($args);
		?>
		<p class="kkart-result-count">
			<?php
			// phpcs:disable WordPress.Security
			if ( 1 === intval( $total ) ) {
				_e( 'Showing the single result', 'kkart' );
			} elseif ( $total <= $per_page || -1 === $per_page ) {
				/* translators: %d: total results */
				printf( _n( 'Showing all %d result', 'Showing all %d results', $total, 'kkart' ), $total );
			} else {
				$first = ( $per_page * $current ) - $per_page + 1;
				$last  = min( $total, $per_page * $current );
				/* translators: 1: first result 2: last result 3: total results */
				printf( _nx( 'Showing %1$d&ndash;%2$d of %3$d result', 'Showing %1$d&ndash;%2$d of %3$d results', $total, 'with first and last result', 'kkart' ), $first, $last, $total );
			}
			?>
		</p>
		<?php
	}
}

if ( ! function_exists( 'kkart_catalog_ordering' ) ) {

	/**
	 * Output the product sorting options.
	 */
	function kkart_catalog_ordering() {
		if ( ! kkart_get_loop_prop( 'is_paginated' ) || ! kkart_products_will_display() ) {
			return;
		}
		$show_default_orderby    = 'menu_order' === apply_filters( 'kkart_default_catalog_orderby', get_option( 'kkart_default_catalog_orderby', 'menu_order' ) );
		$catalog_orderby_options = apply_filters(
			'kkart_catalog_orderby',
			array(
				'menu_order' => __( 'Default sorting', 'kkart' ),
				'popularity' => __( 'Sort by popularity', 'kkart' ),
				'rating'     => __( 'Sort by average rating', 'kkart' ),
				'date'       => __( 'Sort by latest', 'kkart' ),
				'price'      => __( 'Sort by price: low to high', 'kkart' ),
				'price-desc' => __( 'Sort by price: high to low', 'kkart' ),
			)
		);

		$default_orderby = kkart_get_loop_prop( 'is_search' ) ? 'relevance' : apply_filters( 'kkart_default_catalog_orderby', get_option( 'kkart_default_catalog_orderby', '' ) );
		// phpcs:disable WordPress.Security.NonceVerification.Recommended
		$orderby = isset( $_GET['orderby'] ) ? kkart_clean( wp_unslash( $_GET['orderby'] ) ) : $default_orderby;
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		if ( kkart_get_loop_prop( 'is_search' ) ) {
			$catalog_orderby_options = array_merge( array( 'relevance' => __( 'Relevance', 'kkart' ) ), $catalog_orderby_options );

			unset( $catalog_orderby_options['menu_order'] );
		}

		if ( ! $show_default_orderby ) {
			unset( $catalog_orderby_options['menu_order'] );
		}

		if ( ! kkart_review_ratings_enabled() ) {
			unset( $catalog_orderby_options['rating'] );
		}

		if ( ! array_key_exists( $orderby, $catalog_orderby_options ) ) {
			$orderby = current( array_keys( $catalog_orderby_options ) );
		}

		kkart_get_template(
			'loop/orderby.php',
			array(
				'catalog_orderby_options' => $catalog_orderby_options,
				'orderby'                 => $orderby,
				'show_default_orderby'    => $show_default_orderby,
			)
		);
	}
}

if ( ! function_exists( 'kkart_pagination' ) ) {

	/**
	 * Output the pagination.
	 */
	function kkart_pagination() {
		if ( ! kkart_get_loop_prop( 'is_paginated' ) || ! kkart_products_will_display() ) {
			return;
		}

		$args = array(
			'total'   => kkart_get_loop_prop( 'total_pages' ),
			'current' => kkart_get_loop_prop( 'current_page' ),
			'base'    => esc_url_raw( add_query_arg( 'product-page', '%#%', false ) ),
			'format'  => '?product-page=%#%',
		);

		if ( ! kkart_get_loop_prop( 'is_shortcode' ) ) {
			$args['format'] = '';
			$args['base']   = esc_url_raw( str_replace( 999999999, '%#%', remove_query_arg( 'add-to-cart', get_pagenum_link( 999999999, false ) ) ) );
		}

		extract($args);
		
		$total   = isset( $total ) ? $total : kkart_get_loop_prop( 'total_pages' );
		$current = isset( $current ) ? $current : kkart_get_loop_prop( 'current_page' );
		$base    = isset( $base ) ? $base : esc_url_raw( str_replace( 999999999, '%#%', remove_query_arg( 'add-to-cart', get_pagenum_link( 999999999, false ) ) ) );
		$format  = isset( $format ) ? $format : '';

		if ( $total <= 1 ) {
			return;
		}
		?>
		<nav class="kkart-pagination">
			<?php
			echo paginate_links(
				apply_filters(
					'kkart_pagination_args',
					array( // WPCS: XSS ok.
						'base'      => $base,
						'format'    => $format,
						'add_args'  => false,
						'current'   => max( 1, $current ),
						'total'     => $total,
						'prev_text' => '&larr;',
						'next_text' => '&rarr;',
						'type'      => 'list',
						'end_size'  => 3,
						'mid_size'  => 3,
					)
				)
			);
			?>
		</nav>
		<?php
	}
}

/**
 * Single Product
 */

if ( ! function_exists( 'kkart_show_product_images' ) ) {

	/**
	 * Output the product image before the single product summary.
	 */
	function kkart_show_product_images() {
		
		global $product;

		$columns           = apply_filters( 'kkart_product_thumbnails_columns', 4 );
		$post_thumbnail_id = $product->get_image_id();
		$wrapper_classes   = apply_filters(
			'kkart_single_product_image_gallery_classes',
			array(
				'kkart-product-gallery',
				'kkart-product-gallery--' . ( $product->get_image_id() ? 'with-images' : 'without-images' ),
				'kkart-product-gallery--columns-' . absint( $columns ),
				'images',
			)
		);
		
		?>
		<div class="<?php echo esc_attr( implode( ' ', array_map( 'sanitize_html_class', $wrapper_classes ) ) ); ?>" data-columns="<?php echo esc_attr( $columns ); ?>" style="opacity: 0; transition: opacity .25s ease-in-out;">
			<figure class="kkart-product-gallery__wrapper">
				<?php
				if ( $product->get_image_id() ) {
					$html = kkart_get_gallery_image_html( $post_thumbnail_id, true );
				} else {
					$html  = '<div class="kkart-product-gallery__image--placeholder">';
					$html .= sprintf( '<img src="%s" alt="%s" class="wp-post-image" />', esc_url( kkart_placeholder_img_src( 'kkart_single' ) ), esc_html__( 'Awaiting product image', 'kkart' ) );
					$html .= '</div>';
				}

				echo apply_filters( 'kkart_single_product_image_thumbnail_html', $html, $post_thumbnail_id ); // phpcs:disable WordPress.XSS.EscapeOutput.OutputNotEscaped

				do_action( 'kkart_product_thumbnails' );
				?>
			</figure>
		</div>

<?php
	}
}
if ( ! function_exists( 'kkart_show_product_thumbnails' ) ) {

	/**
	 * Output the product thumbnails.
	 */
	function kkart_show_product_thumbnails() {
		
		global $product;

		$attachment_ids = $product->get_gallery_image_ids();

		if ( $attachment_ids && $product->get_image_id() ) {
			foreach ( $attachment_ids as $attachment_id ) {
				echo apply_filters( 'kkart_single_product_image_thumbnail_html', kkart_get_gallery_image_html( $attachment_id ), $attachment_id ); // phpcs:disable WordPress.XSS.EscapeOutput.OutputNotEscaped
			}
		}

	}
}

/**
 * Get HTML for a gallery image.
 *
 * Hooks: kkart_gallery_thumbnail_size, kkart_gallery_image_size and kkart_gallery_full_size accept name based image sizes, or an array of width/height values.
 *
 * @since 3.3.2
 * @param int  $attachment_id Attachment ID.
 * @param bool $main_image Is this the main image or a thumbnail?.
 * @return string
 */
function kkart_get_gallery_image_html( $attachment_id, $main_image = false ) {
	$flexslider        = (bool) apply_filters( 'kkart_single_product_flexslider_enabled', get_theme_support( 'kkart-product-gallery-slider' ) );
	$gallery_thumbnail = kkart_get_image_size( 'gallery_thumbnail' );
	$thumbnail_size    = apply_filters( 'kkart_gallery_thumbnail_size', array( $gallery_thumbnail['width'], $gallery_thumbnail['height'] ) );
	$image_size        = apply_filters( 'kkart_gallery_image_size', $flexslider || $main_image ? 'kkart_single' : $thumbnail_size );
	$full_size         = apply_filters( 'kkart_gallery_full_size', apply_filters( 'kkart_product_thumbnails_large_size', 'full' ) );
	$thumbnail_src     = wp_get_attachment_image_src( $attachment_id, $thumbnail_size );
	$full_src          = wp_get_attachment_image_src( $attachment_id, $full_size );
	$alt_text          = trim( wp_strip_all_tags( get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ) );
	$image             = wp_get_attachment_image(
		$attachment_id,
		$image_size,
		false,
		apply_filters(
			'kkart_gallery_image_html_attachment_image_params',
			array(
				'title'                   => _wp_specialchars( get_post_field( 'post_title', $attachment_id ), ENT_QUOTES, 'UTF-8', true ),
				'data-caption'            => _wp_specialchars( get_post_field( 'post_excerpt', $attachment_id ), ENT_QUOTES, 'UTF-8', true ),
				'data-src'                => esc_url( $full_src[0] ),
				'data-large_image'        => esc_url( $full_src[0] ),
				'data-large_image_width'  => esc_attr( $full_src[1] ),
				'data-large_image_height' => esc_attr( $full_src[2] ),
				'class'                   => esc_attr( $main_image ? 'wp-post-image' : '' ),
			),
			$attachment_id,
			$image_size,
			$main_image
		)
	);

	return '<div data-thumb="' . esc_url( $thumbnail_src[0] ) . '" data-thumb-alt="' . esc_attr( $alt_text ) . '" class="kkart-product-gallery__image"><a href="' . esc_url( $full_src[0] ) . '">' . $image . '</a></div>';
}

if ( ! function_exists( 'kkart_output_product_data_tabs' ) ) {

	/**
	 * Output the product tabs.
	 */
	function kkart_output_product_data_tabs() {
		$product_tabs = apply_filters( 'kkart_product_tabs', array() );

		if ( ! empty( $product_tabs ) ) : ?>

			<div class="kkart-tabs kkart-tabs-wrapper">
				<ul class="tabs kkart-tabs" role="tablist">
					<?php foreach ( $product_tabs as $key => $product_tab ) : ?>
						<li class="<?php echo esc_attr( $key ); ?>_tab" id="tab-title-<?php echo esc_attr( $key ); ?>" role="tab" aria-controls="tab-<?php echo esc_attr( $key ); ?>">
							<a href="#tab-<?php echo esc_attr( $key ); ?>">
								<?php echo wp_kses_post( apply_filters( 'kkart_product_' . $key . '_tab_title', $product_tab['title'], $key ) ); ?>
							</a>
						</li>
					<?php endforeach; ?>
				</ul>
				<?php foreach ( $product_tabs as $key => $product_tab ) : ?>
					<div class="kkart-Tabs-panel kkart-Tabs-panel--<?php echo esc_attr( $key ); ?> panel entry-content kkart-tab" id="tab-<?php echo esc_attr( $key ); ?>" role="tabpanel" aria-labelledby="tab-title-<?php echo esc_attr( $key ); ?>">
						<?php
						if ( isset( $product_tab['callback'] ) ) {
							call_user_func( $product_tab['callback'], $key, $product_tab );
						}
						?>
					</div>
				<?php endforeach; ?>

				<?php do_action( 'kkart_product_after_tabs' ); ?>
			</div>

		<?php endif;
	}
}
if ( ! function_exists( 'kkart_template_single_title' ) ) {

	/**
	 * Output the product title.
	 */
	function kkart_template_single_title() {
		the_title( '<h1 class="product_title entry-title">', '</h1>' );
	}
}
if ( ! function_exists( 'kkart_template_single_rating' ) ) {

	/**
	 * Output the product rating.
	 */
	function kkart_template_single_rating() {
		if ( post_type_supports( 'product', 'comments' ) ) {
			global $product;

			if ( ! kkart_review_ratings_enabled() ) {
				return;
			}

			$rating_count = $product->get_rating_count();
			$review_count = $product->get_review_count();
			$average      = $product->get_average_rating();

			if ( $rating_count > 0 ){ ?>

				<div class="kkart-product-rating">
					<?php echo kkart_get_rating_html( $average, $rating_count ); // WPCS: XSS ok. ?>
					<?php if ( comments_open() ) : ?>
						<?php //phpcs:disable ?>
						<a href="#reviews" class="kkart-review-link" rel="nofollow">(<?php printf( _n( '%s customer review', '%s customer reviews', $review_count, 'kkart' ), '<span class="count">' . esc_html( $review_count ) . '</span>' ); ?>)</a>
						<?php // phpcs:enable ?>
					<?php endif ?>
				</div>

		<?php
			} 
		}
	}
}
if ( ! function_exists( 'kkart_template_single_price' ) ) {

	/**
	 * Output the product price.
	 */
	function kkart_template_single_price() {
		
		global $product;

		echo '<p class="'. esc_attr( apply_filters( 'kkart_product_price_class', 'price' ) ) .'">'. $product->get_price_html() .'</p>';
	}
}
if ( ! function_exists( 'kkart_template_single_excerpt' ) ) {

	/**
	 * Output the product short description (excerpt).
	 */
	function kkart_template_single_excerpt() {
		global $post;

		$short_description = apply_filters( 'kkart_short_description', $post->post_excerpt );

		if ( ! $short_description ) {
			return;
		}

		?>
		<div class="kkart-product-details__short-description">
			<?php echo $short_description; ?>
		</div>
		<?php
	}
}
if ( ! function_exists( 'kkart_template_single_meta' ) ) {

	/**
	 * Output the product meta.
	 */
	function kkart_template_single_meta() {
		global $product;
		?>
		<div class="product_meta">

			<?php do_action( 'kkart_product_meta_start' ); ?>

			<?php if ( kkart_product_sku_enabled() && ( $product->get_sku() || $product->is_type( 'variable' ) ) ) : ?>

				<span class="sku_wrapper"><?php esc_html_e( 'SKU:', 'kkart' ); ?> <span class="sku"><?php echo ( $sku = $product->get_sku() ) ? $sku : esc_html__( 'N/A', 'kkart' ); ?></span></span>

			<?php endif; ?>

			<?php echo kkart_get_product_category_list( $product->get_id(), ', ', '<span class="posted_in">' . _n( 'Category:', 'Categories:', count( $product->get_category_ids() ), 'kkart' ) . ' ', '</span>' ); ?>

			<?php echo kkart_get_product_tag_list( $product->get_id(), ', ', '<span class="tagged_as">' . _n( 'Tag:', 'Tags:', count( $product->get_tag_ids() ), 'kkart' ) . ' ', '</span>' ); ?>

			<?php do_action( 'kkart_product_meta_end' ); ?>

		</div>
		<?php
	}
}
if ( ! function_exists( 'kkart_template_single_sharing' ) ) {

	/**
	 * Output the product sharing.
	 */
	function kkart_template_single_sharing() {
		do_action( 'kkart_share' ); // Sharing plugins can hook into here.
	}
}
if ( ! function_exists( 'kkart_show_product_sale_flash' ) ) {

	/**
	 * Output the product sale flash.
	 */
	function kkart_show_product_sale_flash() {
		global $post, $product;

		if ( $product->is_on_sale() ){
			echo apply_filters( 'kkart_sale_flash', '<span class="onsale">' . esc_html__( 'Sale!', 'kkart' ) . '</span>', $post, $product );
		}
	}
}

if ( ! function_exists( 'kkart_template_single_add_to_cart' ) ) {

	/**
	 * Trigger the single product add to cart action.
	 */
	function kkart_template_single_add_to_cart() {
		global $product;
		do_action( 'kkart_' . $product->get_type() . '_add_to_cart' );
	}
}
if ( ! function_exists( 'kkart_simple_add_to_cart' ) ) {

	/**
	 * Output the simple product add to cart area.
	 */
	function kkart_simple_add_to_cart($el = array()) {
		global $product;

		if ( ! $product->is_purchasable() ) {
			return;
		}

		echo kkart_get_stock_html( $product ); // WPCS: XSS ok.

		if ( $product->is_in_stock() ) : ?>

			<?php 
				do_action( 'kkart_before_add_to_cart_form' );
			?>
			
			<form class="cart kkart-add-to-cart-form" action="<?php echo esc_url( apply_filters( 'kkart_add_to_cart_form_action', $product->get_permalink() ) ); ?>" method="post" enctype='multipart/form-data'>
				<?php do_action( 'kkart_before_add_to_cart_button' ); ?>

				<?php
				do_action( 'kkart_before_add_to_cart_quantity' );
				
				echo '<div class="kkart-product-quantity-holder">';
				if(!empty($el['atts']['show_quantity'])){
					kkart_quantity_input(
						array(
							'min_value'   => apply_filters( 'kkart_quantity_input_min', $product->get_min_purchase_quantity(), $product ),
							'max_value'   => apply_filters( 'kkart_quantity_input_max', $product->get_max_purchase_quantity(), $product ),
							'input_value' => isset( $_POST['quantity'] ) ? kkart_stock_amount( wp_unslash( $_POST['quantity'] ) ) : $product->get_min_purchase_quantity(), // WPCS: CSRF ok, input var ok.
						)
					);
				}
				echo '</div>';
				
				do_action( 'kkart_after_add_to_cart_quantity' );
				?>

				<button type="submit" name="add-to-cart" value="<?php echo esc_attr( $product->get_id() ); ?>" class="kkart-cart-btn-holder">
				
				<?php 
				if( !empty($el['atts']['cart_icon'])){
					echo '<i class="'. $el['atts']['cart_icon'].' kkart-cart-btn-icon"></i>';
				}
				
				echo '<span class="kkart-cart-btn-text">'. esc_html( empty($el['atts']['cart_text']) ? $product->single_add_to_cart_text() : $el['atts']['cart_text'] ) .'</span>';
				
				if( !empty($el['atts']['cart_icon'])){
					echo '<i class="'. $el['atts']['cart_icon'].' kkart-cart-btn-icon"></i>';
				}
				?>
				
				</button>

				<?php do_action( 'kkart_after_add_to_cart_button' ); ?>
			</form>

			<?php do_action( 'kkart_after_add_to_cart_form' ); ?>

		<?php endif;
	}
}
if ( ! function_exists( 'kkart_grouped_add_to_cart' ) ) {

	/**
	 * Output the grouped product add to cart area.
	 */
	function kkart_grouped_add_to_cart($el = array()) {
		global $product;

		$products = array_filter( array_map( 'kkart_get_product', $product->get_children() ), 'kkart_products_array_filter_visible_grouped' );

		if ( $products ) {
			kkart_get_template(
				'single-product/add-to-cart/grouped.php',
				array(
					'grouped_product'    => $product,
					'grouped_products'   => $products,
					'quantites_required' => false,
					'el'  => $el,
				)
			);
		}
	}
}
if ( ! function_exists( 'kkart_variable_add_to_cart' ) ) {

	/**
	 * Output the variable product add to cart area.
	 */
	function kkart_variable_add_to_cart($el = array()) {
		global $product;

		// Enqueue variation scripts.
		wp_enqueue_script( 'kkart-add-to-cart-variation' );

		// Get Available variations?
		$get_variations = count( $product->get_children() ) <= apply_filters( 'kkart_ajax_variation_threshold', 30, $product );

		// Load the template.
		kkart_get_template(
			'single-product/add-to-cart/variable.php',
			array(
				'available_variations' => $get_variations ? $product->get_available_variations() : false,
				'attributes'           => $product->get_variation_attributes(),
				'selected_attributes'  => $product->get_default_attributes(),
				'el'  => $el,
			)
		);
	}
}
if ( ! function_exists( 'kkart_external_add_to_cart' ) ) {

	/**
	 * Output the external product add to cart area.
	 */
	function kkart_external_add_to_cart($el = array()) {
		global $product;

		if ( ! $product->add_to_cart_url() ) {
			return;
		}

		kkart_get_template(
			'single-product/add-to-cart/external.php',
			array(
				'product_url' => $product->add_to_cart_url(),
				'button_text' => $product->single_add_to_cart_text(),
				'el'  => $el,
			)
		);
	}
}

if ( ! function_exists( 'kkart_quantity_input' ) ) {

	/**
	 * Output the quantity input for add to cart forms.
	 *
	 * @param  array           $args Args for the input.
	 * @param  KKART_Product|null $product Product.
	 * @param  boolean         $echo Whether to return or echo|string.
	 *
	 * @return string
	 */
	function kkart_quantity_input( $args = array(), $product = null, $echo = true ) {
		if ( is_null( $product ) ) {
			$product = $GLOBALS['product'];
		}

		$defaults = array(
			'input_id'     => uniqid( 'quantity_' ),
			'input_name'   => 'quantity',
			'input_value'  => '1',
			'classes'      => apply_filters( 'kkart_quantity_input_classes', array( 'input-text', 'qty', 'text' ), $product ),
			'max_value'    => apply_filters( 'kkart_quantity_input_max', -1, $product ),
			'min_value'    => apply_filters( 'kkart_quantity_input_min', 0, $product ),
			'step'         => apply_filters( 'kkart_quantity_input_step', 1, $product ),
			'pattern'      => apply_filters( 'kkart_quantity_input_pattern', has_filter( 'kkart_stock_amount', 'intval' ) ? '[0-9]*' : '' ),
			'inputmode'    => apply_filters( 'kkart_quantity_input_inputmode', has_filter( 'kkart_stock_amount', 'intval' ) ? 'numeric' : '' ),
			'product_name' => $product ? $product->get_title() : '',
			'placeholder'  => apply_filters( 'kkart_quantity_input_placeholder', '', $product ),
		);

		$args = apply_filters( 'kkart_quantity_input_args', wp_parse_args( $args, $defaults ), $product );

		// Apply sanity to min/max args - min cannot be lower than 0.
		$args['min_value'] = max( $args['min_value'], 0 );
		$args['max_value'] = 0 < $args['max_value'] ? $args['max_value'] : '';

		// Max cannot be lower than min if defined.
		if ( '' !== $args['max_value'] && $args['max_value'] < $args['min_value'] ) {
			$args['max_value'] = $args['min_value'];
		}

		ob_start();

		kkart_get_template( 'global/quantity-input.php', $args );

		if ( $echo ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo ob_get_clean();
		} else {
			return ob_get_clean();
		}
	}
}

if ( ! function_exists( 'kkart_product_description_tab' ) ) {

	/**
	 * Output the description tab content.
	 */
	function kkart_product_description_tab() {
		global $post;

		$heading = apply_filters( 'kkart_product_description_heading', __( 'Description', 'kkart' ) );
		
		if ( $heading ) : ?>
		<h2><?php echo esc_html( $heading ); ?></h2>
		<?php endif;
		the_content();
	}
}
if ( ! function_exists( 'kkart_product_additional_information_tab' ) ) {

	/**
	 * Output the attributes tab content.
	 */
	function kkart_product_additional_information_tab() {
		global $product;

		$heading = apply_filters( 'kkart_product_additional_information_heading', __( 'Additional information', 'kkart' ) );

		if ( $heading ) : ?>
			<h2><?php echo esc_html( $heading ); ?></h2>
		<?php endif; 
		
		do_action( 'kkart_product_additional_information', $product );
	}
}
if ( ! function_exists( 'kkart_default_product_tabs' ) ) {

	/**
	 * Add default product tabs to product pages.
	 *
	 * @param array $tabs Array of tabs.
	 * @return array
	 */
	function kkart_default_product_tabs( $tabs = array() ) {
		global $product, $post;

		// Description tab - shows product content.
		if ( $post->post_content ) {
			$tabs['description'] = array(
				'title'    => __( 'Description', 'kkart' ),
				'priority' => 10,
				'callback' => 'kkart_product_description_tab',
			);
		}

		// Additional information tab - shows attributes.
		if ( $product && ( $product->has_attributes() || apply_filters( 'kkart_product_enable_dimensions_display', $product->has_weight() || $product->has_dimensions() ) ) ) {
			$tabs['additional_information'] = array(
				'title'    => __( 'Additional information', 'kkart' ),
				'priority' => 20,
				'callback' => 'kkart_product_additional_information_tab',
			);
		}

		// Reviews tab - shows comments.
		if ( comments_open() ) {
			$tabs['reviews'] = array(
				/* translators: %s: reviews count */
				'title'    => sprintf( __( 'Reviews (%d)', 'kkart' ), $product->get_review_count() ),
				'priority' => 30,
				'callback' => 'comments_template',
			);
		}

		return $tabs;
	}
}

if ( ! function_exists( 'kkart_sort_product_tabs' ) ) {

	/**
	 * Sort tabs by priority.
	 *
	 * @param array $tabs Array of tabs.
	 * @return array
	 */
	function kkart_sort_product_tabs( $tabs = array() ) {

		// Make sure the $tabs parameter is an array.
		if ( ! is_array( $tabs ) ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_trigger_error
			trigger_error( 'Function kkart_sort_product_tabs() expects an array as the first parameter. Defaulting to empty array.' );
			$tabs = array();
		}

		// Re-order tabs by priority.
		if ( ! function_exists( '_sort_priority_callback' ) ) {
			/**
			 * Sort Priority Callback Function
			 *
			 * @param array $a Comparison A.
			 * @param array $b Comparison B.
			 * @return bool
			 */
			function _sort_priority_callback( $a, $b ) {
				if ( ! isset( $a['priority'], $b['priority'] ) || $a['priority'] === $b['priority'] ) {
					return 0;
				}
				return ( $a['priority'] < $b['priority'] ) ? -1 : 1;
			}
		}

		uasort( $tabs, '_sort_priority_callback' );

		return $tabs;
	}
}

if ( ! function_exists( 'kkart_comments' ) ) {

	/**
	 * Output the Review comments template.
	 *
	 * @param WP_Comment $comment Comment object.
	 * @param array      $args Arguments.
	 * @param int        $depth Depth.
	 */
	function kkart_comments( $comment, $args, $depth ) {
		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		$GLOBALS['comment'] = $comment;
		kkart_get_template(
			'single-product/review.php',
			array(
				'comment' => $comment,
				'args'    => $args,
				'depth'   => $depth,
			)
		);
	}
}

if ( ! function_exists( 'kkart_review_display_gravatar' ) ) {
	/**
	 * Display the review authors gravatar
	 *
	 * @param array $comment WP_Comment.
	 * @return void
	 */
	function kkart_review_display_gravatar( $comment ) {
		echo get_avatar( $comment, apply_filters( 'kkart_review_gravatar_size', '60' ), '' );
	}
}

if ( ! function_exists( 'kkart_review_display_rating' ) ) {
	/**
	 * Display the reviewers star rating
	 *
	 * @return void
	 */
	function kkart_review_display_rating() {
		global $comment;
		
		if ( post_type_supports( 'product', 'comments' ) ) {
			
			$rating = intval( get_comment_meta( $comment->comment_ID, 'rating', true ) );

			if ( $rating && kkart_review_ratings_enabled() ) {
				echo kkart_get_rating_html( $rating ); // WPCS: XSS ok.
			}
		}
	}
}

if ( ! function_exists( 'kkart_review_display_meta' ) ) {
	/**
	 * Display the review authors meta (name, verified owner, review date)
	 *
	 * @return void
	 */
	function kkart_review_display_meta() {
		global $comment;
		
		$verified = kkart_review_is_from_verified_owner( $comment->comment_ID );

		if ( '0' === $comment->comment_approved ) { ?>

			<p class="meta">
				<em class="kkart-review__awaiting-approval">
					<?php esc_html_e( 'Your review is awaiting approval', 'kkart' ); ?>
				</em>
			</p>

		<?php } else { ?>

			<p class="meta">
				<strong class="kkart-review__author"><?php comment_author(); ?> </strong>
				<?php
				if ( 'yes' === get_option( 'kkart_review_rating_verification_label' ) && $verified ) {
					echo '<em class="kkart-review__verified verified">(' . esc_attr__( 'verified owner', 'kkart' ) . ')</em> ';
				}

				?>
				<span class="kkart-review__dash">&ndash;</span> <time class="kkart-review__published-date" datetime="<?php echo esc_attr( get_comment_date( 'c' ) ); ?>"><?php echo esc_html( get_comment_date( kkart_date_format() ) ); ?></time>
			</p>

			<?php
		}
	}
}

if ( ! function_exists( 'kkart_review_display_comment_text' ) ) {

	/**
	 * Display the review content.
	 */
	function kkart_review_display_comment_text() {
		echo '<div class="description">';
		comment_text();
		echo '</div>';
	}
}

if ( ! function_exists( 'kkart_output_related_products' ) ) {

	/**
	 * Output the related products.
	 */
	function kkart_output_related_products() {

		$args = array(
			'posts_per_page' => 4,
			'columns'        => 4,
			'orderby'        => 'rand', // @codingStandardsIgnoreLine.
		);

		kkart_related_products( apply_filters( 'kkart_output_related_products_args', $args ) );
	}
}

if ( ! function_exists( 'kkart_related_products' ) ) {

	/**
	 * Output the related products.
	 *
	 * @param array $args Provided arguments.
	 */
	function kkart_related_products( $args = array() ) {
		global $product;

		if ( ! $product ) {
			return;
		}

		$defaults = array(
			'posts_per_page' => 2,
			'columns'        => 2,
			'orderby'        => 'rand', // @codingStandardsIgnoreLine.
			'order'          => 'desc',
		);

		$args = wp_parse_args( $args, $defaults );

		// Get visible related products then sort them at random.
		$args['related_products'] = array_filter( array_map( 'kkart_get_product', kkart_get_related_products( $product->get_id(),$args['posts_per_page'], $product->get_upsell_ids() ) ), 'kkart_products_array_filter_visible' );

		// Handle orderby.
		$args['related_products'] = kkart_products_array_orderby( $args['related_products'], $args['orderby'], $args['order'] );

		// Set global loop values.
		kkart_set_loop_prop( 'name', 'related' );
		kkart_set_loop_prop( 'columns', apply_filters( 'kkart_related_products_columns', $args['columns'] ) );
		
		extract( $args );
		
		if ( $related_products ) : ?>

			<section class="related products">

				<?php
				$heading = apply_filters( 'kkart_product_related_products_heading', __( 'Related products', 'kkart' ) );

				if ( $heading ) :
					?>
					<h2><?php echo esc_html( $heading ); ?></h2>
				<?php endif; ?>
				
				<?php kkart_product_loop_start(); ?>

					<?php foreach ( $related_products as $related_product ) : ?>

							<?php
							$post_object = get_post( $related_product->get_id() );

							setup_postdata( $GLOBALS['post'] =& $post_object ); // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited, Squiz.PHP.DisallowMultipleAssignments.Found

							kkart_get_template_part( 'content', 'product' );
							?>

					<?php endforeach; ?>

				<?php kkart_product_loop_end(); ?>

			</section>
			<?php
		endif;

		wp_reset_postdata();
	}
}

if ( ! function_exists( 'kkart_upsell_display' ) ) {

	/**
	 * Output product up sells.
	 *
	 * @param int    $limit (default: -1).
	 * @param int    $columns (default: 4).
	 * @param string $orderby Supported values - rand, title, ID, date, modified, menu_order, price.
	 * @param string $order Sort direction.
	 */
	function kkart_upsell_display( $limit = '-1', $columns = 4, $orderby = 'rand', $order = 'desc' ) {
		global $product;

		if ( ! $product ) {
			return;
		}

		// Handle the legacy filter which controlled posts per page etc.
		$args = apply_filters(
			'kkart_upsell_display_args',
			array(
				'posts_per_page' => $limit,
				'orderby'        => $orderby,
				'order'          => $order,
				'columns'        => $columns,
			)
		);
		kkart_set_loop_prop( 'name', 'up-sells' );
		kkart_set_loop_prop( 'columns', apply_filters( 'kkart_upsells_columns', isset( $args['columns'] ) ? $args['columns'] : $columns ) );

		$orderby = apply_filters( 'kkart_upsells_orderby', isset( $args['orderby'] ) ? $args['orderby'] : $orderby );
		$order   = apply_filters( 'kkart_upsells_order', isset( $args['order'] ) ? $args['order'] : $order );
		$limit   = apply_filters( 'kkart_upsells_total', isset( $args['posts_per_page'] ) ? $args['posts_per_page'] : $limit );

		// Get visible upsells then sort them at random, then limit result set.
		$upsells = kkart_products_array_orderby( array_filter( array_map( 'kkart_get_product', $product->get_upsell_ids() ), 'kkart_products_array_filter_visible' ), $orderby, $order );
		$upsells = $limit > 0 ? array_slice( $upsells, 0, $limit ) : $upsells;

		kkart_get_template(
			'single-product/up-sells.php',
			array(
				'upsells'        => $upsells,

				// Not used now, but used in previous version of up-sells.php.
				'posts_per_page' => $limit,
				'orderby'        => $orderby,
				'columns'        => $columns,
			)
		);
	}
}

/** Cart */

if ( ! function_exists( 'kkart_shipping_calculator' ) ) {

	/**
	 * Output the cart shipping calculator.
	 *
	 * @param string $button_text Text for the shipping calculation toggle.
	 */
	function kkart_shipping_calculator( $button_text = '' ) {
		if ( 'no' === get_option( 'kkart_enable_shipping_calc' ) || ! KKART()->cart->needs_shipping() ) {
			return;
		}
		wp_enqueue_script( 'kkart-country-select' );
		kkart_get_template(
			'cart/shipping-calculator.php',
			array(
				'button_text' => $button_text,
			)
		);
	}
}

if ( ! function_exists( 'kkart_cart_totals' ) ) {

	/**
	 * Output the cart totals.
	 */
	function kkart_cart_totals() {
		if ( is_checkout() ) {
			return;
		}
		
		?>
		<div class="cart_totals <?php echo ( KKART()->customer->has_calculated_shipping() ) ? 'calculated_shipping' : ''; ?>">

			<?php do_action( 'kkart_before_cart_totals' ); ?>

			<h2><?php esc_html_e( 'Cart totals', 'kkart' ); ?></h2>

			<table cellspacing="0" class="shop_table shop_table_responsive">

				<tr class="cart-subtotal">
					<th><?php esc_html_e( 'Subtotal', 'kkart' ); ?></th>
					<td data-title="<?php esc_attr_e( 'Subtotal', 'kkart' ); ?>"><?php kkart_cart_totals_subtotal_html(); ?></td>
				</tr>

				<?php foreach ( KKART()->cart->get_coupons() as $code => $coupon ) : ?>
					<tr class="cart-discount coupon-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
						<th><?php kkart_cart_totals_coupon_label( $coupon ); ?></th>
						<td data-title="<?php echo esc_attr( kkart_cart_totals_coupon_label( $coupon, false ) ); ?>"><?php kkart_cart_totals_coupon_html( $coupon ); ?></td>
					</tr>
				<?php endforeach; ?>

				<?php if ( KKART()->cart->needs_shipping() && KKART()->cart->show_shipping() ) : ?>

					<?php do_action( 'kkart_cart_totals_before_shipping' ); ?>

					<?php kkart_cart_totals_shipping_html(); ?>

					<?php do_action( 'kkart_cart_totals_after_shipping' ); ?>

				<?php elseif ( KKART()->cart->needs_shipping() && 'yes' === get_option( 'kkart_enable_shipping_calc' ) ) : ?>

					<tr class="shipping">
						<th><?php esc_html_e( 'Shipping', 'kkart' ); ?></th>
						<td data-title="<?php esc_attr_e( 'Shipping', 'kkart' ); ?>"><?php kkart_shipping_calculator(); ?></td>
					</tr>

				<?php endif; ?>

				<?php foreach ( KKART()->cart->get_fees() as $fee ) : ?>
					<tr class="fee">
						<th><?php echo esc_html( $fee->name ); ?></th>
						<td data-title="<?php echo esc_attr( $fee->name ); ?>"><?php kkart_cart_totals_fee_html( $fee ); ?></td>
					</tr>
				<?php endforeach; ?>

				<?php
				if ( kkart_tax_enabled() && ! KKART()->cart->display_prices_including_tax() ) {
					$taxable_address = KKART()->customer->get_taxable_address();
					$estimated_text  = '';

					if ( KKART()->customer->is_customer_outside_base() && ! KKART()->customer->has_calculated_shipping() ) {
						/* translators: %s location. */
						$estimated_text = sprintf( ' <small>' . esc_html__( '(estimated for %s)', 'kkart' ) . '</small>', KKART()->countries->estimated_for_prefix( $taxable_address[0] ) . KKART()->countries->countries[ $taxable_address[0] ] );
					}

					if ( 'itemized' === get_option( 'kkart_tax_total_display' ) ) {
						foreach ( KKART()->cart->get_tax_totals() as $code => $tax ) { // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
							?>
							<tr class="tax-rate tax-rate-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
								<th><?php echo esc_html( $tax->label ) . $estimated_text; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></th>
								<td data-title="<?php echo esc_attr( $tax->label ); ?>"><?php echo wp_kses_post( $tax->formatted_amount ); ?></td>
							</tr>
							<?php
						}
					} else {
						?>
						<tr class="tax-total">
							<th><?php echo esc_html( KKART()->countries->tax_or_vat() ) . $estimated_text; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></th>
							<td data-title="<?php echo esc_attr( KKART()->countries->tax_or_vat() ); ?>"><?php kkart_cart_totals_taxes_total_html(); ?></td>
						</tr>
						<?php
					}
				}
				?>

				<?php do_action( 'kkart_cart_totals_before_order_total' ); ?>

				<tr class="order-total">
					<th><?php esc_html_e( 'Total', 'kkart' ); ?></th>
					<td data-title="<?php esc_attr_e( 'Total', 'kkart' ); ?>"><?php kkart_cart_totals_order_total_html(); ?></td>
				</tr>

				<?php do_action( 'kkart_cart_totals_after_order_total' ); ?>

			</table>

			<div class="kkart-proceed-to-checkout">
				<?php do_action( 'kkart_proceed_to_checkout' ); ?>
			</div>

			<?php do_action( 'kkart_after_cart_totals' ); ?>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kkart_cross_sell_display' ) ) {

	/**
	 * Output the cart cross-sells.
	 *
	 * @param  int    $limit (default: 2).
	 * @param  int    $columns (default: 2).
	 * @param  string $orderby (default: 'rand').
	 * @param  string $order (default: 'desc').
	 */
	function kkart_cross_sell_display( $limit = 2, $columns = 2, $orderby = 'rand', $order = 'desc' ) {
		if ( is_checkout() ) {
			return;
		}
		// Get visible cross sells then sort them at random.
		$cross_sells = array_filter( array_map( 'kkart_get_product', KKART()->cart->get_cross_sells() ), 'kkart_products_array_filter_visible' );

		kkart_set_loop_prop( 'name', 'cross-sells' );
		kkart_set_loop_prop( 'columns', apply_filters( 'kkart_cross_sells_columns', $columns ) );

		// Handle orderby and limit results.
		$orderby     = apply_filters( 'kkart_cross_sells_orderby', $orderby );
		$order       = apply_filters( 'kkart_cross_sells_order', $order );
		$cross_sells = kkart_products_array_orderby( $cross_sells, $orderby, $order );
		$limit       = apply_filters( 'kkart_cross_sells_total', $limit );
		$cross_sells = $limit > 0 ? array_slice( $cross_sells, 0, $limit ) : $cross_sells;

		kkart_get_template(
			'cart/cross-sells.php',
			array(
				'cross_sells'    => $cross_sells,

				// Not used now, but used in previous version of up-sells.php.
				'posts_per_page' => $limit,
				'orderby'        => $orderby,
				'columns'        => $columns,
			)
		);
	}
}

if ( ! function_exists( 'kkart_button_proceed_to_checkout' ) ) {

	/**
	 * Output the proceed to checkout button.
	 */
	function kkart_button_proceed_to_checkout() {
		?>
		<a href="<?php echo esc_url( kkart_get_checkout_url() ); ?>" class="checkout-button button alt kkart-forward">
			<?php esc_html_e( 'Proceed to checkout', 'kkart' ); ?>
		</a>
		<?php
	}
}

if ( ! function_exists( 'kkart_widget_shopping_cart_button_view_cart' ) ) {

	/**
	 * Output the view cart button.
	 */
	function kkart_widget_shopping_cart_button_view_cart() {
		echo '<a href="' . esc_url( kkart_get_cart_url() ) . '" class="button kkart-forward">' . esc_html__( 'View cart', 'kkart' ) . '</a>';
	}
}

if ( ! function_exists( 'kkart_widget_shopping_cart_proceed_to_checkout' ) ) {

	/**
	 * Output the proceed to checkout button.
	 */
	function kkart_widget_shopping_cart_proceed_to_checkout() {
		echo '<a href="' . esc_url( kkart_get_checkout_url() ) . '" class="button checkout kkart-forward">' . esc_html__( 'Checkout', 'kkart' ) . '</a>';
	}
}

if ( ! function_exists( 'kkart_widget_shopping_cart_subtotal' ) ) {
	/**
	 * Output to view cart subtotal.
	 *
	 * @since 3.7.0
	 */
	function kkart_widget_shopping_cart_subtotal() {
		echo '<strong>' . esc_html__( 'Subtotal:', 'kkart' ) . '</strong> ' . KKART()->cart->get_cart_subtotal(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

/** Mini-Cart */

if ( ! function_exists( 'kkart_mini_cart' ) ) {

	/**
	 * Output the Mini-cart - used by cart widget.
	 *
	 * @param array $args Arguments.
	 */
	function kkart_mini_cart( $args = array() ) {

		$defaults = array(
			'list_class' => '',
		);

		$args = wp_parse_args( $args, $defaults );
		
		extract($args);
		
		do_action( 'kkart_before_mini_cart' ); ?>

		<?php if ( ! KKART()->cart->is_empty() ) : ?>

			<ul class="kkart-mini-cart cart_list product_list_widget <?php echo esc_attr( $args['list_class'] ); ?>">
				<?php
				do_action( 'kkart_before_mini_cart_contents' );

				foreach ( KKART()->cart->get_cart() as $cart_item_key => $cart_item ) {
					$_product   = apply_filters( 'kkart_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );
					$product_id = apply_filters( 'kkart_cart_item_product_id', $cart_item['product_id'], $cart_item, $cart_item_key );

					if ( $_product && $_product->exists() && $cart_item['quantity'] > 0 && apply_filters( 'kkart_widget_cart_item_visible', true, $cart_item, $cart_item_key ) ) {
						$product_name      = apply_filters( 'kkart_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key );
						$thumbnail         = apply_filters( 'kkart_cart_item_thumbnail', $_product->get_image(), $cart_item, $cart_item_key );
						$product_price     = apply_filters( 'kkart_cart_item_price', KKART()->cart->get_product_price( $_product ), $cart_item, $cart_item_key );
						$product_permalink = apply_filters( 'kkart_cart_item_permalink', $_product->is_visible() ? $_product->get_permalink( $cart_item ) : '', $cart_item, $cart_item_key );
						?>
						<li class="kkart-mini-cart-item <?php echo esc_attr( apply_filters( 'kkart_mini_cart_item_class', 'mini_cart_item', $cart_item, $cart_item_key ) ); ?>">
							<?php
							echo apply_filters( // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
								'kkart_cart_item_remove_link',
								sprintf(
									'<a href="%s" class="remove remove_from_cart_button" aria-label="%s" data-product_id="%s" data-cart_item_key="%s" data-product_sku="%s">&times;</a>',
									esc_url( kkart_get_cart_remove_url( $cart_item_key ) ),
									esc_attr__( 'Remove this item', 'kkart' ),
									esc_attr( $product_id ),
									esc_attr( $cart_item_key ),
									esc_attr( $_product->get_sku() )
								),
								$cart_item_key
							);
							?>
							<?php if ( empty( $product_permalink ) ) : ?>
								<?php echo $thumbnail . $product_name; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
							<?php else : ?>
								<a href="<?php echo esc_url( $product_permalink ); ?>">
									<?php echo $thumbnail . $product_name; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
								</a>
							<?php endif; ?>
							<?php echo kkart_get_formatted_cart_item_data( $cart_item ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
							<?php echo apply_filters( 'kkart_widget_cart_item_quantity', '<span class="quantity">' . sprintf( '%s &times; %s', $cart_item['quantity'], $product_price ) . '</span>', $cart_item, $cart_item_key ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
						</li>
						<?php
					}
				}

				do_action( 'kkart_mini_cart_contents' );
				?>
			</ul>

			<p class="kkart-mini-cart__total total">
				<?php
				/**
				 * Hook: kkart_widget_shopping_cart_total.
				 *
				 * @hooked kkart_widget_shopping_cart_subtotal - 10
				 */
				do_action( 'kkart_widget_shopping_cart_total' );
				?>
			</p>

			<?php do_action( 'kkart_widget_shopping_cart_before_buttons' ); ?>

			<p class="kkart-mini-cart__buttons buttons"><?php do_action( 'kkart_widget_shopping_cart_buttons' ); ?></p>

			<?php do_action( 'kkart_widget_shopping_cart_after_buttons' ); ?>

		<?php else : ?>

			<p class="kkart-mini-cart__empty-message"><?php esc_html_e( 'No products in the cart.', 'kkart' ); ?></p>

		<?php endif; ?>

		<?php do_action( 'kkart_after_mini_cart' );
	}
}

/** Registration */

if ( ! function_exists( 'kkart_registration_form' ) ) {

	/**
	 * Output the Kkart Login Form.
	 *
	 * @param array $args Arguments.
	 */
	function kkart_register_form( $args = array(), $echo = true) {

		$defaults = array(
			'message'  => '',
			'redirect' => '',
			'hidden'   => false,
		);

		$args = wp_parse_args( $args, $defaults );

		
		if ( is_user_logged_in() ) {
			return;
		}
		
		ob_start();
		kkart_print_notices();
		?>
		<form method="post" class="kkart-form kkart-form-register register" <?php do_action( 'kkart_register_form_tag' ); ?> >

			<?php do_action( 'kkart_register_form_start' ); ?>

			<?php if ( 'no' === get_option( 'kkart_registration_generate_username' ) ) : ?>

				<p class="kkart-form-row kkart-form-row--wide form-row form-row-wide">
					<label for="reg_username"><?php esc_html_e( 'Username', 'kkart' ); ?>&nbsp;<span class="required">*</span></label>
					<input type="text" class="kkart-Input kkart-Input--text input-text" name="username" id="reg_username" autocomplete="username" value="<?php echo ( ! empty( $_POST['username'] ) ) ? esc_attr( wp_unslash( $_POST['username'] ) ) : ''; ?>" /><?php // @codingStandardsIgnoreLine ?>
				</p>

			<?php endif; ?>

			<p class="kkart-form-row kkart-form-row--wide form-row form-row-wide">
				<label for="reg_email"><?php esc_html_e( 'Email address', 'kkart' ); ?>&nbsp;<span class="required">*</span></label>
				<input type="email" class="kkart-Input kkart-Input--text input-text" name="email" id="reg_email" autocomplete="email" value="<?php echo ( ! empty( $_POST['email'] ) ) ? esc_attr( wp_unslash( $_POST['email'] ) ) : ''; ?>" /><?php // @codingStandardsIgnoreLine ?>
			</p>

			<?php if ( 'no' === get_option( 'kkart_registration_generate_password' ) ) : ?>

				<p class="kkart-form-row kkart-form-row--wide form-row form-row-wide">
					<label for="reg_password"><?php esc_html_e( 'Password', 'kkart' ); ?>&nbsp;<span class="required">*</span></label>
					<input type="password" class="kkart-Input kkart-Input--text input-text" name="password" id="reg_password" autocomplete="new-password" />
				</p>

			<?php else : ?>

				<p><?php esc_html_e( 'A password will be sent to your email address.', 'kkart' ); ?></p>

			<?php endif; ?>

			<?php do_action( 'register_form' ); ?>
			<?php do_action( 'kkart_register_form' ); ?>

			<p class="kkart-form-row form-row">
				<?php wp_nonce_field( 'kkart-register', 'kkart-register-nonce' ); ?>
				<button type="submit" class="kkart-Button kkart-button button kkart-form-register__submit" name="register" value="<?php esc_attr_e( 'Register', 'kkart' ); ?>"><?php esc_html_e( 'Register', 'kkart' ); ?></button>
				<a href="<?php echo esc_url( wp_login_url() ); ?>" class="kkart-login-form-show"><?php esc_html_e( 'Login?', 'kkart' ); ?></a>
			</p>

		</form>
	<?php

		if(empty($echo)){
			return ob_get_clean();
		}
		
		echo ob_get_clean();
	}
}

/** Login */

if ( ! function_exists( 'kkart_login_form' ) ) {

	/**
	 * Output the Kkart Login Form.
	 *
	 * @param array $args Arguments.
	 */
	function kkart_login_form( $args = array(), $echo = true) {

		$defaults = array(
			'message'  => '',
			'redirect' => '',
			'hidden'   => false,
		);

		$args = wp_parse_args( $args, $defaults );

		
		if ( is_user_logged_in() ) {
			return;
		}
		
		ob_start();
		kkart_print_notices();
		?>
		
		<form class="kkart-form kkart-form-login login" method="post" <?php echo ( $hidden ) ? 'style="display:none;"' : ''; ?>>

			<?php 
			if(!did_action( 'login_enqueue_scripts' )){
				do_action( 'login_enqueue_scripts' );
			}
			
			do_action( 'kkart_login_form_start' );
			?>

			<?php echo ( $message ) ? wpautop( wptexturize( $message ) ) : ''; // @codingStandardsIgnoreLine ?>

			<p class="form-row">
				<label for="username"><?php esc_html_e( 'Username or email', 'kkart' ); ?>&nbsp;<span class="required">*</span></label>
				<input type="text" class="input-text" name="username" id="username" autocomplete="username" />
			</p>
			<p class="form-row user-pass-wrap">
				<label for="password"><?php esc_html_e( 'Password', 'kkart' ); ?>&nbsp;<span class="required">*</span></label>
				<input class="input-text" type="password" name="password" id="password" autocomplete="current-password" />
			</p>
			<div class="clear"></div>

			<?php do_action( 'login_form' ); ?>
			<?php do_action( 'kkart_login_form' ); ?>

			<p class="form-row">
				<label class="kkart-form__label kkart-form__label-for-checkbox kkart-form-login__rememberme">
					<input class="kkart-form__input kkart-form__input-checkbox" name="rememberme" type="checkbox" id="rememberme" value="forever" /> <span><?php esc_html_e( 'Remember me', 'kkart' ); ?></span>
				</label>
				<?php wp_nonce_field( 'kkart-login', 'kkart-login-nonce' ); ?>
				<input type="hidden" name="redirect" value="<?php echo esc_url( $redirect ); ?>" />
				<button type="submit" class="kkart-button button kkart-form-login__submit" name="login" value="<?php esc_attr_e( 'Login', 'kkart' ); ?>"><?php esc_html_e( 'Login', 'kkart' ); ?></button>
			</p>
			<p class="kkart-inline-block lost_password">
				<a href="<?php echo esc_url( wp_lostpassword_url() ); ?>"><?php esc_html_e( 'Lost your password?', 'kkart' ); ?></a>
			</p>
			<p class="kkart-inline-block signup">
				<a href="<?php echo esc_url( wp_registration_url() ); ?>"><?php esc_html_e( 'Signup?', 'kkart' ); ?></a>
			</p>

			<div class="clear"></div>

			<?php do_action( 'kkart_login_form_end' ); ?>

		</form>
	<?php

		if(empty($echo)){
			return ob_get_clean();
		}
		
		echo ob_get_clean();
	}
}

if ( ! function_exists( 'kkart_checkout_login_form' ) ) {

	/**
	 * Output the Kkart Checkout Login Form.
	 */
	function kkart_checkout_login_form($el = array()) {
		
		$div = '';
		
		if(is_user_logged_in()){
			$current_user = wp_get_current_user();
						
			$div .= '<div class="kkart-checkout-user-details"> '. __('Name').' : '.$current_user->display_name.'<br/>'. __('Email').' : '.$current_user->user_email .'</div>
			<div class="kkart-checkout-logout"><a href="'.str_replace( '&amp;', '&', wp_logout_url( kkart_get_page_permalink( 'myaccount' ) ) ).'">'.__('Signin with other account').'</a></div>';
			
		
		// Show Login form
		}else{
			ob_start();
			
			echo '<div class="kkart-checkout-login-holder">';
			kkart_login_form(
				array(
					'message'  => esc_html__( 'If you have shopped with us before, please enter your details below. If you are a new customer, please proceed to the Billing section.', 'kkart' ),
					'redirect' => kkart_get_checkout_url(),
					'hidden'   => true,
				)
			);
			
			echo '</div>
			<div class="kkart-checkout-register-holder" style="display:none">';
			
			kkart_register_form(
				array(
					'message'  => esc_html__( 'If you have shopped with us before, please enter your details below. If you are a new customer, please proceed to the Billing section.', 'kkart' ),
					'redirect' => kkart_get_checkout_url(),
					'hidden'   => true,
				)
			);
			
			echo '</div>';
			$div .= ob_get_clean();
		}
		
		return $div;
	}
}

if ( ! function_exists( 'kkart_checkout_billing' ) ) {

	/**
	 * Output the Kkart Checkout Billing Form.
	 */
	function kkart_checkout_billing($el = array(), $type = 'billing') {
		$div = '';
		
		$checkout = KKART()->checkout();
		$fields = $checkout->get_checkout_fields( 'billing' );
		
		$div .= '<div class="kkart-billing-fields__field-wrapper">';
		foreach ( $fields as $key => $field ) {
			$field['return'] = true;
			$div .= kkart_form_field( $key, $field, $checkout->get_value( $key ) );
		}
		$div .= '</div>';
		
		$holder = '<div class="kkart-address-holder kkart-address-done" data-form-type="'. $type .'">';
		
		if($type == 'shipping'){
			$holder .= '<div class="kkart-shipping-to">
				<input type="checkbox" name="ship_to_billing_address" class="ship_to_billing_address" checked="checked"/> '.__('Shipping address same as the Billing address').'
			</div>';
		}
		
		$holder .= '<div class="kkart-addresses-container">
				<div class="kkart-addresses-holder" data-formate="">
				</div>
				<a class="kkart-address-form-headding">'.__('Add New Address').'</a>
				<div class="kkart-address-form-container">
					<div class="kkart-address-form-holder">
					'.$div.'
					</div>
					<div class="kkart-address-btn-holder">
						<input type="hidden" name="address_id"/>
						<button class="kkart-save-address-form">'.__('Save').'</button>
						<button class="kkart-save-and-usethis-address-form">'.__('Save and use this').'</button>
						<button class="kkart-cancel-address-form">'.__('Cancel').'</button>
					</div>
				</div>
			</div>
		</div>';
		
		return $holder;
	}
}

if ( ! function_exists( 'kkart_checkout_shipping' ) ) {

	/**
	 * Output the Kkart Checkout Shipping Form.
	 */
	function kkart_checkout_shipping($el = array()) {
		return kkart_checkout_billing($el, 'shipping');
	}
}

if ( ! function_exists( 'kkart_checkout_cart' ) ) {

	/**
	 * Output the Kkart Checkout cart.
	 */
	function kkart_checkout_cart($el = array()) {
		
		ob_start();
		?>
		<table class="shop_table kkart-checkout-review-order-table">
			<thead>
				<tr>
					<th class="product-name"><?php esc_html_e( 'Product', 'kkart' ); ?></th>
					<th class="product-total"><?php esc_html_e( 'Subtotal', 'kkart' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php
				foreach ( KKART()->cart->get_cart() as $cart_item_key => $cart_item ){
					$_product = apply_filters( 'kkart_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key );

					if ( $_product && $_product->exists() && $cart_item['quantity'] > 0 && apply_filters( 'kkart_checkout_cart_item_visible', true, $cart_item, $cart_item_key ) ){
						?>
						<tr class="<?php echo esc_attr( apply_filters( 'kkart_cart_item_class', 'cart_item', $cart_item, $cart_item_key ) ); ?>">
	
							<td class="product-name">
								<a href="<?php echo get_permalink($_product->get_id()); ?>">
									<?php echo apply_filters( 'kkart_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key ) . '&nbsp;'; ?>
									<?php echo apply_filters( 'kkart_checkout_cart_item_quantity', ' <strong class="product-quantity">' . sprintf( '&times;&nbsp;%s', $cart_item['quantity'] ) . '</strong>', $cart_item, $cart_item_key ); ?>
									<?php echo kkart_get_formatted_cart_item_data( $cart_item ); ?>
								</a>
							</td>
							
							<td class="product-total">
								<?php echo apply_filters( 'kkart_cart_item_subtotal', KKART()->cart->get_product_subtotal( $_product, $cart_item['quantity'] ), $cart_item, $cart_item_key ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
							</td>
						</tr>
						<?php
					}
				}
				?>
			</tbody>
			<tfoot>

				<tr class="cart-subtotal">
					<th><?php esc_html_e( 'Subtotal', 'kkart' ); ?></th>
					<td><?php kkart_cart_totals_subtotal_html(); ?></td>
				</tr>

				<?php foreach ( KKART()->cart->get_coupons() as $code => $coupon ) : ?>
					<tr class="cart-discount coupon-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
						<th><?php kkart_cart_totals_coupon_label( $coupon ); ?></th>
						<td><?php kkart_cart_totals_coupon_html( $coupon ); ?></td>
					</tr>
				<?php endforeach; ?>

				<?php if ( KKART()->cart->needs_shipping() && KKART()->cart->show_shipping() ) : ?>
					<?php kkart_cart_totals_shipping_html(); ?>
				<?php endif; ?>

				<?php foreach ( KKART()->cart->get_fees() as $fee ) : ?>
					<tr class="fee">
						<th><?php echo esc_html( $fee->name ); ?></th>
						<td><?php kkart_cart_totals_fee_html( $fee ); ?></td>
					</tr>
				<?php endforeach; ?>

				<?php if ( kkart_tax_enabled() && ! KKART()->cart->display_prices_including_tax() ) : ?>
					<?php if ( 'itemized' === get_option( 'kkart_tax_total_display' ) ) : ?>
						<?php foreach ( KKART()->cart->get_tax_totals() as $code => $tax ) : ?>
							<tr class="tax-rate tax-rate-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
								<th><?php echo esc_html( $tax->label ); ?></th>
								<td><?php echo wp_kses_post( $tax->formatted_amount ); ?></td>
							</tr>
						<?php endforeach; ?>
					<?php else : ?>
						<tr class="tax-total">
							<th><?php echo esc_html( KKART()->countries->tax_or_vat() ); ?></th>
							<td><?php kkart_cart_totals_taxes_total_html(); ?></td>
						</tr>
					<?php endif; ?>
				<?php endif; ?>
				
				<tr class="order-total">
					<th><?php esc_html_e( 'Total', 'kkart' ); ?></th>
					<td><?php kkart_cart_totals_order_total_html(); ?></td>
				</tr>
			</tfoot>
		</table>
		<?php
		return ob_get_clean();
	}
}

// TODO: verify the template
if( ! function_exists('kkart_cart_shipping_tmpl') ) {
	
	/*
	* params is an array
	*/
	function kkart_cart_shipping_tmpl( $params ){
		
		extract( $params );
		
		$formatted_destination    = isset( $formatted_destination ) ? $formatted_destination : KKART()->countries->get_formatted_address( $package['destination'], ', ' );
		$has_calculated_shipping  = ! empty( $has_calculated_shipping );
		//$show_shipping_calculator = ! empty( $show_shipping_calculator );

		$cart = is_cart();
?>
		<tr class="kkart-shipping-totals shipping">
			<?php if(!$cart){?>
			<th><?php  echo wp_kses_post( $package_name ); ?></th>
			<?php } ?>
			<td data-title="<?php echo esc_attr( $package_name ); ?>">
				<?php if ( $available_methods ) : ?>
					<ul id="shipping_method" class="kkart-shipping-methods " cart="<?php echo $cart ?>">
						<?php foreach ( $available_methods as $method ) : ?>
							<li>
								<?php
								if ( 1 < count( $available_methods ) ) {
									printf( '<input type="radio" name="shipping_method[%1$d]" data-index="%1$d" id="shipping_method_%1$d_%2$s" value="%3$s" class="shipping_method" %4$s />', $index, esc_attr( sanitize_title( $method->id ) ), esc_attr( $method->id ), checked( $method->id, $chosen_method, false ) ); // WPCS: XSS ok.
								} else {
									printf( '<input type="hidden" name="shipping_method[%1$d]" data-index="%1$d" id="shipping_method_%1$d_%2$s" value="%3$s" class="shipping_method" />', $index, esc_attr( sanitize_title( $method->id ) ), esc_attr( $method->id ) ); // WPCS: XSS ok.
								}
								printf( '<label for="shipping_method_%1$s_%2$s">%3$s</label>', $index, esc_attr( sanitize_title( $method->id ) ), kkart_cart_totals_shipping_method_label( $method ) ); // WPCS: XSS ok.
								do_action( 'kkart_after_shipping_rate', $method, $index );
								?>
							</li>
						<?php endforeach; ?>
					</ul>
					<?php if ( is_cart() ) : ?>
						<p class="kkart-shipping-destination">
							<?php
							if ( $formatted_destination ) {
								// Translators: $s shipping destination.
								printf( esc_html__( 'Shipping to %s.', 'kkart' ) . ' ', '<strong>' . esc_html( $formatted_destination ) . '</strong>' );
								$calculator_text = esc_html__( 'Change address', 'kkart' );
							} else {
								echo wp_kses_post( apply_filters( 'kkart_shipping_estimate_html', __( 'Shipping options will be updated during checkout.', 'kkart' ) ) );
							}
							?>
						</p>
					<?php endif; ?>
					<?php
				elseif ( ! $has_calculated_shipping || ! $formatted_destination ) :
					if ( is_cart() && 'no' === get_option( 'kkart_enable_shipping_calc' ) ) {
						echo wp_kses_post( apply_filters( 'kkart_shipping_not_enabled_on_cart_html', __( 'Shipping costs are calculated during checkout.', 'kkart' ) ) );
					} else {
						echo wp_kses_post( apply_filters( 'kkart_shipping_may_be_available_html', sprintf( __( 'No shipping options were found for %s.', 'kkart' ) . ' ', '<strong>' . esc_html( $formatted_destination ) . '</strong>' , 'kkart' )  ) );
					}
				elseif ( ! is_cart() ) :
					echo wp_kses_post( apply_filters( 'kkart_no_shipping_available_html', __( 'There are no shipping options available. Please ensure that your address has been entered correctly, or contact us if you need any help.', 'kkart' ) ) );
				else :
					// Translators: $s shipping destination.
					echo wp_kses_post( apply_filters( 'kkart_cart_no_shipping_available_html', sprintf( esc_html__( 'No shipping options were found for %s.', 'kkart' ) . ' ', '<strong>' . esc_html( $formatted_destination ) . '</strong>' ) ) );
					$calculator_text = esc_html__( 'Enter a different address', 'kkart' );
				endif;
				?>

				<?php if ( $show_package_details ) : ?>
					<?php echo '<p class="kkart-shipping-contents"><small>' . esc_html( $package_details ) . '</small></p>'; ?>
				<?php endif; ?>

				<?php if ( is_cart() && get_option('kkart_enable_shipping_calc') =="yes") : ?>
					<?php 
					$address_text = __('Add New Address');
					$address_class = 'kkart-address-form-headding';
					$get_address = kkart_get_user_address();
					
					if(!empty($get_address)){
						$address_text = __('Change Address');
						$address_class = 'kkart-address-selection';
					}
					
					echo '<a class="'.$address_class.'">'. $address_text .'</a>
					<div class="kkart-cart-modal-wrapper">
					<div class="kkart-cart-modal">
					<div class="shipping-headding">
						<h3>'.__('Shipping Address').'</h3>
						<a class="kkart-address-form-adder">'.__('Add New Address').'</a>
						<span class="kkart-close-address">&#10006</span>
					</div>
					'.kkart_checkout_billing( array(), 'shipping_cart').'</div></div>';
					?>
				<?php endif; ?>
			</td>
		</tr>

<?php
	}
}

if ( ! function_exists( 'kkart_breadcrumb' ) ) {

	/**
	 * Output the Kkart Breadcrumb.
	 *
	 * @param array $args Arguments.
	 */
	function kkart_breadcrumb( $args = array() ) {
		$args = wp_parse_args(
			$args,
			apply_filters(
				'kkart_breadcrumb_defaults',
				array(
					'delimiter'   => '&nbsp;&#47;&nbsp;',
					'wrap_before' => '<nav class="kkart-breadcrumb">',
					'wrap_after'  => '</nav>',
					'before'      => '',
					'after'       => '',
					'home'        => _x( 'Home', 'breadcrumb', 'kkart' ),
				)
			)
		);

		$breadcrumbs = new KKART_Breadcrumb();

		if ( ! empty( $args['home'] ) ) {
			$breadcrumbs->add_crumb( $args['home'], apply_filters( 'kkart_breadcrumb_home_url', home_url() ) );
		}

		$args['breadcrumb'] = $breadcrumbs->generate();

		/**
		 * Kkart Breadcrumb hook
		 *
		 * @hooked KKART_Structured_Data::generate_breadcrumblist_data() - 10
		 */
		do_action( 'kkart_breadcrumb', $breadcrumbs, $args );

		kkart_get_template( 'global/breadcrumb.php', $args );
	}
}

if ( ! function_exists( 'kkart_order_review' ) ) {

	/**
	 * Output the Order review table for the checkout.
	 *
	 * @param bool $deprecated Deprecated param.
	 */
	function kkart_order_review( $deprecated = false ) {
		kkart_get_template(
			'checkout/review-order.php',
			array(
				'checkout' => KKART()->checkout(),
			)
		);
	}
}

if ( ! function_exists( 'kkart_checkout_payment' ) ) {

	/**
	 * Output the Payment Methods on the checkout.
	 */
	function kkart_checkout_payment() {
		if ( KKART()->cart->needs_payment() ) {
			$available_gateways = KKART()->payment_gateways()->get_available_payment_gateways();
			KKART()->payment_gateways()->set_current_gateway( $available_gateways );
		} else {
			$available_gateways = array();
		}
		
		ob_start();
		
		foreach($available_gateways as $gateway){
						
			if($gateway->enabled == 'yes'): ?>
				<div class="kkart-payment-method">
					<div class="kkart-payment-method-header">
						<div>
							<input type="radio" name="payment_method" id="payment_method_<?php echo esc_attr( $gateway->id ) ?>" class="payment_method_radio" value="<?php echo esc_attr( $gateway->id ) ?>" <?php checked($gateway->chosen, true, false) ?> data-order_button_text="<?php echo esc_attr( $gateway->order_button_text ) ?>" />
							<span><?php echo esc_html( $gateway->title ) ?></span>
						</div>
						<span class="kkart_payment_icon"><?php echo $gateway->get_icon() ?></span>
					</div>
					<div class="kkart-payment-method-desc">
						<?php echo $gateway->description ?>
						<?php if( $gateway->has_fields() ){
							$gateway->payment_fields();
						}
						?>
					</div>
				</div>
			<?php endif; 
		}
		
		return ob_get_clean();
	}
}

if ( ! function_exists( 'kkart_tax_amount_html' ) ) {
	function kkart_tax_amount_html(){
		
		$div = '';
		
		if( kkart_tax_enabled() && ! KKART()->cart->display_prices_including_tax() ){ 
			if( 'itemized' === get_option( 'kkart_tax_total_display' ) ){
				foreach( KKART()->cart->get_tax_totals() as $code => $tax ){
					$div .= '<tr class="tax-rate tax-rate-'. esc_attr( sanitize_title( $code ) ).'">
						<th>'. esc_html( $tax->label ).'</th>
						<td>'. wp_kses_post( $tax->formatted_amount ) .'</td>
					</tr>';
				}
			}else{
				$div .= '<tr class="tax-total">
					<th>'. esc_html( KKART()->countries->tax_or_vat() ) .'</th>
					<td>'. kkart_cart_totals_taxes_total_html(false) .'</td>
				</tr>';
			}
		}
		
		return $div;
	}
}

if ( ! function_exists( 'kkart_coupon_price_html' ) ) {
	
	function kkart_coupon_price_html() {
		ob_start(); ?>
		<?php foreach ( KKART()->cart->get_coupons() as $code => $coupon ) : ?>
			<div class="kkart-cart-discount coupon-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
				<span><?php kkart_cart_totals_coupon_label( $coupon ); ?></span>
				<span data-title="<?php echo esc_attr( kkart_cart_totals_coupon_label( $coupon, false ) ); ?>"><?php kkart_cart_totals_coupon_html( $coupon ); ?></span>
			</div>
		<?php endforeach; 
		
		return ob_get_clean();
	}
}

if ( ! function_exists( 'kkart_checkout_coupon_form' ) ) {

	/**
	 * Output the Coupon form for the checkout.
	 */
	function kkart_checkout_coupon_form($el = array(), $echo = true){
		ob_start(); 
		?>
		<div class="kkart-form-coupon-toggle">
		<a href="#" class="showcoupon"><?php echo esc_html__( 'Add a Coupon', 'kkart' ); ?></a>

		</div>

		<form class="checkout_coupon kkart-form-coupon" method="post" style="display:none">
			<p class="form-row form-row-first">
				<input type="text" name="coupon_code" class="input-text" placeholder="<?php esc_attr_e( 'Coupon code', 'kkart' ); ?>" id="coupon_code" value="" />
			</p>

			<p class="form-row form-row-last">
				<button type="submit" class="button" name="apply_coupon" value="<?php esc_attr_e( 'Apply coupon', 'kkart' ); ?>"><?php esc_html_e( 'Apply', 'kkart' ); ?></button>
			</p>

			<div class="clear"></div>
		</form>
		<?php
		
		if(!$echo){
			return ob_get_clean();
		}
		
		echo ob_get_clean();
	}
}

if ( ! function_exists( 'kkart_checkout_terms' ) ) {

	/**
	 * Output the Coupon form for the checkout.
	 */
	function kkart_checkout_terms() {
		
		if ( apply_filters( 'kkart_checkout_show_terms', true ) && function_exists( 'kkart_terms_and_conditions_checkbox_enabled' ) ) {
			do_action( 'kkart_checkout_before_terms_and_conditions' );

			?>
			<div class="kkart-terms-and-conditions-wrapper">
				<?php
				/**
				 * Terms and conditions hook used to inject content.
				 *
				 * @since 3.4.0.
				 * @hooked kkart_checkout_privacy_policy_text() Shows custom privacy policy text. Priority 20.
				 * @hooked kkart_terms_and_conditions_page_content() Shows t&c page content. Priority 30.
				 */
				do_action( 'kkart_checkout_terms_and_conditions' );
				?>

				<?php if ( kkart_terms_and_conditions_checkbox_enabled() ) : ?>
					<p class="form-row validate-required">
						<label class="kkart-form__label kkart-form__label-for-checkbox checkbox">
						<input type="checkbox" class="kkart-form__input kkart-form__input-checkbox input-checkbox" name="terms" <?php checked( apply_filters( 'kkart_terms_is_checked_default', isset( $_POST['terms'] ) ), true ); // WPCS: input var ok, csrf ok. ?> id="terms" />
							<span class="kkart-terms-and-conditions-checkbox-text"><?php kkart_terms_and_conditions_checkbox_text(); ?></span>&nbsp;<span class="required">*</span>
						</label>
						<input type="hidden" name="terms-field" value="1" />
					</p>
				<?php endif; ?>
			</div>
			<?php

			do_action( 'kkart_checkout_after_terms_and_conditions' );
		}
	}
}

if ( ! function_exists( 'kkart_checkout_payment_method' ) ) {

	/**
	 * Output the Coupon form for the checkout.
	 */
	function kkart_checkout_payment_method() {
		
		?>
		<li class="kkart_payment_method payment_method_<?php echo esc_attr( $gateway->id ); ?>">
			<input id="payment_method_<?php echo esc_attr( $gateway->id ); ?>" type="radio" class="input-radio" name="payment_method" value="<?php echo esc_attr( $gateway->id ); ?>" <?php checked( $gateway->chosen, true ); ?> data-order_button_text="<?php echo esc_attr( $gateway->order_button_text ); ?>" />

			<label for="payment_method_<?php echo esc_attr( $gateway->id ); ?>">
				<?php echo $gateway->get_title(); /* phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped */ ?> <?php echo $gateway->get_icon(); /* phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped */ ?>
			</label>
			<?php if ( $gateway->has_fields() || $gateway->get_description() ) : ?>
				<div class="payment_box payment_method_<?php echo esc_attr( $gateway->id ); ?>" <?php if ( ! $gateway->chosen ) : /* phpcs:ignore Squiz.ControlStructures.ControlSignature.NewlineAfterOpenBrace */ ?>style="display:none;"<?php endif; /* phpcs:ignore Squiz.ControlStructures.ControlSignature.NewlineAfterOpenBrace */ ?>>
					<?php $gateway->payment_fields(); ?>
				</div>
			<?php endif; ?>
		</li>
		<?php
	}
}

if ( ! function_exists( 'kkart_products_will_display' ) ) {

	/**
	 * Check if we will be showing products or not (and not sub-categories only).
	 *
	 * @return bool
	 */
	function kkart_products_will_display() {
		$display_type = kkart_get_loop_display_mode();

		return 0 < kkart_get_loop_prop( 'total', 0 ) && 'subcategories' !== $display_type;
	}
}

if ( ! function_exists( 'kkart_get_loop_display_mode' ) ) {

	/**
	 * See what is going to display in the loop.
	 *
	 * @since 3.3.0
	 * @return string Either products, subcategories, or both, based on current page.
	 */
	function kkart_get_loop_display_mode() {
		// Only return products when filtering things.
		if ( kkart_get_loop_prop( 'is_search' ) || kkart_get_loop_prop( 'is_filtered' ) ) {
			return 'products';
		}

		$parent_id    = 0;
		$display_type = '';

		if ( is_shop() ) {
			$display_type = get_option( 'kkart_shop_page_display', '' );
		} elseif ( is_product_category() ) {
			$parent_id    = get_queried_object_id();
			$display_type = get_term_meta( $parent_id, 'display_type', true );
			$display_type = '' === $display_type ? get_option( 'kkart_category_archive_display', '' ) : $display_type;
		}

		if ( ( ! is_shop() || 'subcategories' !== $display_type ) && 1 < kkart_get_loop_prop( 'current_page' ) ) {
			return 'products';
		}

		// Ensure valid value.
		if ( '' === $display_type || ! in_array( $display_type, array( 'products', 'subcategories', 'both' ), true ) ) {
			$display_type = 'products';
		}

		// If we're showing categories, ensure we actually have something to show.
		if ( in_array( $display_type, array( 'subcategories', 'both' ), true ) ) {
			$subcategories = kkart_get_product_subcategories( $parent_id );

			if ( empty( $subcategories ) ) {
				$display_type = 'products';
			}
		}

		return $display_type;
	}
}

if ( ! function_exists( 'kkart_maybe_show_product_subcategories' ) ) {

	/**
	 * Maybe display categories before, or instead of, a product loop.
	 *
	 * @since 3.3.0
	 * @param string $loop_html HTML.
	 * @return string
	 */
	function kkart_maybe_show_product_subcategories( $loop_html = '' ) {
		if ( kkart_get_loop_prop( 'is_shortcode' ) && ! KKART_Template_Loader::in_content_filter() ) {
			return $loop_html;
		}

		$display_type = kkart_get_loop_display_mode();

		// If displaying categories, append to the loop.
		if ( 'subcategories' === $display_type || 'both' === $display_type ) {
			ob_start();
			kkart_output_product_categories(
				array(
					'parent_id' => is_product_category() ? get_queried_object_id() : 0,
				)
			);
			$loop_html .= ob_get_clean();

			if ( 'subcategories' === $display_type ) {
				kkart_set_loop_prop( 'total', 0 );

				// This removes pagination and products from display for themes not using kkart_get_loop_prop in their product loops.  @todo Remove in future major version.
				global $wp_query;

				if ( $wp_query->is_main_query() ) {
					$wp_query->post_count    = 0;
					$wp_query->max_num_pages = 0;
				}
			}
		}

		return $loop_html;
	}
}

if ( ! function_exists( 'kkart_product_subcategories' ) ) {
	/**
	 * This is a legacy function which used to check if we needed to display subcats and then output them. It was called by templates.
	 *
	 * From 3.3 onwards this is all handled via hooks and the kkart_maybe_show_product_subcategories function.
	 *
	 * Since some templates have not updated compatibility, to avoid showing incorrect categories this function has been deprecated and will
	 * return nothing. Replace usage with kkart_output_product_categories to render the category list manually.
	 *
	 * This is a legacy function which also checks if things should display.
	 * Themes no longer need to call these functions. It's all done via hooks.
	 *
	 * @deprecated 3.3.1 @todo Add a notice in a future version.
	 * @param array $args Arguments.
	 * @return null|boolean
	 */
	function kkart_product_subcategories( $args = array() ) {
		$defaults = array(
			'before'        => '',
			'after'         => '',
			'force_display' => false,
		);

		$args = wp_parse_args( $args, $defaults );

		if ( $args['force_display'] ) {
			// We can still render if display is forced.
			kkart_output_product_categories(
				array(
					'before'    => $args['before'],
					'after'     => $args['after'],
					'parent_id' => is_product_category() ? get_queried_object_id() : 0,
				)
			);
			return true;
		} else {
			// Output nothing. kkart_maybe_show_product_subcategories will handle the output of cats.
			$display_type = kkart_get_loop_display_mode();

			if ( 'subcategories' === $display_type ) {
				// This removes pagination and products from display for themes not using kkart_get_loop_prop in their product loops. @todo Remove in future major version.
				global $wp_query;

				if ( $wp_query->is_main_query() ) {
					$wp_query->post_count    = 0;
					$wp_query->max_num_pages = 0;
				}
			}

			return 'subcategories' === $display_type || 'both' === $display_type;
		}
	}
}

if ( ! function_exists( 'kkart_output_product_categories' ) ) {
	/**
	 * Display product sub categories as thumbnails.
	 *
	 * This is a replacement for kkart_product_subcategories which also does some logic
	 * based on the loop. This function however just outputs when called.
	 *
	 * @since 3.3.1
	 * @param array $args Arguments.
	 * @return boolean
	 */
	function kkart_output_product_categories( $args = array() ) {
		$args = wp_parse_args(
			$args,
			array(
				'before'    => apply_filters( 'kkart_before_output_product_categories', '' ),
				'after'     => apply_filters( 'kkart_after_output_product_categories', '' ),
				'parent_id' => 0,
			)
		);

		$product_categories = kkart_get_product_subcategories( $args['parent_id'] );

		if ( ! $product_categories ) {
			return false;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo $args['before'];

		foreach ( $product_categories as $category ) {
			kkart_get_template(
				'content-product_cat.php',
				array(
					'category' => $category,
				)
			);
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo $args['after'];

		return true;
	}
}

if ( ! function_exists( 'kkart_get_product_subcategories' ) ) {
	/**
	 * Get (and cache) product subcategories.
	 *
	 * @param int $parent_id Get subcategories of this ID.
	 * @return array
	 */
	function kkart_get_product_subcategories( $parent_id = 0 ) {
		$parent_id          = absint( $parent_id );
		$cache_key          = apply_filters( 'kkart_get_product_subcategories_cache_key', 'product-category-hierarchy-' . $parent_id, $parent_id );
		$product_categories = $cache_key ? wp_cache_get( $cache_key, 'product_cat' ) : false;

		if ( false === $product_categories ) {
			// NOTE: using child_of instead of parent - this is not ideal but due to a WP bug ( https://core.trac.wordpress.org/ticket/15626 ) pad_counts won't work.
			$product_categories = get_categories(
				apply_filters(
					'kkart_product_subcategories_args',
					array(
						'parent'       => $parent_id,
						'hide_empty'   => 0,
						'hierarchical' => 1,
						'taxonomy'     => 'product_cat',
						'pad_counts'   => 1,
					)
				)
			);

			if ( $cache_key ) {
				wp_cache_set( $cache_key, $product_categories, 'product_cat' );
			}
		}

		if ( apply_filters( 'kkart_product_subcategories_hide_empty', true ) ) {
			$product_categories = wp_list_filter( $product_categories, array( 'count' => 0 ), 'NOT' );
		}

		return $product_categories;
	}
}

if ( ! function_exists( 'kkart_subcategory_thumbnail' ) ) {

	/**
	 * Show subcategory thumbnails.
	 *
	 * @param mixed $category Category.
	 */
	function kkart_subcategory_thumbnail( $category ) {
		$small_thumbnail_size = apply_filters( 'subcategory_archive_thumbnail_size', 'kkart_thumbnail' );
		$dimensions           = kkart_get_image_size( $small_thumbnail_size );
		$thumbnail_id         = get_term_meta( $category->term_id, 'thumbnail_id', true );

		if ( $thumbnail_id ) {
			$image        = wp_get_attachment_image_src( $thumbnail_id, $small_thumbnail_size );
			$image        = $image[0];
			$image_srcset = function_exists( 'wp_get_attachment_image_srcset' ) ? wp_get_attachment_image_srcset( $thumbnail_id, $small_thumbnail_size ) : false;
			$image_sizes  = function_exists( 'wp_get_attachment_image_sizes' ) ? wp_get_attachment_image_sizes( $thumbnail_id, $small_thumbnail_size ) : false;
		} else {
			$image        = kkart_placeholder_img_src();
			$image_srcset = false;
			$image_sizes  = false;
		}

		if ( $image ) {
			// Prevent esc_url from breaking spaces in urls for image embeds.
			// Ref: https://core.trac.wordpress.org/ticket/23605.
			$image = str_replace( ' ', '%20', $image );

			// Add responsive image markup if available.
			if ( $image_srcset && $image_sizes ) {
				echo '<img src="' . esc_url( $image ) . '" alt="' . esc_attr( $category->name ) . '" width="' . esc_attr( $dimensions['width'] ) . '" height="' . esc_attr( $dimensions['height'] ) . '" srcset="' . esc_attr( $image_srcset ) . '" sizes="' . esc_attr( $image_sizes ) . '" />';
			} else {
				echo '<img src="' . esc_url( $image ) . '" alt="' . esc_attr( $category->name ) . '" width="' . esc_attr( $dimensions['width'] ) . '" height="' . esc_attr( $dimensions['height'] ) . '" />';
			}
		}
	}
}

if ( ! function_exists( 'kkart_order_details_table' ) ) {

	/**
	 * Displays order details in a table.
	 *
	 * @param mixed $order_id Order ID.
	 */
	function kkart_order_details_table( $order_id ) {
		if ( ! $order_id ) {
			return;
		}

		kkart_get_template(
			'order/order-details.php',
			array(
				'order_id' => $order_id,
			)
		);
	}
}

if ( ! function_exists( 'kkart_order_downloads_table' ) ) {

	/**
	 * Displays order downloads in a table.
	 *
	 * @since 3.2.0
	 * @param array $downloads Downloads.
	 */
	function kkart_order_downloads_table( $downloads ) {
		if ( ! $downloads ) {
			return;
		}
		kkart_get_template(
			'order/order-downloads.php',
			array(
				'downloads' => $downloads,
			)
		);
	}
}

if ( ! function_exists( 'kkart_order_again_button' ) ) {

	/**
	 * Display an 'order again' button on the view order page.
	 *
	 * @param object $order Order.
	 */
	function kkart_order_again_button( $order ) {
		if ( ! $order || ! $order->has_status( apply_filters( 'kkart_valid_order_statuses_for_order_again', array( 'completed' ) ) ) || ! is_user_logged_in() ) {
			return;
		}

		kkart_get_template(
			'order/order-again.php',
			array(
				'order'           => $order,
				'order_again_url' => wp_nonce_url( add_query_arg( 'order_again', $order->get_id(), kkart_get_cart_url() ), 'kkart-order_again' ),
			)
		);
	}
}

/** Forms */

if ( ! function_exists( 'kkart_form_field' ) ) {

	/**
	 * Outputs a checkout/address form field.
	 *
	 * @param string $key Key.
	 * @param mixed  $args Arguments.
	 * @param string $value (default: null).
	 * @return string
	 */
	function kkart_form_field( $key, $args, $value = null ) {
		$defaults = array(
			'type'              => 'text',
			'label'             => '',
			'description'       => '',
			'placeholder'       => '',
			'maxlength'         => false,
			'required'          => false,
			'autocomplete'      => false,
			'id'                => $key,
			'class'             => array(),
			'label_class'       => array(),
			'input_class'       => array(),
			'return'            => false,
			'options'           => array(),
			'custom_attributes' => array(),
			'validate'          => array(),
			'default'           => '',
			'autofocus'         => '',
			'priority'          => '',
		);

		$args = wp_parse_args( $args, $defaults );
		$args = apply_filters( 'kkart_form_field_args', $args, $key, $value );

		if ( $args['required'] ) {
			$args['class'][] = 'validate-required';
			$required        = '&nbsp;<abbr class="required" title="' . esc_attr__( 'required', 'kkart' ) . '">*</abbr>';
		} else {
			$required = '&nbsp;<span class="optional">(' . esc_html__( 'optional', 'kkart' ) . ')</span>';
		}

		if ( is_string( $args['label_class'] ) ) {
			$args['label_class'] = array( $args['label_class'] );
		}

		if ( is_null( $value ) ) {
			$value = $args['default'];
		}

		// Custom attribute handling.
		$custom_attributes         = array();
		$args['custom_attributes'] = array_filter( (array) $args['custom_attributes'], 'strlen' );

		if ( $args['maxlength'] ) {
			$args['custom_attributes']['maxlength'] = absint( $args['maxlength'] );
		}

		if ( ! empty( $args['autocomplete'] ) ) {
			$args['custom_attributes']['autocomplete'] = $args['autocomplete'];
		}

		if ( true === $args['autofocus'] ) {
			$args['custom_attributes']['autofocus'] = 'autofocus';
		}

		if ( $args['description'] ) {
			$args['custom_attributes']['aria-describedby'] = $args['id'] . '-description';
		}

		if ( ! empty( $args['custom_attributes'] ) && is_array( $args['custom_attributes'] ) ) {
			foreach ( $args['custom_attributes'] as $attribute => $attribute_value ) {
				$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
			}
		}

		if ( ! empty( $args['validate'] ) ) {
			foreach ( $args['validate'] as $validate ) {
				$args['class'][] = 'validate-' . $validate;
			}
		}

		$field           = '';
		$label_id        = $args['id'];
		$sort            = $args['priority'] ? $args['priority'] : '';
		$field_container = '<p class="form-row %1$s" id="%2$s" data-priority="' . esc_attr( $sort ) . '">%3$s</p>';

		switch ( $args['type'] ) {
			case 'country':
				$countries = 'shipping_country' === $key ? KKART()->countries->get_shipping_countries() : KKART()->countries->get_allowed_countries();

				if ( 1 === count( $countries ) ) {

					$field .= '<strong>' . current( array_values( $countries ) ) . '</strong>';

					$field .= '<input type="hidden" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" value="' . current( array_keys( $countries ) ) . '" ' . implode( ' ', $custom_attributes ) . ' class="country_to_state" readonly="readonly" />';

				} else {

					$field = '<select name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" class="country_to_state country_select ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" ' . implode( ' ', $custom_attributes ) . '><option value="default">' . esc_html__( 'Select a country / region&hellip;', 'kkart' ) . '</option>';

					foreach ( $countries as $ckey => $cvalue ) {
						$field .= '<option value="' . esc_attr( $ckey ) . '" ' . selected( $value, $ckey, false ) . '>' . esc_html( $cvalue ) . '</option>';
					}

					$field .= '</select>';

					$field .= '<noscript><button type="submit" name="kkart_checkout_update_totals" value="' . esc_attr__( 'Update country / region', 'kkart' ) . '">' . esc_html__( 'Update country / region', 'kkart' ) . '</button></noscript>';

				}

				break;
			case 'state':
				/* Get country this state field is representing */
				$for_country = isset( $args['country'] ) ? $args['country'] : KKART()->checkout->get_value( 'billing_state' === $key ? 'billing_country' : 'shipping_country' );
				$states = KKART()->countries->get_states( $for_country );

				if ( is_array( $states ) && empty( $states ) ) {

					$field_container = '<p class="form-row %1$s" id="%2$s" style="display: none">%3$s</p>';

					$field .= '<input type="hidden" class="hidden" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" value="" ' . implode( ' ', $custom_attributes ) . ' placeholder="' . esc_attr( $args['placeholder'] ) . '" readonly="readonly" data-input-classes="' . esc_attr( implode( ' ', $args['input_class'] ) ) . '"/>';

				} elseif ( ! is_null( $for_country ) && is_array( $states ) ) {

					$field .= '<select name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" class="state_select ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" ' . implode( ' ', $custom_attributes ) . ' data-placeholder="' . esc_attr( $args['placeholder'] ? $args['placeholder'] : esc_html__( 'Select an option&hellip;', 'kkart' ) ) . '"  data-input-classes="' . esc_attr( implode( ' ', $args['input_class'] ) ) . '">
						<option value="">' . esc_html__( 'Select an option&hellip;', 'kkart' ) . '</option>';

					foreach ( $states as $ckey => $cvalue ) {
						$field .= '<option value="' . esc_attr( $ckey ) . '" ' . selected( $value, $ckey, false ) . '>' . esc_html( $cvalue ) . '</option>';
					}

					$field .= '</select>';

				} else {

					$field .= '<input type="text" class="input-text ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" value="' . esc_attr( $value ) . '"  placeholder="' . esc_attr( $args['placeholder'] ) . '" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" ' . implode( ' ', $custom_attributes ) . ' data-input-classes="' . esc_attr( implode( ' ', $args['input_class'] ) ) . '"/>';

				}

				break;
			case 'textarea':
				$field .= '<textarea name="' . esc_attr( $key ) . '" class="input-text ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" id="' . esc_attr( $args['id'] ) . '" placeholder="' . esc_attr( $args['placeholder'] ) . '" ' . ( empty( $args['custom_attributes']['rows'] ) ? ' rows="2"' : '' ) . ( empty( $args['custom_attributes']['cols'] ) ? ' cols="5"' : '' ) . implode( ' ', $custom_attributes ) . '>' . esc_textarea( $value ) . '</textarea>';

				break;
			case 'checkbox':
				$field = '<label class="checkbox ' . implode( ' ', $args['label_class'] ) . '" ' . implode( ' ', $custom_attributes ) . '>
						<input type="' . esc_attr( $args['type'] ) . '" class="input-checkbox ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" value="1" ' . checked( $value, 1, false ) . ' /> ' . $args['label'] . $required . '</label>';

				break;
			case 'text':
			case 'password':
			case 'datetime':
			case 'datetime-local':
			case 'date':
			case 'month':
			case 'time':
			case 'week':
			case 'number':
			case 'email':
			case 'url':
			case 'tel':
				$field .= '<input type="' . esc_attr( $args['type'] ) . '" class="input-text ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" placeholder="' . esc_attr( $args['placeholder'] ) . '"  value="' . esc_attr( $value ) . '" ' . implode( ' ', $custom_attributes ) . ' />';

				break;
			case 'hidden':
				$field .= '<input type="' . esc_attr( $args['type'] ) . '" class="input-hidden ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" value="' . esc_attr( $value ) . '" ' . implode( ' ', $custom_attributes ) . ' />';

				break;
			case 'select':
				$field   = '';
				$options = '';

				if ( ! empty( $args['options'] ) ) {
					foreach ( $args['options'] as $option_key => $option_text ) {
						if ( '' === $option_key ) {
							// If we have a blank option, select2 needs a placeholder.
							if ( empty( $args['placeholder'] ) ) {
								$args['placeholder'] = $option_text ? $option_text : __( 'Choose an option', 'kkart' );
							}
							$custom_attributes[] = 'data-allow_clear="true"';
						}
						$options .= '<option value="' . esc_attr( $option_key ) . '" ' . selected( $value, $option_key, false ) . '>' . esc_html( $option_text ) . '</option>';
					}

					$field .= '<select name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" class="select ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" ' . implode( ' ', $custom_attributes ) . ' data-placeholder="' . esc_attr( $args['placeholder'] ) . '">
							' . $options . '
						</select>';
				}

				break;
			case 'radio':
				$label_id .= '_' . current( array_keys( $args['options'] ) );

				if ( ! empty( $args['options'] ) ) {
					foreach ( $args['options'] as $option_key => $option_text ) {
						$field .= '<input type="radio" class="input-radio ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" value="' . esc_attr( $option_key ) . '" name="' . esc_attr( $key ) . '" ' . implode( ' ', $custom_attributes ) . ' id="' . esc_attr( $args['id'] ) . '_' . esc_attr( $option_key ) . '"' . checked( $value, $option_key, false ) . ' />';
						$field .= '<label for="' . esc_attr( $args['id'] ) . '_' . esc_attr( $option_key ) . '" class="radio ' . implode( ' ', $args['label_class'] ) . '">' . esc_html( $option_text ) . '</label>';
					}
				}

				break;
		}

		if ( ! empty( $field ) ) {
			$field_html = '';

			if ( $args['label'] && 'checkbox' !== $args['type'] ) {
				$field_html .= '<label for="' . esc_attr( $label_id ) . '" class="' . esc_attr( implode( ' ', $args['label_class'] ) ) . '">' . wp_kses_post( $args['label'] ) . $required . '</label>';
			}

			$field_html .= '<span class="kkart-input-wrapper">' . $field;

			if ( $args['description'] ) {
				$field_html .= '<span class="description" id="' . esc_attr( $args['id'] ) . '-description" aria-hidden="true">' . wp_kses_post( $args['description'] ) . '</span>';
			}

			$field_html .= '</span>';

			$container_class = esc_attr( implode( ' ', $args['class'] ) );
			$container_id    = esc_attr( $args['id'] ) . '_field';
			$field           = sprintf( $field_container, $container_class, $container_id, $field_html );
		}

		/**
		 * Filter by type.
		 */
		$field = apply_filters( 'kkart_form_field_' . $args['type'], $field, $key, $args, $value );

		/**
		 * General filter on form fields.
		 *
		 * @since 3.4.0
		 */
		$field = apply_filters( 'kkart_form_field', $field, $key, $args, $value );

		if ( $args['return'] ) {
			return $field;
		} else {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $field;
		}
	}
}

if ( ! function_exists( 'get_product_search_form' ) ) {

	/**
	 * Display product search form.
	 *
	 * Will first attempt to locate the product-searchform.php file in either the child or.
	 * the parent, then load it. If it doesn't exist, then the default search form.
	 * will be displayed.
	 *
	 * The default searchform uses html5.
	 *
	 * @param bool $echo (default: true).
	 * @return string
	 */
	function get_product_search_form( $echo = true ) {
		global $product_search_form_index;

		ob_start();

		if ( empty( $product_search_form_index ) ) {
			$product_search_form_index = 0;
		}

		do_action( 'pre_get_product_search_form' );

		kkart_get_template(
			'product-searchform.php',
			array(
				'index' => $product_search_form_index++,
			)
		);

		$form = apply_filters( 'get_product_search_form', ob_get_clean() );

		if ( ! $echo ) {
			return $form;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo $form;
	}
}

if ( ! function_exists( 'kkart_output_auth_header' ) ) {

	/**
	 * Output the Auth header.
	 */
	function kkart_output_auth_header() {
		kkart_get_template( 'auth/header.php' );
	}
}

if ( ! function_exists( 'kkart_output_auth_footer' ) ) {

	/**
	 * Output the Auth footer.
	 */
	function kkart_output_auth_footer() {
		kkart_get_template( 'auth/footer.php' );
	}
}

if ( ! function_exists( 'kkart_single_variation' ) ) {

	/**
	 * Output placeholders for the single variation.
	 */
	function kkart_single_variation() {
		echo '<div class="kkart-variation single_variation"></div>';
	}
}

if ( ! function_exists( 'kkart_single_variation_add_to_cart_button' ) ) {

	/**
	 * Output the add to cart button for variations.
	 */
	function kkart_single_variation_add_to_cart_button($el = array()) {
		global $product;
		?>
		<div class="kkart-variation-add-to-cart variations_button">
			<?php do_action( 'kkart_before_add_to_cart_button' ); ?>

			<?php
			do_action( 'kkart_before_add_to_cart_quantity' );
			echo '<div class="kkart-product-quantity-holder">';
			
			if(!empty($el['atts']['show_quantity'])){
				kkart_quantity_input(
					array(
						'min_value'   => apply_filters( 'kkart_quantity_input_min', $product->get_min_purchase_quantity(), $product ),
						'max_value'   => apply_filters( 'kkart_quantity_input_max', $product->get_max_purchase_quantity(), $product ),
						'input_value' => isset( $_POST['quantity'] ) ? kkart_stock_amount( wp_unslash( $_POST['quantity'] ) ) : $product->get_min_purchase_quantity(), // WPCS: CSRF ok, input var ok.
					)
				);
			}
			
			echo '</div>';
			do_action( 'kkart_after_add_to_cart_quantity' );
			?>
			
			<button type="submit" class="kkart-cart-btn-holder single_add_to_cart_button">
				
				<?php 
				if( !empty($el['atts']['cart_icon'])){
					echo '<i class="'. $el['atts']['cart_icon'].' kkart-cart-btn-icon"></i>';
				}
				
				echo '<span class="kkart-cart-btn-text">'. esc_html( empty($el['atts']['cart_text']) ? $product->single_add_to_cart_text() : $el['atts']['cart_text'] ) .'</span>';
				
				if( !empty($el['atts']['cart_icon'])){
					echo '<i class="'. $el['atts']['cart_icon'].' kkart-cart-btn-icon"></i>';
				}
				?>
				
			</button>

			<?php do_action( 'kkart_after_add_to_cart_button' ); ?>

			<input type="hidden" name="add-to-cart" value="<?php echo absint( $product->get_id() ); ?>" />
			<input type="hidden" name="product_id" value="<?php echo absint( $product->get_id() ); ?>" />
			<input type="hidden" name="variation_id" class="variation_id" value="0" />
		</div>
<?php
	}
}

if ( ! function_exists( 'kkart_dropdown_variation_attribute_options' ) ) {

	/**
	 * Output a list of variation attributes for use in the cart forms.
	 *
	 * @param array $args Arguments.
	 * @since 2.4.0
	 */
	function kkart_dropdown_variation_attribute_options( $args = array() ) {
		$args = wp_parse_args(
			apply_filters( 'kkart_dropdown_variation_attribute_options_args', $args ),
			array(
				'options'          => false,
				'attribute'        => false,
				'product'          => false,
				'selected'         => false,
				'name'             => '',
				'id'               => '',
				'class'            => '',
				'show_option_none' => __( 'Choose an option', 'kkart' ),
			)
		);

		// Get selected value.
		if ( false === $args['selected'] && $args['attribute'] && $args['product'] instanceof KKART_Product ) {
			$selected_key = 'attribute_' . sanitize_title( $args['attribute'] );
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			$args['selected'] = isset( $_REQUEST[ $selected_key ] ) ? kkart_clean( wp_unslash( $_REQUEST[ $selected_key ] ) ) : $args['product']->get_variation_default_attribute( $args['attribute'] );
			// phpcs:enable WordPress.Security.NonceVerification.Recommended
		}

		$options               = $args['options'];
		$product               = $args['product'];
		$attribute             = $args['attribute'];
		$name                  = $args['name'] ? $args['name'] : 'attribute_' . sanitize_title( $attribute );
		$id                    = $args['id'] ? $args['id'] : sanitize_title( $attribute );
		$class                 = $args['class'];
		$show_option_none      = (bool) $args['show_option_none'];
		$show_option_none_text = $args['show_option_none'] ? $args['show_option_none'] : __( 'Choose an option', 'kkart' ); // We'll do our best to hide the placeholder, but we'll need to show something when resetting options.

		if ( empty( $options ) && ! empty( $product ) && ! empty( $attribute ) ) {
			$attributes = $product->get_variation_attributes();
			$options    = $attributes[ $attribute ];
		}

		$html  = '<select id="' . esc_attr( $id ) . '" class="' . esc_attr( $class ) . '" name="' . esc_attr( $name ) . '" data-attribute_name="attribute_' . esc_attr( sanitize_title( $attribute ) ) . '" data-show_option_none="' . ( $show_option_none ? 'yes' : 'no' ) . '">';
		$html .= '<option value="">' . esc_html( $show_option_none_text ) . '</option>';

		if ( ! empty( $options ) ) {
			if ( $product && taxonomy_exists( $attribute ) ) {
				// Get terms if this is a taxonomy - ordered. We need the names too.
				$terms = kkart_get_product_terms(
					$product->get_id(),
					$attribute,
					array(
						'fields' => 'all',
					)
				);

				foreach ( $terms as $term ) {
					if ( in_array( $term->slug, $options, true ) ) {
						$html .= '<option value="' . esc_attr( $term->slug ) . '" ' . selected( sanitize_title( $args['selected'] ), $term->slug, false ) . '>' . esc_html( apply_filters( 'kkart_variation_option_name', $term->name, $term, $attribute, $product ) ) . '</option>';
					}
				}
			} else {
				foreach ( $options as $option ) {
					// This handles < 2.4.0 bw compatibility where text attributes were not sanitized.
					$selected = sanitize_title( $args['selected'] ) === $args['selected'] ? selected( $args['selected'], sanitize_title( $option ), false ) : selected( $args['selected'], $option, false );
					$html    .= '<option value="' . esc_attr( $option ) . '" ' . $selected . '>' . esc_html( apply_filters( 'kkart_variation_option_name', $option, null, $attribute, $product ) ) . '</option>';
				}
			}
		}

		$html .= '</select>';

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo apply_filters( 'kkart_dropdown_variation_attribute_options_html', $html, $args );
	}
}

if ( ! function_exists( 'kkart_account_content' ) ) {

	/**
	 * My Account content output.
	 */
	function kkart_account_content() {
		global $wp;

		if ( ! empty( $wp->query_vars ) ) {
			foreach ( $wp->query_vars as $key => $value ) {
				// Ignore pagename param.
				if ( 'pagename' === $key ) {
					continue;
				}

				if ( has_action( 'kkart_account_' . $key . '_endpoint' ) ) {
					do_action( 'kkart_account_' . $key . '_endpoint', $value );
					return;
				}
			}
		}

		// No endpoint found? Default to dashboard.
		kkart_get_template(
			'myaccount/dashboard.php',
			array(
				'current_user' => get_user_by( 'id', get_current_user_id() ),
			)
		);
	}
}

if ( ! function_exists( 'kkart_account_navigation' ) ) {

	/**
	 * My Account navigation template.
	 */
	function kkart_account_navigation() {
		kkart_get_template( 'myaccount/navigation.php' );
	}
}

if ( ! function_exists( 'kkart_account_orders' ) ) {

	/**
	 * My Account > Orders template.
	 *
	 * @param int $current_page Current page number.
	 */
	function kkart_account_orders( $current_page,  $order_col = array()) {
		$current_page    = empty( $current_page ) ? 1 : absint( $current_page );
		/* $customer_orders = kkart_get_orders(
			apply_filters(
				'kkart_my_account_my_orders_query',
				array(
					'customer' => get_current_user_id(),
					'page'     => $current_page,
					'paginate' => true,
				)
			)
		); */
		//TODO Pagination
		$current_user_id = get_current_user_id();
		
		$order_ids = kkart_get_orders(
			array(
				'limit'    => -1,
				'customer' => $current_user_id,
				'return'   => 'ids',
			)
		);

		$order_content = '';
		if(!empty($order_ids)){
			
			$order_content .= '<div class="kkart-myaccount-details kkart-myaccount-orderdetails"></div>';
			$order_content .= '<table class="kkart-myaccount-table kkart-myaccount-orders"><thead><tr>';
			foreach ( kkart_get_account_orders_columns() as $column_id => $column_name ) :
					
				if(in_array($column_id, $order_col)){
						
					$order_content .= '<th><span>'. esc_html( $column_name ) .'</span></th>';
				}
					
			endforeach;
				
			$order_content .= '</tr></thead>';
			$order_content .= '<tbody>';
				
			foreach ( $order_ids as $ik => $iv ) {
				
				$order      = kkart_get_order( $iv ); 
				$item_count = $order->get_item_count() - $order->get_item_count_refunded();
				$order_content .= '<tr>';
				
				foreach ( kkart_get_account_orders_columns() as $column_id => $column_name ){
					
					if(in_array($column_id, $order_col)){
						
						$order_content .= '<td data-title="'. esc_attr( $column_name ) .'">';
						
						if ( has_action( 'kkart_my_account_my_orders_column_' . $column_id ) ){
							
							do_action( 'kkart_my_account_my_orders_column_' . $column_id, $order );
						}elseif ( 'order-number' === $column_id ){
							
							$order_content .= '<a href="javascript:void(0)" class="kkart-order-view" data-order-key="'.$order->get_order_number().'">'. esc_html( _x( '#', 'hash before order number', 'kkart' ) . $order->get_order_number() ) .'</a>';
						}elseif ( 'order-date' === $column_id ){
							
							$order_content .= '<time datetime="'. esc_attr( $order->get_date_created()->date( 'c' ) ) .'">'. esc_html( kkart_format_datetime( $order->get_date_created() ) ) .'</time>';
						}elseif ( 'order-status' === $column_id ){
							
							$order_content .= esc_html( kkart_get_order_status_name( $order->get_status() ) );
						}elseif ( 'order-total' === $column_id ){
							
							$order_content .= wp_kses_post( sprintf( _n( '%1$s for %2$s item', '%1$s for %2$s items', $item_count, 'kkart' ), $order->get_formatted_order_total(), $item_count ) );
						}elseif ( 'order-actions' === $column_id ){
							
							$actions = kkart_get_account_orders_actions( $order );
							
							if ( ! empty( $actions ) ) {
								
								foreach ( $actions as $key => $action ) { 
									
									$order_content .= '<a href="javascript:void(0)" class="kkart-order-'.sanitize_html_class( $key ).'" data-order-key="'.$order->get_order_number().'">' . esc_html( $action['name'] ) . '</a>';
								}
							}
						}
						$order_content .= '</td>';
					}
				}
			}
			$order_content .= '</tbody></table>';
		}else{
			
			$order_content .= '<div class="kkart-message kkart-message--info kkart-Message kkart-Message--info kkart-info"><a class="kkart-Button button" href="'.esc_url(kkart_get_page_permalink( 'shop' )).'">Browse</a>No order has been made yet.</div>';
		}
		
		return $order_content;
	}
}

if ( ! function_exists( 'kkart_account_view_order' ) ) {

	/**
	 * My Account > View order template.
	 *
	 * @param int $order_id Order ID.
	 */
	function kkart_account_view_order( $order_id ) {
		//KKART_Shortcode_My_Account::view_order( absint( $order_id ) );
		$order = kkart_get_order( absint( $order_id ) );
		$result = '';
		
		if ( ! $order || ! current_user_can( 'view_order', $order_id ) ) {
			
			$result = '<div class="kkart-error">' . esc_html__( 'Invalid order.', 'kkart' ) . ' <a href="' . esc_url( kkart_get_page_permalink( 'myaccount' ) ) . '" class="kkart-forward">' . esc_html__( 'My account', 'kkart' ) . '</a></div>';
			$result = '<button class="kkart-close-orderdetails">Close</button>';
		}else{

			// Backwards compatibility.
			$status       = new stdClass();
			$status->name = kkart_get_order_status_name( $order->get_status() );
			
			$notes = $order->get_customer_order_notes();
			
			$result .= sprintf(esc_html__( 'Order #%1$s was placed on %2$s and is currently %3$s.', 'kkart' ),$order->get_order_number(), kkart_format_datetime( $order->get_date_created() ), kkart_get_order_status_name( $order->get_status() ));
			
			if ( $notes ) :
				$result .= '<p>'.esc_html_e( 'Order updates', 'kkart' ).'</p><ol>';
				foreach ( $notes as $note ) :
					$result .= '<li><div><div><p>'.date_i18n( esc_html__( 'l jS \o\f F Y, h:ia', 'kkart' ), strtotime( $note->comment_date ) ).'</p><div>'.wpautop( wptexturize( $note->comment_content ) ).'</div><div></div></div><div></div></div></li>';
				endforeach;
				$result .= '</ol>';
			endif;
			
			$order_items           = $order->get_items( apply_filters( 'kkart_purchase_order_item_types', 'line_item' ) );
			$show_purchase_note    = $order->has_status( apply_filters( 'kkart_purchase_note_order_statuses', array( 'completed', 'processing' ) ) );
			$show_customer_details = is_user_logged_in() && $order->get_user_id() === get_current_user_id();
			$downloads             = $order->get_downloadable_items();
			$show_downloads        = $order->has_downloadable_item() && $order->is_download_permitted();
			
			$result .= '<section><p>order details</p>';
			$result .= '<table class="kkart-myaccount-table"><thead><tr>
			<th>product</th>
			<th>Total</th></tr></thead><tbody>';
			foreach ( $order_items as $item_id => $item ) {
				
				$product = $item->get_product();
				if ( ! apply_filters( 'kkart_order_item_visible', true, $item ) ) {
					return $result;
				}
				
				$result .= '<tr><td>';
				$is_visible        = $product && $product->is_visible();
				$product_permalink = apply_filters( 'kkart_order_item_permalink', $is_visible ? $product->get_permalink( $item ) : '', $item, $order );

				$result .= apply_filters( 'kkart_order_item_name', $product_permalink ? sprintf( '<a href="%s">%s</a>', $product_permalink, $item->get_name() ) : $item->get_name(), $item, $is_visible );

				$qty          = $item->get_quantity();
				$refunded_qty = $order->get_qty_refunded_for_item( $item_id );

				if ( $refunded_qty ) {
					$qty_display = '<del>' . esc_html( $qty ) . '</del> <ins>' . esc_html( $qty - ( $refunded_qty * -1 ) ) . '</ins>';
				} else {
					$qty_display = esc_html( $qty );
				}

				$result .= apply_filters( 'kkart_order_item_quantity_html', ' &nbsp;<strong>(' . sprintf( '&times;&nbsp;%s', $qty_display ) . ')</strong>', $item );

				$result .= kkart_display_item_meta( $item ); 

				$result .= '</td><td>'.$order->get_formatted_line_subtotal( $item ).'</td></tr>';
				
				if ( $show_purchase_note && $purchase_note ) :
					$result .= '<tr><td colspan="2">'.wpautop( do_shortcode( wp_kses_post( $purchase_note ) ) ).'</td></tr>';
				endif;
			}
			$result .= '</tbody><tfoot>';
			foreach ( $order->get_order_item_totals() as $key => $total ) {
				$result .= '<tr><th scope="row">'.$total['label'].'</th><td>'.$total['value'].'</td></tr>';
			}
			if ( $order->get_customer_note() ) : 
				$result .= '<tr><th>'.esc_html_e( 'Note:', 'kkart' ).'</th><td>'.wp_kses_post( nl2br( wptexturize( $order->get_customer_note() ) ) ).'</td></tr>';
			endif;
			$result .= '</tfoot></table></section>';
			$result .= '<button class="kkart-close-orderdetails kkart-close-button">Close</button>';
		}
		
		return $result;
	}
}

if ( ! function_exists( 'kkart_account_downloads' ) ) {

	/**
	 * My Account > Downloads template.
	 */
	function kkart_account_downloads($download_col = array()) {
		
		$downloads = KKART()->customer->get_downloadable_products();
		$has_downloads = (bool) $downloads;
		
		$download_content = '';
		
		if($has_downloads){
			
			$download_content .= '<table class="kkart-myaccount-table kkart-myaccount-downloads">
				<thead><tr>';
			
			foreach ( kkart_get_account_downloads_columns() as $column_id => $column_name ){
				if(in_array($column_id, $download_col)){
					$download_content .= '<th>'.$column_name.'</th>';
				}
			}
			
			$download_content .= '</tr></thead>
				<tbody>';
				
			foreach($downloads as $dk => $dv){
				
				$download_content .= '<tr>';
				foreach(kkart_get_account_downloads_columns() as $column_id => $column_name){
					
					if(in_array($column_id, $download_col)){
						$download_content .= '<td>';
						if('download-product' == $column_id){
							if ( $dv['product_url'] ) {
								$download_content .= '<a href="' . esc_url( $dv['product_url'] ) . '">' . esc_html( $dv['product_name'] ) . '</a>';
							} else {
								$download_content .= esc_html( $dv['product_name'] );
							}
						}
						if('download-file' == $column_id){
							$download_content .= '<a href="' . esc_url( $dv['download_url'] ) . '" class="kkart-MyAccount-downloads-file button alt">' . esc_html( $dv['download_name'] ) . '</a>';
						}
						if('download-remaining' == $column_id){
							$download_content .= is_numeric( $dv['downloads_remaining'] ) ? $dv['downloads_remaining'] : 'infinity';
						}
						if('download-expires' == $column_id){
							if ( ! empty( $dv['access_expires'] ) ) {
								$download_content .= '<time datetime="' . esc_attr( date( 'Y-m-d', strtotime( $dv['access_expires'] ) ) ) . '" title="' . esc_attr( strtotime( $dv['access_expires'] ) ) . '">' . date_i18n( get_option( 'date_format' ), strtotime( $dv['access_expires'] ) ) . '</time>';
							} else {
								$download_content .= esc_html__( 'Never', 'kkart' );
							}
						}
						$download_content .= '</td>';
					}
				}
				$download_content .= '</tr>';
			}
			$download_content .= '</tbody>
				</table>';
		}else{
			$download_content .= '<div class="kkart-message kkart-message--info kkart-Message kkart-Message--info kkart-info"><a class="kkart-Button button kkart-inline-button" href="'.esc_url(kkart_get_page_permalink( 'shop' )).'">Go to Shop</a>No downloads are available yet.</div>';
		}
		
		return $download_content;
	}
}

if ( ! function_exists( 'kkart_account_edit_address' ) ) {

	/**
	 * My Account > Edit address template.
	 *
	 * @param string $type Address type.
	 */
	function kkart_account_edit_address( $type ) {
		$type = kkart_edit_address_i18n( sanitize_title( $type ), true );

		KKART_Shortcode_My_Account::edit_address( $type );
	}
}

if ( ! function_exists( 'kkart_account_payment_methods' ) ) {

	/**
	 * My Account > Downloads template.
	 */
	function kkart_account_payment_methods() {
		kkart_get_template( 'myaccount/payment-methods.php' );
	}
}

if ( ! function_exists( 'kkart_account_add_payment_method' ) ) {

	/**
	 * My Account > Add payment method template.
	 */
	function kkart_account_add_payment_method() {
		KKART_Shortcode_My_Account::add_payment_method();
	}
}

if ( ! function_exists( 'kkart_account_edit_account' ) ) {

	/**
	 * My Account > Edit account template.
	 */
	function kkart_account_edit_account() {
		KKART_Shortcode_My_Account::edit_account();
	}
}

if ( ! function_exists( 'kkart_no_products_found' ) ) {

	/**
	 * Handles the loop when no products were found/no product exist.
	 */
	function kkart_no_products_found() {
		echo '<p class="kkart-info">'. esc_html( 'No products were found matching your selection.', 'kkart' ) .'</p>';
	}
}


if ( ! function_exists( 'kkart_get_email_order_items' ) ) {
	/**
	 * Get HTML for the order items to be shown in emails.
	 *
	 * @param KKART_Order $order Order object.
	 * @param array    $args Arguments.
	 *
	 * @since 3.0.0
	 * @return string
	 */
	function kkart_get_email_order_items( $order, $args = array() ) {
		ob_start();

		$defaults = array(
			'show_sku'      => false,
			'show_image'    => false,
			'image_size'    => array( 32, 32 ),
			'plain_text'    => false,
			'sent_to_admin' => false,
		);

		$args     = wp_parse_args( $args, $defaults );
		$template = $args['plain_text'] ? 'emails/plain/email-order-items.php' : 'emails/email-order-items.php';

		kkart_get_template(
			$template,
			apply_filters(
				'kkart_email_order_items_args',
				array(
					'order'               => $order,
					'items'               => $order->get_items(),
					'show_download_links' => $order->is_download_permitted() && ! $args['sent_to_admin'],
					'show_sku'            => $args['show_sku'],
					'show_purchase_note'  => $order->is_paid() && ! $args['sent_to_admin'],
					'show_image'          => $args['show_image'],
					'image_size'          => $args['image_size'],
					'plain_text'          => $args['plain_text'],
					'sent_to_admin'       => $args['sent_to_admin'],
				)
			)
		);

		return apply_filters( 'kkart_email_order_items_table', ob_get_clean(), $order );
	}
}

if ( ! function_exists( 'kkart_display_item_meta' ) ) {
	/**
	 * Display item meta data.
	 *
	 * @since  3.0.0
	 * @param  KKART_Order_Item $item Order Item.
	 * @param  array         $args Arguments.
	 * @return string|void
	 */
	function kkart_display_item_meta( $item, $args = array() ) {
		$strings = array();
		$html    = '';
		$args    = wp_parse_args(
			$args,
			array(
				'before'       => '<ul class="kkart-item-meta"><li>',
				'after'        => '</li></ul>',
				'separator'    => '</li><li>',
				'echo'         => true,
				'autop'        => false,
				'label_before' => '<strong class="kkart-item-meta-label">',
				'label_after'  => ':</strong> ',
			)
		);

		foreach ( $item->get_formatted_meta_data() as $meta_id => $meta ) {
			$value     = $args['autop'] ? wp_kses_post( $meta->display_value ) : wp_kses_post( make_clickable( trim( $meta->display_value ) ) );
			$strings[] = $args['label_before'] . wp_kses_post( $meta->display_key ) . $args['label_after'] . $value;
		}

		if ( $strings ) {
			$html = $args['before'] . implode( $args['separator'], $strings ) . $args['after'];
		}

		$html = apply_filters( 'kkart_display_item_meta', $html, $item, $args );

		if ( $args['echo'] ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $html;
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'kkart_display_item_downloads' ) ) {
	/**
	 * Display item download links.
	 *
	 * @since  3.0.0
	 * @param  KKART_Order_Item $item Order Item.
	 * @param  array         $args Arguments.
	 * @return string|void
	 */
	function kkart_display_item_downloads( $item, $args = array() ) {
		$strings = array();
		$html    = '';
		$args    = wp_parse_args(
			$args,
			array(
				'before'    => '<ul class ="kkart-item-downloads"><li>',
				'after'     => '</li></ul>',
				'separator' => '</li><li>',
				'echo'      => true,
				'show_url'  => false,
			)
		);

		$downloads = is_object( $item ) && $item->is_type( 'line_item' ) ? $item->get_item_downloads() : array();

		if ( $downloads ) {
			$i = 0;
			foreach ( $downloads as $file ) {
				$i ++;

				if ( $args['show_url'] ) {
					$strings[] = '<strong class="kkart-item-download-label">' . esc_html( $file['name'] ) . ':</strong> ' . esc_html( $file['download_url'] );
				} else {
					/* translators: %d: downloads count */
					$prefix    = count( $downloads ) > 1 ? sprintf( __( 'Download %d', 'kkart' ), $i ) : __( 'Download', 'kkart' );
					$strings[] = '<strong class="kkart-item-download-label">' . $prefix . ':</strong> <a href="' . esc_url( $file['download_url'] ) . '" target="_blank">' . esc_html( $file['name'] ) . '</a>';
				}
			}
		}

		if ( $strings ) {
			$html = $args['before'] . implode( $args['separator'], $strings ) . $args['after'];
		}

		$html = apply_filters( 'kkart_display_item_downloads', $html, $item, $args );

		if ( $args['echo'] ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo $html;
		} else {
			return $html;
		}
	}
}

if ( ! function_exists( 'kkart_photoswipe' ) ) {

	/**
	 * Get the shop sidebar template.
	 */
	function kkart_photoswipe() {
		if ( current_theme_supports( 'kkart-product-gallery-lightbox' ) ) {
		?>	
			<div class="pswp" tabindex="-1" role="dialog" aria-hidden="true">
				<div class="pswp__bg"></div>
				<div class="pswp__scroll-wrap">
					<div class="pswp__container">
						<div class="pswp__item"></div>
						<div class="pswp__item"></div>
						<div class="pswp__item"></div>
					</div>
					<div class="pswp__ui pswp__ui--hidden">
						<div class="pswp__top-bar">
							<div class="pswp__counter"></div>
							<button class="pswp__button pswp__button--close" aria-label="<?php esc_attr_e( 'Close (Esc)', 'kkart' ); ?>"></button>
							<button class="pswp__button pswp__button--share" aria-label="<?php esc_attr_e( 'Share', 'kkart' ); ?>"></button>
							<button class="pswp__button pswp__button--fs" aria-label="<?php esc_attr_e( 'Toggle fullscreen', 'kkart' ); ?>"></button>
							<button class="pswp__button pswp__button--zoom" aria-label="<?php esc_attr_e( 'Zoom in/out', 'kkart' ); ?>"></button>
							<div class="pswp__preloader">
								<div class="pswp__preloader__icn">
									<div class="pswp__preloader__cut">
										<div class="pswp__preloader__donut"></div>
									</div>
								</div>
							</div>
						</div>
						<div class="pswp__share-modal pswp__share-modal--hidden pswp__single-tap">
							<div class="pswp__share-tooltip"></div>
						</div>
						<button class="pswp__button pswp__button--arrow--left" aria-label="<?php esc_attr_e( 'Previous (arrow left)', 'kkart' ); ?>"></button>
						<button class="pswp__button pswp__button--arrow--right" aria-label="<?php esc_attr_e( 'Next (arrow right)', 'kkart' ); ?>"></button>
						<div class="pswp__caption">
							<div class="pswp__caption__center"></div>
						</div>
					</div>
				</div>
			</div>
		<?php
		}
	}
}

/**
 * Outputs a list of product attributes for a product.
 *
 * @since  3.0.0
 * @param  KKART_Product $product Product Object.
 */
function kkart_display_product_attributes( $product ) {
	$product_attributes = array();

	// Display weight and dimensions before attribute list.
	$display_dimensions = apply_filters( 'kkart_product_enable_dimensions_display', $product->has_weight() || $product->has_dimensions() );

	if ( $display_dimensions && $product->has_weight() ) {
		$product_attributes['weight'] = array(
			'label' => __( 'Weight', 'kkart' ),
			'value' => kkart_format_weight( $product->get_weight() ),
		);
	}

	if ( $display_dimensions && $product->has_dimensions() ) {
		$product_attributes['dimensions'] = array(
			'label' => __( 'Dimensions', 'kkart' ),
			'value' => kkart_format_dimensions( $product->get_dimensions( false ) ),
		);
	}

	// Add product attributes to list.
	$attributes = array_filter( $product->get_attributes(), 'kkart_attributes_array_filter_visible' );

	foreach ( $attributes as $attribute ) {
		$values = array();

		if ( $attribute->is_taxonomy() ) {
			$attribute_taxonomy = $attribute->get_taxonomy_object();
			$attribute_values   = kkart_get_product_terms( $product->get_id(), $attribute->get_name(), array( 'fields' => 'all' ) );

			foreach ( $attribute_values as $attribute_value ) {
				$value_name = esc_html( $attribute_value->name );

				if ( $attribute_taxonomy->attribute_public ) {
					$values[] = '<a href="' . esc_url( get_term_link( $attribute_value->term_id, $attribute->get_name() ) ) . '" rel="tag">' . $value_name . '</a>';
				} else {
					$values[] = $value_name;
				}
			}
		} else {
			$values = $attribute->get_options();

			foreach ( $values as &$value ) {
				$value = make_clickable( esc_html( $value ) );
			}
		}

		$product_attributes[ 'attribute_' . sanitize_title_with_dashes( $attribute->get_name() ) ] = array(
			'label' => kkart_attribute_label( $attribute->get_name() ),
			'value' => apply_filters( 'kkart_attribute', wpautop( wptexturize( implode( ', ', $values ) ) ), $attribute, $values ),
		);
	}

	/**
	 * Hook: kkart_display_product_attributes.
	 *
	 * @since 3.6.0.
	 * @param array $product_attributes Array of atributes to display; label, value.
	 * @param KKART_Product $product Showing attributes for this product.
	 */
	$product_attributes = apply_filters( 'kkart_display_product_attributes', $product_attributes, $product );

	kkart_get_template(
		'single-product/product-attributes.php',
		array(
			'product_attributes' => $product_attributes,
			// Legacy params.
			'product'            => $product,
			'attributes'         => $attributes,
			'display_dimensions' => $display_dimensions,
		)
	);
}

/**
 * Get HTML to show product stock.
 *
 * @since  3.0.0
 * @param  KKART_Product $product Product Object.
 * @return string
 */
function kkart_get_stock_html( $product ) {
	$html         = '';
	$availability = $product->get_availability();

	if ( ! empty( $availability['availability'] ) ) {
		ob_start();

		kkart_get_template(
			'single-product/stock.php',
			array(
				'product'      => $product,
				'class'        => $availability['class'],
				'availability' => $availability['availability'],
			)
		);

		$html = ob_get_clean();
	}

	if ( has_filter( 'kkart_stock_html' ) ) {
		kkart_deprecated_function( 'The kkart_stock_html filter', '', 'kkart_get_stock_html' );
		$html = apply_filters( 'kkart_stock_html', $html, $availability['availability'], $product );
	}

	return apply_filters( 'kkart_get_stock_html', $html, $product );
}

/**
 * Get HTML for ratings.
 *
 * @since  3.0.0
 * @param  float $rating Rating being shown.
 * @param  int   $count  Total number of ratings.
 * @return string
 */
function kkart_get_rating_html( $rating, $count = 0 ) {
	$html = '';

	if ( 0 < $rating ) {
		/* translators: %s: rating */
		$label = sprintf( __( 'Rated %s out of 5', 'kkart' ), $rating );
		$html  = '<div class="star-rating" role="img" aria-label="' . esc_attr( $label ) . '">' . kkart_get_star_rating_html( $rating, $count ) . '</div>';
	}

	return apply_filters( 'kkart_product_get_rating_html', $html, $rating, $count );
}

/**
 * Get HTML for star rating.
 *
 * @since  3.1.0
 * @param  float $rating Rating being shown.
 * @param  int   $count  Total number of ratings.
 * @return string
 */
function kkart_get_star_rating_html( $rating, $count = 0 ) {
	$html = '<span style="width:' . ( ( $rating / 5 ) * 100 ) . '%">';

	if ( 0 < $count ) {
		/* translators: 1: rating 2: rating count */
		$html .= sprintf( _n( 'Rated %1$s out of 5 based on %2$s customer rating', 'Rated %1$s out of 5 based on %2$s customer ratings', $count, 'kkart' ), '<strong class="rating">' . esc_html( $rating ) . '</strong>', '<span class="rating">' . esc_html( $count ) . '</span>' );
	} else {
		/* translators: %s: rating */
		$html .= sprintf( esc_html__( 'Rated %s out of 5', 'kkart' ), '<strong class="rating">' . esc_html( $rating ) . '</strong>' );
	}

	$html .= '</span>';

	return apply_filters( 'kkart_get_star_rating_html', $html, $rating, $count );
}

/**
 * Returns a 'from' prefix if you want to show where prices start at.
 *
 * @since  3.0.0
 * @return string
 */
function kkart_get_price_html_from_text() {
	return apply_filters( 'kkart_get_price_html_from_text', '<span class="from">' . _x( 'From:', 'min_price', 'kkart' ) . ' </span>' );
}

/**
 * Get logout endpoint.
 *
 * @since  2.6.9
 *
 * @param string $redirect Redirect URL.
 *
 * @return string
 */
function kkart_logout_url( $redirect = '' ) {
	$redirect = $redirect ? $redirect : apply_filters( 'kkart_logout_default_redirect_url', kkart_get_page_permalink( 'myaccount' ) );

	if ( get_option( 'kkart_logout_endpoint' ) ) {
		return wp_nonce_url( kkart_get_endpoint_url( 'customer-logout', '', $redirect ), 'customer-logout' );
	}

	return wp_logout_url( $redirect );
}

/**
 * Show notice if cart is empty.
 *
 * @since 3.1.0
 */
function kkart_empty_cart_message() {
	echo '<p class="cart-empty kkart-info">' . wp_kses_post( apply_filters( 'kkart_empty_cart_message', __( 'Your cart is currently empty.', 'kkart' ) ) ) . '</p>';
}

/**
 * Disable search engines indexing core, dynamic, cart/checkout pages.
 *
 * @since 3.2.0
 */
function kkart_page_noindex() {
	if ( is_page( kkart_get_page_id( 'cart' ) ) || is_page( kkart_get_page_id( 'checkout' ) ) || is_page( kkart_get_page_id( 'myaccount' ) ) ) {
		wp_no_robots();
	}
}
add_action( 'wp_head', 'kkart_page_noindex' );

/**
 * Get a slug identifying the current theme.
 *
 * @since 3.3.0
 * @return string
 */
function kkart_get_theme_slug_for_templates() {
	return apply_filters( 'kkart_theme_slug_for_templates', get_option( 'template' ) );
}

/**
 * Gets and formats a list of cart item data + variations for display on the frontend.
 *
 * @since 3.3.0
 * @param array $cart_item Cart item object.
 * @param bool  $flat Should the data be returned flat or in a list.
 * @return string
 */
function kkart_get_formatted_cart_item_data( $cart_item, $flat = false, $nicename_check = true ) {
	$item_data = array();

	// Variation values are shown only if they are not found in the title as of 3.0.
	// This is because variation titles display the attributes.
	if ( $cart_item['data']->is_type( 'variation' ) && is_array( $cart_item['variation'] ) ) {
		foreach ( $cart_item['variation'] as $name => $value ) {
			$taxonomy = kkart_attribute_taxonomy_name( str_replace( 'attribute_pa_', '', urldecode( $name ) ) );

			if ( taxonomy_exists( $taxonomy ) ) {
				// If this is a term slug, get the term's nice name.
				$term = get_term_by( 'slug', $value, $taxonomy );
				if ( ! is_wp_error( $term ) && $term && $term->name ) {
					$value = $term->name;
				}
				$label = kkart_attribute_label( $taxonomy );
			} else {
				// If this is a custom option slug, get the options name.
				$value = apply_filters( 'kkart_variation_option_name', $value, null, $taxonomy, $cart_item['data'] );
				$label = kkart_attribute_label( str_replace( 'attribute_', '', $name ), $cart_item['data'] );
			}

			// Check the nicename against the title.
			if ( '' === $value || ($nicename_check && kkart_is_attribute_in_product_name( $value, $cart_item['data']->get_name() )) ) {
				continue;
			}

			$item_data[] = array(
				'key'   => $label,
				'value' => $value,
			);
		}
	}

	// Filter item data to allow 3rd parties to add more to the array.
	$item_data = apply_filters( 'kkart_get_item_data', $item_data, $cart_item );

	// Format item data ready to display.
	foreach ( $item_data as $key => $data ) {
		// Set hidden to true to not display meta on cart.
		if ( ! empty( $data['hidden'] ) ) {
			unset( $item_data[ $key ] );
			continue;
		}
		$item_data[ $key ]['key']     = ! empty( $data['key'] ) ? $data['key'] : $data['name'];
		$item_data[ $key ]['display'] = ! empty( $data['display'] ) ? $data['display'] : $data['value'];
	}

	// Output flat or in list format.
	if ( count( $item_data ) > 0 ) {
		ob_start();

		if ( $flat ) {
			foreach ( $item_data as $data ) {
				echo esc_html( $data['key'] ) . ': ' . wp_kses_post( $data['display'] ) . "\n";
			}
		} else {
			extract(array( 'item_data' => $item_data ) );
			?>
			<div class="kkart-variations">
				<?php foreach ( $item_data as $data ) : ?>
				<div class="kkart-variation-holder">
					<strong class="<?php echo sanitize_html_class( 'variation-' . $data['key'] ); ?>"><?php echo wp_kses_post( $data['key'] ); ?>:</strong>
					<span class="<?php echo sanitize_html_class( 'variation-' . $data['key'] ); ?>"><?php echo wp_kses_post( $data['display'] ); ?></span>
				</div>
				<?php endforeach; ?>
			</div>
			<?php
		}

		return ob_get_clean();
	}

	return '';
}

/**
 * Gets the url to remove an item from the cart.
 *
 * @since 3.3.0
 * @param string $cart_item_key contains the id of the cart item.
 * @return string url to page
 */
function kkart_get_cart_remove_url( $cart_item_key ) {
	$cart_page_url = kkart_get_cart_url();
	return apply_filters( 'kkart_get_remove_url', $cart_page_url ? wp_nonce_url( add_query_arg( 'remove_item', $cart_item_key, $cart_page_url ), 'kkart-cart' ) : '' );
}

/**
 * Gets the url to re-add an item into the cart.
 *
 * @since 3.3.0
 * @param  string $cart_item_key Cart item key to undo.
 * @return string url to page
 */
function kkart_get_cart_undo_url( $cart_item_key ) {
	$cart_page_url = kkart_get_cart_url();

	$query_args = array(
		'undo_item' => $cart_item_key,
	);

	return apply_filters( 'kkart_get_undo_url', $cart_page_url ? wp_nonce_url( add_query_arg( $query_args, $cart_page_url ), 'kkart-cart' ) : '', $cart_item_key );
}

/**
 * Outputs all queued notices on KKART pages.
 *
 * @since 3.5.0
 */
function kkart_output_all_notices() {
	echo '<div class="kkart-notices-wrapper">';
	kkart_print_notices();
	echo '</div>';
}

/**
 * Products RSS Feed.
 *
 * @deprecated 2.6
 */
function kkart_products_rss_feed() {
	kkart_deprecated_function( 'kkart_products_rss_feed', '2.6' );
}

if ( ! function_exists( 'kkart_reset_loop' ) ) {

	/**
	 * Reset the loop's index and columns when we're done outputting a product loop.
	 *
	 * @deprecated 3.3
	 */
	function kkart_reset_loop() {
		kkart_reset_loop();
	}
}

if ( ! function_exists( 'kkart_product_reviews_tab' ) ) {
	/**
	 * Output the reviews tab content.
	 *
	 * @deprecated 2.4.0 Unused.
	 */
	function kkart_product_reviews_tab() {
		kkart_deprecated_function( 'kkart_product_reviews_tab', '2.4' );
	}
}

/**
 * Display pay buttons HTML.
 *
 * @since 3.9.0
 */
function kkart_get_pay_buttons() {
	$supported_gateways = array();
	$available_gateways = KKART()->payment_gateways()->get_available_payment_gateways();

	foreach ( $available_gateways as $gateway ) {
		if ( $gateway->supports( 'pay_button' ) ) {
			$supported_gateways[] = $gateway->get_pay_button_id();
		}
	}

	if ( ! $supported_gateways ) {
		return;
	}

	echo '<div class="kkart-pay-buttons">';
	foreach ( $supported_gateways as $pay_button_id ) {
		echo sprintf( '<div class="kkart-pay-button__%1$s %1$s" id="%1$s"></div>', esc_attr( $pay_button_id ) );
	}
	echo '</div>';
}

// phpcs:enable Generic.Commenting.Todo.TaskFound
