<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package DataStore - MySQL plugins
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Creates a new module DataStore
 * @param string $module Module to create DataStore for
 * @param string $prefix Key Prefix in DataStore
 * @return bool
 */
function _jrCore_db_create_datastore($module, $prefix)
{
    // Items
    $_tmp = array(
        "`_item_id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY"
    );
    jrCore_db_verify_table($module, 'item', $_tmp);

    // Item
    $_tmp = array(
        "`_item_id` INT(11) UNSIGNED NOT NULL DEFAULT '0'",
        "`_profile_id` INT(11) UNSIGNED NOT NULL DEFAULT '0'",
        "`key` VARCHAR(128) NOT NULL DEFAULT ''",
        "`index` SMALLINT(5) UNSIGNED NOT NULL DEFAULT '0'",
        "`value` VARCHAR(512) NOT NULL DEFAULT ''",
        "PRIMARY KEY (`key`, `_item_id`, `index`)",
        "INDEX `_item_id` (`_item_id`)",
        "INDEX `_profile_id` (`_profile_id`)",
        "INDEX `value` (`value`(128))",
    );
    jrCore_db_verify_table($module, 'item_key', $_tmp);

    return true;
}

/**
 * Deletes an existing module datastore
 * @param string $module Module to delete DataStore for
 * @return bool
 */
function _jrCore_db_delete_datastore($module)
{
    $tbl = jrCore_db_table_name($module, 'item');
    $req = "DROP TABLE IF EXISTS {$tbl}";
    jrCore_db_query($req);

    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "DROP TABLE IF EXISTS {$tbl}";
    jrCore_db_query($req);
    return true;
}

/**
 * Migrate profile_ids in datastore
 * @param $module string
 * @return int
 */
function _jrCore_db_sync_datastore_profile_ids($module)
{
    $cnt = 0;
    if (!jrCore_db_table_is_empty($module, 'item')) {

        // Has this DS been setup properly?
        $_ti = jrCore_db_table_columns($module, 'item_key');
        if (is_array($_ti) && isset($_ti['_profile_id'])) {

            // Fix up DS items where the _profile_id has not been set
            $lim = 500;
            $tbl = jrCore_db_table_name($module, 'item_key');
            while (true) {
                $req = "SELECT `_item_id`, `value` FROM {$tbl} WHERE `key` = '_profile_id' AND `value` > 0 AND (`_profile_id` = 0 OR `_profile_id` != `value`) LIMIT {$lim}";
                $_rt = jrCore_db_query($req, '_item_id', false, 'value');
                if (is_array($_rt)) {
                    $req = "UPDATE {$tbl} SET `_profile_id` = CASE `_item_id`\n";
                    foreach ($_rt as $k => $v) {
                        if (is_numeric($v)) {
                            $req .= "WHEN {$k} THEN {$v}\n";
                            $cnt++;
                        }
                    }
                    $req .= "ELSE `_profile_id` END WHERE dispute_id IN(" . implode(',', array_keys($_rt)) . ')';
                    jrCore_db_query($req);
                    if (count($_rt) < $lim) {
                        // No more items
                        break;
                    }
                }
                else {
                    // No more items
                    break;
                }
            }
        }
    }
    return $cnt;
}

/**
 * Run repair operations on a DataStore
 * @param string $module
 * @return bool
 */
function _jrCore_db_repair_datastore($module)
{
    return true;
}

/**
 * Core DS Plugin
 * @param $module
 * @return bool
 */
function _jrCore_db_truncate_datastore($module)
{
    $tbl = jrCore_db_table_name($module, 'item');
    $req = "TRUNCATE TABLE {$tbl}";
    jrCore_db_query($req);
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "TRUNCATE TABLE {$tbl}";
    jrCore_db_query($req);
    return true;
}

/**
 * Core DS Plugin
 * @param $module
 * @return int
 */
function _jrCore_db_get_datastore_item_count($module)
{
    if (jrCore_is_datastore_module($module)) {
        return jrCore_db_number_rows($module, 'item');
    }
    return 0;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @param $match
 * @param $function
 * @return bool
 */
function _jrCore_db_run_key_function($module, $key, $match, $function)
{
    switch (strtolower($function)) {
        case 'sum':
        case 'avg':
        case 'min':
        case 'max':
        case 'std':
            $fnc = strtoupper($function) . '(`value`)';
            break;
        case 'count':
            $fnc = 'COUNT(`_item_id`)';
            break;
        default:
            return false;
    }
    $tbl = jrCore_db_table_name($module, 'item_key');
    if ($match == '*') {
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT {$fnc} AS tc FROM {$tbl}";
        }
        else {
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "'";
        }
    }
    elseif (strpos(' ' . $match, '%')) {
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `value` LIKE '" . jrCore_db_escape($match) . "'";
        }
        else {
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` LIKE '" . jrCore_db_escape($match) . "'";
        }
    }
    elseif (strpos($match, '<') === 0) {
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `value` " . jrCore_db_escape($match);
        }
        else {
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` " . jrCore_db_escape($match);
        }
    }
    elseif (strpos($match, '>') === 0) {
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `value` " . jrCore_db_escape($match);
        }
        else {
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` " . jrCore_db_escape($match);
        }
    }
    elseif (stripos($match, 'IN(') === 0) {
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `value` " . jrCore_db_escape($match);
        }
        else {
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` " . jrCore_db_escape($match);
        }
    }
    else {
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `value` = '" . jrCore_db_escape($match) . "'";
        }
        else {
            $req = "SELECT {$fnc} AS tc FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` = '" . jrCore_db_escape($match) . "'";
        }
    }

    $_rt = jrCore_db_query($req, 'SINGLE');
    if ($_rt && is_array($_rt)) {
        return $_rt['tc'];
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $_ids
 * @return bool
 */
function _jrCore_db_set_display_order($module, $_ids)
{
    $_pi = array();
    $_rt = jrCore_db_get_multiple_items($module, array_keys($_ids), array('_item_id', '_profile_id'));
    if ($_rt && is_array($_rt)) {
        foreach ($_rt as $v) {
            $iid       = (int) $v['_item_id'];
            $_pi[$iid] = (int) $v['_profile_id'];
        }
    }

    $_rq = array();
    $pfx = jrCore_db_get_prefix($module);
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`) VALUES ";
    foreach ($_ids as $iid => $ord) {
        $ord = (int) $ord;
        $iid = (int) $iid;
        $pid = (int) $_pi[$iid];
        $req .= "({$iid},{$pid},'{$pfx}_display_order',0,'{$ord}'),";
    }
    $_rq[] = substr($req, 0, strlen($req) - 1) . " ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)";
    unset($req);

    // If display_order is setup as a key index - update
    if (jrCore_db_key_has_index_table($module, "{$pfx}_display_order")) {
        $tbl = jrCore_db_get_index_table_name($module, "{$pfx}_display_order");
        $req = "INSERT INTO {$tbl} (`_item_id`,`value`) VALUES ";
        foreach ($_ids as $iid => $ord) {
            $ord = (int) $ord;
            $iid = (int) $iid;
            $req .= "({$iid},{$ord}),";
        }
        $_rq[] = substr($req, 0, strlen($req) - 1) . " ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)";
        jrCore_db_multi_query($_rq, false);
        unset($req);
    }
    else {
        jrCore_db_query($_rq[0], null, false, null, false);
    }
    return true;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @param $value
 * @return int|false
 */
function _jrCore_db_create_default_key($module, $key, $value)
{
    $tbl = jrCore_db_table_name($module, 'item_key');
    $key = jrCore_db_escape($key);
    $val = jrCore_db_escape($value);
    $req = "INSERT IGNORE INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`value`) SELECT DISTINCT(`_item_id`),`_profile_id`,'{$key}','{$val}' FROM {$tbl} WHERE `key` = '_created' AND `_item_id` > 0";
    return jrCore_db_query($req, 'COUNT', false, null, false);
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @param $value
 * @param $default
 * @return int|false
 */
function _jrCore_db_update_default_key($module, $key, $value, $default)
{
    $tbl = jrCore_db_table_name($module, 'item_key');
    $key = jrCore_db_escape($key);
    $val = jrCore_db_escape($value);
    $def = jrCore_db_escape($default);
    $req = "UPDATE {$tbl} SET `value` = '{$val}' WHERE `key` = '{$key}' AND (`value` IS NULL OR `value` = '' OR `value` = '{$def}')";
    return jrCore_db_query($req, 'COUNT');
}

/**
 * Core DS Plugin
 * @param $module
 * @param $id
 * @param $key
 * @param $value
 * @param bool|false $update
 * @param array $profile_ids item_id => profile_id array for profile ID matching
 * @return bool
 */
function _jrCore_db_increment_key($module, $id, $key, $value, $update = false, $profile_ids = null)
{
    // Get profile_ids
    if (is_array($profile_ids)) {
        $_pi = $profile_ids;
    }
    elseif ($module == 'jrProfile') {
        $_pi = array();
        foreach ($id as $uid) {
            $_pi[$uid] = $uid;
        }
    }
    else {
        $_pi = array();
        $_rt = jrCore_db_get_multiple_items($module, $id, array('_profile_id'));
        if ($_rt && is_array($_rt)) {
            foreach ($_rt as $v) {
                if (is_numeric($v['_profile_id'])) {
                    $uid       = (int) $v['_item_id'];
                    $iid       = (int) $v['_profile_id'];
                    $_pi[$uid] = $iid;
                }
            }
        }
        else {
            // could not get any of these items - cannot increment
            return false;
        }
    }

    // Remove any item ids we do not have a profile_id for
    foreach ($id as $k => $uid) {
        if (!isset($_pi[$uid])) {
            unset($id[$k]);
        }
    }
    if (count($id) === 0) {
        return false;
    }

    $_in = array();
    $key = jrCore_db_escape($key);
    foreach ($id as $uid) {
        $pid   = (isset($_pi[$uid])) ? $_pi[$uid] : 0;
        $_in[] = "({$uid},{$pid},'{$key}',0,'{$value}')";
        if ($update) {
            $_in[] = "({$uid},{$pid},'_updated',0,UNIX_TIMESTAMP())";
        }
    }

    $_rq = array();
    $tbl = jrCore_db_table_name($module, 'item_key');
    if ($update) {
        $_rq[] = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`)
                  VALUES " . implode(',', $_in) . "
                  ON DUPLICATE KEY UPDATE `value` = IF(`key` = '{$key}',`value` + {$value}, VALUES(`value`))";
    }
    else {
        $_rq[] = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`)
                  VALUES " . implode(',', $_in) . "
                  ON DUPLICATE KEY UPDATE `value` = `value` + {$value}";
    }
    // Do we have a dedicated key index?
    if (jrCore_db_key_has_index_table($module, $key)) {
        $_in = array();
        foreach ($id as $uid) {
            $_in[] = "({$uid},'{$value}')";
        }
        $tbl   = jrCore_db_get_index_table_name($module, $key);
        $_rq[] = "INSERT INTO {$tbl} (`_item_id`,`value`) VALUES " . implode(',', $_in) . " ON DUPLICATE KEY UPDATE `value` = `value` + {$value}";
        jrCore_db_multi_query($_rq);
    }
    else {
        $cnt = jrCore_db_query($_rq[0], 'COUNT', false, null, false);
        if (!$cnt || $cnt < 1) {
            return false;
        }
    }
    return true;
}

/**
 * Core DS Plugin
 * @param $module string Module Name
 * @param $id mixed Unique Item ID OR Array of Item IDs
 * @param $key string Key to decrement
 * @param $value number Integer/Float to decrement by
 * @param $min_value number Lowest Value allowed for Key (default 0)
 * @param $update bool set to TRUE to update updated timed
 * @param array $profile_ids item_id => profile_id array for profile ID matching
 * @return bool
 */
function _jrCore_db_decrement_key($module, $id, $key, $value, $min_value = null, $update = false, $profile_ids = null)
{
    // Get profile_ids
    if (is_array($profile_ids)) {
        $_pi = $profile_ids;
    }
    elseif ($module == 'jrProfile') {
        $_pi = array();
        foreach ($id as $uid) {
            $_pi[$uid] = $uid;
        }
    }
    else {
        $_pi = array();
        $_rt = jrCore_db_get_multiple_items($module, $id, array('_profile_id'));
        if ($_rt && is_array($_rt)) {
            foreach ($_rt as $v) {
                if (is_numeric($v['_profile_id'])) {
                    $uid       = (int) $v['_item_id'];
                    $iid       = (int) $v['_profile_id'];
                    $_pi[$uid] = $iid;
                }
            }
        }
        else {
            // could not get any of these items - cannot increment
            return false;
        }
    }

    // Remove any item ids we do not have a profile_id for
    foreach ($id as $k => $uid) {
        if (!isset($_pi[$uid])) {
            unset($id[$k]);
        }
    }
    if (count($id) === 0) {
        return false;
    }

    $_in = array();
    $key = jrCore_db_escape($key);
    foreach ($id as $uid) {
        $pid   = (isset($_pi[$uid])) ? $_pi[$uid] : 0;
        $_in[] = "({$uid},{$pid},'{$key}',0,0)";
        if ($update) {
            $_in[] = "({$uid},{$pid},'_updated',0,UNIX_TIMESTAMP())";
        }
    }
    $_rq = array();
    $val = ($min_value + $value);
    $tbl = jrCore_db_table_name($module, 'item_key');
    if ($update) {
        $_rq[] = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`)
                  VALUES " . implode(',', $_in) . "
                  ON DUPLICATE KEY UPDATE `value` = IF(`key` = '_updated', VALUES(`value`), IF(CAST(`value` AS DECIMAL) >= {$val},`value` - {$value}, `value`))";
    }
    else {
        $_rq[] = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`)
                  VALUES " . implode(',', $_in) . "
                  ON DUPLICATE KEY UPDATE `value` = IF(CAST(`value` AS DECIMAL) >= {$val},`value` - {$value}, `value`)";
    }

    // Do we have a dedicated key index?
    if (jrCore_db_key_has_index_table($module, $key)) {
        $_in = array();
        foreach ($id as $uid) {
            $_in[] = "({$uid},'{$value}')";
        }
        $tbl   = jrCore_db_get_index_table_name($module, $key);
        $_rq[] = "INSERT INTO {$tbl} (`_item_id`,`value`) VALUES " . implode(',', $_in) . " ON DUPLICATE KEY UPDATE `value` = IF(CAST(`value` AS DECIMAL) >= {$val},`value` - {$value}, `value`)";
        jrCore_db_multi_query($_rq);
    }
    else {
        $cnt = jrCore_db_query($_rq[0], 'COUNT', false, null, false);
        if (!$cnt || $cnt < 1) {
            return false;
        }
    }
    return true;
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $key string
 * @param $limit int
 * @return array|bool
 */
function _jrCore_db_get_items_missing_key($module, $key, $limit)
{
    ini_set('memory_limit', '1024M');
    $lim = ($limit > 0) ? " LIMIT {$limit}" : '';
    $tb1 = jrCore_db_table_name($module, 'item');
    $tb2 = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id` AS i FROM {$tb1} WHERE `_item_id` NOT IN(SELECT `_item_id` FROM {$tb2} WHERE `key` = '" . jrCore_db_escape($key) . "'){$lim}";
    if ($res = jrCore_db_query($req)) {
        $num = mysqli_num_rows($res);
        if ($num > 0) {
            $_rt = array();
            while ($row = mysqli_fetch_assoc($res)) {
                $_rt[] = $row['i'];
            }
            mysqli_free_result($res);
            return $_rt;
        }
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @return bool
 */
function _jrCore_db_item_key_exists($module, $key)
{
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' LIMIT 1";
    $_rt = jrCore_db_query($req, 'SINGLE');
    if (is_array($_rt)) {
        return true;
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module
 * @return array|bool
 */
function _jrCore_db_get_unique_keys($module)
{
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `key` FROM {$tbl} GROUP BY `key` ORDER BY `key` ASC";
    $_rt = jrCore_db_query($req, 'key', false, 'key');
    if ($_rt && is_array($_rt)) {
        return array_keys($_rt);
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @return array|bool
 */
function _jrCore_db_get_all_key_values($module, $key)
{
    if (jrCore_is_ds_index_needed($key)) {
        $tbl = jrCore_db_table_name($module, 'item_key');
        $req = "SELECT `_item_id` AS i FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "'";
        $_rt = jrCore_db_query($req, 'i', false, 'i');
        if ($_rt && is_array($_rt)) {
            switch ($module) {
                case 'jrUser':
                    $iid = '_user_id';
                    break;
                case 'jrProfile':
                    $iid = '_profile_id';
                    break;
                default:
                    $iid = '_item_id';
                    break;
            }
            // We pass through jrCore_db_get_multiple_items here on purpose -
            // it contains the logic to reconstruct all key values (i.e. over 512 bytes)
            $_rt = jrCore_db_get_multiple_items($module, $_rt, array($iid, $key));
            if ($_rt && is_array($_rt)) {
                $_tm = array();
                foreach ($_rt as $v) {
                    $_tm["{$v[$iid]}"] = $v[$key];
                }
                unset($_rt);
                return $_tm;
            }
        }
        return false;
    }

    // Fall through - If no index is needed this means our values cannot
    // consist of multiple entries in the DS - return id/val
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "'";
    return jrCore_db_query($req, 'i', false, 'v');
}

/**
 * Core DS Plugin
 * @param int $profile_id
 * @param string $module
 * @param string $key
 * @return array|bool
 */
function _jrCore_db_get_all_values_for_key_by_profile_id($profile_id, $module, $key)
{
    $pid = (int) $profile_id;
    if (jrCore_is_ds_index_needed($key)) {
        $tbl = jrCore_db_table_name($module, 'item_key');
        $req = "SELECT `_item_id` AS i FROM {$tbl} WHERE `_profile_id` = {$pid} AND `key` = '" . jrCore_db_escape($key) . "'";
        $_rt = jrCore_db_query($req, 'i', false, 'i');
        if ($_rt && is_array($_rt)) {
            // We pass through jrCore_db_get_multiple_items here on purpose -
            // it contains the logic to reconstruct all key values (i.e. over 512 bytes)
            $_rt = jrCore_db_get_multiple_items($module, $_rt, array('_item_id', $key));
            if (is_array($_rt)) {
                $_tm = array();
                foreach ($_rt as $v) {
                    $_tm["{$v['_item_id']}"] = $v[$key];
                }
                unset($_rt);
                return $_tm;
            }
        }
        return false;
    }

    // Fall through - If no index is needed this means our values cannot
    // consist of multiple entries in the DS - return id/val
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `_profile_id` = {$pid} AND `key` = '" . jrCore_db_escape($key) . "'";
    return jrCore_db_query($req, 'i', false, 'v');
}

/**
 * Core DS Plugin
 * @param $module
 * @param $id
 * @param $_keys
 * @param bool $core_check
 * @param bool $cache_reset
 * @param bool $update set to FALSE to prevent updating of _updated key
 * @return bool
 */
function _jrCore_db_delete_multiple_item_keys($module, $id, $_keys, $core_check = true, $cache_reset = true, $update = true)
{
    // Delete keys
    if (count($_keys) > 0) {

        foreach ($_keys as $k => $key) {
            $_keys[$k] = jrCore_db_escape($key);
        }

        $_rq = array();
        $uid = intval($id);
        $tbl = jrCore_db_table_name($module, 'item_key');
        if ($update) {
            $_rq[] = "UPDATE {$tbl} SET `value` = UNIX_TIMESTAMP() WHERE `_item_id` = {$uid} AND `key` = '_updated'";
        }
        $_rq[] = "DELETE FROM {$tbl} WHERE `_item_id` = {$uid} AND `key` IN('" . implode("','", $_keys) . "')";

        // Check for key indexes
        foreach ($_keys as $key) {
            if (jrCore_db_key_has_index_table($module, $key)) {
                $tbl   = jrCore_db_get_index_table_name($module, $key);
                $_rq[] = "DELETE FROM {$tbl} WHERE `_item_id` = {$uid}";
            }
        }
        jrCore_db_multi_query($_rq, false);
        return true;
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $id
 * @param $key
 * @param bool $core_check
 * @param bool $cache_reset
 * @param bool $update
 * @return bool
 */
function _jrCore_db_delete_item_key($module, $id, $key, $core_check = true, $cache_reset = true, $update = true)
{
    return jrCore_db_delete_multiple_item_keys($module, $id, array($key), $core_check, $cache_reset, $update);
}

/**
 * Core DS Plugin
 * @param string $module
 * @param array $_ids
 * @param string $key
 * @param bool $update
 * @return bool
 */
function _jrCore_db_delete_key_from_multiple_items($module, $_ids, $key, $update)
{
    $_rq = array();
    if (is_array($key)) {
        $_ky = array();
        foreach ($key as $k) {
            $_ky[] = jrCore_db_escape($k);
            if (jrCore_db_key_has_index_table($module, $k)) {
                $tbl   = jrCore_db_get_index_table_name($module, $k);
                $_rq[] = "DELETE FROM {$tbl} WHERE `_item_id` IN(" . implode(',', $_ids) . ")";
            }
        }
        $tbl = jrCore_db_table_name($module, 'item_key');
        if ($update) {
            $_rq[] = "UPDATE {$tbl} SET `value` = UNIX_TIMESTAMP() WHERE `_item_id` IN(" . implode(',', $_ids) . ") AND `key` = '_updated'";
        }
        $_rq[] = "DELETE FROM {$tbl} WHERE `key` IN('" . implode("','", $_ky) . "') AND `_item_id` IN(" . implode(',', $_ids) . ")";
    }
    else {
        $tbl = jrCore_db_table_name($module, 'item_key');
        if ($update) {
            $_rq[] = "UPDATE {$tbl} SET `value` = UNIX_TIMESTAMP() WHERE `_item_id` IN(" . implode(',', $_ids) . ") AND `key` = '_updated'";
        }
        $_rq[] = "DELETE FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `_item_id` IN(" . implode(',', $_ids) . ")";
        if (jrCore_db_key_has_index_table($module, $key)) {
            $tbl   = jrCore_db_get_index_table_name($module, $key);
            $_rq[] = "DELETE FROM {$tbl} WHERE `_item_id` IN(" . implode(',', $_ids) . ")";
        }
    }
    jrCore_db_multi_query($_rq, false);
    return true;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @return bool
 */
function _jrCore_db_delete_key_from_all_items($module, $key)
{
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "DELETE FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "'";
    $cnt = jrCore_db_query($req, 'COUNT');
    // If this key has a dedicated index table, truncate it
    if (jrCore_db_key_has_index_table($module, $key)) {
        $tbl = jrCore_db_get_index_table_name($module, $key);
        $req = "TRUNCATE TABLE {$tbl}";
        jrCore_db_query($req, null, false, null, false, null, false);
    }
    return $cnt;
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $count int
 * @return int|false
 */
function _jrCore_db_create_unique_item_id($module, $count = 1)
{
    // Get our unique item id
    $tbl = jrCore_db_table_name($module, 'item');
    if ($count > 1) {
        $ins = str_repeat('(0),', $count);
        $req = "INSERT INTO {$tbl} (`_item_id`) VALUES " . substr($ins, 0, strlen($ins) - 1);
    }
    else {
        $req = "INSERT INTO {$tbl} (`_item_id`) VALUES (0)";
    }
    return jrCore_db_query($req, 'INSERT_ID');
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $profile_id int
 * @return true
 */
function _jrCore_db_update_profile_item_count($module, $profile_id)
{
    $pid = (int) $profile_id;

    // First - get DS item count for profile
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '_profile_id' AND `value` = {$pid}";
    $_cq = jrCore_db_query($req, 'NUMERIC');
    $cnt = (is_array($_cq)) ? count($_cq) : 0;

    // Update
    $tbl = jrCore_db_table_name('jrProfile', 'item_key');
    $req = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`) VALUES ({$pid},{$pid},'profile_{$module}_item_count',0,'{$cnt}') ON DUPLICATE KEY UPDATE `value` = {$cnt}";
    jrCore_db_query($req, null, false, null, false);
    return true;
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $profile_id int
 * @param $user_id int
 * @return true
 */
function _jrCore_db_update_user_item_count($module, $profile_id, $user_id)
{
    $uid = (int) $user_id;
    $pid = (int) $profile_id;

    // First - get DS item count for profile
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '_user_id' AND `value` = {$uid}";
    $_cq = jrCore_db_query($req, 'NUMERIC');
    $cnt = (is_array($_cq)) ? count($_cq) : 0;

    // Update
    $tbl = jrCore_db_table_name('jrUser', 'item_key');
    $req = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`) VALUES ({$uid},{$pid},'user_{$module}_item_count',0,'{$cnt}') ON DUPLICATE KEY UPDATE `value` = {$cnt}";
    jrCore_db_query($req, null, false, null, false);
    return true;
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $item_id int
 * @param $_data array
 * @param null $_core
 * @param bool $profile_count
 * @param bool $skip_trigger
 * @return bool
 */
function _jrCore_db_create_item($module, $item_id, $_data, $_core = null, $profile_count = true, $skip_trigger = false)
{
    // Get our unique item id
    $iid = (int) $item_id;
    $pfx = jrCore_db_get_prefix($module);
    if ($module == 'jrProfile') {
        // If this is the Profile module we always set _profile_id = _item_id
        $pid = $iid;
    }
    else {
        $pid = (int) $_data['_profile_id'];
    }

    // Check for item_order_support
    if (isset($_data["{$pfx}_display_order"]) && $_data["{$pfx}_display_order"] === 0) {

        // Any other items of this type need to have their order incremented by ONE
        // @note This is done in 2 queries since you cannot UPDATE a table that you SELECT FROM
        // http://dev.mysql.com/doc/refman/5.6/en/update.html
        $tbl = jrCore_db_table_name($module, 'item_key');
        $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '_profile_id' AND `value` = '{$pid}'";
        $_ei = jrCore_db_query($req, '_item_id', false, '_item_id');
        if ($_ei && is_array($_ei)) {
            $req = "UPDATE {$tbl} SET `value` = (`value` + 1) WHERE _item_id IN(" . implode(',', $_ei) . ") AND `key` = '{$pfx}_display_order'";
            jrCore_db_query($req);
        }

    }

    // Max 10,000 indexes in a single DS key (bailout)
    $max = jrCore_get_config_value('jrCore', 'max_ds_key_index', 10000);

    $_rq = array();
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`) VALUES ";
    foreach ($_data as $k => $v) {

        $val = false;
        if ($v === 'UNIX_TIMESTAMP()') {
            $req .= "({$iid},{$pid},'" . jrCore_db_escape($k) . "','0',UNIX_TIMESTAMP()),";
            $val = 'UNIX_TIMESTAMP()';
        }
        else {
            // If our value is longer than 508 bytes we split it up
            if (!is_numeric($v)) {
                $v = jrCore_strip_emoji($v);
            }
            $len = strlen($v);
            if ($len > 508) {
                $idx = 0;
                $_tm = array();
                while ($len) {
                    $_tm[] = mb_strcut($v, 0, 508, "UTF-8");
                    $v     = mb_strcut($v, 508, $len, "UTF-8");
                    $len   = strlen($v);
                    $idx++;
                    if ($idx >= $max) {
                        $siz = jrCore_format_size(strlen($v));
                        jrCore_logger('CRI', "core: max_ds_key_index count hit for {$module}/{$k} - truncating value (length: {$siz})");
                        break;
                    }
                }
                foreach ($_tm as $idx => $part) {
                    $req .= "({$iid},{$pid},'" . jrCore_db_escape($k) . "','" . ($idx + 1) . "','" . jrCore_db_escape($part) . "'),";
                }
            }
            else {
                if (is_numeric($v)) {
                    $val = $v;
                }
                else {
                    $val = jrCore_db_escape($v);
                }
                $req .= "({$iid},{$pid},'" . jrCore_db_escape($k) . "','0','" . $val . "'),";
            }
        }

        if ($val && jrCore_db_key_has_index_table($module, $k)) {
            $tbi = jrCore_db_get_index_table_name($module, $k);
            if ($val != 'UNIX_TIMESTAMP()') {
                $val = "'{$val}'";
            }
            $_rq[] = "INSERT INTO {$tbi} (`_item_id`,`value`) VALUES ({$iid},{$val})";
        }

    }
    $req = substr($req, 0, strlen($req) - 1);
    $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
    if ($cnt && $cnt > 0) {
        if (count($_rq) > 0) {
            jrCore_db_multi_query($_rq, false);
        }
        return true;
    }
    return false;
}

/**
 * Core DS plugin
 * @param $module string
 * @param $iid int
 * @param $_data array
 * @param null $_core
 * @param bool|false $skip_trigger
 * @return array|bool
 */
function _jrCore_db_create_multiple_items($module, $iid, $_data, $_core = null, $skip_trigger = false)
{
    $_rq = array();
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`) VALUES ";
    $uid = $iid;
    foreach ($_data as $_dt) {
        $pid = (int) $_dt['_profile_id'];
        foreach ($_dt as $k => $v) {
            $val = false;
            if ($v === 'UNIX_TIMESTAMP()') {
                $req .= "({$uid},{$pid},'" . jrCore_db_escape($k) . "','0',UNIX_TIMESTAMP()),";
                $val = 'UNIX_TIMESTAMP()';
            }
            else {
                // If our value is longer than 508 bytes we split it up
                $len = strlen($v);
                $v   = jrCore_strip_emoji($v);
                if ($len > 508) {
                    $_tm = array();
                    while ($len) {
                        $_tm[] = mb_strcut($v, 0, 508, "UTF-8");
                        $v     = mb_strcut($v, 508, $len, "UTF-8");
                        $len   = strlen($v);
                    }
                    foreach ($_tm as $idx => $part) {
                        $req .= "({$uid},{$pid},'" . jrCore_db_escape($k) . "','" . ($idx + 1) . "','" . jrCore_db_escape($part) . "'),";
                    }
                }
                else {
                    if (is_numeric($v)) {
                        $val = $v;
                    }
                    else {
                        $val = jrCore_db_escape($v);
                    }
                    $req .= "({$uid},{$pid},'" . jrCore_db_escape($k) . "','0','" . jrCore_db_escape($v) . "'),";
                }
            }
            if ($val && jrCore_db_key_has_index_table($module, $k)) {
                $tbi = jrCore_db_get_index_table_name($module, $k);
                if ($val != 'UNIX_TIMESTAMP()') {
                    $val = "'{$val}'";
                }
                $_rq[] = "INSERT INTO {$tbi} (`_item_id`,`value`) VALUES ({$iid},{$val})";
            }
        }
        $uid++;
    }
    $req = substr($req, 0, strlen($req) - 1);
    $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
    if ($cnt && $cnt > 0) {
        if (count($_rq) > 0) {
            jrCore_db_multi_query($_rq, false);
        }
        return $cnt;
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $key string
 * @param $value mixed
 * @param bool|false $item_id_array
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @return array|bool
 */
function _jrCore_db_get_multiple_items_by_key($module, $key, $value, $item_id_array = false, $skip_caching = false)
{
    $idx = jrCore_db_key_has_index_table($module, $key);
    if (is_array($value)) {
        $esc = jrCore_db_escape($key);
        $_rq = array();
        foreach ($value as $k => $v) {
            if ($idx) {
                $_rq[] = "(`value` = '" . jrCore_db_escape($v) . "')";
            }
            else {
                if ($key == '_profile_id') {
                    $_rq[] = intval($v);
                }
                elseif (is_numeric($k)) {
                    $_rq[] = "(`key` = '{$esc}' AND `value` = '" . jrCore_db_escape($v) . "')";
                }
                else {
                    $_rq[] = "(`key` = '" . jrCore_db_escape($k) . "' AND `value` = '" . jrCore_db_escape($v) . "')";
                }
            }
        }
        if (count($_rq) > 0) {
            if ($idx) {
                $tbl = jrCore_db_get_index_table_name($module, $key);
                $req = "SELECT `_item_id` FROM {$tbl} WHERE " . implode(' OR ', $_rq);
            }
            elseif ($key == '_profile_id') {
                $tbl = jrCore_db_table_name($module, 'item_key');
                $req = "SELECT `_item_id` FROM {$tbl} WHERE `_profile_id` IN(" . implode(',', $_rq) . ')';
            }
            else {
                $tbl = jrCore_db_table_name($module, 'item_key');
                $req = "SELECT `_item_id` FROM {$tbl} WHERE " . implode(' OR ', $_rq);
            }
        }
        else {
            return false;
        }
    }
    else {
        if ($idx) {
            $tbl = jrCore_db_get_index_table_name($module, $key);
            $req = "SELECT `_item_id` FROM {$tbl} WHERE `value` = '" . jrCore_db_escape($value) . "'";
        }
        elseif ($key == '_profile_id') {
            $tbl = jrCore_db_table_name($module, 'item_key');
            $req = "SELECT `_item_id` FROM {$tbl} WHERE `_profile_id` = " . intval($value);
        }
        else {
            $tbl = jrCore_db_table_name($module, 'item_key');
            $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` = '" . jrCore_db_escape($value) . "'";
        }
    }
    $_rt = jrCore_db_query($req, '_item_id');
    if (!$_rt || !is_array($_rt)) {
        return false;
    }
    if ($item_id_array) {
        return array_keys($_rt);
    }
    return jrCore_db_get_multiple_items($module, array_keys($_rt), null, $skip_caching);
}

/**
 * Core DS Plugin
 * @param $module string
 * @param $key string
 * @param $match string
 * @param $value mixed
 * @param bool|false $item_id_array
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @return array|bool
 */
function _jrCore_db_search_multiple_items_by_key($module, $key, $match, $value, $item_id_array = false, $skip_caching = false)
{
    $idx = jrCore_db_key_has_index_table($module, $key);
    $imp = ($match == '!=') ? 'AND' : 'OR';
    if (is_array($value)) {
        $esc = jrCore_db_escape($key);
        $_rq = array();
        foreach ($value as $v) {
            if ($idx) {
                if (!is_numeric($v)) {
                    $v = "'" . jrCore_db_escape($v) . "'";
                }
                $_rq[] = "(`value` {$match} {$v})";
            }
            else {
                if ($key == '_profile_id') {
                    $_rq[] = intval($v);
                }
                else {
                    if (!is_numeric($v)) {
                        $v = "'" . jrCore_db_escape($v) . "'";
                    }
                    $_rq[] = "`value` {$match} {$v}";
                }
            }
        }
        if (count($_rq) > 0) {
            if ($idx) {
                $tbl = jrCore_db_get_index_table_name($module, $key);
                $req = "SELECT `_item_id` FROM {$tbl} WHERE " . implode(" {$imp} ", $_rq);
            }
            else {
                $tbl = jrCore_db_table_name($module, 'item_key');
                $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '{$esc}' AND (" . implode(" {$imp} ", $_rq) . ')';
            }
        }
        else {
            return false;
        }
    }
    else {
        if ($key == '_profile_id') {
            $tbl = jrCore_db_table_name($module, 'item_key');
            $req = "SELECT `_item_id` FROM {$tbl} WHERE `_profile_id` {$match} " . intval($value);
        }
        else {
            if (!is_numeric($value)) {
                $value = "'" . jrCore_db_escape($value) . "'";
            }
            if ($idx) {
                $tbl = jrCore_db_get_index_table_name($module, $key);
                $req = "SELECT `_item_id` FROM {$tbl} WHERE `value` {$match} {$value}";
            }
            else {
                $tbl = jrCore_db_table_name($module, 'item_key');
                $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` {$match} {$value}";
            }
        }
    }
    $_rt = jrCore_db_query($req, '_item_id');
    if (!is_array($_rt)) {
        return false;
    }
    if ($item_id_array) {
        return array_keys($_rt);
    }
    return jrCore_db_get_multiple_items($module, array_keys($_rt), null, $skip_caching);
}

/**
 * Core DS Plugin
 * @param $module
 * @param $key
 * @param $value
 * @param bool|false $skip_trigger
 * @param bool|false $skip_caching
 * @return bool|mixed
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function _jrCore_db_get_item_by_key($module, $key, $value, $skip_trigger = false, $skip_caching = false)
{
    if (jrCore_db_key_has_index_table($module, $key)) {
        $tbl = jrCore_db_get_index_table_name($module, $key);
        $req = "SELECT `_item_id` FROM {$tbl} WHERE `value` = '" . jrCore_db_escape($value) . "'";
    }
    else {
        $tbl = jrCore_db_table_name($module, 'item_key');
        $req = "SELECT `_item_id` FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($key) . "' AND `value` = '" . jrCore_db_escape($value) . "' LIMIT 1";
    }
    if (!$_rt = jrCore_db_query($req, 'SINGLE')) {
        return false;
    }
    return jrCore_db_get_item($module, $_rt['_item_id'], $skip_trigger, $skip_caching);
}

/**
 * Core DS plugin
 * @param $module
 * @param $id
 * @param bool|false $skip_trigger
 * @param bool|false $skip_caching
 * @return array|bool
 */
function _jrCore_db_get_item($module, $id, $skip_trigger = false, $skip_caching = false)
{
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `key` AS k,`index` AS x,`value` AS v FROM {$tbl} WHERE `_item_id` = " . intval($id);
    jrCore_start_timer("get_item_{$module}");
    $_rt = jrCore_db_query($req, 'NUMERIC');
    jrCore_stop_timer("get_item_{$module}");
    if (is_array($_rt)) {

        // Construct item
        $_ot = array('_item_id' => intval($id));
        $_ix = array();
        foreach ($_rt as $k => $v) {
            if ($v['x'] > 0) {
                if (isset($_ot["{$v['k']}"]) && !is_array($_ot["{$v['k']}"])) {
                    // We already saw index 0 - move it to our array
                    $_ot["{$v['k']}"] = array($_ot["{$v['k']}"]);
                }
                $_ot["{$v['k']}"]["{$v['x']}"] = $v['v'];
                $_ix["{$v['k']}"]              = 1;
            }
            else {
                $_ot["{$v['k']}"] = $v['v'];
            }
            unset($_rt[$k]);
        }
        if (count($_ix) > 0) {
            // We have index keys that need to be sorted and re-assembled
            foreach ($_ix as $k => $i) {
                ksort($_ot[$k], SORT_NUMERIC);
                $_ot[$k] = implode('', $_ot[$k]);
                unset($_ix[$id]);
            }
        }
        // Make sure _item_id did not get changed
        $_ot['_item_id'] = intval($id);
        unset($_ix, $_rt);
        return $_ot;
    }
    return false;
}

/**
 * Core DS Plugin
 * @param string $module Module the item belongs to
 * @param array $_ids array array of _item_id's to get
 * @param array $_keys Array of key names to get, default is all keys for each item
 * @param bool $skip_caching Set to true to force item reload (skip caching)
 * @return array|false
 */
function _jrCore_db_get_multiple_items($module, $_ids, $_keys = null, $skip_caching = false)
{
    $_ky = array();
    $tbl = jrCore_db_table_name($module, 'item_key');
    if (is_array($_keys) && count($_keys) > 0) {
        foreach ($_keys as $k) {
            // We handle _item_id down below...
            if ($k != '_item_id') {
                $_ky[] = jrCore_db_escape($k);
            }
        }
    }
    $req = "SELECT `_item_id` AS i,`key` AS k,`index` AS x,`value` AS v FROM {$tbl} WHERE `_item_id` IN(" . implode(',', $_ids) . ")";
    if (!empty($_ky)) {
        $req .= " AND `key` IN('" . implode("','", $_ky) . "')";
    }
    jrCore_start_timer("get_multiple_items_{$module}");
    $_rt = jrCore_db_query($req, 'NUMERIC');
    jrCore_stop_timer("get_multiple_items_{$module}");
    if (is_array($_rt)) {

        $_nw = array();
        $_ix = array();
        foreach ($_rt as $k => $v) {
            if ($v['x'] > 0) {
                if (isset($_nw["{$v['i']}"]["{$v['k']}"]) && !is_array($_nw["{$v['i']}"]["{$v['k']}"])) {
                    // We already saw index 0 - move it to our array
                    $_nw["{$v['i']}"]["{$v['k']}"] = array($_nw["{$v['i']}"]["{$v['k']}"]);
                }
                $_nw["{$v['i']}"]["{$v['k']}"]["{$v['x']}"] = $v['v'];
                $_ix["{$v['i']}"]["{$v['k']}"]              = 1;
            }
            else {
                $_nw["{$v['i']}"]["{$v['k']}"] = $v['v'];
            }
            unset($_rt[$k]);
        }
        if (count($_ix) > 0) {
            // We have index keys that need to be sorted and re-assembled
            foreach ($_ix as $id => $keys) {
                foreach ($keys as $k => $i) {
                    ksort($_nw[$id][$k], SORT_NUMERIC);
                    $_nw[$id][$k] = implode('', $_nw[$id][$k]);
                }
                unset($_ix[$id]);
            }
        }
        unset($_ix);

        // Put things back into our incoming order
        $add_id = false;
        if ($module == 'jrUser') {
            $add_id = '_user_id';
        }
        elseif ($module == 'jrProfile') {
            $add_id = '_profile_id';
        }
        $i   = 0;
        $_rs = array();
        foreach ($_ids as $id) {
            if (isset($_nw[$id])) {
                $_rs[$i]             = $_nw[$id];
                $_rs[$i]['_item_id'] = $id;
                if ($add_id) {
                    $_rs[$i][$add_id] = $id;
                }
                $i++;
            }
        }
        unset($_nw, $_rt);
        return $_rs;
    }
    return false;
}

/**
 * Core DS Plugin
 * @param $module
 * @param $id
 * @param $key
 * @return bool|string
 */
function _jrCore_db_get_item_key($module, $id, $key)
{
    if (!jrCore_is_ds_index_needed($key) && jrCore_db_key_has_index_table($module, $key)) {
        $tbl = jrCore_db_get_index_table_name($module, $key);
        $req = "SELECT `value` AS v FROM {$tbl} WHERE `_item_id` = " . intval($id);
        $_rt = jrCore_db_query($req, 'SINGLE');
        if ($_rt && is_array($_rt)) {
            return $_rt['v'];
        }
    }
    else {
        $tbl = jrCore_db_table_name($module, 'item_key');
        $req = "SELECT `index` AS i, `value` AS v FROM {$tbl} WHERE `_item_id` = " . intval($id) . " AND `key` = '" . jrCore_db_escape($key) . "'";
        $_rt = jrCore_db_query($req, 'NUMERIC');
        if ($_rt && is_array($_rt)) {
            if (!isset($_rt[1])) {
                return $_rt[0]['v'];
            }
            $_ot = array();
            foreach ($_rt as $v) {
                if (isset($_ot["{$v['i']}"])) {
                    $_ot["{$v['i']}"] .= $v['v'];
                }
                else {
                    $_ot["{$v['i']}"] = $v['v'];
                }
            }
            ksort($_ot, SORT_NUMERIC);
            return implode('', $_ot);
        }
    }
    return false;
}

/**
 * Core DS Plugin
 * @param string $module
 * @param array $_data
 * @param bool $exist_check
 * @return bool
 */
function _jrCore_db_update_multiple_items($module, $_data = null, $exist_check = true)
{
    // If $exist_check has been set to FALSE, we may not know the _profile_id for any
    // NEW key that we may have to insert - get it here if needed
    $_pi = array();
    if (!$exist_check) {
        foreach ($_data as $_vals) {
            if (empty($_vals['_profile_id'])) {
                $exist_check = true;
                break;
            }
        }
    }
    // Get profile ids
    if ($exist_check) {
        if (jrCore_db_key_has_index_table($module, '_profile_id')) {
            $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
            $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `_item_id` IN(" . implode(',', array_keys($_data)) . ')';
        }
        else {
            $tbl = jrCore_db_table_name($module, 'item_key');
            $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `key` = '_profile_id' AND `_item_id` IN(" . implode(',', array_keys($_data)) . ')';
        }
        $_pi = jrCore_db_query($req, 'i', false, 'v');
        if (!$_pi) {
            // items do not exist
            return false;
        }
    }

    // Are we updating _profile_id ?
    $_up = array();
    foreach ($_data as $id => $_vals) {
        if ($exist_check && !isset($_pi[$id])) {
            // This item does not exist - skip
            unset($_data[$id]);
        }
        if (isset($_vals['_profile_id'])) {
            // We are setting the _profile_id for this item - make sure ALL keys
            // for this item have been updated to the correct _profile_id
            $_vals['_profile_id'] = (int) $_vals['_profile_id'];
            if ($_vals['_profile_id'] > 0) {
                $_pi[$id] = $_vals['_profile_id'];
                $_up[$id] = $_vals['_profile_id'];
            }
        }
    }
    if (count($_data) === 0) {
        // nothing to update
        return false;
    }

    // Update
    $_rq = array();
    $_mx = array();
    $_zo = array();
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "INSERT INTO {$tbl} (`_item_id`,`_profile_id`,`key`,`index`,`value`) VALUES ";
    foreach ($_data as $id => $_vals) {
        $_mx[$id] = array();
        $_zo[$id] = array();
        $pid      = (isset($_pi[$id])) ? intval($_pi[$id]) : 0;
        foreach ($_vals as $k => $v) {

            $val = false;
            if ($v === 'UNIX_TIMESTAMP()') {
                $req          .= "({$id},{$pid},'" . jrCore_db_escape($k) . "',0,UNIX_TIMESTAMP()),";
                $_mx[$id][$k] = '0';
                $val          = 'UNIX_TIMESTAMP()';
            }
            else {
                if (!is_numeric($v)) {
                    $v = jrCore_strip_emoji($v);
                }
                $len = strlen($v);
                // If our value is longer than 508 bytes we split it up
                if ($len > 508) {
                    $_tm = array();
                    while ($len) {
                        $_tm[] = mb_strcut($v, 0, 508, "UTF-8");
                        $v     = mb_strcut($v, 508, $len, "UTF-8");
                        $len   = strlen($v);
                    }
                    $idx = 0;
                    foreach ($_tm as $i => $part) {
                        $idx = ($i + 1);
                        $req .= "({$id},{$pid},'" . jrCore_db_escape($k) . "','{$idx}','" . jrCore_db_escape($part) . "'),";
                    }
                    $_mx[$id][$k] = $idx;
                    // We have to also delete any previous 0 index
                    $_zo[$id][] = $k;
                }
                else {
                    if (is_numeric($v)) {
                        $val = $v;
                    }
                    else {
                        $val = jrCore_db_escape($v);
                    }
                    $req          .= "({$id},{$pid},'" . jrCore_db_escape($k) . "',0,'" . $val . "'),";
                    $_mx[$id][$k] = '0';
                }
            }
            if ($val && jrCore_db_key_has_index_table($module, $k)) {
                $tbi = jrCore_db_get_index_table_name($module, $k);
                if ($val != 'UNIX_TIMESTAMP()') {
                    $val = "'{$val}'";
                }
                $_rq[] = "INSERT INTO {$tbi} (`_item_id`,`value`) VALUES ({$id},{$val}) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)";
            }
        }
    }
    $_rq[] = substr($req, 0, strlen($req) - 1) . " ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)";

    // Cleanup
    $_tm = array();
    foreach ($_mx as $id => $_vals) {
        foreach ($_vals as $fld => $max) {
            if (jrCore_is_ds_index_needed($fld)) {
                $_tm[] = "(`_item_id` = {$id} AND `key` = '" . jrCore_db_escape($fld) . "' AND `index` > {$max})";
            }
        }
    }
    if (count($_zo) > 0) {
        // We have fields where we need to delete the ZERO index
        foreach ($_zo as $id => $_vals) {
            foreach ($_vals as $fld) {
                $_tm[] = "(`_item_id` = {$id} AND `key` = '" . jrCore_db_escape($fld) . "' AND `index` = 0)";
            }
        }
    }

    // Set _profile_id's for items that need changing
    if (count($_up) > 0) {
        foreach ($_up as $k => $v) {
            $_rq[] = "UPDATE {$tbl} SET `_profile_id` = {$v} WHERE `_item_id` = " . intval($k);
        }
    }

    // Delete key indexes that are no longer used
    if (count($_tm) > 0) {
        $_rq[] = "DELETE FROM {$tbl} WHERE " . implode(' OR ', $_tm);
    }
    unset($_tm);

    // @note Do not check the return of jrCore_db_multi_query - you will
    // only get a return for SELECT statements - not UPDATE or DELETE
    jrCore_db_multi_query($_rq, false);
    return true;
}

/**
 * Core DS plugin
 * @param $module
 * @param $_ids
 * @param bool $delete_media
 * @param bool $profile_count
 * @return true
 */
function _jrCore_db_delete_multiple_items($module, $_ids, $delete_media = true, $profile_count = true)
{
    // Delete items
    $ids = implode(',', $_ids);
    $tb1 = jrCore_db_table_name($module, 'item');
    $tb2 = jrCore_db_table_name($module, 'item_key');
    $_rq = array(
        "DELETE FROM {$tb1} WHERE `_item_id` IN({$ids})",
        "DELETE FROM {$tb2} WHERE `_item_id` IN({$ids})"
    );
    if ($_tb = jrCore_db_get_all_index_tables_for_module($module)) {
        foreach ($_tb as $key) {
            $tbl   = jrCore_db_get_index_table_name($module, $key);
            $_rq[] = "DELETE FROM {$tbl} WHERE `_item_id` IN({$ids})";
        }
    }
    jrCore_db_multi_query($_rq, false);
    return true;
}

/**
 * Get an item order array for a given module and profile_id
 * @param string $module
 * @param int $profile_id
 * @return array|bool
 */
function _jrCore_db_get_display_order_for_profile_id($module, $profile_id)
{
    $pid = (int) $profile_id;
    $pfx = jrCore_db_get_prefix($module);
    $tbl = jrCore_db_table_name($module, 'item_key');
    $req = "SELECT `_item_id`, `value` FROM {$tbl} WHERE `_profile_id` = {$pid} AND `key` = '{$pfx}_display_order'";
    if ($_rt = jrCore_db_query($req, '_item_id', false, 'value')) {
        return $_rt;
    }
    return false;
}

/**
 * Get Private Profiles
 * @return array|bool
 */
function _jrCore_db_get_private_profiles()
{
    $tbl = jrCore_db_table_name('jrProfile', 'item_key');
    $req = "SELECT `_item_id` AS i, `value` AS v FROM {$tbl} WHERE `key` = 'profile_private' AND `value` != '1'";
    return jrCore_db_query($req, 'i', false, 'v');
}

/**
 * PLUGIN - Get an array of pending item_ids for a module
 * @param string $module
 * @return array|bool
 */
function _jrCore_db_get_pending_item_ids($module)
{
    $pfx = jrCore_db_get_prefix($module);
    if (jrCore_db_key_has_index_table($module, "{$pfx}_pending")) {
        $tbl = jrCore_db_get_index_table_name($module, "{$pfx}_pending");
        $req = "SELECT `_item_id` AS i FROM {$tbl} WHERE `value` IN(1,2)";
    }
    else {
        $tbl = jrCore_db_table_name($module, 'item_key');
        $req = "SELECT `_item_id` AS i FROM {$tbl} WHERE `key` = '{$pfx}_pending' AND `value` IN(1,2)";
    }
    return jrCore_db_query($req, 'i');
}

/**
 * jrCore DS plugin - db_search_items
 * @param $module string module
 * @param $_params array params
 * @return array|bool
 */
function _jrCore_db_search_items($module, $_params)
{
    global $_user, $_conf;
    $_params['module'] = $module;
    // Backup copy of original params
    $_backup = $_params;

    // Config values
    $optimized_order = jrCore_get_config_value('jrCore', 'optimize_order', 'on');
    $optimized_limit = jrCore_get_config_value('jrCore', 'pager_limit', 'on');

    // If we are optimizing our pagebreak we switch to simplepagebreak at the simplepagebreak_cutoff
    if ($optimized_limit == 'on' && isset($_params['pagebreak']) && jrCore_checktype($_params['pagebreak'], 'number_nz')) {
        $nr = jrCore_db_get_ds_row_count($module);
        $mr = (!empty($_params['simplepagebreak_cutoff'])) ? intval($_params['simplepagebreak_cutoff']) : ((!empty($_conf['jrCore_simplepagebreak_cutoff'])) ? intval($_conf['jrCore_simplepagebreak_cutoff']) : 50000);
        if ($nr > $mr) {
            $_params['simplepagebreak'] = (int) $_params['pagebreak'];
            unset($_params['pagebreak']);
        }
    }

    // Other modules can provide supported parameters for searching - send
    // our trigger so those events can be added in.
    if (!isset($_params['skip_triggers']) || jrCore_checktype($_params['skip_triggers'], 'is_false')) {

        $_params = jrCore_trigger_event('jrCore', 'db_search_params', $_params, array('module' => $module));

        // Did our listener return a full result set?
        if (isset($_params['full_result_set'])) {
            jrCore_db_search_fdebug($_backup, $_params, 'no query run - full_result_set privided by listening module', 0, array()); // OK
            return $_params['full_result_set'];
        }

        // See if a listener switched modules on us
        $_change = jrCore_get_flag('jrcore_active_trigger_args');
        if (isset($_change['module']) && $_change['module'] != $module) {
            $module            = $_change['module'];
            $_params['module'] = $module;
        }
        unset($_change);

    }

    // Check for special order_by display_order - if we ARE ordering by display_order
    // we need to turn OFF the quota check - we won't get there unless quota is allowed
    $key = 0;
    $pid = 0;
    if (!empty($_params['order_by']) && is_array($_params['order_by']) && count($_params['order_by']) === 1) {
        if ($pfx = jrCore_db_get_prefix($module)) {
            if (isset($_params['order_by']["{$pfx}_display_order"])) {
                // Yes we are ordering by DISPLAY order - is it for a specific profile_id?
                if (isset($_params['profile_id']) && jrCore_checktype($_params['profile_id'], 'number_nz')) {
                    $pid = (int) $_params['profile_id'];
                }
                if (!empty($_params['search']) && is_array($_params['search'])) {
                    foreach ($_params['search'] as $sk => $sv) {
                        if (strpos($sv, '_profile_id ') === 0) {
                            // Check if we have already been passed the profile_id as a separate parameter
                            if ($pid === 0) {
                                list(, $pid) = explode('=', $sv, 2);
                                $pid = (int) trim($pid);
                            }
                            $key = $sk;
                            break;
                        }
                    }
                }
                $_params['quota_check']            = false;  // Turn off quota check - we won't get this list unless it is allowed
                $_params['jrProfile_active_check'] = false;  // No need to check if profile is active

                if ($pid && $pid > 0) {
                    if ($pfx = jrCore_db_get_prefix($module)) {
                        if ($_ids = jrCore_db_get_display_order_for_profile_id($module, $pid, $_params['order_by']["{$pfx}_display_order"])) {
                            $_params['search'][$key] = "_item_id in " . implode(',', $_ids);
                            if (count($_params['search']) === 1) {
                                // No need for order by if we only have our in list
                                unset($_params['order_by']);
                            }
                        }
                    }
                }

            }
        }
    }

    // Create cache key
    $cky = json_encode($_params);

    // Send out Cache Check event
    $_params = jrCore_trigger_event('jrCore', 'db_search_cache_check', $_params, array('module' => $module, 'cache_key' => $cky));

    // By default results are cached for the requesting user.  This is because our
    // results can change based on the viewing user, and the data they have access to
    // However, if cache_user_id = false, then it means we have the SAME result
    // irregardless of who is running this function, and that allows us to save cache space
    $cache_user_id = true;
    if (isset($_params['cache_user_id']) && jrCore_checktype($_params['cache_user_id'], 'is_false')) {
        $cache_user_id = false;
    }

    $cache_profile_id = 0;
    if (!empty($_params['cache_profile_id']) && jrCore_checktype($_params['cache_profile_id'], 'number_nz')) {
        $cache_profile_id = (int) $_params['cache_profile_id'];
    }

    // Check for cache
    if (!isset($_params['no_cache']) || $_params['no_cache'] === false) {

        if ($tmp = jrCore_is_cached($module, $cky, $cache_user_id, false)) {
            if (isset($tmp['no_results'])) {
                if (isset($_params['return_count']) && jrCore_checktype($_params['return_count'], 'is_true')) {
                    return 0;
                }
                // Return false
                return false;
            }
            jrCore_db_search_fdebug($_backup, $_params, 'no query run - result set was cached', 0, $tmp); // OK
            if (!empty($tmp['info'])) {
                $tmp['info']['cached'] = 1;
            }
            if (isset($_params['return_count']) && jrCore_checktype($_params['return_count'], 'is_true')) {
                $tmp = intval($tmp);
            }
            return $tmp;
        }
    }

    // Changing cache seconds?
    $sec = 0;
    if (isset($_params['cache_seconds']) && jrCore_checktype($_params['cache_seconds'], 'number_nz')) {
        $sec = (int) $_params['cache_seconds'];
    }

    $cqt = false;  // Counter Query Time - used in fdebug logging

    // Allow a listener to provide the actual result set
    if (!isset($_params['result_set'])) {

        // We allow searching on both USER and PROFILE keys for all modules - check for those here
        if (isset($_params['search']) && is_array($_params['search'])) {
            switch ($module) {
                case 'jrProfile':
                    $_ck = array(
                        'user' => 'jrUser'
                    );
                    break;
                case 'jrUser':
                    $_ck = array(
                        'profile' => 'jrProfile'
                    );
                    break;
                default:
                    $_ck = array(
                        'user'    => 'jrUser',
                        'profile' => 'jrProfile'
                    );
                    break;
            }
            foreach ($_ck as $pfx => $mod) {
                foreach ($_params['search'] as $k => $cond) {
                    $_c = array();

                    // OR condition Check
                    if (strpos($cond, '||')) {
                        $tbl = jrCore_db_table_name($mod, 'item_key');
                        foreach (explode('||', $cond) as $part) {
                            if (strpos(trim($part), "{$pfx}_") === 0) {
                                if ($_sc = jrCore_db_check_for_supported_operator($part)) {
                                    // There are keys in this condition we need to handle
                                    if (strpos(' ' . $_sc[0], '%')) {
                                        $_c[] = "`key` LIKE '" . jrCore_db_escape($_sc[0]) . "' AND `value` {$_sc[1]} {$_sc[2]}";
                                    }
                                    elseif ($_sc[1] == 'between') {
                                        $_c[] = "`key` = '" . jrCore_db_escape($_sc[0]) . "' AND `value` BETWEEN {$_sc[2]} AND {$_sc[3]}";
                                    }
                                    elseif ($_sc[1] == 'not_between') {
                                        $_c[] = "`key` = '" . jrCore_db_escape($_sc[0]) . "' AND `value` NOT BETWEEN {$_sc[2]} AND {$_sc[3]}";
                                    }
                                    else {
                                        $_c[] = "`key` = '" . jrCore_db_escape($_sc[0]) . "' AND `value` {$_sc[1]} {$_sc[2]}";
                                    }
                                    unset($_params['search'][$k]);
                                }
                                else {
                                    jrCore_logger('MAJ', 'core: invalid OR search operator in jrCore_db_search_items parameters', array($module, $_params));
                                    return false;
                                }
                            }
                        }
                        if (count($_c) > 0) {
                            $_params['search'][] = "_{$pfx}_id IN (SELECT `_item_id` FROM {$tbl} WHERE (" . implode(' OR ', $_c) . '))';
                        }
                    }
                    // Check for "user_" and "profile_" key searches which are allowed when searching any DS
                    elseif (strpos(trim($cond), "{$pfx}_") === 0) {
                        $tbl = jrCore_db_table_name($mod, 'item_key');
                        if ($_sc = jrCore_db_check_for_supported_operator($cond)) {
                            $in_mod = 'in';
                            $in_cmp = $_sc[1];
                            if (isset($_params['ignore_missing']) && jrCore_checktype($_params['ignore_missing'], 'is_true')) {
                                // With ignore_missing set, not all items could have this key - that means we cannot
                                // test for items that have the key set to a !=, not_in or not_like value
                                switch ($_sc[1]) {
                                    case '!=':
                                        $in_mod = 'not_in';
                                        $in_cmp = '=';
                                        break;
                                    case 'not_in':
                                        $in_mod = 'not_in';
                                        $in_cmp = 'IN';
                                        break;
                                    case 'not_like':
                                        $in_mod = 'not_in';
                                        $in_cmp = 'LIKE';
                                        break;
                                }
                            }
                            $_params['search'][] = "_{$pfx}_id {$in_mod} (SELECT `_item_id` FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($_sc[0]) . "' AND `value` {$in_cmp} {$_sc[2]})";
                            unset($_params['search'][$k]);
                        }
                        else {
                            jrCore_logger('MAJ', 'core: invalid OR search operator in jrCore_db_search_items parameters', array($module, $_params));
                            return false;
                        }
                    }
                }
                unset($_c);
            }
        }

        $uik = true; // Use index key - we will set this FALSE for specific _profile_id queries
        $prf = '';
        $dob = '_created';
        $_in = array();
        $_ob = array();
        $_sc = array();
        $_ky = array();
        $_eq = array();
        $_ne = array();
        $ino = false;
        $sgb = false;
        $_so = false;
        $tic = 0;  // "Total IN() Count"

        if (isset($_params['search']) && is_array($_params['search']) && count($_params['search']) > 0) {

            // Pre check for OR search conditions
            if (strpos(json_encode($_params['search']), '||')) {
                $_so = array();
                foreach ($_params['search'] as $k => $v) {
                    if (strpos($v, '||')) {
                        foreach (explode('||', $v) as $cond) {
                            if (!$tmp = jrCore_db_check_for_supported_operator($cond)) {
                                jrCore_logger('MAJ', 'core: invalid OR search operator in jrCore_db_search_items parameters', array($module, $_params));
                                return false;
                            }
                            if (strpos(' ' . $tmp[0], '%')) {
                                // Wildcard key
                                $tmp[1] = 'LIKE';
                            }
                            $_so[$k][] = $tmp;
                        }
                        if (count($_so) > 0) {
                            $_params['search'][$k] = "{$k} OR COND";
                        }
                    }
                }
            }

            // We need to be sure that != and not_in search conditions come
            // last in the search array, since if we have an = or in search
            // we may be able to exclude the != or not_in search entirely
            $_lk = array();
            $_hk = array();
            foreach ($_params['search'] as $k => $v) {
                $v = trim($v);
                if (isset($_dc[$v])) {
                    // Already seen this one...
                    unset($_params['search'][$k]);
                    continue;
                }
                $_dc[$v] = 1;
                @list($key, $opt,) = explode(' ', $v, 3);
                if (!isset($opt) || strlen($opt) === 0) {
                    // Bad Search
                    jrCore_logger('MAJ', 'core: invalid search criteria in jrCore_db_search_items parameters', array($module, $_params));
                    return false;
                }
                switch (jrCore_str_to_lower($opt)) {
                    case '!=':
                    case 'not_in':
                        if (trim($key) == '_profile_id') {
                            $_hk[] = $v;
                        }
                        else {
                            $_lk[] = $v;
                        }
                        break;
                    default:
                        $_lk[] = $v;
                        break;
                }
            }
            if (count($_hk) > 0) {
                $_params['search'] = array_merge($_lk, $_hk);
            }
            unset($_lk, $_hk);

            // Search prep
            $_dc = array();
            foreach ($_params['search'] as $k => $v) {
                @list($key, $opt, $val) = explode(' ', $v, 3);
                $key = jrCore_str_to_lower($key);
                if (!strpos(' ' . $key, '%')) {
                    $_ky[$key] = 1;
                }
                if (!empty($val) && strpos($val, '(SELECT ') === 0) {
                    // We have a sub query as our match condition
                    // If this is a sub query for _profile_id we can skip a join
                    if (strpos($v, '_profile_id ') === 0) {
                        // We are looking for specific profile id's
                        $prf .= ' AND a.`_profile_id` ' . str_replace('not_in', 'NOT IN', substr($v, 12)) . ' ';
                        $uik = false;
                    }
                    else {
                        switch (jrCore_str_to_lower($opt)) {
                            case 'not_in':
                                $_sc[] = array($key, 'NOT IN', $val, 'no_quotes');
                                break;
                            case 'not_like':
                                $_sc[] = array($key, 'NOT LIKE', $val, 'no_quotes');
                                break;
                            default:
                                $_sc[] = array($key, $opt, $val, 'no_quotes');
                                break;
                        }
                    }
                    continue;
                }
                // Check for OR conditions (||)
                elseif ($opt == 'OR' && $val == 'COND') {
                    // We have an OR condition as our match condition
                    $_sc[] = array($key, $opt, $val, 'parens');
                    continue;
                }

                switch (jrCore_str_to_lower($opt)) {
                    case '>':
                    case '>=':
                    case '<':
                    case '<=':
                        if (strpos($val, '.')) {
                            $_sc[] = array($key, $opt, floatval($val), 'no_quotes');
                        }
                        else {
                            $_sc[] = array($key, $opt, intval($val), 'no_quotes');
                        }
                        break;

                    // Not Equal To
                    case '!=':
                        // With a NOT EQUAL operator on non _item_id, we also need to include items where the key may be MISSING or NULL
                        if ($key == '_item_id' || $key == '_profile_id' || $key == '_user_id') {
                            if ($key == '_profile_id') {
                                $val = (int) $val;
                                // Did we already get an = search for a specific profile_id?  If so then
                                // we can skip this != search condition
                                if (isset($_eq[$key][$val])) {
                                    jrCore_db_search_fdebug_error('invalid search condition - _profile_id search values exclude each other', $_backup, $_params);
                                    return false;
                                }
                                // Note: we handle _profile_id searches separately
                                $_ne[$key][$val] = $val;
                            }
                            else {
                                $_sc[] = array($key, $opt, intval($val), 'no_quotes');
                            }
                        }
                        elseif (jrCore_db_key_found_on_all_items($key) || (isset($_params['ignore_missing']) && jrCore_checktype($_params['ignore_missing'], 'is_true'))) {
                            $_sc[] = array($key, $opt, jrCore_db_escape($val));
                        }
                        else {
                            $tbl   = jrCore_db_table_name($module, 'item_key');
                            $vrq   = "(SELECT yy.`_item_id` FROM {$tbl} yy LEFT JOIN {$tbl} zz ON (zz.`_item_id` = yy.`_item_id` AND zz.`key` = '" . jrCore_db_escape($key) . "') WHERE yy.`key` = '_created' AND (zz.`value` != '" . jrCore_db_escape($val) . "' OR zz.`value` IS NULL))";
                            $_sc[] = array('_item_id', 'IN', $vrq, 'no_quotes', $key);
                            unset($_ky[$key]);
                        }
                        break;

                    // Equal To
                    case '=':
                        if (ctype_digit("{$val}")) {
                            if ($key == '_profile_id') {
                                // NOTE: we do not add _profile_id searches to our $_sc (search conditions) array since they are handled separately
                                $val             = (int) $val;
                                $_eq[$key][$val] = $val;
                            }
                            else {
                                $_sc[] = array($key, $opt, $val);
                            }
                        }
                        else {
                            $_sc[] = array($key, $opt, jrCore_db_escape($val));
                        }
                        break;

                    // Between | Not_Between
                    case 'between':
                    case 'not_between':
                        if (strpos($val, ',')) {
                            list($vl1, $vl2) = explode(',', $val);
                            $vl1 = trim($vl1);
                            $vl2 = trim($vl2);
                            if (is_numeric($vl1) && is_numeric($vl2)) {
                                if (strpos(' ' . $vl1, '.')) {
                                    $vl1 = floatval($vl1);
                                }
                                else {
                                    $vl1 = intval($vl1);
                                }
                                if (strpos(' ' . $vl2, '.')) {
                                    $vl2 = floatval($vl2);
                                }
                                else {
                                    $vl2 = intval($vl2);
                                }
                                if ($vl2 < $vl1) {
                                    $val = "{$vl2},{$vl1}";
                                }
                                else {
                                    $val = "{$vl1},{$vl2}";
                                }
                                $_sc[] = array($key, jrCore_str_to_lower($opt), $val);
                            }
                        }
                        else {
                            jrCore_logger('MAJ', "core: invalid {$opt} search condition in jrCore_db_search_items search: {$opt}", array($module, $_params));
                            return false;
                        }
                        break;

                    // Like
                    case 'like':
                        if (strpos($val, '\%')) {
                            // We are looking explicitly for a percent sign
                            $_ps = explode('\%', $val);
                            if ($_ps && is_array($_ps)) {
                                $_pt = array();
                                foreach ($_ps as $pprt) {
                                    $_pt[] = jrCore_db_escape($pprt);
                                }
                                $_sc[] = array($key, strtoupper($opt), implode('\%', $_pt));
                                unset($_pt);
                            }
                            else {
                                $_sc[] = array($key, strtoupper($opt), jrCore_db_escape($val));
                            }
                            unset($_ps);
                        }
                        else {
                            $_sc[] = array($key, strtoupper($opt), jrCore_db_escape($val));
                        }
                        // If we do NOT get a group_by parameter, and we are doing a
                        // wildcard search on the KEY, we have to group by _item_id
                        if (!isset($_params['group_by']) && strpos(' ' . $key, '%')) {
                            $ino = '_item_id';
                        }
                        break;

                    // Not_Like
                    case 'not_like':
                        if (strpos($val, '\%')) {
                            // We are looking explicitly for a percent sign
                            $_ps = explode('\%', $val);
                            if ($_ps && is_array($_ps)) {
                                $_pt = array();
                                foreach ($_ps as $pprt) {
                                    $_pt[] = jrCore_db_escape($pprt);
                                }
                                $val = implode('\%', $_pt);
                                unset($_pt);
                            }
                        }
                        $tbl = jrCore_db_table_name($module, 'item_key');
                        if (isset($_params['ignore_missing']) && jrCore_checktype($_params['ignore_missing'], 'is_true')) {
                            if (jrCore_db_key_has_index_table($module, $key)) {
                                $vrq = "(SELECT yy.`_item_id` FROM " . jrCore_db_get_index_table_name($module, $key) . " yy WHERE yy.`value` NOT LIKE '" . jrCore_db_escape($val) . "')";
                            }
                            else {
                                $vrq = "(SELECT yy.`_item_id` FROM {$tbl} yy WHERE yy.key = '" . jrCore_db_escape($key) . "' AND yy.`value` NOT LIKE '" . jrCore_db_escape($val) . "')";
                            }
                        }
                        else {
                            $vrq = "(SELECT yy.`_item_id` FROM {$tbl} yy LEFT JOIN {$tbl} zz ON (zz.`_item_id` = yy.`_item_id` AND zz.`key` = '" . jrCore_db_escape($key) . "') WHERE yy.`key` = '_created' AND (zz.`value` NOT LIKE '" . jrCore_db_escape($val) . "' OR zz.`value` IS NULL))";
                        }
                        $_sc[] = array('_item_id', 'IN', $vrq, 'no_quotes', $key);
                        unset($_ky[$key]);
                        break;

                    case 'regexp':
                        $_sc[] = array($key, strtoupper($opt), jrCore_db_escape($val));
                        break;

                    case 'in':
                        $_vl = array();
                        foreach (explode(',', $val) as $iv) {
                            if (ctype_digit("{$iv}")) {
                                if ($key == '_item_id' || $key == '_profile_id' || $key == '_user_id') {
                                    $_vl[] = intval($iv);
                                }
                                else {
                                    // Don't (int) here - strips leading zeros
                                    $_vl[] = "'{$iv}'";
                                }
                            }
                            else {
                                $_vl[] = "'" . jrCore_db_escape($iv) . "'";
                            }
                            $_in[$iv] = $iv;
                        }

                        // By default if we do NOT get an ORDER BY clause on an IN, order by FIELD unless specifically set NOT to
                        if (!isset($_params['order_by']) && !isset($_params['return_item_id_only'])) {
                            // If our key is _item_id, we can skip the GROUP BY
                            if ($key == '_item_id') {
                                $sgb = true;
                            }
                            $ino = $key;
                            if (isset($_do)) {
                                $_do = array_merge($_do, $_vl);
                            }
                            else {
                                // Our values...
                                if (count($_params['search']) === 1) {
                                    // We set out Total Item Count here - this is used below when computing our pagebreak
                                    $tic = count($_vl);
                                    $_do = jrCore_db_get_item_id_in_list($_vl, $_params);
                                    if (empty($_do)) {
                                        // Results on this page do not exist - no data
                                        jrCore_add_to_cache($module, $cky, array('no_results' => 1), $sec, $cache_profile_id, $cache_user_id, false);
                                        jrCore_db_search_fdebug_error("no items returned from get_item_id_in_list function call", $_backup, $_params);
                                        return false;
                                    }
                                }
                                else {
                                    $_do = $_vl;
                                }
                            }
                            $_vl = $_do;
                        }

                        if ($key == '_item_id' || $key == '_profile_id' || $key == '_user_id') {
                            if ($key == '_profile_id') {
                                if (!isset($_eq[$key])) {
                                    $_eq[$key] = array();
                                }
                                foreach ($_vl as $vpid) {
                                    $_eq[$key][$vpid] = $vpid;
                                }
                            }
                            else {
                                if (count($_vl) > 1) {
                                    $val   = "(" . implode(',', $_vl) . ')';
                                    $_sc[] = array($key, 'IN', $val, 'no_quotes');
                                }
                                else {
                                    $_sc[] = array($key, '=', reset($_vl), 'no_quotes');
                                }
                            }
                        }
                        else {
                            $val   = "(" . implode(',', $_vl) . ')';
                            $_sc[] = array($key, 'IN', $val, 'no_quotes');
                        }
                        unset($_vl);
                        break;

                    case 'not_in':
                        // If we are excluding specific _profile_id's then we can see if we have already included them
                        if ($key == '_profile_id' && isset($_eq[$key])) {
                            $_pval = explode(',', $val);
                            foreach ($_pval as $ik => $iv) {
                                $iv = intval($iv);
                                if (isset($_eq['_profile_id'][$iv])) {
                                    // We have both an EQUAL and a NOT EQUAL for this profile_id - exclude
                                    unset($_eq['_profile_id'][$iv]);
                                    unset($_pval[$ik]);
                                }
                            }
                            // If we come out with equals left, we can exclude all the not_in's left since we would never match them
                            if (count($_eq['_profile_id']) > 0) {
                                jrCore_db_search_fdebug_error("removing search condition {$k} due to equals exclusion", $_backup, $_params);
                                unset($_params['search'][$k]);
                                unset($_pval);
                                continue 2;
                            }
                            if (isset($_do)) {
                                $_do = $_eq['_profile_id'];
                            }
                            // We have some not_in profile_id's left - restore $val
                            if (count($_pval) > 0) {
                                $val = implode(',', $_pval);
                                unset($_pval);
                            }
                            else {
                                // We have no not-in conditions left - unset and continue
                                unset($_params['search'][$k]);
                                unset($_pval);
                                continue 2;
                            }
                        }

                        $_vl = array();
                        foreach (explode(',', $val) as $iv) {
                            switch ($key) {
                                case '_item_id':
                                case '_user_id':
                                case '_profile_id':
                                case '_created':
                                case '_updated':
                                    $_vl[] = intval($iv);
                                    break;
                                default:
                                    if (ctype_digit("{$iv}")) {
                                        // Don't int here - strips leading zeros
                                        $_vl[] = "'{$iv}'";
                                    }
                                    else {
                                        $_vl[] = "'" . jrCore_db_escape($iv) . "'";
                                    }
                                    break;
                            }
                        }
                        $val = '(' . implode(',', $_vl) . ')';
                        // ALL items have a _item_id/_profile_id/_user_id so no need to do the extra join here
                        if ((jrCore_db_key_found_on_all_items($key) && (!isset($_params['ignore_missing']) || jrCore_checktype($_params['ignore_missing'], 'is_true'))) || (isset($_params['ignore_missing']) && jrCore_checktype($_params['ignore_missing'], 'is_true'))) {
                            // If we have a _profile_id NOT IN, and we are running our privacy check down below,
                            // we can skip creating another JOIN condition and just add to the existing profile_id check
                            if ($key == '_profile_id') {
                                if (!isset($_ne[$key])) {
                                    $_ne[$key] = array();
                                }
                                foreach ($_vl as $vpid) {
                                    $_ne[$key][$vpid] = $vpid;
                                }
                            }
                            else {
                                // NOTE: We use "no_quotes" here since the values in $_vl have been quoted as needed
                                if (count($_vl) == 1) {
                                    $_sc[] = array($key, '!=', reset($_vl), 'no_quotes');
                                }
                                else {
                                    $_sc[] = array($key, 'NOT IN', $val, 'no_quotes');
                                }
                            }
                        }
                        else {
                            $tbl   = jrCore_db_table_name($module, 'item_key');
                            $vrq   = "(SELECT yy.`_item_id` FROM {$tbl} yy LEFT JOIN {$tbl} zz ON (zz.`_item_id` = yy.`_item_id` AND zz.`key` = '" . jrCore_db_escape($key) . "') WHERE yy.`key` = '_created' AND (zz.`value` NOT IN{$val} OR zz.`value` IS NULL))";
                            $_sc[] = array('_item_id', 'IN', $vrq, 'no_quotes', $key);
                            unset($_ky[$key]);
                        }
                        unset($_vl);
                        break;

                    default:
                        jrCore_logger('MAJ', "core: invalid search operator in jrCore_db_search_items search: {$opt}", array($module, $_params));
                        return false;
                }
            }
            unset($_dc);
        }

        // Module prefix
        $pfx = jrCore_db_get_prefix($module);

        // Check for Pending Support
        $apn = false;
        $_pn = jrCore_get_registered_module_features('jrCore', 'pending_support');
        if (!jrUser_is_admin() && isset($_pn) && isset($_pn[$module]) && (!isset($_params['ignore_pending']) || jrCore_checktype($_params['ignore_pending'], 'is_false'))) {
            // If this is a profile owner viewing their own profile we show them everything
            if (!isset($_params['profile_id']) || !jrProfile_is_profile_owner($_params['profile_id'])) {
                // Pending support is on for this module - check status
                // 0 = immediately active
                // 1 = review needed
                // Let's see if anything is pending
                if ($_pi = jrCore_db_get_pending_item_ids($module)) {
                    $apn = "AND a.`_item_id` NOT IN(" . implode(',', $_pi) . ")\n";
                }
            }
        }

        // in order to properly ORDER BY, we must be including the key we are
        // ordering by in our JOIN - thus, if the user specifies an ORDER BY on
        // a key that they did not search on, then we must add in an IS NOT NULL
        // condition for the order by key
        $custom_order_by = array();
        if (isset($_params['order_by']) && $_params['order_by'] !== false) {
            if (is_array($_params['order_by'])) {

                // Check for some special orders
                if (count($_params['order_by']) === 1) {
                    if (isset($_params['order_by']["{$pfx}_display_order"])) {
                        // We are ordering by DISPLAY ORDER
                        // Sort by display order, _item_id desc default
                        $_params['order_by']['_item_id'] = 'numerical_desc';
                    }
                    elseif ($module == 'jrProfile' && isset($_params['order_by']['_profile_id'])) {
                        // Order by _item_id
                        $_params['order_by'] = array('_item_id' => $_params['order_by']['_profile_id']);
                    }
                    elseif ($module == 'jrUser' && isset($_params['order_by']['_user_id'])) {
                        // Order by _item_id
                        $_params['order_by'] = array('_item_id' => $_params['order_by']['_user_id']);
                    }
                }

                foreach ($_params['order_by'] as $k => $v) {
                    if ($k != '_item_id') {
                        $dob = $k;
                    }
                    // Check for random order - no need to join
                    if (!isset($_ky[$k]) && $k != '_item_id' && $v != 'random') {

                        // See if we have existing search queries.  This sub query is needed
                        // since if there are NO search conditions, then the table will NOT
                        // have been joined in order to do the order - but if we join on a
                        // not exists condition (!=, not_in, not_like) we have to include items
                        // that do NOT have the DS key set
                        switch ($k) {

                            // We know these keys exist on all items
                            case '_created':
                            case '_updated':
                            case '_user_id':
                            case '_profile_id':
                                $_sc[] = array($k, '>', 0);
                                break;

                            // Any other key may NOT exist on all items
                            default:
                                if ((jrCore_db_key_found_on_all_items($k) && (!isset($_params['ignore_missing']) || jrCore_checktype($_params['ignore_missing'], 'is_true'))) || (isset($_params['ignore_missing']) && jrCore_checktype($_params['ignore_missing'], 'is_true'))) {
                                    // This key is found on all items OR we've been specifically told to ignore any items that are missing this key
                                    $vrq = "`key` = '" . jrCore_db_escape($k) . "'";
                                }
                                else {
                                    // NOTE: Do not use an index table here since an index table
                                    // only contains entries for items that HAVE the key
                                    $tbl                 = jrCore_db_table_name($module, 'item_key');
                                    $vrq                 = "((a.`key` = '" . jrCore_db_escape($k) . "') OR (a.`key` = '_created' AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '" . jrCore_db_escape($k) . "')))";
                                    $custom_order_by[$k] = 1;
                                }
                                $_sc[] = array($k, 'CUSTOM', $vrq);
                                break;
                        }
                        $_ky[$k] = 1;
                    }
                }
            }
            else {
                // We have a bad order_by
                jrCore_logger('DBG', "core: invalid order_by in jrCore_db_search_items - value must be an array", array($module, $_params, $_REQUEST));
            }
        }

        // Lastly - if we get a group_by parameter, we have to make sure the field
        // that is being grouped on is joined to the query so it can be grouped
        $_gb = array();
        if (isset($_params['group_by']) && !is_array($_params['group_by']) && strlen($_params['group_by']) > 0 && strpos($_params['group_by'], '_item_id') !== 0) {
            // Check for special UNIQUE in our group_by
            if (strpos($_params['group_by'], ' UNIQUE')) {
                list($gfd,) = explode(' ', $_params['group_by']);
                $gfd = trim($gfd);
                $gtb = jrCore_db_table_name($module, 'item_key');
                $grq = "SELECT MAX(`_item_id`) AS iid FROM {$gtb} WHERE `key` = '" . jrCore_db_escape($gfd) . "' GROUP BY `value`";
                jrCore_start_timer("search_items_gb_{$module}");
                $_gr = jrCore_db_query($grq, 'iid', false, 'iid');
                jrCore_stop_timer("search_items_gb_{$module}");
                if ($_gr && is_array($_gr)) {
                    $_sc[] = array('_item_id', 'IN', '(' . implode(',', $_gr) . ')', 'no_quotes');
                }
                unset($_params['group_by']);
            }
            else {
                foreach (explode(',', $_params['group_by']) as $k => $gby) {
                    $gby = trim($gby);
                    switch ($gby) {

                        case '_item_id':
                            // _item_id never needs a join since that is the default
                            break;

                        case '_profile_id':
                            // _profile_id MAY NOT need a join - it depends on dedicated key indexes
                            break;

                        default:
                            if (!isset($_ky[$gby])) {
                                $_sc[] = array($gby, 'IS NOT', 'NULL');
                            }
                            break;
                    }
                    if (!isset($_gb[$gby])) {
                        $_gb[$gby] = $k;
                    }
                }
            }
        }
        // Make sure we got something
        if (!isset($_sc) || !is_array($_sc) || count($_sc) === 0) {
            $_sc[] = array('_item_id', '>', 0);
        }

        // To try and avoiding creating temp tables, we need to make sure if we have
        // an ORDER BY clause, the table that is being ordered on needs to be the
        // first table in the query
        // https://dev.mysql.com/doc/refman/5.0/en/order-by-optimization.html
        if (isset($_params['order_by']) && is_array($_params['order_by'])) {
            $o_key = array_keys($_params['order_by']);
            $o_key = reset($o_key);
            $_stmp = array();
            $found = false;
            foreach ($_sc as $k => $v) {
                if (!$found && $v[0] == $o_key) {
                    $_stmp[0] = $v;
                    $found    = true;
                }
                else {
                    $t_key         = ($k + 1);
                    $_stmp[$t_key] = $v;
                }
            }
            ksort($_stmp, SORT_NUMERIC);
            $_sc = array_values($_stmp);
            unset($_stmp, $o_key, $found, $t_key);
        }

        // If the FIRST JOIN has a dedicated index table, and we are GROUPING by _profile_id, the query
        // will fail since the dedicated index table does not have a _profile_id column
        $join_p = false;
        if (isset($_gb['_profile_id'])) {
            // If we have more than 1 search condition, we can use the FIRST join condition
            // that is not joining on a dedicated index table
            if (count($_sc) > 0) {
                foreach ($_sc as $v) {
                    if (!jrCore_db_key_has_index_table($module, $v[0])) {
                        // We found another join we can use - save the key
                        $join_p = $v[0];
                        break;
                    }
                }
            }
            if (!$join_p) {
                // There are no other joins we are doing where we can get the _profile_id - join
                $_sc[] = array('_profile_id', 'IS NOT', 'NULL');
            }
        }

        // Build query and get data
        $idx = true;
        $_al = range('a', 'z');

        // Allow modules to tell us what key names do NOT need an "index check" - i.e.
        // if a key is always going to contain a short value (i.e. a number, etc.)
        // then there is no need for the index < 2 check to be added on the JOIN
        // condition.  This can save a temporary table creation
        $_ii = array();
        $_ag = array(
            'module'    => $module,
            'params'    => $_params,
            'search'    => $_sc,
            'cache_key' => $cky
        );
        $_ii = jrCore_trigger_event('jrCore', 'db_search_simple_keys', $_ii, $_ag);

        $req = '';       // Main data Query
        $_jc = array();  // saves key values we matched in our JOIN condition so we can skip them in the WHERE condition
        $_di = array();

        foreach ($_sc as $k => $v) {

            // Save for our "order by" below - we must be searching on a column to order by it
            $als            = $_al[$k];
            $_ob["{$v[0]}"] = $als;

            // Does this key have a dedicated index column?
            $kdx = false;
            $tba = jrCore_db_table_name($module, 'item_key');
            if ($uik && jrCore_db_key_has_index_table($module, $v[0])) {
                $kdx     = true;
                $tba     = jrCore_db_get_index_table_name($module, $v[0]);
                $_di[$k] = $v[0];
            }

            if ($k == 0) {

                $force = (!empty($_params['force_index'])) ? " FORCE INDEX ({$_params['force_index']})" : '';
                if (is_array($_so)) {
                    // With an OR condition we have to group on the item_id or we can
                    // get multiple results for the same key
                    $req .= "SELECT DISTINCT(a.`_item_id`) AS _item_id FROM {$tba} a{$force}\n";
                    $idx = false;
                }
                else {
                    $req .= "SELECT a.`_item_id` AS _item_id FROM {$tba} a{$force}\n";
                }

            }
            // wildcard
            elseif (strpos(' ' . $v[0], '%') || $v[1] == 'OR') {
                if (!$idx) {
                    // We're already doing a DISTINCT so no need for index requirement
                    $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                }
                else {
                    if (is_array($_ii) && isset($_ii["{$v[0]}"]) || !jrCore_is_ds_index_needed($v[0])) {
                        $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                    }
                    else {
                        if ($kdx) {
                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                        }
                        else {
                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`index` < 2)\n";
                        }
                    }
                }
            }
            elseif ($v[0] !== '_item_id') {
                if (!$idx) {
                    // We're already doing a DISTINCT so no need for index requirement
                    if (isset($custom_order_by["{$v[0]}"])) {
                        if ($kdx) {
                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                        }
                        else {
                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}')\n";
                        }
                    }
                    else {
                        if (isset($v[3]) && $v[3] == 'no_quotes' || $v[2] == 'NULL') {
                            if ($kdx) {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` {$v[1]} {$v[2]})\n";
                            }
                            else {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` {$v[1]} {$v[2]})\n";
                            }
                        }
                        elseif ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            if ($kdx) {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` BETWEEN {$v1} AND {$v2})\n";
                            }
                            else {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` BETWEEN {$v1} AND {$v2})\n";
                            }
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            if ($kdx) {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2})\n";
                            }
                            else {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2})\n";
                            }
                        }
                        else {
                            if ($kdx) {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` {$v[1]} '{$v[2]}')\n";
                            }
                            else {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` {$v[1]} '{$v[2]}')\n";
                            }
                        }
                    }
                }
                else {
                    switch ($v[0]) {
                        case '_item_id':
                        case '_user_id':
                        case '_created':
                        case '_updated':
                            // No index needed on keys we know cannot be longer than 512
                            if ($kdx) {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                            }
                            else {
                                $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}')\n";
                            }
                            break;
                        default:
                            if (is_array($_ii) && isset($_ii["{$v[0]}"]) || !jrCore_is_ds_index_needed($v[0])) {
                                if (isset($custom_order_by["{$v[0]}"])) {
                                    if ($kdx) {
                                        $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                                    }
                                    else {
                                        $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}')\n";
                                    }
                                }
                                else {
                                    if (isset($v[3]) && $v[3] == 'no_quotes' || $v[2] == 'NULL') {
                                        if ($kdx) {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` {$v[1]} {$v[2]})\n";
                                        }
                                        else {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` {$v[1]} {$v[2]})\n";
                                        }
                                    }
                                    elseif ($v[1] == 'between') {
                                        list($v1, $v2) = explode(',', $v[2]);
                                        if ($kdx) {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` BETWEEN {$v1} AND {$v2})\n";
                                        }
                                        else {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` BETWEEN {$v1} AND {$v2})\n";
                                        }
                                    }
                                    elseif ($v[1] == 'not_between') {
                                        list($v1, $v2) = explode(',', $v[2]);
                                        if ($kdx) {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2})\n";
                                        }
                                        else {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2})\n";
                                        }
                                    }
                                    elseif ($v[1] == 'CUSTOM') {
                                        // [0] => forum_updated
                                        // [1] => CUSTOM
                                        // [2] => a.`key` = 'forum_updated'
                                        if ($kdx) {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                                        }
                                        else {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.{$v[2]})\n";
                                        }
                                    }
                                    else {
                                        if ($kdx) {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`value` {$v[1]} '{$v[2]}')\n";
                                        }
                                        else {
                                            $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`value` {$v[1]} '{$v[2]}')\n";
                                        }
                                    }
                                    $_jc["{$v[0]}"] = $v;
                                }
                            }
                            else {
                                if ($kdx) {
                                    $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id`)\n";
                                }
                                else {
                                    $req .= "JOIN {$tba} {$als} ON ({$als}.`_item_id` = a.`_item_id` AND {$als}.`key` = '{$v[0]}' AND {$als}.`index` < 2)\n";
                                }
                            }
                            break;
                    }
                }
            }

            // See if this is our group_by column
            if (isset($_gb["{$v[0]}"])) {
                if (!isset($group_by)) {
                    $group_by = "GROUP BY {$als}.`value`";
                }
                else {
                    $group_by .= ",{$als}.`value`";
                }
            }
            elseif ($join_p && $v[0] == $join_p) {
                $group_by = "GROUP BY {$als}.`_profile_id`";
                $join_p   = false;
            }
        }

        // Privacy Check - non admin users
        // 0 = Private
        // 1 = Global
        // 2 = Shared
        // 3 = Shared but Visible in Search
        $add = '';
        $aeq = false; // if $aeq is TRUE, we "add our equals" SQL to the query
        if (!jrUser_is_admin() && (!isset($_params['privacy_check']) || jrCore_checktype($_params['privacy_check'], 'is_true'))) {

            // Get profiles that are NOT public and are allowed to change their profile privacy
            $_pp = jrCore_db_get_private_profiles();

            // Do we have any non-public profiles?
            if (is_array($_pp)) {

                // We have SOME private profiles
                // $_pp is in the format profile_id => profile_private
                $npp = count($_pp);
                if ($npp > 0) {

                    if (!jrUser_is_logged_in() || !isset($_user['_user_id'])) {

                        // If we are searching for a specific _profile_id we can check those here
                        // and see if any of them are PRIVATE profiles - if so we exclude those
                        if (isset($_eq['_profile_id']) && is_array($_eq['_profile_id']) && count($_eq['_profile_id']) > 0) {

                            // If we received a NOT EQUALS _profile_id, remove those here
                            if (isset($_ne['_profile_id']) && is_array($_ne['_profile_id'])) {
                                $_eq['_profile_id'] = jrCore_create_combined_equal_array($_eq['_profile_id'], $_ne['_profile_id']);
                                unset($_ne['_profile_id']);
                            }
                            if (count($_eq['_profile_id']) === 0) {
                                // We have no _profile_id's left over that this user can view - exit
                                jrCore_db_search_fdebug_error("privacy check excluded _profile_ids resulted in no profile_ids left in _eq array", $_backup, $_params);
                                return false;
                            }

                            foreach ($_eq['_profile_id'] as $k => $epid) {
                                if (isset($_pp[$epid])) {
                                    // This profile is a PRIVATE profile - exclude
                                    unset($_eq['_profile_id'][$k]);
                                }
                            }
                            if (count($_eq['_profile_id']) === 0) {
                                // We have no _profile_id's left over that this user can view - exit
                                jrCore_db_search_fdebug_error("privacy check excluded all _profile_id matches", $_backup, $_params);
                                return false;
                            }

                            if (isset($_di[0])) {
                                if (jrCore_db_key_has_index_table($module, '_profile_id')) {
                                    $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
                                    $add = "AND a.`_item_id` IN(SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', $_eq['_profile_id']) . "))\n";
                                }
                                else {
                                    $tbl = jrCore_db_table_name($module, 'item_key');
                                    $add = "AND a.`_item_id` IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '_created' AND `_profile_id` IN(" . implode(',', $_eq['_profile_id']) . "))\n";
                                }
                            }
                            else {
                                $add = "AND a.`_profile_id` IN(" . implode(',', $_eq['_profile_id']) . ")\n";
                            }
                            unset($_eq['_profile_id']);
                        }
                        else {

                            // We did not get a profile_id EQUALS - if we got a NOT EQUALS
                            // let's add those into $_pp so we can exclude the extra query
                            if (isset($_ne['_profile_id']) && is_array($_ne['_profile_id'])) {
                                foreach ($_ne['_profile_id'] as $npid) {
                                    $_pp[$npid] = $npid;
                                }
                                unset($_ne['_profile_id']);
                            }

                            // Users that are not logged in only see global profiles
                            if (isset($_di[0])) {
                                if (jrCore_db_key_has_index_table($module, '_profile_id')) {
                                    $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
                                    $add = "AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', array_keys($_pp)) . "))\n";
                                }
                                else {
                                    $tbl = jrCore_db_table_name($module, 'item_key');
                                    $add = "AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '_created' AND `_profile_id` IN(" . implode(',', array_keys($_pp)) . "))\n";
                                }
                            }
                            else {
                                $add .= "AND a.`_profile_id` NOT IN(" . implode(',', array_keys($_pp)) . ")\n";
                            }
                        }

                    }
                    else {

                        // Users that are logged in see:
                        // global profiles
                        // their own profiles
                        // any profiles they follow
                        // (jrProfile list only) any profiles with profile_private set to "3"
                        $_pr = array();
                        $hid = (int) jrUser_get_profile_home_key('_profile_id');
                        if ($hid > 0) {
                            $_pr[] = $hid;
                        }
                        if (isset($_user['user_active_profile_id']) && jrCore_checktype($_user['user_active_profile_id'], 'number_nz') && $_user['user_active_profile_id'] != $hid) {
                            $_pr[] = (int) $_user['user_active_profile_id'];
                        }
                        if (jrCore_module_is_active('jrFollower')) {
                            // If we are logged in, we can see GLOBAL profiles as well as profiles we are followers of
                            if ($_ff = jrFollower_get_profiles_followed($_user['_user_id'])) {
                                $_pr = array_merge($_ff, $_pr);
                                unset($_ff);
                            }
                        }
                        // Power/Multi users can always see the profiles they manage
                        // $_tm will be an array of profile_id => user_id entries
                        $_tm = jrProfile_get_user_linked_profiles($_user['_user_id']);
                        if ($_tm && is_array($_tm)) {
                            foreach ($_tm as $lpid => $luid) {
                                if (!$_ne || !isset($_ne[$lpid])) {
                                    $_pr[] = $lpid;
                                }
                            }
                            unset($_tm);
                        }
                        if (count($_pr) > 0) {

                            if ($module == 'jrProfile') {
                                // This is a jrProfile list - any profile's with profile_private set to 3 must be removed
                                foreach ($_pp as $pid => $ppi) {
                                    if ($ppi == 3) {
                                        unset($_pp[$pid]);
                                    }
                                }
                            }

                            // Check profiles we have access to
                            foreach ($_pr as $pid) {
                                if (isset($_pp[$pid])) {
                                    // We have access to this profile - remove from private list
                                    unset($_pp[$pid]);
                                }
                            }
                            if (count($_pp) > 0) {

                                // We still have profiles in the private list that we cannot see - if these
                                // profile_id's are in our EQUAL array, remove them
                                if (isset($_eq['_profile_id'])) {
                                    foreach ($_eq['_profile_id'] as $k => $pid) {
                                        if (isset($_pp[$pid])) {
                                            unset($_eq['_profile_id'][$k]);
                                        }
                                    }
                                    // If we received a NOT EQUALS _profile_id, remove those here
                                    if (isset($_ne['_profile_id']) && is_array($_ne['_profile_id'])) {
                                        $_eq['_profile_id'] = jrCore_create_combined_equal_array($_eq['_profile_id'], $_ne['_profile_id']);
                                        unset($_ne['_profile_id']);
                                    }
                                    if (count($_eq['_profile_id']) === 0) {
                                        // We have no _profile_id's left over that this user can view - exit
                                        jrCore_db_search_fdebug_error("privacy check excluded all _profile_id matches (2)", $_backup, $_params);
                                        return false;
                                    }

                                    if (isset($_di[0])) {
                                        if (jrCore_db_key_has_index_table($module, '_profile_id')) {
                                            $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
                                            $add = "AND a.`_item_id` IN(SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', $_eq['_profile_id']) . "))\n";
                                        }
                                        else {
                                            $tbl = jrCore_db_table_name($module, 'item_key');
                                            $add = "AND a.`_item_id` IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '_created' AND `_profile_id` IN(" . implode(',', $_eq['_profile_id']) . "))\n";
                                        }
                                    }
                                    else {
                                        $add = "AND a.`_profile_id` IN(" . implode(',', $_eq['_profile_id']) . ")\n";
                                    }
                                    unset($_eq['_profile_id']);
                                }
                                else {
                                    // Add any NOT EQUALS profiles_id's into our privacy check
                                    if (isset($_ne['_profile_id']) && is_array($_ne['_profile_id'])) {
                                        foreach ($_ne['_profile_id'] as $npid) {
                                            $_pp[$npid] = 1;
                                        }
                                        unset($_ne['_profile_id']);
                                    }
                                    // Make sure we exclude those in our privacy list
                                    if (isset($_di[0])) {
                                        if (jrCore_db_key_has_index_table($module, '_profile_id')) {
                                            $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
                                            $add = "AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', array_keys($_pp)) . "))\n";
                                        }
                                        else {
                                            $tbl = jrCore_db_table_name($module, 'item_key');
                                            $add = "AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '_created' AND `_profile_id` IN(" . implode(',', array_keys($_pp)) . "))\n";
                                        }
                                    }
                                    else {
                                        $add .= " AND a.`_profile_id` NOT IN(" . implode(',', array_keys($_pp)) . ")\n";
                                    }
                                }
                            }
                            else {
                                // User has access to all profile id's in $_pp
                                $aeq = true;
                            }
                        }
                        else {
                            // User belongs to no profiles - should not get here
                            $aeq = true;
                        }
                    }
                }
                else {
                    // There are no private profiles on the system (should not get here)
                    $aeq = true;
                }
            }
            else {
                // There are no private profiles on the system
                $aeq = true;
            }
        }
        else {
            // Admin user - bypass privacy checking
            $aeq = true;
        }

        if ($aeq && isset($_eq['_profile_id']) && is_array($_eq['_profile_id']) && count($_eq['_profile_id']) > 0) {

            // If we have been given BOTH equals and NOT equals for profiles, we want to get rid
            // of the NOT equals - go through the equals and remove any that are NOT equals
            if (isset($_ne['_profile_id']) && is_array($_ne['_profile_id']) && count($_ne['_profile_id']) > 0) {
                $_eq['_profile_id'] = jrCore_create_combined_equal_array($_eq['_profile_id'], $_ne['_profile_id']);
                if (count($_eq['_profile_id']) === 0) {
                    // We've removed all profile id's - short circuit
                    jrCore_db_search_fdebug_error("both include and exclude _profile_id params resulted in no profile_ids", $_backup, $_params);
                    return false;
                }
                unset($_ne['_profile_id']);  // We no longer need to add the NOT EQUALS
            }

            if (isset($_di[0])) {
                if (jrCore_db_key_has_index_table($module, '_profile_id')) {
                    $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
                    $add = "AND a.`_item_id` IN(SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', $_eq['_profile_id']) . "))\n";
                }
                else {
                    $tbl = jrCore_db_table_name($module, 'item_key');
                    $add = "AND a.`_item_id` IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '_created' AND `_profile_id` IN(" . implode(',', $_eq['_profile_id']) . "))\n";
                }
            }
            else {
                $add = "AND a.`_profile_id` IN(" . implode(',', $_eq['_profile_id']) . ")\n";
            }
        }

        // We're excluding specific profile_id's from our search
        if (isset($_ne['_profile_id']) && is_array($_ne['_profile_id']) && count($_ne['_profile_id']) > 0) {
            if (isset($_di[0])) {
                if (jrCore_db_key_has_index_table($module, '_profile_id')) {
                    $tbl = jrCore_db_get_index_table_name($module, '_profile_id');
                    $add = "AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `value` IN(" . implode(',', $_ne['_profile_id']) . "))\n";
                }
                else {
                    $tbl = jrCore_db_table_name($module, 'item_key');
                    $add = "AND a.`_item_id` NOT IN(SELECT `_item_id` FROM {$tbl} WHERE `key` = '_created' AND `_profile_id` IN(" . implode(',', $_ne['_profile_id']) . "))\n";
                }
            }
            else {
                $add = "AND a.`_profile_id` NOT IN(" . implode(',', $_ne['_profile_id']) . ")\n";
            }
        }

        $_sc = array_values($_sc);
        $req .= 'WHERE ';
        foreach ($_sc as $k => $v) {

            if ($k > 0 && isset($_jc["{$v[0]}"])) {
                continue;
            }

            if ($k == 0) {
                if ($v[0] == '_item_id') {
                    if ($v[2] == 'NULL' || (isset($v[3]) && $v[3] == 'no_quotes')) {
                        if ($v[1] == '>' && $v[2] == '0') {
                            $req .= "a.`key` = '{$dob}'\n";
                        }
                        else {
                            $req .= "(a.`key` = '{$dob}' AND a.`_item_id` {$v[1]} {$v[2]})\n";
                        }
                    }
                    else {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "(a.`key` = '{$dob}' AND a.`_item_id` BETWEEN {$v1} AND {$v2})\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "(a.`key` = '{$dob}' AND a.`_item_id` NOT BETWEEN {$v1} AND {$v2})\n";
                        }
                        else {
                            $req .= "(a.`key` = '{$dob}' AND a.`_item_id` {$v[1]} '{$v[2]}')\n";
                        }
                    }
                }
                elseif ($v[1] == 'CUSTOM') {
                    if (isset($_di[$k])) {
                        $req .= "1 = 1\n";
                    }
                    else {
                        if (strpos($v[2], '`key`') === 0) {
                            $req .= "{$_al[$k]}.{$v[2]}\n";
                        }
                        else {
                            $req .= "{$v[2]}\n";
                        }
                    }
                }
                elseif ($v[1] == 'IS OR IS NOT') {
                    $req .= "a.`key` = '{$v[0]}'\n";
                }
                elseif (isset($v[3]) && $v[3] == 'parens' && isset($_so["{$v[0]}"])) {
                    $_bd = array();
                    // ((a.key = 'something' AND value = 1) OR (a.key = 'other' AND value = 2))
                    $req .= '(';
                    foreach ($_so["{$v[0]}"] as $_part) {
                        if ($_part[0] == '_item_id') {
                            if ($_part[1] == 'between') {
                                $req .= "(a.`key` = '_created' AND a.`_item_id` BETWEEN {$_part[2]} AND {$_part[3]})\n";
                            }
                            elseif ($_part[1] == 'not_between') {
                                $req .= "(a.`key` = '_created' AND a.`_item_id` NOT BETWEEN {$_part[2]} AND {$_part[3]})\n";
                            }
                            else {
                                $_bd[] = "(a.`key` = '_created' AND a.`_item_id` {$_part[1]} {$_part[2]})";
                            }
                        }
                        elseif ($_part[1] == 'LIKE') {
                            $_bd[] = "(a.`key` LIKE '{$_part[0]}' AND a.`value` {$_part[1]} {$_part[2]})";
                        }
                        else {
                            if ($_part[1] == 'between') {
                                $_bd[] = "(a.`key` = '{$_part[0]}' AND a.`value` BETWEEN {$_part[2]} AND {$_part[3]})\n";
                            }
                            elseif ($_part[1] == 'not_between') {
                                $_bd[] = "(a.`key` = '{$_part[0]}' AND a.`value` NOT BETWEEN {$_part[2]} AND {$_part[3]})\n";
                            }
                            else {
                                if (isset($_di[$k])) {
                                    $_bd[] = "(a.`value` {$_part[1]} {$_part[2]})";
                                }
                                else {
                                    $_bd[] = "(a.`key` = '{$_part[0]}' AND a.`value` {$_part[1]} {$_part[2]})";
                                }
                            }
                        }
                    }
                    $req .= implode(' OR ', $_bd) . ")\n";
                }
                elseif ($v[0] == "{$pfx}_visible") {
                    $req .= "a.`key` = '{$v[0]}' AND (a.`value` IS NULL OR a.`value` != 'off')\n";
                }
                // wildcard (all keys)
                elseif ($v[0] == '%') {
                    if (isset($v[3]) && $v[3] == 'no_quotes') {
                        $req .= "a.`value` {$v[1]} {$v[2]}\n";
                    }
                    else {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "a.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "a.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "a.`value` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                }
                // wildcard match on key
                elseif (strpos(' ' . $v[0], '%')) {
                    if (isset($v[3]) && $v[3] == 'no_quotes') {
                        $req .= "a.`key` LIKE '{$v[0]}' AND a.`value` {$v[1]} {$v[2]}\n";
                    }
                    else {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "a.`key` LIKE '{$v[0]}' AND a.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "a.`key` LIKE '{$v[0]}' AND a.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "a.`key` LIKE '{$v[0]}' AND a.`value` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                }
                // IN / NOT IN (no quotes) or NULL
                elseif ($v[2] == 'NULL' || (isset($v[3]) && $v[3] == 'no_quotes')) {
                    if (isset($_di[$k])) {
                        $req .= "a.`value` {$v[1]} {$v[2]}\n";
                    }
                    else {
                        $req .= "a.`key` = '{$v[0]}' AND a.`value` {$v[1]} {$v[2]}\n";
                    }
                }
                else {
                    if ($v[1] == 'between') {
                        list($v1, $v2) = explode(',', $v[2]);
                        if (isset($_di[$k])) {
                            $req .= "a.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "a.`key` = '{$v[0]}' AND a.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                    }
                    elseif ($v[1] == 'not_between') {
                        list($v1, $v2) = explode(',', $v[2]);
                        if (isset($_di[$k])) {
                            $req .= "a.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "a.`key` = '{$v[0]}' AND a.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                    }
                    else {
                        if (isset($_di[$k])) {
                            $req .= "a.`value` {$v[1]} '{$v[2]}'\n";
                        }
                        else {
                            $req .= "a.`key` = '{$v[0]}' AND a.`value` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                }
            }

            // keys beyond the first key...
            elseif ($v[1] !== 'CUSTOM') {
                // If we are searching by _item_id we always use "a" for our prefix
                if ($v[0] == '_item_id') {
                    if ($v[2] == 'NULL' || (isset($v[3]) && $v[3] == 'no_quotes')) {
                        $req .= "AND a.`_item_id` {$v[1]} {$v[2]}\n";
                    }
                    else {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND a.`_item_id` BETWEEN {$v1} AND {$v2}\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND a.`_item_id` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "AND a.`_item_id` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                }
                else {
                    $als = $_ob["{$v[0]}"];
                    // Special is or is not condition
                    // (e.`value` IS NOT NULL OR e.`value` IS NULL)
                    // This allows an ORDER_BY on a column that may not be set in all DS entries
                    if ($v[1] == 'IS OR IS NOT') {
                        $req .= "AND ({$als}.`value` > '' OR {$als}.`value` IS NULL)\n";
                    }
                    elseif (isset($v[3]) && $v[3] == 'parens' && isset($_so["{$v[0]}"])) {
                        $_bd = array();
                        // ((a.key = 'something' AND value = 1) OR (a.key = 'other' AND value = 2))
                        $req .= 'AND (';
                        foreach ($_so["{$v[0]}"] as $_part) {
                            if ($_part[0] == '_item_id') {
                                $_bd[] = "(a.`_item_id` {$_part[1]} {$_part[2]})";
                            }
                            elseif ($_part[1] == 'LIKE') {
                                $_bd[] = "({$als}.`key` LIKE '{$_part[0]}' AND {$als}.`value` {$_part[1]} {$_part[2]})";
                            }
                            else {
                                if ($_part[1] == 'between') {
                                    $_bd[] = "({$als}.`key` = '{$_part[0]}' AND {$als}.`value` BETWEEN {$_part[2]} AND {$_part[3]})\n";
                                }
                                elseif ($_part[1] == 'not_between') {
                                    $_bd[] = "({$als}.`key` = '{$_part[0]}' AND {$als}.`value` NOT BETWEEN {$_part[2]} AND {$_part[3]})\n";
                                }
                                else {
                                    $_bd[] = "({$als}.`key` = '{$_part[0]}' AND {$als}.`value` {$_part[1]} {$_part[2]})";
                                }
                            }
                        }
                        $req .= implode(' OR ', $_bd) . ")\n";
                    }
                    // wildcard (all keys)
                    elseif ($v[0] == '%') {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND {$als}.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "AND {$als}.`value` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                    // wildcard match on key
                    elseif (strpos(' ' . $v[0], '%')) {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND {$als}.`key` LIKE '{$v[0]}' AND {$als}.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND {$als}.`key` LIKE '{$v[0]}' AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "AND {$als}.`key` LIKE '{$v[0]}' AND {$als}.`value` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                    elseif ($v[2] == 'NULL' || (isset($v[3]) && $v[3] == 'no_quotes')) {
                        if (strpos($v[2], '(SELECT ') !== 0) {
                            $req .= "AND {$als}.`value` {$v[1]} {$v[2]}\n";
                        }
                    }
                    else {
                        if ($v[1] == 'between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND {$als}.`value` BETWEEN {$v1} AND {$v2}\n";
                        }
                        elseif ($v[1] == 'not_between') {
                            list($v1, $v2) = explode(',', $v[2]);
                            $req .= "AND {$als}.`value` NOT BETWEEN {$v1} AND {$v2}\n";
                        }
                        else {
                            $req .= "AND {$als}.`value` {$v[1]} '{$v[2]}'\n";
                        }
                    }
                }
            }
        }

        // Bring in privacy additions if set...
        if (!empty($add)) {
            $req .= $add;
        }
        // Bring in pending items exclusion
        if (!empty($apn)) {
            $req .= $apn;
        }
        // Bring in profile_id search
        if (!empty($prf)) {
            $req .= $prf;
        }

        // For our counting query
        $re2 = $req;

        // Group by
        if (isset($group_by) && strlen($group_by) > 0) {
            $req .= $group_by . ' ';
            $re2 .= $group_by . ' ';
        }

        elseif (!strpos($req, 'RAND()')) {
            // Default - group by item_id
            if (!$sgb && $ino == '_item_id') {
                if (isset($_params['pagebreak'])) {
                    $req .= "GROUP BY a._item_id ";
                }
                $re2 .= "GROUP BY a._item_id ";
            }
        }

        // Some items are not needed in our counting query
        if (!isset($_params['return_count']) || $_params['return_count'] === false) {

            // Count of IN items
            $do_count = 0;
            if (isset($_do) && is_array($_do)) {
                $do_count = count($_do);
            }

            // Order by
            if (isset($_params['order_by']) && is_array($_params['order_by']) && count($_params['order_by']) > 0) {
                $_ov = array();
                $oby = "ORDER BY ";

                foreach ($_params['order_by'] as $k => $v) {
                    $v = strtoupper($v);
                    switch ($v) {

                        case 'RAND':
                        case 'RANDOM':
                            if (isset($_params['limit']) && intval($_params['limit']) === 1) {
                                jrCore_db_query("SET @rnd := RAND()");
                                $req .= "AND a.`_item_id` >= FLOOR(1 + @rnd * (SELECT MAX(_item_id) FROM " . jrCore_db_table_name($module, 'item') . ")) ";
                                $oby = false;
                            }
                            else {
                                $_ov[] = 'RAND()';
                            }
                            // With random ordering we ignore all other orders...
                            continue 2;

                        case 'ASC':
                        case 'DESC':
                            if (!isset($_ob[$k]) && $k != '_item_id') {
                                jrCore_db_search_fdebug_error("you must include the {$k} field in your search criteria to order_by it", $_backup, $_params);
                                jrCore_logger('MAJ', 'core: invalid order_by criteria in jrCore_db_search_items parameters', array("error: you must include the {$k} field in your search criteria to order_by it", $module, $_params, $_backup));
                                return false;
                            }
                            // If we are ordering by _item_id, we do not order by value...
                            if (count($custom_order_by) > 0 && $k != '_item_id' && $k != '_created') {
                                // Check for index tables
                                $fld = true;
                                if (count($_di) > 0) {
                                    foreach ($_di as $itv) {
                                        if ($itv == $k) {
                                            $fld = false;
                                            break;
                                        }
                                    }
                                }
                                if ($fld) {
                                    if ($v == 'ASC') {
                                        $_ov[] = "FIELD(" . $_ob[$k] . ".`key`, '_created', '{$k}') ASC, " . $_ob[$k] . ".`value` {$v}";
                                    }
                                    else {
                                        $_ov[] = "FIELD(" . $_ob[$k] . ".`key`, '{$k}', '_created') ASC, " . $_ob[$k] . ".`value` {$v}";
                                    }
                                }
                                else {
                                    $_ov[] = " " . $_ob[$k] . ".`value` {$v}";
                                }
                            }
                            elseif ($k == '_item_id') {
                                $_ov[] = "a.`_item_id` {$v}";
                            }
                            else {
                                switch ($k) {
                                    case '_user_id':
                                        $_ov[] = "`_item_id` {$v}";
                                        break;
                                    case '_profile_id':
                                        if (count($_sc) === 1 && jrCore_db_key_has_index_table($module, $_sc[0][0])) {
                                            $_ov[] = "`_item_id` {$v}";
                                        }
                                        else {
                                            if ($module == 'jrProfile') {
                                                $_ov[] = "a.`_item_id` {$v}";
                                            }
                                            else {
                                                $_ov[] = "a.`_profile_id` {$v}";
                                            }
                                        }
                                        break;
                                    case '_created':
                                        if ($optimized_order == 'on') {
                                            $_ov[] = "a.`_item_id` {$v}";
                                        }
                                        else {
                                            $_ov[] = $_ob[$k] . ".`value` {$v}";
                                        }
                                        break;
                                    default:
                                        $_ov[] = $_ob[$k] . ".`value` {$v}";
                                        break;
                                }

                            }
                            break;

                        case 'NUMERICAL_ASC':
                            if (!isset($_ob[$k]) && $k != '_item_id') {
                                jrCore_db_search_fdebug_error("you must include the {$k} field in your search criteria to order_by it (2)", $_backup, $_params);
                                jrCore_logger('MAJ', 'invalid order_by criteria in jrCore_db_search_items parameters', array("error: you must include the {$k} field in your search criteria to order_by it", $module, $_params, $_backup));
                                return false;
                            }
                            if (count($custom_order_by) > 0 && $k != '_item_id' && $k != '_created') {
                                // Check for index tables
                                $fld = true;
                                if (count($_di) > 0) {
                                    foreach ($_di as $itv) {
                                        if ($itv == $k) {
                                            $fld = false;
                                            break;
                                        }
                                    }
                                }
                                if ($fld) {
                                    $_ov[] = "FIELD(" . $_ob[$k] . ".`key`, '_created', '{$k}') ASC, (" . $_ob[$k] . ".`value` + 0) ASC";
                                }
                                else {
                                    $_ov[] = "(" . $_ob[$k] . ".`value` + 0) ASC";
                                }
                            }
                            else {

                                switch ($k) {
                                    case '_item_id':
                                    case '_user_id':
                                        $_ov[] = "`_item_id` ASC";
                                        break;
                                    case '_profile_id':
                                        if (count($_sc) === 1 && jrCore_db_key_has_index_table($module, $_sc[0][0])) {
                                            $_ov[] = "`_item_id` ASC";
                                        }
                                        else {
                                            if ($module == 'jrProfile') {
                                                $_ov[] = "a.`_item_id` ASC";
                                            }
                                            else {
                                                $_ov[] = "a.`_profile_id` ASC";
                                            }
                                        }
                                        break;
                                    case '_created':
                                        if ($optimized_order == 'on') {
                                            $_ov[] = "a.`_item_id` ASC";
                                        }
                                        else {
                                            $_ov[] = '(' . $_ob[$k] . ".`value` + 0) ASC";
                                        }
                                        break;
                                    default:
                                        $_ov[] = '(' . $_ob[$k] . ".`value` + 0) ASC";
                                        break;
                                }

                            }
                            break;

                        case 'NUMERICAL_DESC':
                            if (!isset($_ob[$k]) && $k != '_item_id') {
                                jrCore_db_search_fdebug_error("invalid order_by criteria", $_backup, $_params);
                                jrCore_logger('MAJ', 'invalid order_by criteria in jrCore_db_search_items parameters', array("error: you must include the {$k} field in your search criteria to order_by it", $module, $_params, $_backup));
                                return false;
                            }
                            if (count($custom_order_by) > 0 && $k != '_item_id' && $k != '_created') {
                                // Check for index tables
                                $fld = true;
                                if (count($_di) > 0) {
                                    foreach ($_di as $itv) {
                                        if ($itv == $k) {
                                            $fld = false;
                                            break;
                                        }
                                    }
                                }
                                if ($fld) {
                                    $_ov[] = "FIELD(" . $_ob[$k] . ".`key`, '{$k}', '_created') ASC, (" . $_ob[$k] . ".`value` + 0) DESC";
                                }
                                else {
                                    $_ov[] = "(" . $_ob[$k] . ".`value` + 0) DESC";
                                }
                            }
                            else {
                                switch ($k) {
                                    case '_item_id':
                                    case '_user_id':
                                        $_ov[] = "`_item_id` DESC";
                                        break;
                                    case '_profile_id':
                                        if (count($_sc) === 1 && jrCore_db_key_has_index_table($module, $_sc[0][0])) {
                                            $_ov[] = "`_item_id` DESC";
                                        }
                                        else {
                                            if ($module == 'jrProfile') {
                                                $_ov[] = "a.`_item_id` DESC";
                                            }
                                            else {
                                                $_ov[] = "a.`_profile_id` DESC";
                                            }
                                        }
                                        break;
                                    case '_created':
                                        if ($optimized_order == 'on') {
                                            $_ov[] = "a.`_item_id` DESC";
                                        }
                                        else {
                                            $_ov[] = '(' . $_ob[$k] . ".`value` + 0) DESC";
                                        }
                                        break;
                                    default:
                                        $_ov[] = '(' . $_ob[$k] . ".`value` + 0) DESC";
                                        break;
                                }
                            }
                            break;

                        default:

                            jrCore_db_search_fdebug_error("invalid order direction: {$v} received for {$k} - must be one of: ASC, DESC, NUMERICAL_ASC, NUMERICAL_DESC, RANDOM", $_backup, $_params);
                            jrCore_logger('MAJ', 'invalid order_by criteria in jrCore_db_search_items parameters', array("invalid order direction: {$v} received for {$k} - must be one of: ASC, DESC, NUMERICAL_ASC, NUMERICAL_DESC, RANDOM", $module, $_params, $_backup));
                            return false;
                    }
                }
                if (strlen($oby) > 0) {
                    $req .= $oby . implode(', ', $_ov) . ' ';
                }
            }

            elseif ($ino && isset($_do)) {

                // If there are multiple search conditions we can't optimize
                if (count($_params['search']) > 1) {
                    // No need to order if we're only getting 1 result from the DS
                    if (!isset($_params['limit']) || $_params['limit'] > 1) {
                        $field = false;
                        if ($ino == '_item_id') {
                            $field = "a.`_item_id`";
                        }
                        elseif ($ino == '_user_id' && $module == 'jrUser') {
                            $field = "a.`_item_id`";
                        }
                        elseif ($ino == '_profile_id' && $module == 'jrProfile') {
                            $field = "a.`_item_id`";
                        }
                        elseif (isset($_ob[$ino])) {
                            $field = $_ob[$ino] . ".`value`";
                        }
                        if ($field && count($_do) > 0) {
                            $req .= "ORDER BY FIELD({$field}," . implode(',', $_do) . ") ";
                        }
                    }
                }

                // If we get a LIST of items - i.e. IN() - we order by the list order
                elseif ($do_count > 1) {
                    // No need to order if we're only getting 1 result from the DS
                    if (!isset($_params['limit']) || $_params['limit'] > 1) {
                        $field = false;
                        if ($ino == '_item_id') {
                            $field = $ino;
                        }
                        elseif ($module == 'jrUser' && $ino == '_user_id') {
                            $field = '_item_id';
                        }
                        elseif ($module == 'jrProfile' && $ino == '_profile_id') {
                            $field = '_item_id';
                        }
                        elseif ($ino == '_profile_id') {
                            $field = '_profile_id';
                        }
                        elseif (isset($_ob[$ino])) {
                            $field = $ino;
                        }
                        if ($field) {
                            $_after_order_by = array($field, array_values($_in));
                        }
                    }
                }

            }
        }

        // Start our result set.  When doing a search an array with 2 keys is returned:
        // "_items" - contains the actual search results numerically indexed
        // "info" - contains meta information about the result set
        $_rs = array(
            'info' => array()
        );

        //-----------------------------------
        // Limit
        $slow_count = false;
        if (isset($_params['limit']) && !isset($_params['pagebreak']) && !isset($_params['simplepagebreak'])) {
            if (!jrCore_checktype($_params['limit'], 'number_nz')) {
                jrCore_db_search_fdebug_error("error: invalid limit value - must be a number greater than 0", $_backup, $_params);
                return false;
            }
            $req                  .= " LIMIT " . intval($_params['limit']) . ' ';
            $_rs['info']['limit'] = intval($_params['limit']);
            $re2                  = null;
        }

        //-----------------------------------
        // Simple Pagebreak - no COUNT
        elseif ((!isset($_params['return_count']) || $_params['return_count'] === false) && isset($_params['simplepagebreak']) && jrCore_checktype($_params['simplepagebreak'], 'number_nz')) {

            // Check for good page num
            if (!isset($_params['page']) || !jrCore_checktype($_params['page'], 'number_nz')) {
                $_params['page'] = 1;
            }
            if ($tic > 0) {
                $req .= " LIMIT " . ($_params['simplepagebreak'] + 1);
            }
            else {
                $req .= " LIMIT " . intval(($_params['page'] - 1) * $_params['simplepagebreak']) . "," . ($_params['simplepagebreak'] + 1);
            }
            $_rs['info']['next_page']       = intval($_params['page'] + 1);
            $_rs['info']['pagebreak']       = (int) $_params['simplepagebreak'];
            $_rs['info']['simplepagebreak'] = (int) $_params['simplepagebreak'];
            $_rs['info']['page']            = (int) $_params['page'];
            $_rs['info']['this_page']       = (int) $_params['page'];
            $_rs['info']['prev_page']       = ($_params['page'] > 1) ? intval($_params['page'] - 1) : 0;
            $_rs['info']['page_base_url']   = htmlspecialchars(jrCore_strip_url_params(jrCore_get_current_url(), array('p')));
            if (isset($_params['use_total_row_count']) && $_params['use_total_row_count'] > 0) {
                $_rs['info']['total_items'] = (int) $_params['use_total_row_count'];
            }
            $re2 = null;
        }

        //-----------------------------------
        // Pagebreak
        elseif ((!isset($_params['return_count']) || $_params['return_count'] === false) && isset($_params['pagebreak']) && jrCore_checktype($_params['pagebreak'], 'number_nz')) {

            // Check for good page num
            if (!isset($_params['page']) || !jrCore_checktype($_params['page'], 'number_nz')) {
                $_params['page'] = 1;
            }

            // We can be told to use the TOTAL ROW COUNT of the entire table OR passed a number
            if (!empty($_params['use_total_row_count'])) {
                if (is_numeric($_params['use_total_row_count'])) {
                    $_ct = array('tc' => (int) $_params['use_total_row_count']);
                }
                else {
                    $_ct = array('tc' => jrCore_db_get_datastore_item_count($module));
                }
            }
            elseif ($tic > 0) {
                // We have been given the exact items to search for using an IN() clause
                // this means we know the highest count we can have BEFORE we run our
                // SQL query - if the count is LESS than our pagebreak we just use count()
                $_ct = array('tc' => $tic);
            }
            else {
                if (is_array($_so)) {
                    $re2 = str_replace('SELECT DISTINCT(a.`_item_id`) AS _item_id', 'SELECT COUNT(DISTINCT(a.`_item_id`)) AS tc', $re2);
                }
                else {
                    $re2 = str_replace('SELECT a.`_item_id` AS _item_id', 'SELECT COUNT(a.`_item_id`) AS tc', $re2);
                }

                // Do we have a cached result set for this COUNT?
                $_ct             = false;
                $count_cache_key = md5($re2);
                if ($optimized_limit == 'on' && (!isset($_params['no_cache']) || $_params['no_cache'] === false)) {
                    $_ct = jrCore_is_cached('jrCore', $count_cache_key);
                }
                if (!$_ct) {

                    $beg = explode(' ', microtime());
                    $beg = $beg[1] + $beg[0];

                    jrCore_start_timer("search_items_rc_{$module}");
                    if (strpos($req, 'GROUP BY')) {
                        $_ct = array(
                            'tc' => jrCore_db_query($re2, 'NUM_ROWS', false, null, false)
                        );
                    }
                    else {
                        $_ct = jrCore_db_query($re2, 'SINGLE', false, null, false);
                    }
                    jrCore_stop_timer("search_items_rc_{$module}");

                    $end = explode(' ', microtime());
                    $end = $end[1] + $end[0];
                    $end = round(($end - $beg), 3);

                    // Query Trigger
                    $_qp = array(
                        'module'     => $module,
                        'query'      => $re2,
                        'query_time' => $end,
                        'count'      => (isset($_ct['tc'])) ? intval($_ct['tc']) : 0,
                        'cache_key'  => $cky
                    );
                    jrCore_trigger_event('jrCore', 'db_search_count_query', $_params, $_qp);

                    if (isset($_params['slow_query_time']) && $_params['slow_query_time'] > 0 && $end >= $_params['slow_query_time']) {
                        $slow_count = $end;
                    }
                    elseif ($end > 0.24 && isset($_conf['jrDeveloper_slow_queries']) && $_conf['jrDeveloper_slow_queries'] > 0 && $end > $_conf['jrDeveloper_slow_queries']) {
                        $slow_count = $end;
                    }
                    $cqt = $end;
                    if ($optimized_limit == 'on') {
                        jrCore_add_to_cache('jrCore', $count_cache_key, $_ct);
                    }
                }

            }

            if (is_array($_ct) && isset($_ct['tc']) && $_ct['tc'] > 0) {

                // Check if we also have a limit - this is going to limit the total
                // result set to a specific size, but still allow pagination
                if (isset($_params['limit'])) {
                    // We need to see WHERE we are in the requested set
                    $_rs['info']['total_items'] = (jrCore_checktype($_ct['tc'], 'number_nz')) ? intval($_ct['tc']) : 0;
                    if ($_rs['info']['total_items'] > $_params['limit']) {
                        $_rs['info']['total_items'] = $_params['limit'];
                    }
                    // Find out how many we are returning on this query...
                    $pnum = $_params['pagebreak'];
                    if (($_params['page'] * $_params['pagebreak']) > $_params['limit']) {
                        $pnum = ($_params['limit'] % $_params['pagebreak']);
                        // See if the request range is completely outside the last page
                        if ($_params['pagebreak'] < $_params['limit'] && $_params['page'] > ceil($_params['limit'] / $_params['pagebreak'])) {
                            // invalid set - no results - cache empty result set
                            jrCore_add_to_cache($module, $cky, array('no_results' => 1), $sec, $cache_profile_id, $cache_user_id, false);
                            jrCore_db_search_fdebug_error("no items returned from COUNT query", $_backup, $_params);
                            return false;
                        }
                    }
                    if ($tic > 0) {
                        $req .= " LIMIT {$pnum}";
                    }
                    else {
                        $req .= " LIMIT " . intval(($_params['page'] - 1) * $_params['pagebreak']) . ",{$pnum}";
                    }
                }
                else {
                    $_rs['info']['total_items'] = (jrCore_checktype($_ct['tc'], 'number_nz')) ? intval($_ct['tc']) : 0;
                    if ($tic > 0) {
                        $req .= " LIMIT {$_params['pagebreak']}";
                    }
                    else {
                        $req .= " LIMIT " . intval(($_params['page'] - 1) * $_params['pagebreak']) . ",{$_params['pagebreak']}";
                    }
                }
                $_rs['info']['total_pages']   = (int) ceil($_rs['info']['total_items'] / $_params['pagebreak']);
                $_rs['info']['next_page']     = ($_rs['info']['total_pages'] > $_params['page']) ? intval($_params['page'] + 1) : 0;
                $_rs['info']['pagebreak']     = (int) $_params['pagebreak'];
                $_rs['info']['page']          = (int) $_params['page'];
                $_rs['info']['this_page']     = $_params['page'];
                $_rs['info']['prev_page']     = ($_params['page'] > 1) ? intval($_params['page'] - 1) : 0;
                $_rs['info']['page_base_url'] = htmlspecialchars(jrCore_strip_url_params(jrCore_get_current_url(), array('p')));
            }
            else {
                // No items
                jrCore_add_to_cache($module, $cky, array('no_results' => 1), $sec, $cache_profile_id, $cache_user_id, false);
                jrCore_db_search_fdebug_error("no items returned from COUNT query (2)", $_backup, $_params);
                return false;
            }
        }
        else {
            // Default limit of 10
            $req .= " LIMIT 10";
        }

        $beg = explode(' ', microtime());
        $beg = $beg[1] + $beg[0];

        jrCore_start_timer("search_items_{$module}");
        $_rt = jrCore_db_query($req, 'NUMERIC', false, null, false);
        jrCore_stop_timer("search_items_{$module}");

        $end = explode(' ', microtime());
        $end = $end[1] + $end[0];
        $end = round(($end - $beg), 3);

        // Slow Query logging
        $slow_time = 0;
        if (isset($_params['slow_query_time']) && $_params['slow_query_time'] > 0) {
            $slow_time = $_params['slow_query_time'];
        }
        elseif (isset($_conf['jrDeveloper_slow_queries']) && $_conf['jrDeveloper_slow_queries'] > 0) {
            $slow_time = $_conf['jrDeveloper_slow_queries'];
        }
        if ($slow_count || ($slow_time > 0 && $end >= $slow_time)) {
            global $_post;
            $_rq = array(
                'process'    => (jrCore_client_is_detached()) ? 'worker (background)' : 'client',
                '_post'      => $_post,
                '_params'    => $_params,
                'query_time' => $end,
                'query'      => $req
            );
            $tag = '';
            if ($slow_count && isset($_ct)) {
                $_rq['pagination_query_time'] = $slow_count;
                $_rq['pagination_query']      = $re2;
                $tag                          = ' pagination';
            }
            // Show whichever is longer
            $pri = (jrCore_client_is_detached()) ? 'MIN' : 'MAJ';
            $tim = ($slow_count && $slow_count > $end) ? $slow_count : $end;
            jrCore_logger($pri, "slow jrCore_db_search_items{$tag} query: {$tim} seconds", $_rq);
        }

        // Query Trigger
        $_qp = array(
            'module'     => $module,
            'query'      => $req,
            'query_time' => $end,
            'results'    => $_rt,
            'cache_key'  => $cky
        );
        jrCore_trigger_event('jrCore', 'db_search_items_query', $_params, $_qp);

    }
    else {
        $_rt = $_params['result_set'];
        jrCore_db_search_fdebug_error("result_set was provided by event listener - no query was run", $_backup, $_params);
        $end = false;
        $req = false;
        $_rs = false;
        $re2 = null;
    }

    if ($_rt && is_array($_rt)) {

        // See if we are only providing a count - no need for triggers here
        if (isset($_params['return_count']) && jrCore_checktype($_params['return_count'], 'is_true')) {
            $count = count($_rt);
            if (!isset($_params['no_cache']) || jrCore_checktype($_params['no_cache'], 'is_false')) {
                jrCore_add_to_cache($module, $cky, $count, $sec, $cache_profile_id, $cache_user_id, false);
            }
            jrCore_db_search_fdebug($_backup, $_params, $req, $end, $_rs, $re2, $cqt); // OK
            return $count;
        }

        // Item _ids
        $_id = array();
        foreach ($_rt as $v) {
            $iid       = (int) $v['_item_id'];
            $_id[$iid] = $iid;
        }
        $_id = array_values($_id);

        // See if we are returning an array of item_ids only
        if (isset($_params['return_item_id_only']) && jrCore_checktype($_params['return_item_id_only'], 'is_true')) {
            if (!isset($_params['no_cache']) || jrCore_checktype($_params['no_cache'], 'is_false')) {
                jrCore_add_to_cache($module, $cky, $_id, $sec, $cache_profile_id, $cache_user_id, false);
            }
            jrCore_db_search_fdebug($_backup, $_params, $req, $end, $_rs, $re2, $cqt); // OK
            return $_id;
        }

        $_ky = null;
        if (isset($_params['return_keys']) && is_array($_params['return_keys']) && count($_params['return_keys']) > 0) {
            $_params['return_keys'][] = '_item_id';    // Always include _item_id
            $_params['return_keys'][] = '_user_id';    // We must include _user_id or jrUser search items trigger does not know the user to include
            $_params['return_keys'][] = '_profile_id'; // We must include _profile_id or jrProfile search items trigger does not know the profile to include
            $_ky                      = $_params['return_keys'];
        }

        // NOTE: We force $skip_caching here since we are being cached separately
        $_rs['_items'] = jrCore_db_get_multiple_items($module, $_id, $_ky, true, false, $cache_profile_id);
        if ($_rs['_items'] && is_array($_rs['_items'])) {

            if (isset($_after_order_by) && is_array($_after_order_by)) {
                // We have to ORDER THIS ARRAY of DS items - this happens when we
                // get an order by FIELD - we know the order to put the items in
                // ahead of the query, so we can do our ordering post-query - this
                // prevents a TMP table from being created in the DB
                $_ordered = array();
                foreach ($_after_order_by[1] as $order_id) {
                    foreach ($_rs['_items'] as $k => $v) {
                        if (isset($v["{$_after_order_by[0]}"]) && $v["{$_after_order_by[0]}"] == $order_id) {
                            $_ordered[] = $v;
                            unset($_rs['_items'][$k]);
                        }
                    }
                }
                $_rs['_items'] = $_ordered;
                unset($_ordered, $_after_order_by, $order_id);
            }

            // If we are using the SIMPLE pagebreak setup, if we have LESS
            // items than our pagebreak, we have NO next page
            if (isset($_params['simplepagebreak'])) {
                // When we have simplepagebreak, we actually get ONE MORE item than requested - this
                // lets us know if there is a next page or not
                if (count($_rs['_items']) <= $_params['simplepagebreak']) {
                    // There is no next page
                    $_rs['info']['next_page'] = 0;
                }
                else {
                    // We need to remove the last element in the array since it was not asked for
                    array_pop($_rs['_items']);
                }
            }

            // Add in some meta data
            if (!isset($_rs['info']['total_items'])) {
                $_rs['info']['total_items'] = count($_rs['_items']);
            }

            // Trigger search event
            if (!isset($_params['skip_triggers']) || jrCore_checktype($_params['skip_triggers'], 'is_false')) {
                $_params['cache_key'] = $cky;
                $_rs                  = jrCore_trigger_event('jrCore', 'db_search_items', $_rs, $_params);
            }

            // Check for index_key
            if (!empty($_params['index_key'])) {
                $_rs['_items'] = jrCore_db_create_item_id_index_array($_rs['_items'], $_params['index_key']);
            }

            // Check for return keys and save profile_id's
            $_ci = array();
            if ($_ky) {
                $_ky = array_flip($_ky);
                foreach ($_rs['_items'] as $k => $v) {
                    if (!empty($v['_profile_id'])) {
                        $_ci["{$v['_profile_id']}"] = $v['_profile_id'];
                    }
                    foreach ($v as $ky => $kv) {
                        if (!isset($_ky[$ky])) {
                            unset($_rs['_items'][$k][$ky]);
                        }
                    }
                }
            }
            else {
                // Save profile_ids for use in caching
                foreach ($_rs['_items'] as $v) {
                    if (!empty($v['_profile_id'])) {
                        $_ci["{$v['_profile_id']}"] = $v['_profile_id'];
                    }
                }
            }
            $pid = 0;
            if (!empty($_params['cache_profile_id']) && jrCore_checktype($_params['cache_profile_id'], 'number_nz')) {
                // We've been given the profile_id to cache these results under
                $pid = (int) $_params['cache_profile_id'];
            }
            elseif (count($_ci) === 1) {
                $pid = reset($_ci);
            }
            else {
                jrCore_set_flag('datastore_cache_profile_ids', $_ci);
            }
            unset($_ci);

            $_rs['_params']               = $_backup;
            $_rs['_params']['module']     = $module;
            $_rs['_params']['module_url'] = jrCore_get_module_url($module);
            if (!isset($_params['no_cache']) || jrCore_checktype($_params['no_cache'], 'is_false')) {
                jrCore_add_to_cache($module, $cky, $_rs, $sec, $pid, $cache_user_id, false);
            }
            if (!isset($_params['skip_triggers']) || jrCore_checktype($_params['skip_triggers'], 'is_false')) {
                $_rs = jrCore_trigger_event('jrCore', 'db_search_results', $_rs, $_params);
            }

            // fdebug log
            jrCore_db_search_fdebug($_backup, $_params, $req, $end, $_rs, $re2, $cqt); // OK
            return $_rs;
        }
    }

    // Fall through - no results - cache empty result set
    if (!isset($_params['no_cache']) || jrCore_checktype($_params['no_cache'], 'is_false')) {
        jrCore_add_to_cache($module, $cky, array('no_results' => 1), $sec, $cache_profile_id, $cache_user_id, false);
    }
    jrCore_db_search_fdebug($_backup, $_params, $req, $end, $_rs, $re2, $cqt); // OK
    if (!isset($_params['skip_triggers']) || jrCore_checktype($_params['skip_triggers'], 'is_false')) {
        jrCore_trigger_event('jrCore', 'db_search_results', array('no_results' => true), $_params);
    }
    if (isset($_params['return_count']) && jrCore_checktype($_params['return_count'], 'is_true')) {
        return 0;
    }
    return false;
}
