<?php
 /**
 * Jamroom Image Support module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2021 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * view: limit_image_size_save
 * @param array $_post Posted Data
 * @param array $_user Viewing User data
 * @param array $_conf Global Config
 */
function view_jrImage_limit_image_size_save($_post, $_user, $_conf)
{
    global $_mods;
    jrUser_master_only();
    jrCore_form_validate($_post);

    @ini_set('max_execution_time', 82800); // 23 hours max
    @ini_set('memory_limit', '1024M');

    $pfx       = jrCore_db_get_prefix($_post['image_module']);
    $lid       = 0;
    $ctr       = 1;
    $checked   = 0;
    $old_tsize = 0;
    $new_tsize = 0;
    $mod_name  = $_mods["{$_post['image_module']}"]['module_name'];
    $cache_dir = jrCore_get_module_cache_dir('jrImage');

    $total = jrCore_db_get_datastore_item_count($_post['image_module']);
    if ($total > 0) {
        jrCore_form_modal_notice('update', 'Checking ' . jrCore_number_format($total) . ' ' . $mod_name . ' items for images...');

        // Get all module item images that need to be resized
        while (true) {
            $_rt = array(
                "search"         => array(
                    "_item_id > {$lid}"
                ),
                'return_keys'    => array(
                    "_item_id",
                    "{$pfx}_title",
                    "{$pfx}_image_size",
                    "{$pfx}_image_width",
                    "{$pfx}_image_height",
                    "{$pfx}_image_extension",
                    "{$pfx}_file_name",
                    "{$pfx}_file_size",
                    "{$pfx}_file_width",
                    "{$pfx}_file_height",
                    "{$pfx}_file_extension",
                    "{$pfx}_file_item_price"
                ),
                'skip_triggers'  => true,
                'ignore_pending' => true,
                'limit'          => 1000
            );
            $_rt = jrCore_db_search_items($_post['image_module'], $_rt);
            if ($_rt && isset($_rt['_items']) && is_array($_rt['_items']) && count($_rt['_items']) > 0) {

                $checked += count($_rt['_items']);
                jrCore_form_modal_notice('update', 'Found ' . jrCore_number_format($checked) . ' ' . $mod_name . ' items - scanning for images to resize');
                if ($_post['dry_run'] != 'on') {
                    jrCore_logger('INF', 'image: resize item images tool - scanning ' . jrCore_number_format($checked) . ' ' . $mod_name . ' images');
                }
                foreach ($_rt['_items'] as $rt) {

                    if (isset($rt["{$pfx}_title"])) {
                        $name = ' &quot;' . $rt["{$pfx}_title"] . '&quot;';
                    }
                    elseif (!empty($rt["{$pfx}_file_name"])) {
                        $name = ' &quot;' . $rt["{$pfx}_file_name"] . '&quot;';
                    }
                    else {
                        $name = '';
                    }
                    $lid = $rt['_item_id'];

                    // If this item is FOR SALE, skip it
                    if (isset($rt["{$pfx}_file_item_price"]) && $rt["{$pfx}_file_item_price"] > 0) {
                        jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}{$name} has an item price - skipping");
                        continue;
                    }

                    $ext = false;
                    $fld = false;
                    if (isset($rt["{$pfx}_image_size"]) && $rt["{$pfx}_image_size"] > 0) {
                        // Image File
                        $ext = (isset($rt["{$pfx}_image_extension"])) ? $rt["{$pfx}_image_extension"] : false;
                        $fld = "{$pfx}_image";
                    }
                    elseif (isset($rt["{$pfx}_file_size"]) && $rt["{$pfx}_file_size"] > 0) {
                        // Attachment
                        $ext = (isset($rt["{$pfx}_file_extension"])) ? $rt["{$pfx}_file_extension"] : false;
                        $fld = "{$pfx}_file";
                    }
                    if (!$ext) {
                        // Item does not have an image
                        jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}{$name} - does not have an attached image");
                        continue;
                    }

                    // See if we have an image file
                    switch ($ext) {
                        case 'png':
                        case 'jpg':
                        case 'jpe':
                        case 'jpeg':
                        case 'jif':
                        case 'jfi':
                        case 'jfif':

                            // Calculate new width and length
                            if ($rt["{$fld}_width"] >= $rt["{$fld}_height"]) {
                                $w = $_post['max_item_size'];
                                // Make sure this image is LARGER than what we have requested
                                if ($w >= $rt["{$fld}_width"]) {
                                    // This image is already smaller
                                    continue 2;
                                }
                                $h = floor($_post['max_item_size'] * $rt["{$fld}_height"] / $rt["{$fld}_width"]);
                            }
                            else {
                                $h = $_post['max_item_size'];
                                // Make sure this image is LARGER than what we have requested
                                if ($h >= $rt["{$fld}_height"]) {
                                    // This image is already smaller
                                    continue 2;
                                }
                                $w = floor($_post['max_item_size'] * $rt["{$fld}_width"] / $rt["{$fld}_height"]);
                            }

                            // Source file
                            if ($file = jrCore_confirm_media_file_is_local($rt['_profile_id'], "{$_post['image_module']}_{$rt['_item_id']}_{$fld}.{$rt["{$fld}_extension"]}")) {

                                $target = "{$cache_dir}/resized_" . jrCore_create_unique_string(10) . ".{$rt["{$fld}_extension"]}";
                                $result = jrImage_resize_image($file, $target, $w);
                                if ($result && strpos($result, 'ERROR') === 0) {
                                    jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}: {$result}");
                                    continue 2;
                                }
                                $new_size = filesize($target);
                                $old_size = filesize($file);
                                if ($new_size < $old_size) {

                                    // @note: $target is now the RESIZED file
                                    $ctr++;
                                    $saved = ($old_size - $new_size);
                                    if ($_post['dry_run'] == 'on') {
                                        // dry run - just report size savings
                                        jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}{$name} resized would save <b>" . jrCore_format_size($saved) . "</b> disk space");
                                    }
                                    else {
                                        // For real
                                        jrCore_write_media_file($rt['_profile_id'], "{$_post['image_module']}_{$rt['_item_id']}_{$fld}.{$rt["{$fld}_extension"]}", $target);
                                        $_tmp = array(
                                            "{$fld}_size"   => $new_size,
                                            "{$fld}_width"  => $w,
                                            "{$fld}_height" => $h,
                                        );
                                        jrCore_db_update_item($_post['image_module'], $rt['_item_id'], $_tmp, null, false);
                                        jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}{$name} was successfully resized saving <b>" . jrCore_format_size($saved) . "</b> disk space");
                                    }
                                    $new_tsize += $new_size;
                                    $old_tsize += $old_size;
                                }
                                else {
                                    jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}{$name} - is already smaller than the requested size");
                                }
                                if (is_file($target)) {
                                    unlink($target);
                                }
                            }
                            else {
                                jrCore_form_modal_notice('update', "{$ctr}: ID {$lid}{$name} - unable to locate media file");
                            }
                            break;
                    }
                }
                if (count($_rt['_items']) < 1000) {
                    // We're at the end
                    break;
                }
            }
            else {
                break;
            }
        }

        // All done - Show results
        if ($new_tsize > 0) {
            $ctr--;
            jrCore_form_modal_notice('update', "Total size of {$ctr} matched {$mod_name} image files BEFORE resizing: " . jrCore_format_size($old_tsize));
            jrCore_form_modal_notice('update', "Total size of {$ctr} matched {$mod_name} image files AFTER resizing: " . jrCore_format_size($new_tsize));
            $saved = $old_tsize - $new_tsize;
            if ($_post['dry_run'] == 'on') {
                jrCore_form_modal_notice('complete', "Test Run: A total of " . jrCore_format_size($saved) . " of disk space could be saved");
            }
            else {
                jrCore_form_modal_notice('complete', "Success: A total of " . jrCore_format_size($saved) . " of disk space has been saved");
                jrCore_logger('INF', 'image: resize item images tool completed - freed ' . jrCore_format_size($saved) . " of disk space");
            }
        }
        else {
            jrCore_form_modal_notice('complete', "There were no image files found to resize for {$mod_name}");
        }
    }
    else {
        jrCore_form_modal_notice('complete', "There were no image files found to resize for {$mod_name}");
    }
    jrCore_form_result();
}
