<?php
/**
 * Adds options to the customizer for Kkart.
 *
 * @version 3.3.0
 * @package Kkart
 */

defined( 'ABSPATH' ) || exit;

/**
 * KKART_Shop_Customizer class.
 */
class KKART_Shop_Customizer {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'customize_register', array( $this, 'add_sections' ) );
		add_action( 'customize_controls_print_styles', array( $this, 'add_styles' ) );
		add_action( 'customize_controls_print_scripts', array( $this, 'add_scripts' ), 30 );
		add_action( 'wp_enqueue_scripts', array( $this, 'add_frontend_scripts' ) );
	}

	/**
	 * Add settings to the customizer.
	 *
	 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
	 */
	public function add_sections( $wp_customize ) {
		$wp_customize->add_panel(
			'kkart',
			array(
				'priority'       => 200,
				'capability'     => 'manage_kkart',
				'theme_supports' => '',
				'title'          => __( 'Kkart', 'kkart' ),
			)
		);

		$this->add_store_notice_section( $wp_customize );
		$this->add_product_catalog_section( $wp_customize );
		$this->add_product_images_section( $wp_customize );
		$this->add_checkout_section( $wp_customize );
	}

	/**
	 * Frontend CSS styles.
	 */
	public function add_frontend_scripts() {
		if ( ! is_customize_preview() || ! is_store_notice_showing() ) {
			return;
		}

		$css = '.kkart-store-notice, p.demo_store { display: block !important; }';
		wp_add_inline_style( 'customize-preview', $css );
	}

	/**
	 * CSS styles to improve our form.
	 */
	public function add_styles() {
		?>
		<style type="text/css">
			.kkart-cropping-control {
				margin: 0 40px 1em 0;
				padding: 0;
				display:inline-block;
				vertical-align: top;
			}

			.kkart-cropping-control input[type=radio] {
				margin-top: 1px;
			}

			.kkart-cropping-control span.kkart-cropping-control-aspect-ratio {
				margin-top: .5em;
				display:block;
			}

			.kkart-cropping-control span.kkart-cropping-control-aspect-ratio input {
				width: auto;
				display: inline-block;
			}
		</style>
		<?php
	}

	/**
	 * Scripts to improve our form.
	 */
	public function add_scripts() {
		$min_rows    = kkart_get_theme_support( 'product_grid::min_rows', 1 );
		$max_rows    = kkart_get_theme_support( 'product_grid::max_rows', '' );
		$min_columns = kkart_get_theme_support( 'product_grid::min_columns', 1 );
		$max_columns = kkart_get_theme_support( 'product_grid::max_columns', '' );

		/* translators: %d: Setting value */
		$min_notice = __( 'The minimum allowed setting is %d', 'kkart' );
		/* translators: %d: Setting value */
		$max_notice = __( 'The maximum allowed setting is %d', 'kkart' );
		?>
		<script type="text/javascript">
			jQuery( function( $ ) {
				$( document.body ).on( 'change', '.kkart-cropping-control input[type="radio"]', function() {
					var $wrapper = $( this ).closest( '.kkart-cropping-control' ),
						value    = $wrapper.find( 'input:checked' ).val();

					if ( 'custom' === value ) {
						$wrapper.find( '.kkart-cropping-control-aspect-ratio' ).slideDown( 200 );
					} else {
						$wrapper.find( '.kkart-cropping-control-aspect-ratio' ).hide();
					}

					return false;
				} );

				wp.customize.bind( 'ready', function() { // Ready?
					$( '.kkart-cropping-control' ).find( 'input:checked' ).change();
				} );

				wp.customize( 'kkart_demo_store', function( setting ) {
					setting.bind( function( value ) {
						var notice = wp.customize( 'kkart_demo_store_notice' );

						if ( value && ! notice.callbacks.has( notice.preview ) ) {
							notice.bind( notice.preview );
						} else if ( ! value ) {
							notice.unbind( notice.preview );
						}
					} );
				} );

				wp.customize( 'kkart_demo_store_notice', function( setting ) {
					setting.bind( function( value ) {
						var checkbox = wp.customize( 'kkart_demo_store' );

						if ( checkbox.get() ) {
							$( '.kkart-store-notice' ).text( value );
						}
					} );
				} );

				wp.customize.section( 'kkart_store_notice', function( section ) {
					section.expanded.bind( function( isExpanded ) {
						if ( isExpanded ) {
							var notice   = wp.customize( 'kkart_demo_store_notice' ),
								checkbox = wp.customize( 'kkart_demo_store' );

							if ( checkbox.get() && ! notice.callbacks.has( notice.preview ) ) {
								notice.bind( notice.preview );
							} else if ( ! checkbox.get() ) {
								notice.unbind( notice.preview );
							}
						}
					} );
				} );

				wp.customize.section( 'kkart_product_catalog', function( section ) {
					section.expanded.bind( function( isExpanded ) {
						if ( isExpanded ) {
							wp.customize.previewer.previewUrl.set( '<?php echo esc_js( kkart_get_page_permalink( 'shop' ) ); ?>' );
						}
					} );
				} );

				wp.customize.section( 'kkart_product_images', function( section ) {
					section.expanded.bind( function( isExpanded ) {
						if ( isExpanded ) {
							wp.customize.previewer.previewUrl.set( '<?php echo esc_js( kkart_get_page_permalink( 'shop' ) ); ?>' );
						}
					} );
				} );

				wp.customize.section( 'kkart_checkout', function( section ) {
					section.expanded.bind( function( isExpanded ) {
						if ( isExpanded ) {
							wp.customize.previewer.previewUrl.set( '<?php echo esc_js( kkart_get_page_permalink( 'checkout' ) ); ?>' );
						}
					} );
				} );

				wp.customize( 'kkart_catalog_columns', function( setting ) {
					setting.bind( function( value ) {
						var min = parseInt( '<?php echo esc_js( $min_columns ); ?>', 10 );
						var max = parseInt( '<?php echo esc_js( $max_columns ); ?>', 10 );

						value = parseInt( value, 10 );

						if ( max && value > max ) {
							setting.notifications.add( 'max_columns_error', new wp.customize.Notification(
								'max_columns_error',
								{
									type   : 'error',
									message: '<?php echo esc_js( sprintf( $max_notice, $max_columns ) ); ?>'
								}
							) );
						} else {
							setting.notifications.remove( 'max_columns_error' );
						}

						if ( min && value < min ) {
							setting.notifications.add( 'min_columns_error', new wp.customize.Notification(
								'min_columns_error',
								{
									type   : 'error',
									message: '<?php echo esc_js( sprintf( $min_notice, $min_columns ) ); ?>'
								}
							) );
						} else {
							setting.notifications.remove( 'min_columns_error' );
						}
					} );
				} );

				wp.customize( 'kkart_catalog_rows', function( setting ) {
					setting.bind( function( value ) {
						var min = parseInt( '<?php echo esc_js( $min_rows ); ?>', 10 );
						var max = parseInt( '<?php echo esc_js( $max_rows ); ?>', 10 );

						value = parseInt( value, 10 );

						if ( max && value > max ) {
							setting.notifications.add( 'max_rows_error', new wp.customize.Notification(
								'max_rows_error',
								{
									type   : 'error',
									message: '<?php echo esc_js( sprintf( $max_notice, $max_rows ) ); ?>'
								}
							) );
						} else {
							setting.notifications.remove( 'max_rows_error' );
						}

						if ( min && value < min ) {
							setting.notifications.add( 'min_rows_error', new wp.customize.Notification(
								'min_rows_error',
								{
									type   : 'error',
									message: '<?php echo esc_js( sprintf( $min_notice, $min_rows ) ); ?>'
								}
							) );
						} else {
							setting.notifications.remove( 'min_rows_error' );
						}
					} );
				} );
			} );
		</script>
		<?php
	}

	/**
	 * Sanitize the shop page & category display setting.
	 *
	 * @param string $value '', 'subcategories', or 'both'.
	 * @return string
	 */
	public function sanitize_archive_display( $value ) {
		$options = array( '', 'subcategories', 'both' );

		return in_array( $value, $options, true ) ? $value : '';
	}

	/**
	 * Sanitize the catalog orderby setting.
	 *
	 * @param string $value An array key from the below array.
	 * @return string
	 */
	public function sanitize_default_catalog_orderby( $value ) {
		$options = apply_filters(
			'kkart_default_catalog_orderby_options',
			array(
				'menu_order' => __( 'Default sorting (custom ordering + name)', 'kkart' ),
				'popularity' => __( 'Popularity (sales)', 'kkart' ),
				'rating'     => __( 'Average rating', 'kkart' ),
				'date'       => __( 'Sort by most recent', 'kkart' ),
				'price'      => __( 'Sort by price (asc)', 'kkart' ),
				'price-desc' => __( 'Sort by price (desc)', 'kkart' ),
			)
		);

		return array_key_exists( $value, $options ) ? $value : 'menu_order';
	}

	/**
	 * Store notice section.
	 *
	 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
	 */
	private function add_store_notice_section( $wp_customize ) {
		$wp_customize->add_section(
			'kkart_store_notice',
			array(
				'title'    => __( 'Store Notice', 'kkart' ),
				'priority' => 10,
				'panel'    => 'kkart',
			)
		);

		$wp_customize->add_setting(
			'kkart_demo_store',
			array(
				'default'              => 'no',
				'type'                 => 'option',
				'capability'           => 'manage_kkart',
				'sanitize_callback'    => 'kkart_bool_to_string',
				'sanitize_js_callback' => 'kkart_string_to_bool',
			)
		);

		$wp_customize->add_setting(
			'kkart_demo_store_notice',
			array(
				'default'           => __( 'This is a demo store for testing purposes &mdash; no orders shall be fulfilled.', 'kkart' ),
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => 'wp_kses_post',
				'transport'         => 'postMessage',
			)
		);

		$wp_customize->add_control(
			'kkart_demo_store_notice',
			array(
				'label'       => __( 'Store notice', 'kkart' ),
				'description' => __( 'If enabled, this text will be shown site-wide. You can use it to show events or promotions to visitors!', 'kkart' ),
				'section'     => 'kkart_store_notice',
				'settings'    => 'kkart_demo_store_notice',
				'type'        => 'textarea',
			)
		);

		$wp_customize->add_control(
			'kkart_demo_store',
			array(
				'label'    => __( 'Enable store notice', 'kkart' ),
				'section'  => 'kkart_store_notice',
				'settings' => 'kkart_demo_store',
				'type'     => 'checkbox',
			)
		);

		if ( isset( $wp_customize->selective_refresh ) ) {
			$wp_customize->selective_refresh->add_partial(
				'kkart_demo_store_notice',
				array(
					'selector'            => '.kkart-store-notice',
					'container_inclusive' => true,
					'render_callback'     => 'kkart_demo_store',
				)
			);
		}
	}

	/**
	 * Product catalog section.
	 *
	 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
	 */
	public function add_product_catalog_section( $wp_customize ) {
		$wp_customize->add_section(
			'kkart_product_catalog',
			array(
				'title'    => __( 'Product Catalog', 'kkart' ),
				'priority' => 10,
				'panel'    => 'kkart',
			)
		);

		$wp_customize->add_setting(
			'kkart_shop_page_display',
			array(
				'default'           => '',
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => array( $this, 'sanitize_archive_display' ),
			)
		);

		$wp_customize->add_control(
			'kkart_shop_page_display',
			array(
				'label'       => __( 'Shop page display', 'kkart' ),
				'description' => __( 'Choose what to display on the main shop page.', 'kkart' ),
				'section'     => 'kkart_product_catalog',
				'settings'    => 'kkart_shop_page_display',
				'type'        => 'select',
				'choices'     => array(
					''              => __( 'Show products', 'kkart' ),
					'subcategories' => __( 'Show categories', 'kkart' ),
					'both'          => __( 'Show categories &amp; products', 'kkart' ),
				),
			)
		);

		$wp_customize->add_setting(
			'kkart_category_archive_display',
			array(
				'default'           => '',
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => array( $this, 'sanitize_archive_display' ),
			)
		);

		$wp_customize->add_control(
			'kkart_category_archive_display',
			array(
				'label'       => __( 'Category display', 'kkart' ),
				'description' => __( 'Choose what to display on product category pages.', 'kkart' ),
				'section'     => 'kkart_product_catalog',
				'settings'    => 'kkart_category_archive_display',
				'type'        => 'select',
				'choices'     => array(
					''              => __( 'Show products', 'kkart' ),
					'subcategories' => __( 'Show subcategories', 'kkart' ),
					'both'          => __( 'Show subcategories &amp; products', 'kkart' ),
				),
			)
		);

		$wp_customize->add_setting(
			'kkart_default_catalog_orderby',
			array(
				'default'           => 'menu_order',
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => array( $this, 'sanitize_default_catalog_orderby' ),
			)
		);

		$wp_customize->add_control(
			'kkart_default_catalog_orderby',
			array(
				'label'       => __( 'Default product sorting', 'kkart' ),
				'description' => __( 'How should products be sorted in the catalog by default?', 'kkart' ),
				'section'     => 'kkart_product_catalog',
				'settings'    => 'kkart_default_catalog_orderby',
				'type'        => 'select',
				'choices'     => apply_filters(
					'kkart_default_catalog_orderby_options',
					array(
						'menu_order' => __( 'Default sorting (custom ordering + name)', 'kkart' ),
						'popularity' => __( 'Popularity (sales)', 'kkart' ),
						'rating'     => __( 'Average rating', 'kkart' ),
						'date'       => __( 'Sort by most recent', 'kkart' ),
						'price'      => __( 'Sort by price (asc)', 'kkart' ),
						'price-desc' => __( 'Sort by price (desc)', 'kkart' ),
					)
				),
			)
		);

		// The following settings should be hidden if the theme is declaring the values.
		if ( has_filter( 'loop_shop_columns' ) ) {
			return;
		}

		$wp_customize->add_setting(
			'kkart_catalog_columns',
			array(
				'default'              => 4,
				'type'                 => 'option',
				'capability'           => 'manage_kkart',
				'sanitize_callback'    => 'absint',
				'sanitize_js_callback' => 'absint',
			)
		);

		$wp_customize->add_control(
			'kkart_catalog_columns',
			array(
				'label'       => __( 'Products per row', 'kkart' ),
				'description' => __( 'How many products should be shown per row?', 'kkart' ),
				'section'     => 'kkart_product_catalog',
				'settings'    => 'kkart_catalog_columns',
				'type'        => 'number',
				'input_attrs' => array(
					'min'  => kkart_get_theme_support( 'product_grid::min_columns', 1 ),
					'max'  => kkart_get_theme_support( 'product_grid::max_columns', '' ),
					'step' => 1,
				),
			)
		);

		// Only add this setting if something else isn't managing the number of products per page.
		if ( ! has_filter( 'loop_shop_per_page' ) ) {
			$wp_customize->add_setting(
				'kkart_catalog_rows',
				array(
					'default'              => 4,
					'type'                 => 'option',
					'capability'           => 'manage_kkart',
					'sanitize_callback'    => 'absint',
					'sanitize_js_callback' => 'absint',
				)
			);
		}

		$wp_customize->add_control(
			'kkart_catalog_rows',
			array(
				'label'       => __( 'Rows per page', 'kkart' ),
				'description' => __( 'How many rows of products should be shown per page?', 'kkart' ),
				'section'     => 'kkart_product_catalog',
				'settings'    => 'kkart_catalog_rows',
				'type'        => 'number',
				'input_attrs' => array(
					'min'  => kkart_get_theme_support( 'product_grid::min_rows', 1 ),
					'max'  => kkart_get_theme_support( 'product_grid::max_rows', '' ),
					'step' => 1,
				),
			)
		);
	}

	/**
	 * Product images section.
	 *
	 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
	 */
	private function add_product_images_section( $wp_customize ) {
		if ( class_exists( 'Jetpack' ) && Jetpack::is_module_active( 'photon' ) ) {
			$regen_description = ''; // Nothing to report; Jetpack will handle magically.
		} elseif ( apply_filters( 'kkart_background_image_regeneration', true ) && ! is_multisite() ) {
			$regen_description = __( 'After publishing your changes, new image sizes will be generated automatically.', 'kkart' );
		} elseif ( apply_filters( 'kkart_background_image_regeneration', true ) && is_multisite() ) {
			/* translators: 1: tools URL 2: regen thumbs url */
			$regen_description = sprintf( __( 'After publishing your changes, new image sizes may not be shown until you regenerate thumbnails. You can do this from the <a href="%1$s" target="_blank">tools section in Kkart</a> or by using a plugin such as <a href="%2$s" target="_blank">Regenerate Thumbnails</a>.', 'kkart' ), admin_url( 'admin.php?page=kkart-status&tab=tools' ), 'https://en-gb.wordpress.org/plugins/regenerate-thumbnails/' );
		} else {
			/* translators: %s: regen thumbs url */
			$regen_description = sprintf( __( 'After publishing your changes, new image sizes may not be shown until you <a href="%s" target="_blank">Regenerate Thumbnails</a>.', 'kkart' ), 'https://en-gb.wordpress.org/plugins/regenerate-thumbnails/' );
		}

		$wp_customize->add_section(
			'kkart_product_images',
			array(
				'title'       => __( 'Product Images', 'kkart' ),
				'description' => $regen_description,
				'priority'    => 20,
				'panel'       => 'kkart',
			)
		);

		if ( ! kkart_get_theme_support( 'single_image_width' ) ) {
			$wp_customize->add_setting(
				'kkart_single_image_width',
				array(
					'default'              => 600,
					'type'                 => 'option',
					'capability'           => 'manage_kkart',
					'sanitize_callback'    => 'absint',
					'sanitize_js_callback' => 'absint',
				)
			);

			$wp_customize->add_control(
				'kkart_single_image_width',
				array(
					'label'       => __( 'Main image width', 'kkart' ),
					'description' => __( 'Image size used for the main image on single product pages. These images will remain uncropped.', 'kkart' ),
					'section'     => 'kkart_product_images',
					'settings'    => 'kkart_single_image_width',
					'type'        => 'number',
					'input_attrs' => array(
						'min'  => 0,
						'step' => 1,
					),
				)
			);
		}

		if ( ! kkart_get_theme_support( 'thumbnail_image_width' ) ) {
			$wp_customize->add_setting(
				'kkart_thumbnail_image_width',
				array(
					'default'              => 300,
					'type'                 => 'option',
					'capability'           => 'manage_kkart',
					'sanitize_callback'    => 'absint',
					'sanitize_js_callback' => 'absint',
				)
			);

			$wp_customize->add_control(
				'kkart_thumbnail_image_width',
				array(
					'label'       => __( 'Thumbnail width', 'kkart' ),
					'description' => __( 'Image size used for products in the catalog.', 'kkart' ),
					'section'     => 'kkart_product_images',
					'settings'    => 'kkart_thumbnail_image_width',
					'type'        => 'number',
					'input_attrs' => array(
						'min'  => 0,
						'step' => 1,
					),
				)
			);
		}

		include_once KKART_ABSPATH . 'includes/customizer/class-kkart-customizer-control-cropping.php';

		$wp_customize->add_setting(
			'kkart_thumbnail_cropping',
			array(
				'default'           => '1:1',
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => 'kkart_clean',
			)
		);

		$wp_customize->add_setting(
			'kkart_thumbnail_cropping_custom_width',
			array(
				'default'              => '4',
				'type'                 => 'option',
				'capability'           => 'manage_kkart',
				'sanitize_callback'    => 'absint',
				'sanitize_js_callback' => 'absint',
			)
		);

		$wp_customize->add_setting(
			'kkart_thumbnail_cropping_custom_height',
			array(
				'default'              => '3',
				'type'                 => 'option',
				'capability'           => 'manage_kkart',
				'sanitize_callback'    => 'absint',
				'sanitize_js_callback' => 'absint',
			)
		);

		$wp_customize->add_control(
			new KKART_Customizer_Control_Cropping(
				$wp_customize,
				'kkart_thumbnail_cropping',
				array(
					'section'  => 'kkart_product_images',
					'settings' => array(
						'cropping'      => 'kkart_thumbnail_cropping',
						'custom_width'  => 'kkart_thumbnail_cropping_custom_width',
						'custom_height' => 'kkart_thumbnail_cropping_custom_height',
					),
					'label'    => __( 'Thumbnail cropping', 'kkart' ),
					'choices'  => array(
						'1:1'       => array(
							'label'       => __( '1:1', 'kkart' ),
							'description' => __( 'Images will be cropped into a square', 'kkart' ),
						),
						'custom'    => array(
							'label'       => __( 'Custom', 'kkart' ),
							'description' => __( 'Images will be cropped to a custom aspect ratio', 'kkart' ),
						),
						'uncropped' => array(
							'label'       => __( 'Uncropped', 'kkart' ),
							'description' => __( 'Images will display using the aspect ratio in which they were uploaded', 'kkart' ),
						),
					),
				)
			)
		);
	}

	/**
	 * Checkout section.
	 *
	 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
	 */
	public function add_checkout_section( $wp_customize ) {
		$wp_customize->add_section(
			'kkart_checkout',
			array(
				'title'       => __( 'Checkout', 'kkart' ),
				'priority'    => 20,
				'panel'       => 'kkart',
				'description' => __( 'These options let you change the appearance of the Kkart checkout.', 'kkart' ),
			)
		);

		// Checkout field controls.
		$fields = array(
			'company'   => __( 'Company name', 'kkart' ),
			'address_2' => __( 'Address line 2', 'kkart' ),
			'phone'     => __( 'Phone', 'kkart' ),
		);
		foreach ( $fields as $field => $label ) {
			$wp_customize->add_setting(
				'kkart_checkout_' . $field . '_field',
				array(
					'default'           => 'phone' === $field ? 'required' : 'optional',
					'type'              => 'option',
					'capability'        => 'manage_kkart',
					'sanitize_callback' => array( $this, 'sanitize_checkout_field_display' ),
				)
			);
			$wp_customize->add_control(
				'kkart_checkout_' . $field . '_field',
				array(
					/* Translators: %s field name. */
					'label'    => sprintf( __( '%s field', 'kkart' ), $label ),
					'section'  => 'kkart_checkout',
					'settings' => 'kkart_checkout_' . $field . '_field',
					'type'     => 'select',
					'choices'  => array(
						'hidden'   => __( 'Hidden', 'kkart' ),
						'optional' => __( 'Optional', 'kkart' ),
						'required' => __( 'Required', 'kkart' ),
					),
				)
			);
		}

		// Register settings.
		$wp_customize->add_setting(
			'kkart_checkout_highlight_required_fields',
			array(
				'default'              => 'yes',
				'type'                 => 'option',
				'capability'           => 'manage_kkart',
				'sanitize_callback'    => 'kkart_bool_to_string',
				'sanitize_js_callback' => 'kkart_string_to_bool',
			)
		);

		$wp_customize->add_setting(
			'kkart_checkout_terms_and_conditions_checkbox_text',
			array(
				/* translators: %s terms and conditions page name and link */
				'default'           => sprintf( __( 'I have read and agree to the website %s', 'kkart' ), '[terms]' ),
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => 'wp_kses_post',
				'transport'         => 'postMessage',
			)
		);

		$wp_customize->add_setting(
			'kkart_checkout_privacy_policy_text',
			array(
				/* translators: %s privacy policy page name and link */
				'default'           => sprintf( __( 'Your personal data will be used to process your order, support your experience throughout this website, and for other purposes described in our %s.', 'kkart' ), '[privacy_policy]' ),
				'type'              => 'option',
				'capability'        => 'manage_kkart',
				'sanitize_callback' => 'wp_kses_post',
				'transport'         => 'postMessage',
			)
		);

		// Register controls.
		$wp_customize->add_control(
			'kkart_checkout_highlight_required_fields',
			array(
				'label'    => __( 'Highlight required fields with an asterisk', 'kkart' ),
				'section'  => 'kkart_checkout',
				'settings' => 'kkart_checkout_highlight_required_fields',
				'type'     => 'checkbox',
			)
		);

		if ( current_user_can( 'manage_privacy_options' ) ) {
			$choose_pages = array(
				'wp_page_for_privacy_policy' => __( 'Privacy policy', 'kkart' ),
				'kkart_terms_page_id'  => __( 'Terms and conditions', 'kkart' ),
			);
		} else {
			$choose_pages = array(
				'kkart_terms_page_id'  => __( 'Terms and conditions', 'kkart' ),
			);
		}
		$pages        = get_pages(
			array(
				'post_type'   => 'page',
				'post_status' => 'publish,private,draft',
				'child_of'    => 0,
				'parent'      => -1,
				'exclude'     => array(
					kkart_get_page_id( 'cart' ),
					kkart_get_page_id( 'checkout' ),
					kkart_get_page_id( 'myaccount' ),
				),
				'sort_order'  => 'asc',
				'sort_column' => 'post_title',
			)
		);
		$page_choices = array( '' => __( 'No page set', 'kkart' ) ) + array_combine( array_map( 'strval', wp_list_pluck( $pages, 'ID' ) ), wp_list_pluck( $pages, 'post_title' ) );

		foreach ( $choose_pages as $id => $name ) {
			$wp_customize->add_setting(
				$id,
				array(
					'default'    => '',
					'type'       => 'option',
					'capability' => 'manage_kkart',
				)
			);
			$wp_customize->add_control(
				$id,
				array(
					/* Translators: %s: page name. */
					'label'    => sprintf( __( '%s page', 'kkart' ), $name ),
					'section'  => 'kkart_checkout',
					'settings' => $id,
					'type'     => 'select',
					'choices'  => $page_choices,
				)
			);
		}

		$wp_customize->add_control(
			'kkart_checkout_privacy_policy_text',
			array(
				'label'           => __( 'Privacy policy', 'kkart' ),
				'description'     => __( 'Optionally add some text about your store privacy policy to show during checkout.', 'kkart' ),
				'section'         => 'kkart_checkout',
				'settings'        => 'kkart_checkout_privacy_policy_text',
				'active_callback' => 'kkart_privacy_policy_page_id',
				'type'            => 'textarea',
			)
		);

		$wp_customize->add_control(
			'kkart_checkout_terms_and_conditions_checkbox_text',
			array(
				'label'           => __( 'Terms and conditions', 'kkart' ),
				'description'     => __( 'Optionally add some text for the terms checkbox that customers must accept.', 'kkart' ),
				'section'         => 'kkart_checkout',
				'settings'        => 'kkart_checkout_terms_and_conditions_checkbox_text',
				'active_callback' => 'kkart_terms_and_conditions_page_id',
				'type'            => 'text',
			)
		);

		if ( isset( $wp_customize->selective_refresh ) ) {
			$wp_customize->selective_refresh->add_partial(
				'kkart_checkout_privacy_policy_text',
				array(
					'selector'            => '.kkart-privacy-policy-text',
					'container_inclusive' => true,
					'render_callback'     => 'kkart_checkout_privacy_policy_text',
				)
			);
			$wp_customize->selective_refresh->add_partial(
				'kkart_checkout_terms_and_conditions_checkbox_text',
				array(
					'selector'            => '.kkart-terms-and-conditions-checkbox-text',
					'container_inclusive' => false,
					'render_callback'     => 'kkart_terms_and_conditions_checkbox_text',
				)
			);
		}
	}

	/**
	 * Sanitize field display.
	 *
	 * @param string $value '', 'subcategories', or 'both'.
	 * @return string
	 */
	public function sanitize_checkout_field_display( $value ) {
		$options = array( 'hidden', 'optional', 'required' );
		return in_array( $value, $options, true ) ? $value : '';
	}
}

new KKART_Shop_Customizer();
