<?php
/**
 * REST API Shipping Zones Controller base
 *
 * Houses common functionality between Shipping Zones and Locations.
 *
 * @package Kkart\RestApi
 * @since    3.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REST API Shipping Zones base class.
 *
 * @package Kkart\RestApi
 * @extends KKART_REST_Controller
 */
abstract class KKART_REST_Shipping_Zones_Controller_Base extends KKART_REST_Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'kkart/v2';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'shipping/zones';

	/**
	 * Retrieve a Shipping Zone by it's ID.
	 *
	 * @param int $zone_id Shipping Zone ID.
	 * @return KKART_Shipping_Zone|WP_Error
	 */
	protected function get_zone( $zone_id ) {
		$zone = KKART_Shipping_Zones::get_zone_by( 'zone_id', $zone_id );

		if ( false === $zone ) {
			return new WP_Error( 'kkart_rest_shipping_zone_invalid', __( 'Resource does not exist.', 'kkart' ), array( 'status' => 404 ) );
		}

		return $zone;
	}

	/**
	 * Check whether a given request has permission to read Shipping Zones.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function get_items_permissions_check( $request ) {
		if ( ! kkart_shipping_enabled() ) {
			return new WP_Error( 'rest_no_route', __( 'Shipping is disabled.', 'kkart' ), array( 'status' => 404 ) );
		}

		if ( ! kkart_rest_check_manager_permissions( 'settings', 'read' ) ) {
			return new WP_Error( 'kkart_rest_cannot_view', __( 'Sorry, you cannot list resources.', 'kkart' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return true;
	}

	/**
	 * Check if a given request has access to create Shipping Zones.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function create_item_permissions_check( $request ) {
		if ( ! kkart_shipping_enabled() ) {
			return new WP_Error( 'rest_no_route', __( 'Shipping is disabled.', 'kkart' ), array( 'status' => 404 ) );
		}

		if ( ! kkart_rest_check_manager_permissions( 'settings', 'edit' ) ) {
			return new WP_Error( 'kkart_rest_cannot_create', __( 'Sorry, you are not allowed to create resources.', 'kkart' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return true;
	}

	/**
	 * Check whether a given request has permission to edit Shipping Zones.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function update_items_permissions_check( $request ) {
		if ( ! kkart_shipping_enabled() ) {
			return new WP_Error( 'rest_no_route', __( 'Shipping is disabled.', 'kkart' ), array( 'status' => 404 ) );
		}

		if ( ! kkart_rest_check_manager_permissions( 'settings', 'edit' ) ) {
			return new WP_Error( 'kkart_rest_cannot_edit', __( 'Sorry, you are not allowed to edit this resource.', 'kkart' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return true;
	}

	/**
	 * Check whether a given request has permission to delete Shipping Zones.
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function delete_items_permissions_check( $request ) {
		if ( ! kkart_shipping_enabled() ) {
			return new WP_Error( 'rest_no_route', __( 'Shipping is disabled.', 'kkart' ), array( 'status' => 404 ) );
		}

		if ( ! kkart_rest_check_manager_permissions( 'settings', 'delete' ) ) {
			return new WP_Error( 'kkart_rest_cannot_edit', __( 'Sorry, you are not allowed to delete this resource.', 'kkart' ), array( 'status' => rest_authorization_required_code() ) );
		}

		return true;
	}

}
