<?php
/**
 * InstalledExtensions class file.
 */

namespace Automattic\Kkart\Admin\Marketing;

use Automattic\Kkart\Admin\Loader;
use Automattic\Kkart\Admin\PluginsHelper;

/**
 * Installed Marketing Extensions class.
 */
class InstalledExtensions {

	/**
	 * Gets an array of plugin data for the "Installed marketing extensions" card.
	 *
	 * Valid extensions statuses are: installed, activated, configured
	 */
	public static function get_data() {
		$data = [];

		$automatewoo = self::get_automatewoo_extension_data();
		$mailchimp   = self::get_mailchimp_extension_data();
		$facebook    = self::get_facebook_extension_data();
		$google      = self::get_google_extension_data();
		$hubspot     = self::get_hubspot_extension_data();
		$amazon_ebay = self::get_amazon_ebay_extension_data();

		if ( $automatewoo ) {
			$data[] = $automatewoo;
		}

		if ( $mailchimp ) {
			$data[] = $mailchimp;
		}

		if ( $facebook ) {
			$data[] = $facebook;
		}

		if ( $google ) {
			$data[] = $google;
		}

		if ( $hubspot ) {
			$data[] = $hubspot;
		}

		if ( $amazon_ebay ) {
			$data[] = $amazon_ebay;
		}

		return $data;
	}

	/**
	 * Get allowed plugins.
	 *
	 * @return array
	 */
	public static function get_allowed_plugins() {
		return [
			'automatewoo',
			'mailchimp-for-kkart',
			'creative-mail-by-constant-contact',
			'facebook-for-kkart',
			'kliken-marketing-for-google',
			'hubspot-for-kkart',
			'kkart-amazon-ebay-integration',
		];
	}

	/**
	 * Get AutomateWoo extension data.
	 *
	 * @return array|bool
	 */
	protected static function get_automatewoo_extension_data() {
		$slug = 'automatewoo';

		if ( ! PluginsHelper::is_plugin_installed( $slug ) ) {
			return false;
		}

		$data         = self::get_extension_base_data( $slug );
		$data['icon'] = plugins_url( 'images/marketing/automatewoo.svg', KKART_ADMIN_PLUGIN_FILE );

		if ( 'activated' === $data['status'] && function_exists( 'AW' ) ) {
			$data['settingsUrl'] = admin_url( 'admin.php?page=automatewoo-settings' );
			$data['docsUrl']     = 'https://automatewoo.com/docs/';
			$data['status']      = 'configured'; // Currently no configuration step.
		}

		return $data;
	}

	/**
	 * Get MailChimp extension data.
	 *
	 * @return array|bool
	 */
	protected static function get_mailchimp_extension_data() {
		$slug = 'mailchimp-for-kkart';

		if ( ! PluginsHelper::is_plugin_installed( $slug ) ) {
			return false;
		}

		$data         = self::get_extension_base_data( $slug );
		$data['icon'] = plugins_url( 'images/marketing/mailchimp.svg', KKART_ADMIN_PLUGIN_FILE );

		if ( 'activated' === $data['status'] && function_exists( 'mailchimp_is_configured' ) ) {
			$data['docsUrl']     = 'https://mailchimp.com/help/connect-or-disconnect-mailchimp-for-kkart/';
			$data['settingsUrl'] = admin_url( 'admin.php?page=mailchimp-kkart' );

			if ( mailchimp_is_configured() ) {
				$data['status'] = 'configured';
			}
		}

		return $data;
	}

	/**
	 * Get Facebook extension data.
	 *
	 * @return array|bool
	 */
	protected static function get_facebook_extension_data() {
		$slug = 'facebook-for-kkart';

		if ( ! PluginsHelper::is_plugin_installed( $slug ) ) {
			return false;
		}

		$data         = self::get_extension_base_data( $slug );
		$data['icon'] = plugins_url( 'images/marketing/facebook.svg', KKART_ADMIN_PLUGIN_FILE );

		if ( 'activated' === $data['status'] && function_exists( 'facebook_for_kkart' ) ) {
			$integration = facebook_for_kkart()->get_integration();

			if ( $integration->is_configured() ) {
				$data['status'] = 'configured';
			}

			$data['settingsUrl'] = facebook_for_kkart()->get_settings_url();
			$data['docsUrl']     = facebook_for_kkart()->get_documentation_url();
		}

		return $data;
	}

	/**
	 * Get Google extension data.
	 *
	 * @return array|bool
	 */
	protected static function get_google_extension_data() {
		$slug = 'kliken-marketing-for-google';

		if ( ! PluginsHelper::is_plugin_installed( $slug ) ) {
			return false;
		}

		$data         = self::get_extension_base_data( $slug );
		$data['icon'] = plugins_url( 'images/marketing/google.svg', KKART_ADMIN_PLUGIN_FILE );

		if ( 'activated' === $data['status'] && function_exists( 'kk_kkart_plugin' ) && class_exists( '\Kliken\WcPlugin\Helper' ) ) {

			$kliken_settings = \Kliken\WcPlugin\Helper::get_plugin_options();

			// Use same check as the Kliken Get Started Page.
			if ( \Kliken\WcPlugin\Helper::is_valid_account_id( $kliken_settings['account_id'] ) ) {
				$data['status'] = 'configured';
			}

			$data['settingsUrl'] = admin_url( 'admin.php?page=kkart-settings&tab=integration&section=kk_wcintegration' );
			$data['docsUrl']     = 'https://docs.kkart.com/document/google-ads/';
		}

		return $data;
	}

	/**
	 * Get Hubspot extension data.
	 *
	 * @return array|bool
	 */
	protected static function get_hubspot_extension_data() {
		$slug = 'hubspot-for-kkart';

		if ( ! PluginsHelper::is_plugin_installed( $slug ) ) {
			return false;
		}

		$data         = self::get_extension_base_data( $slug );
		$data['icon'] = plugins_url( 'images/marketing/hubspot.svg', KKART_ADMIN_PLUGIN_FILE );

		if ( 'activated' === $data['status'] && class_exists( '\Hubwoo' ) ) {

			// Use same check as HubWoo admin.
			if ( \Hubwoo::is_setup_completed() ) {
				$data['status'] = 'configured';
			}

			$data['settingsUrl'] = admin_url( 'admin.php?page=hubwoo' );
			$data['docsUrl']     = 'https://docs.makewebbetter.com/hubspot-integration-for-kkart/';
		}

		return $data;
	}

	/**
	 * Get Amazon / Ebay extension data.
	 *
	 * @return array|bool
	 */
	protected static function get_amazon_ebay_extension_data() {
		$slug = 'kkart-amazon-ebay-integration';

		if ( ! PluginsHelper::is_plugin_installed( $slug ) ) {
			return false;
		}

		$data         = self::get_extension_base_data( $slug );
		$data['icon'] = plugins_url( 'images/marketing/amazon-ebay.svg', KKART_ADMIN_PLUGIN_FILE );

		if ( 'activated' === $data['status'] && class_exists( '\CodistoConnect' ) ) {

			$codisto_merchantid = get_option( 'codisto_merchantid' );

			// Use same check as codisto admin tabs.
			if ( is_numeric( $codisto_merchantid ) ) {
				$data['status'] = 'configured';
			}

			$data['settingsUrl'] = admin_url( 'admin.php?page=codisto-settings' );
			$data['docsUrl']     = 'https://docs.kkart.com/document/getting-started-with-kkart-amazon-ebay-integration/';
		}

		return $data;
	}


	/**
	 * Get an array of basic data for a given extension.
	 *
	 * @param string $slug Plugin slug.
	 *
	 * @return array|false
	 */
	protected static function get_extension_base_data( $slug ) {
		$status      = PluginsHelper::is_plugin_active( $slug ) ? 'activated' : 'installed';
		$plugin_data = PluginsHelper::get_plugin_data( $slug );

		if ( ! $plugin_data ) {
			return false;
		}

		return [
			'slug'        => $slug,
			'status'      => $status,
			'name'        => $plugin_data['Name'],
			'description' => html_entity_decode( wp_trim_words( $plugin_data['Description'], 20 ) ),
			'supportUrl'  => 'https://kkart.com/my-account/create-a-ticket/',
		];
	}

}
