/**
 * External dependencies
 */
import { __, _n, sprintf } from '@wordpress/i18n';
import { SearchListControl } from '@kkart/components';
import PropTypes from 'prop-types';
import { withSearchedProducts } from '@kkart/block-hocs';
import ErrorMessage from '@kkart/editor-components/error-placeholder/error-message.js';

/**
 * The products control exposes a custom selector for searching and selecting
 * products.
 *
 * @param {Object} props Component props.
 * @param {string} props.error
 * @param {Function} props.onChange  Callback fired when the selected item changes
 * @param {Function} props.onSearch  Callback fired when a search is triggered
 * @param {Array}    props.selected  An array of selected products.
 * @param {Array}    props.products  An array of products to select from.
 * @param {boolean}  props.isLoading Whether or not the products are being loaded.
 *
 * @return {Function} A functional component.
 */
const ProductsControl = ( {
	error,
	onChange,
	onSearch,
	selected,
	products,
	isLoading,
} ) => {
	const messages = {
		clear: __( 'Clear all products', 'kkart' ),
		list: __( 'Products', 'kkart' ),
		noItems: __(
			"Your store doesn't have any products.",
			'kkart'
		),
		search: __(
			'Search for products to display',
			'kkart'
		),
		selected: ( n ) =>
			sprintf(
				// Translators: %d is the number of selected products.
				_n(
					'%d product selected',
					'%d products selected',
					n,
					'kkart'
				),
				n
			),
		updated: __(
			'Product search results updated.',
			'kkart'
		),
	};

	if ( error ) {
		return <ErrorMessage error={ error } />;
	}

	return (
		<SearchListControl
			className="kkart-products"
			list={ products }
			isLoading={ isLoading }
			selected={ products.filter( ( { id } ) =>
				selected.includes( id )
			) }
			onSearch={ onSearch }
			onChange={ onChange }
			messages={ messages }
		/>
	);
};

ProductsControl.propTypes = {
	onChange: PropTypes.func.isRequired,
	onSearch: PropTypes.func,
	selected: PropTypes.array,
	products: PropTypes.array,
	isLoading: PropTypes.bool,
};

ProductsControl.defaultProps = {
	selected: [],
	products: [],
	isLoading: true,
};

export default withSearchedProducts( ProductsControl );
