<?php
/**
 * Kkart General Settings
 *
 * @package Kkart\Admin
 */

defined( 'ABSPATH' ) || exit;

if ( class_exists( 'KKART_Settings_General', false ) ) {
	return new KKART_Settings_General();
}

/**
 * KKART_Admin_Settings_General.
 */
class KKART_Settings_General extends KKART_Settings_Page {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id    = 'general';
		$this->label = __( 'General', 'kkart' );

		parent::__construct();
	}

	/**
	 * Get settings array.
	 *
	 * @return array
	 */
	public function get_settings() {

		$currency_code_options = get_kkart_currencies();

		foreach ( $currency_code_options as $code => $name ) {
			$currency_code_options[ $code ] = $name . ' (' . get_kkart_currency_symbol( $code ) . ')';
		}

		$settings = apply_filters(
			'kkart_general_settings',
			array(

				array(
					'title' => __( 'Store Address', 'kkart' ),
					'type'  => 'title',
					'desc'  => __( 'This is where your business is located. Tax rates and shipping rates will use this address.', 'kkart' ),
					'id'    => 'store_address',
				),

				array(
					'title'    => __( 'Address line 1', 'kkart' ),
					'desc'     => __( 'The street address for your business location.', 'kkart' ),
					'id'       => 'kkart_store_address',
					'default'  => '',
					'type'     => 'text',
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Address line 2', 'kkart' ),
					'desc'     => __( 'An additional, optional address line for your business location.', 'kkart' ),
					'id'       => 'kkart_store_address_2',
					'default'  => '',
					'type'     => 'text',
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'City', 'kkart' ),
					'desc'     => __( 'The city in which your business is located.', 'kkart' ),
					'id'       => 'kkart_store_city',
					'default'  => '',
					'type'     => 'text',
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Country / State', 'kkart' ),
					'desc'     => __( 'The country and state or province, if any, in which your business is located.', 'kkart' ),
					'id'       => 'kkart_default_country',
					'default'  => 'US:AZ',
					'type'     => 'single_select_country',
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Postcode / ZIP', 'kkart' ),
					'desc'     => __( 'The postal code, if any, in which your business is located.', 'kkart' ),
					'id'       => 'kkart_store_postcode',
					'css'      => 'min-width:50px;',
					'default'  => '',
					'type'     => 'text',
					'desc_tip' => true,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'store_address',
				),

				array(
					'title' => __( 'General options', 'kkart' ),
					'type'  => 'title',
					'desc'  => '',
					'id'    => 'general_options',
				),

				array(
					'title'    => __( 'Selling location(s)', 'kkart' ),
					'desc'     => __( 'This option lets you limit which countries you are willing to sell to.', 'kkart' ),
					'id'       => 'kkart_allowed_countries',
					'default'  => 'all',
					'type'     => 'select',
					'class'    => 'kkart-enhanced-select',
					'css'      => 'min-width: 350px;',
					'desc_tip' => true,
					'options'  => array(
						'all'        => __( 'Sell to all countries', 'kkart' ),
						'all_except' => __( 'Sell to all countries, except for&hellip;', 'kkart' ),
						'specific'   => __( 'Sell to specific countries', 'kkart' ),
					),
				),

				array(
					'title'   => __( 'Sell to all countries, except for&hellip;', 'kkart' ),
					'desc'    => '',
					'id'      => 'kkart_all_except_countries',
					'css'     => 'min-width: 350px;',
					'default' => '',
					'type'    => 'multi_select_countries',
				),

				array(
					'title'   => __( 'Sell to specific countries', 'kkart' ),
					'desc'    => '',
					'id'      => 'kkart_specific_allowed_countries',
					'css'     => 'min-width: 350px;',
					'default' => '',
					'type'    => 'multi_select_countries',
				),

				array(
					'title'    => __( 'Shipping location(s)', 'kkart' ),
					'desc'     => __( 'Choose which countries you want to ship to, or choose to ship to all locations you sell to.', 'kkart' ),
					'id'       => 'kkart_ship_to_countries',
					'default'  => '',
					'type'     => 'select',
					'class'    => 'kkart-enhanced-select',
					'desc_tip' => true,
					'options'  => array(
						''         => __( 'Ship to all countries you sell to', 'kkart' ),
						'all'      => __( 'Ship to all countries', 'kkart' ),
						'specific' => __( 'Ship to specific countries only', 'kkart' ),
						'disabled' => __( 'Disable shipping &amp; shipping calculations', 'kkart' ),
					),
				),

				array(
					'title'   => __( 'Ship to specific countries', 'kkart' ),
					'desc'    => '',
					'id'      => 'kkart_specific_ship_to_countries',
					'css'     => '',
					'default' => '',
					'type'    => 'multi_select_countries',
				),

				array(
					'title'    => __( 'Default customer location', 'kkart' ),
					'id'       => 'kkart_default_customer_address',
					'desc_tip' => __( 'This option determines a customers default location. The MaxMind GeoLite Database will be periodically downloaded to your wp-content directory if using geolocation.', 'kkart' ),
					'default'  => 'base',
					'type'     => 'select',
					'class'    => 'kkart-enhanced-select',
					'options'  => array(
						''                 => __( 'No location by default', 'kkart' ),
						'base'             => __( 'Shop base address', 'kkart' ),
						'geolocation'      => __( 'Geolocate', 'kkart' ),
						'geolocation_ajax' => __( 'Geolocate (with page caching support)', 'kkart' ),
					),
				),

				array(
					'title'    => __( 'Enable taxes', 'kkart' ),
					'desc'     => __( 'Enable tax rates and calculations', 'kkart' ),
					'id'       => 'kkart_calc_taxes',
					'default'  => 'no',
					'type'     => 'checkbox',
					'desc_tip' => __( 'Rates will be configurable and taxes will be calculated during checkout.', 'kkart' ),
				),

				array(
					'title'           => __( 'Enable coupons', 'kkart' ),
					'desc'            => __( 'Enable the use of coupon codes', 'kkart' ),
					'id'              => 'kkart_enable_coupons',
					'default'         => 'yes',
					'type'            => 'checkbox',
					'checkboxgroup'   => 'start',
					'show_if_checked' => 'option',
					'desc_tip'        => __( 'Coupons can be applied from the cart and checkout pages.', 'kkart' ),
				),

				array(
					'desc'            => __( 'Calculate coupon discounts sequentially', 'kkart' ),
					'id'              => 'kkart_calc_discounts_sequentially',
					'default'         => 'no',
					'type'            => 'checkbox',
					'desc_tip'        => __( 'When applying multiple coupons, apply the first coupon to the full price and the second coupon to the discounted price and so on.', 'kkart' ),
					'show_if_checked' => 'yes',
					'checkboxgroup'   => 'end',
					'autoload'        => false,
				),

				array(
					'type' => 'sectionend',
					'id'   => 'general_options',
				),

				array(
					'title' => __( 'Currency options', 'kkart' ),
					'type'  => 'title',
					'desc'  => __( 'The following options affect how prices are displayed on the frontend.', 'kkart' ),
					'id'    => 'pricing_options',
				),

				array(
					'title'    => __( 'Currency', 'kkart' ),
					'desc'     => __( 'This controls what currency prices are listed at in the catalog and which currency gateways will take payments in.', 'kkart' ),
					'id'       => 'kkart_currency',
					'default'  => 'USD',
					'type'     => 'select',
					'class'    => 'kkart-enhanced-select',
					'desc_tip' => true,
					'options'  => $currency_code_options,
				),

				array(
					'title'    => __( 'Currency position', 'kkart' ),
					'desc'     => __( 'This controls the position of the currency symbol.', 'kkart' ),
					'id'       => 'kkart_currency_pos',
					'class'    => 'kkart-enhanced-select',
					'default'  => 'left',
					'type'     => 'select',
					'options'  => array(
						'left'        => __( 'Left', 'kkart' ),
						'right'       => __( 'Right', 'kkart' ),
						'left_space'  => __( 'Left with space', 'kkart' ),
						'right_space' => __( 'Right with space', 'kkart' ),
					),
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Thousand separator', 'kkart' ),
					'desc'     => __( 'This sets the thousand separator of displayed prices.', 'kkart' ),
					'id'       => 'kkart_price_thousand_sep',
					'css'      => 'width:50px;',
					'default'  => ',',
					'type'     => 'text',
					'desc_tip' => true,
				),

				array(
					'title'    => __( 'Decimal separator', 'kkart' ),
					'desc'     => __( 'This sets the decimal separator of displayed prices.', 'kkart' ),
					'id'       => 'kkart_price_decimal_sep',
					'css'      => 'width:50px;',
					'default'  => '.',
					'type'     => 'text',
					'desc_tip' => true,
				),

				array(
					'title'             => __( 'Number of decimals', 'kkart' ),
					'desc'              => __( 'This sets the number of decimal points shown in displayed prices.', 'kkart' ),
					'id'                => 'kkart_price_num_decimals',
					'css'               => 'width:50px;',
					'default'           => '2',
					'desc_tip'          => true,
					'type'              => 'number',
					'custom_attributes' => array(
						'min'  => 0,
						'step' => 1,
					),
				),

				array(
					'type' => 'sectionend',
					'id'   => 'pricing_options',
				),

				array(
					'title' => __( 'Store Customer Support Options', 'kkart' ),
					'type'  => 'title',
					'desc'  => __( 'The following email will be displayed on My Account > Support tab.', 'kkart' ),
					'id'    => 'store_support',
				),
				
				array(
					'title'    => __( 'Customer Care Email', 'kkart' ),
					'desc'     => __( '', 'kkart' ),
					'id'       => 'kkart_store_customer_care_email',
					'css'      => 'min-width:50px;',
					'default'  => get_option( 'admin_email' ),
					'type'     => 'email',
					'desc_tip' => true,
				),
				
				array(
					'type' => 'sectionend',
					'id'   => 'store_support',
				),

			)
		);

		return apply_filters( 'kkart_get_settings_' . $this->id, $settings );
	}

	/**
	 * Output a color picker input box.
	 *
	 * @param mixed  $name Name of input.
	 * @param string $id ID of input.
	 * @param mixed  $value Value of input.
	 * @param string $desc (default: '') Description for input.
	 */
	public function color_picker( $name, $id, $value, $desc = '' ) {
		echo '<div class="color_box">' . kkart_help_tip( $desc ) . '
			<input name="' . esc_attr( $id ) . '" id="' . esc_attr( $id ) . '" type="text" value="' . esc_attr( $value ) . '" class="colorpick" /> <div id="colorPickerDiv_' . esc_attr( $id ) . '" class="colorpickdiv"></div>
		</div>';
	}

	/**
	 * Output the settings.
	 */
	public function output() {
		$settings = $this->get_settings();

		KKART_Admin_Settings::output_fields( $settings );
	}

	/**
	 * Save settings.
	 */
	public function save() {
		$settings = $this->get_settings();

		KKART_Admin_Settings::save_fields( $settings );
	}
}

return new KKART_Settings_General();
