<?php
/**
 * Grouped Product
 *
 * Grouped products cannot be purchased - they are wrappers for other products.
 *
 * @package Kkart\Classes\Products
 * @version 3.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Product grouped class.
 */
class KKART_Product_Grouped extends KKART_Product {

	/**
	 * Stores product data.
	 *
	 * @var array
	 */
	protected $extra_data = array(
		'children' => array(),
	);

	/**
	 * Get internal type.
	 *
	 * @return string
	 */
	public function get_type() {
		return 'grouped';
	}

	/**
	 * Get the add to cart button text.
	 *
	 * @return string
	 */
	public function add_to_cart_text() {
		return apply_filters( 'kkart_product_add_to_cart_text', __( 'View products', 'kkart' ), $this );
	}

	/**
	 * Get the add to cart button text description - used in aria tags.
	 *
	 * @since 3.3.0
	 * @return string
	 */
	public function add_to_cart_description() {
		/* translators: %s: Product title */
		return apply_filters( 'kkart_product_add_to_cart_description', sprintf( __( 'View products in the &ldquo;%s&rdquo; group', 'kkart' ), $this->get_name() ), $this );
	}

	/**
	 * Returns whether or not the product is on sale.
	 *
	 * @param  string $context What the value is for. Valid values are view and edit.
	 * @return bool
	 */
	public function is_on_sale( $context = 'view' ) {
		$children = array_filter( array_map( 'kkart_get_product', $this->get_children( $context ) ), 'kkart_products_array_filter_visible_grouped' );
		$on_sale  = false;

		foreach ( $children as $child ) {
			if ( $child->is_purchasable() && ! $child->has_child() && $child->is_on_sale() ) {
				$on_sale = true;
				break;
			}
		}

		return 'view' === $context ? apply_filters( 'kkart_product_is_on_sale', $on_sale, $this ) : $on_sale;
	}

	/**
	 * Returns false if the product cannot be bought.
	 *
	 * @return bool
	 */
	public function is_purchasable() {
		return apply_filters( 'kkart_is_purchasable', false, $this );
	}

	/**
	 * Returns the price in html format.
	 *
	 * @param string $price (default: '').
	 * @return string
	 */
	public function get_price_html( $price = '' ) {
		$tax_display_mode = get_option( 'kkart_tax_display_shop' );
		$child_prices     = array();
		$children         = array_filter( array_map( 'kkart_get_product', $this->get_children() ), 'kkart_products_array_filter_visible_grouped' );

		foreach ( $children as $child ) {
			if ( '' !== $child->get_price() ) {
				$child_prices[] = 'incl' === $tax_display_mode ? kkart_get_price_including_tax( $child ) : kkart_get_price_excluding_tax( $child );
			}
		}

		if ( ! empty( $child_prices ) ) {
			$min_price = min( $child_prices );
			$max_price = max( $child_prices );
		} else {
			$min_price = '';
			$max_price = '';
		}

		if ( '' !== $min_price ) {
			if ( $min_price !== $max_price ) {
				$price = kkart_format_price_range( $min_price, $max_price );
			} else {
				$price = kkart_price( $min_price );
			}

			$is_free = 0 === $min_price && 0 === $max_price;

			if ( $is_free ) {
				$price = apply_filters( 'kkart_grouped_free_price_html', __( 'Free!', 'kkart' ), $this );
			} else {
				$price = apply_filters( 'kkart_grouped_price_html', $price . $this->get_price_suffix(), $this, $child_prices );
			}
		} else {
			$price = apply_filters( 'kkart_grouped_empty_price_html', '', $this );
		}

		return apply_filters( 'kkart_get_price_html', $price, $this );
	}

	/*
	|--------------------------------------------------------------------------
	| Getters
	|--------------------------------------------------------------------------
	|
	| Methods for getting data from the product object.
	*/

	/**
	 * Return the children of this product.
	 *
	 * @param  string $context What the value is for. Valid values are view and edit.
	 * @return array
	 */
	public function get_children( $context = 'view' ) {
		return $this->get_prop( 'children', $context );
	}

	/*
	|--------------------------------------------------------------------------
	| Setters
	|--------------------------------------------------------------------------
	|
	| Methods for getting data from the product object.
	*/

	/**
	 * Return the children of this product.
	 *
	 * @param array $children List of product children.
	 */
	public function set_children( $children ) {
		$this->set_prop( 'children', array_filter( wp_parse_id_list( (array) $children ) ) );
	}

	/*
	|--------------------------------------------------------------------------
	| Sync with children.
	|--------------------------------------------------------------------------
	*/

	/**
	 * Sync a grouped product with it's children. These sync functions sync
	 * upwards (from child to parent) when the variation is saved.
	 *
	 * @param KKART_Product|int $product Product object or ID for which you wish to sync.
	 * @param bool           $save If true, the product object will be saved to the DB before returning it.
	 * @return KKART_Product Synced product object.
	 */
	public static function sync( $product, $save = true ) {
		if ( ! is_a( $product, 'KKART_Product' ) ) {
			$product = kkart_get_product( $product );
		}
		if ( is_a( $product, 'KKART_Product_Grouped' ) ) {
			$data_store = KKART_Data_Store::load( 'product-' . $product->get_type() );
			$data_store->sync_price( $product );
			if ( $save ) {
				$product->save();
			}
		}
		return $product;
	}
}
