<?php

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
// SCHLIX WEB CONTENT MANAGEMENT SYSTEM - Copyright (C) SCHLIX WEB INC.
// License: GPLv3
//
// Please read the license for details
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//


/**
 * Installer Class
 */
class schlixInstaller {

    protected $use_env = false;
    //_________________________________________________________//
    public function __construct() {
        date_default_timezone_set('UTC');
        session_name('gusrinstall');
        session_start();
        $this->use_env = ( (int) getenv('SCHLIX_INSTALL_USE_ENV') == 1 );        
    }

    //_________________________________________________________//
    function execsql($sql, $SystemDB, $echo_line = true) {
        $templine = '';
        $SystemDB->query("SET SQL_MODE=''"); // fix most errors reported by microsoft tester

        if ($echo_line)
            echo __HTML::DIV_start(['class' => 'row']);
        foreach ($sql as $line) {
            ///		$line = str_replace('"/images','"'.$_SESSION['site_httpbase'].'/images',$line);
            if (strpos($line, '-- Table structure for table') !== false)
            {
                if ($echo_line)
                {
                    $l = str_replace(['-- Table structure for table', '`'], '', $line);
                    echo __HTML::DIV_start(['class' => 'col-sm-3']);
                    echo '<i class="fas fa-database"></i>'.$l;
                    echo __HTML::DIV_end();
                    //echo __HTML::P($line);
                }
            }
            // Skip it if it's a comment
            if (substr($line, 0, 2) == '--' || $line == '')
                continue;

            // Add this line to the current segment
            $templine .= $line;
            // If it has a semicolon at the end, it's the end of the query
            if (substr(trim($line), -1, 1) == ';') {
                // Perform the query

                $SystemDB->query($templine);
                // Reset temp variable to empty
                $templine = '';
            }
        }
        if ($echo_line)
            echo __HTML::DIV_end();
    }

    private function setSessionVarFromEnv($session_var, $env_name)
    {
        if ($this->use_env)
        {
            if (empty($_SESSION[$session_var]))
                $_SESSION[$session_var] = getenv($env_name);
        }
    }
    //_________________________________________________________//
    public function stepRequirementAndDBSetup() {
        global $__html_title;
        
        $__html_title = '2. System Requirement Check';        
        $p_submit = fpost_string('submit');
        
        if ($p_submit)
            $_SESSION['agree'] = fpost_string('agreement'); 
        if (fsession_string('agree') != 'yes') {
            echo __HTML::DIV_start(['class'=>'alert alert-danger']).__HTML::P('User has not agreed to license term & conditions').__HTML::DIV_end();
            return false;
        }
        
        $req = $this->checkSystemRequirement();

        $_SESSION['met_requirement'] = $req['met_requirement'];
        $sess_db_host = fsession_string('db_host');
        if (empty($sess_db_host))
            $_SESSION['db_host'] = $this->use_env ? getenv('SCHLIX_INSTALL_DB_HOST') : 'localhost';
        $this->setSessionVarFromEnv('db_host', 'SCHLIX_INSTALL_DB_HOST');
        if (empty($_SESSION['db_host']))
            $_SESSION['db_host'] = 'localhost';
        $this->setSessionVarFromEnv('db_database', 'SCHLIX_INSTALL_DB_DATABASE');
        $this->setSessionVarFromEnv('db_username', 'SCHLIX_INSTALL_DB_USERNAME');
        $this->setSessionVarFromEnv('db_password', 'SCHLIX_INSTALL_DB_PASSWORD');
        $this->setSessionVarFromEnv('db_socket', 'SCHLIX_INSTALL_DB_SOCKET');
        $this->setSessionVarFromEnv('db_port', 'SCHLIX_INSTALL_DB_PORT');
        $this->setSessionVarFromEnv('db_use_ssl', 'SCHLIX_INSTALL_DB_USE_SSL');
        $this->setSessionVarFromEnv('db_ssl_ca', 'SCHLIX_INSTALL_DB_SSL_CA');
//
/*            // September 2019
            (int) getenv('SCHLIX_INSTALL_DB_PORT'),
            getenv('SCHLIX_INSTALL_DB_SOCKET'),
            (int) getenv('SCHLIX_INSTALL_DB_USE_SSL'),
            getenv('SCHLIX_INSTALL_DB_SSL_CA')
*/        
        
        
        include('step_requirement.template.php');
    }
    
    private function isCurrentRequestSSL()
    {
        return ( (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off') || 
             (!empty( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https' )
           );
    }
    

    //_________________________________________________________//
    public function stepSiteVariablesSetup() {
        global $__html_title;
        
        $__html_title = '3. Initialize Configuration';
        
        $can_continue =  (fsession_string('agree') == 'yes') && fsession_bool('met_requirement') && (fsession_int('req_database_ok') > 0);
//($_SESSION['agree'] && $_SESSION['met_requirement'] && $_SESSION['req_database_ok']);
        if (!$can_continue) {
            
            echo __HTML::DIV_start(array('class'=>'alert alert-danger')).__HTML::P('Cannot continue - either you have not agreed to the license or your system does not meet the requirements').__HTML::DIV_end();
            return;
        }

        $site_path = str_replace('\\', '/', dirname(realpath('../index.php')));
        $server_port = fserver_int('SERVER_PORT');
        $server_name = fserver_string('SERVER_NAME');
        $server_request_uri = fserver_string('REQUEST_URI');
        $is_not_normal_port = ($server_port != 80) && ($server_port != 443);
        $_SESSION['site_domain_name_and_port'] = $server_name . ($is_not_normal_port ? ':' . $server_port : '');
        
        $sess_site_scheme = fsession_string('site_scheme');
        
        if ($sess_site_scheme != 'http' && $sess_site_scheme != 'https')
        {
            $is_ssl = ($this->isCurrentRequestSSL());
            $_SESSION['site_scheme'] = $is_ssl ? 'https' : 'http' ;
        }            

        $curpath = pathinfo(parse_url($server_request_uri, PHP_URL_PATH), PATHINFO_DIRNAME);
        $pos = strrpos($curpath, '/install');
        $the_httpbase = substr($curpath, 0, $pos);

        if ($the_httpbase == '/')
            $the_httpbase = '';
        $_SESSION['site_httpbase'] = $the_httpbase;

        if (empty($_SESSION['mail_default_sender']))
            $_SESSION['mail_default_sender'] = 'Webmaster';
        if (empty($_SESSION['short_site_name']))
            $_SESSION['short_site_name'] = 'main';
        if (empty($_SESSION['mail_default_email']))
            $_SESSION['mail_default_email'] = 'info@' . str_replace('www.', '', $_SERVER['SERVER_NAME']);
        if (empty($_SESSION['admin_login_time']))
            $_SESSION['admin_login_time'] = 7200;
        if (empty($_SESSION['remember_me_cookie_time']))
            $_SESSION['remember_me_cookie_time'] = 172800;
        
        
        if (empty($_SESSION['schlix_js_mode']))
            $_SESSION['schlix_js_mode'] = 3;
        if (empty($_SESSION['admin_username']))
            $_SESSION['admin_username'] = 'admin';
        
            
        $arr_login_time = [ 
            900 => '15 minutes',
            1800 => '30 minutes',
            3600 => '1 hour',
            7200 => '2 hours',
            14400 => '4 hours',
            28800 => '8 hours',
            43200 => '12 hours',
            86400 => '1 day',
            172800 => '2 days',
            604800 => '1 week',
            1209600 => '2 weeks'

            ]; 

        $arr_cookie_time = $arr_login_time; 
            
        include('step_variables.template.php');
    }

    /**
     * Generate the content of config.inc.php
     * @param string $site_name
     * @param string $db_host
     * @param string $db_database
     * @param string $db_username
     * @param string $db_password
     * @param string $admin_login_time
     * @param string $remember_me_cookie_time
     * @param string $mail_default_sender
     * @param string $mail_default_email
     * @param int $system_timezone
     * @param string $record_404_errors
     * @return string
     */
    private function generateSiteConfigFileContent($site_name, $db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca, $admin_login_time, $remember_me_cookie_time, $mail_default_sender, $mail_default_email, $system_timezone, $record_404_errors, $enable_ssl)
    {
        $str_enable_ssl = $enable_ssl ? 'true' : 'false';
        $data = '<?php' . "\n";
        $data.= "/* Auto generated by installer on " . date('Y-m-d H:j:s') . " */\n";
        $site_name= add_single_quote_slashes($site_name);
        $data.="define('SCHLIX_SITE_NAME','{$site_name}');\n";
        if ($enable_ssl)
        {
            $data.="define('SCHLIX_SITE_HTTP_URL','https://'.\$_SERVER['HTTP_HOST'] ); /* All https */\n";
            $data.="define('SCHLIX_SITE_HTTPS_URL','https://'.\$_SERVER['HTTP_HOST'] ); /* All https */\n";
        } else
        {
            $data.="define('SCHLIX_SITE_HTTP_URL','http://'.\$_SERVER['HTTP_HOST'] ); /* Automatic URL */ \n";
            $data.="define('SCHLIX_SITE_HTTPS_URL',''); /* if you have  a valid SSL certificate,  change this to 'https://'.\$_SERVER['HTTP_HOST'] and set SCHLIX_SITE_SSL_ENABLED to true */ \n";            
        }
        $db_host = add_single_quote_slashes($db_host);
        $db_database = add_single_quote_slashes($db_database);
        $db_username = add_single_quote_slashes($db_username);
        $db_password = add_single_quote_slashes($db_password);
        $mail_default_sender = add_single_quote_slashes($mail_default_sender);
        $mail_default_email = add_single_quote_slashes($mail_default_email);
        
        $admin_login_time = (int) $admin_login_time;
        $remember_me_cookie_time = (int) $remember_me_cookie_time;
        if ($admin_login_time <= 0)
            $admin_login_time = 7200; // 2 hours
        if ($remember_me_cookie_time <= 0)
            $remember_me_cookie_time = 172800; // 2 days
        // new - Sept 2019
        $db_port = (int) $db_port;
        $db_use_ssl = (int) $db_use_ssl;
        if (!file_exists($db_ssl_ca))
            $db_ssl_ca = '';
        if ($db_port <= 0)
            $db_port = '0';
        $db_socket = empty($db_socket) ? "''" : "'".add_single_quote_slashes($db_socket)."'";
        $db_ssl_ca = empty($db_ssl_ca) ?  "''" : "'".add_single_quote_slashes($db_ssl_ca)."'";
        
        $data.="define('SCHLIX_SITE_PATH',str_replace('\\\','/',dirname(__FILE__)));\n";
        $data.="define('SCHLIX_DEFAULT_USER_CLASS','users'); /* Default user class - e.g: LDAP, smf forum, opencart, etc */\n";
        $data.="define('SCHLIX_DEFAULT_LANGUAGE','en_us'); /* Default Language */ \n";
        $data.="define('SCHLIX_DEFAULT_ADMIN_GROUP','Administrators');\n";
        //$data.="define('SCHLIX_ADMIN_LANGUAGE','en_us');\n";
        $data.="define('SCHLIX_SITE_OFFLINE',false); /* Set this to yes if you want to make your site offline for visitors */ \n";
        $data.="define('SCHLIX_SEF_ENABLED', true);\n";
        $data.="define('SCHLIX_SITE_SSL_ENABLED',{$str_enable_ssl}); /* indicate whether you have an SSL cerificate for your website */ \n";
        $data.="define('SCHLIX_FORCE_SSL_ADMIN_LOGIN',{$str_enable_ssl}); /* force admin to login via SSL */\n"; // 
        $data.="define('SCHLIX_DB_HOST','{$db_host}'); /* database host, e.g. localhost */\n";
        $data.="define('SCHLIX_DB_DATABASE','{$db_database}'); /* database name */\n";
        $data.="define('SCHLIX_DB_USERNAME','{$db_username}'); /* database username */\n";
        $data.="define('SCHLIX_DB_PASSWORD','{$db_password}'); /* database password */\n";
        $data.="///////// advanced db param - leave blank for default /////////\n";
        $data.="define('SCHLIX_DB_PORT',{$db_port}); /* if the value is zero then it will use the default port number, usually 3306 for mysql */ \n";
        $data.="define('SCHLIX_DB_SOCKET',{$db_socket}); /* if the value is blank then it will the default value */\n";
        $data.="define('SCHLIX_DB_USE_SSL',{$db_use_ssl}); /* set this to true if MySQL uses SSL connection */\n";
        $data.="define('SCHLIX_DB_SSL_CA',{$db_ssl_ca}); /* specify the SSL CA certificate file if SCHLIX_DB_USE_SSL is set to true */\n";
        $data.="///////// end advanced db param  /////////";
        $data.="define('SCHLIX_PAGE_CACHE_ENABLED', false);\n";
        $data.="define('SCHLIX_SQL_ENFORCE_ROW_LIMIT', true);\n";
        $data.="define('SCHLIX_SQL_CACHE_ENABLED', false);\n";
        $data.="define('SCHLIX_SQL_CACHE_TIME', 0);\n";
        $data.="define('SCHLIX_SESSION_LOGIN_TIME',{$admin_login_time}); /* in seconds - this should be short, e.g. 30 minutes to 3 hours */ \n";
        $data.="define('SCHLIX_SESSION_REMEMBER_COOKIE_TIME', {$remember_me_cookie_time}); /* in seconds - this should be longer than the login time */ \n";
        $data.="define('SCHLIX_COOKIE_SAMESITE', 'none'); /* Samesite=none cookie for HTTPS by default as per new standard in 2020 */ \n";
        $data.="define('SCHLIX_ADMIN_THEME','admindefault');\n";
        $data.="define('SCHLIX_USER_IMAGE_DIRECTORY',SCHLIX_SITE_PATH.'/media/images');\n";
        $data.="define('SCHLIX_MAIL_DEFAULT_SENDER','{$mail_default_sender}'); /* Default sender name, e.g. Webmaster */\n";
        $data.="define('SCHLIX_MAIL_DEFAULT_EMAIL','{$mail_default_email}'); /* Default email address for sending out an email if none specified */\n";
        $data.="define('SCHLIX_SMTP_HOST', ''); /* If you use an external SMTP server, fill in the server host here */\n";
        $data.="define('SCHLIX_SMTP_PORT', 25); /* Some common values: 25, 587 (TLS), 465 (SSL) */\n";
        $data.="define('SCHLIX_SMTP_USE_AUTH', false); /* Specify whether an SMTP authentication is required */\n";
        $data.="define('SCHLIX_SMTP_USERNAME', ''); /* Fill in the SMTP username if SMTP authentication is required */ \n";
        $data.="define('SCHLIX_SMTP_PASSWORD', ''); /* Fill in the password for the SMTP user if SMTP authentication is required */\n";
        $data.="define('SCHLIX_SMTP_SECURITY', ''); /* possible values: tls (port 587), ssl (port 465) or simply blank (port 25) */\n";
        $data.="define('SCHLIX_SMTP_SECURITY_SELF_SIGNED', 0); /* true/false value - allow self signed certificate */\n";
        
        $data.=" /* Oauth2 */\n";
        $data.="define('SCHLIX_SMTP_OAUTH2_PROVIDER', ''); /* OAUTH2 providers - google, microsoft, yahoo */ \n";
        $data.="define('SCHLIX_SMTP_OAUTH2_CLIENT_ID', ''); /* OAUTH2 Client/Application ID */ \n";
        $data.="define('SCHLIX_SMTP_OAUTH2_SECRET', ''); /* OAUTH2 secret */ \n";
        $data.="define('SCHLIX_SMTP_OAUTH2_REFRESH_TOKEN', ''); /* OAUTH2 offline refresh token */ \n";
        
        $data.="define('SCHLIX_SYSTEM_TIMEZONE', '{$system_timezone}'); /* default website timezone */\n";
        //$data.="define('SCHLIX_SITE_META_KEYWORDS',''); /* Default meta keywords if there is none defined in the app/item/category */\n";
        //$data.="define('SCHLIX_SITE_META_DESCRIPTION','{$site_name}'); /* Default meta keywords if there is none defined in the app/item/category */ \n";
        //$data.="define('SCHLIX_JS_MODE','{$schlix_js_mode}'); /* Javascript mode. 1 = Debug, 2 = Minified, 3 = Minified and Gzipped */ \n";
        $data.="define('SCHLIX_JS_MODE',3); /* Javascript mode. 1 = Debug, 2 = Minified, 3 = Minified and Gzipped */ \n";
        $data.="define('SCHLIX_RECORD_404_ERRORS','{$record_404_errors}'); /* Record 404 errors in the database */ \n";

        $data.="/**********************************************************************/\n";
        $data.="/* Do not modify the SCHLIX_SITE_URL - this is for auto http/https switch */\n";
        $data.='define(\'SCHLIX_SITE_URL\', (defined(\'SCHLIX_SITE_HTTPS_URL\') && SCHLIX_SITE_SSL_ENABLED && SCHLIX_SITE_HTTPS_URL !=\'\' && (isset($_SERVER[\'HTTPS\']) && (($_SERVER[\'HTTPS\']===\'on\') ||  ($_SERVER[\'HTTPS\']===1))  || $_SERVER[\'SERVER_PORT\']===443)) ? SCHLIX_SITE_HTTPS_URL : SCHLIX_SITE_HTTP_URL);';
        $data.="\n/**********************************************************************/\n";
        //$data.="error_reporting({$error_report});\n";
        //$data.="@ini_set('display_errors',$display_errors);\n";
        $data.="error_reporting(E_ERROR|E_PARSE|E_CORE_ERROR|E_CORE_WARNING|E_COMPILE_ERROR|E_COMPILE_WARNING|E_USER_ERROR|E_RECOVERABLE_ERROR|E_WARNING);\n";
        $data.="@ini_set('display_errors',1);\n";
        
        return $data;
    }
    
    /**
     * Generate Multisite-config.inc.php file content. If $use_wildcard is specified, 
     * it invalidates the $use_www. If $use_www is unspecified, only the specified domain
     * name 
     * @param string $site_id
     * @param string $server_name
     * @param bool $use_wildcard
     * @param bool $use_www
     * @return string
     */
    private function generateMultisiteConfigFileContent($site_id, $server_name, $use_wildcard, $use_www) 
    {
        $trans_www = [];
        if ($use_wildcard)
        {        
            $trans_www['*'] = $site_id; // this means all domain, including parked domain will go to this site_id
        } else
        {
            if ($use_www)
            {
                $site_host = $server_name;//echo remove_prefix_from_string('www.', $server_name);
                $secondary_host = str_starts_with($site_host, 'www.' ) ? remove_prefix_from_string($server_name, 'www.') : 'www.'.$server_name;
                $trans_www[$site_host] = $site_id;
                $trans_www[$secondary_host] = $site_id;
            } else
            {
                $site_host = $server_name;
                $trans_www[$site_host] = $site_id;            
            }
        }


        $data = '<?php' . "\n";
        $data.='// Automatically generated on ' . get_current_datetime() . "\n\n";
        $data.= '/* 
 * Define multisites here. You **MUST EXPLICITLY** define all domains (including with or without www).
 * If you want to enable all sites pointing to a certain subsite, use the wildcard. Only one wildcard is allowed.*/' . "\n\n";
        $data.='global $SchlixWebsites;' . "\n\n";
        $data.='$SchlixWebsites =' . var_export($trans_www, TRUE) . ';';

        return $data;
    }
    
    /**
     * Return a result array with key: (status, messages), where messages is a
     * list of messages
     * @param string $db_host
     * @param string $db_database
     * @param string $db_username
     * @param string $db_password
     * @return array
     */
    protected function initializeDatabase($db_host, $db_database, $db_username, $db_password, $db_port = NULL, $db_socket  = NULL, $db_use_ssl  = NULL, $db_ssl_ca = NULL)
    {
        $init_result = array();
        $messages = array();
        // now init
        if ($db_use_ssl == 1)
        {
            $link = mysqli_init();
            if ($db_ssl_ca)
                mysqli_ssl_set($link,NULL,NULL, $db_ssl_ca, NULL, NULL); 
            $opt = MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT;
            @mysqli_real_connect($link, $db_host, $db_username, $db_password, NULL, $db_port, $db_socket, $opt);                    
        } else
            $link = @mysqli_connect($db_host, $db_username, $db_password, NULL, $db_port, $db_socket);
        
        
        if ($link) {
            // check if database already exists
            $sql = 'SHOW DATABASES';
            $result = array();
            $last_query_result = mysqli_query($link, $sql);
            while ($row = mysqli_fetch_assoc($last_query_result))
                $result[] = $row;
            mysqli_free_result($last_query_result);
            if ($result) {
                $dbs = array();
                foreach ($result as $db) {
                    $dbs[] = $db['Database'];
                }
                if (in_array($db_database, $dbs)) {
                    $messages[] = "Database {$db_database} exists.";
                    $init_result['status'] = true;
                } else {// if not exist then create
                    $messages[] = "Database {$db_database} does not exist. Trying to create one.";
                    $sql = "CREATE DATABASE {$db_database} DEFAULT CHARACTER SET = utf8mb4 DEFAULT COLLATE utf8mb4_unicode_ci;";
                    $last_query_result = mysqli_query($link, $sql);
                    if (!$last_query_result) {
                        $messages[] = ('Cannot create database. Installation failed. Please create the database first (e.g: from phpMyAdmin) then proceed with this installation step. You do not need to close this window.');
                        $init_result['status'] = false;
                    } else {
                        $messages[] = ('Database creation successful');
                        $init_result['status'] = true;
                    }
                }
            } else {
                $messages[] = ('No Show Database permission - ok but continuing install');
                $init_result['status'] = true;                
            }            
             mysqli_close($link);              
             
        } else
        {
            $init_result['status'] = false;
        }
        $init_result['messages'] = $messages;
        return $init_result;
    }
    //_________________________________________________________//
    public function stepDBDataSampleAndUsersSetup() {
        global $__html_title;
        
        $__html_title = '4. Database Installation';
        //$can_continue = ($_SESSION['agree'] && $_SESSION['met_requirement'] && $_SESSION['req_database_ok'] && ($_SESSION['site_name'] || $_POST['site_name'] ) );
        $can_continue =  (fsession_string('agree') == 'yes') && fsession_bool('met_requirement') && (fsession_int('req_database_ok') > 0) && (!empty(fsession_string('site_name')) || !empty(fpost_string('site_name')));
        
        if (!$can_continue) {
            echo __HTML::DIV_start(array('class'=>'alert alert-danger')).__HTML::P('Cannot continue - please complete the previous steps first').__HTML::DIV_end();
            return false;
        }
        if (!empty(fpost_string('submit'))) {
            $_SESSION['use_wildcard'] = fpost_int('use_wildcard'); 
            $_SESSION['use_www'] = fpost_int('use_www');
            $_SESSION['short_site_name'] = convert_into_sef_friendly_title(fpost_string('short_site_name', 63));
            $_SESSION['site_name'] = fpost_string('site_name', 127);
            $_SESSION['site_scheme'] = fpost_string('site_scheme') ;
            $_SESSION['site_domain_name_and_port'] = fpost_string('site_domain_name_and_port');
            $_SESSION['site_url'] = $_SESSION['site_scheme'].'://'.$_SESSION['site_domain_name_and_port'];
            $_SESSION['mail_default_sender'] = fpost_string('mail_default_sender');
            $_SESSION['mail_default_email'] = fpost_string('mail_default_email');
            $_SESSION['admin_login_time'] = fpost_int('admin_login_time');
            $_SESSION['remember_me_cookie_time'] = fpost_int('remember_me_cookie_time');
            $_SESSION['admin_username'] = fpost_string('admin_username');
            $_SESSION['admin_password'] = fpost_string('admin_password');
            $_SESSION['system_timezone'] = fpost_string('system_timezone');
            //$_SESSION['site_httpbase'] = $_POST['site_httpbase'];
            $_SESSION['record_404_errors'] = fpost_int('record_404_errors');
            $_SESSION['req_configinit_ok'] = 1;
        } 
        
        // Save Multisite config file
        $data = $this->generateMultisiteConfigFileContent($_SESSION['short_site_name'], $_SERVER['SERVER_NAME'], $_SESSION['use_wildcard'], $_SESSION['use_www']);
        $result = file_put_contents(INSTALLER_ROOT_PATH . '/multisite-config.inc.php', $data);
        if (!$result) {
            echo __HTML::H3("Cannot save main multi-site configuration file");
            return;
        }
        // create site skeleton
        $site_folder = $_SESSION['short_site_name'];
        if (!create_site_skeleton(INSTALLER_ROOT_PATH, $site_folder)) {
            echo __HTML::H3("Cannot create subsite folder " . $site_folder);
            return;
        }       
        $use_https_fulltime = ($_SESSION['site_scheme'] == 'https');
        // Save site config file
        $data = $this->generateSiteConfigFileContent($_SESSION['site_name'], $_SESSION['db_host'], $_SESSION['db_database'], $_SESSION['db_username'], $_SESSION['db_password'], $_SESSION['db_port'], $_SESSION['db_socket'], $_SESSION['db_use_ssl'], $_SESSION['db_ssl_ca'], $_SESSION['admin_login_time'], $_SESSION['remember_me_cookie_time'], $_SESSION['mail_default_sender'], $_SESSION['mail_default_email'], $_SESSION['system_timezone'], $_SESSION['record_404_errors'], $use_https_fulltime);
            
        $site_folder = $_SESSION['short_site_name'];
        $config_file_location = INSTALLER_ROOT_PATH . '/web/' . $site_folder . '/config.inc.php';
        $config_save_result = file_put_contents($config_file_location, $data);
        if (!$config_save_result) {
            $_SESSION['settings_save_ok'] = 0;
            echo __HTML::H3("Cannot save subsite configuration file");
            return;
        } else {
            $_SESSION['settings_save_ok'] = 1;
        }

       $db_init_result =  $this->initializeDatabase($_SESSION['db_host'], $_SESSION['db_database'], $_SESSION['db_username'], $_SESSION['db_password'], $_SESSION['db_port'], $_SESSION['db_socket'], $_SESSION['db_use_ssl'], $_SESSION['db_ssl_ca']);
       $_SESSION['db_init_ok'] = $db_init_result['status'];
       if ($db_init_result['messages'])
       {
           foreach ($db_init_result['messages'] as $message)
           {
               echo __HTML::H3($message);
           }
       }
       
        if ($_SESSION['db_init_ok']) {
            echo __HTML::H3('Creating tables...');

            $install_sql = file(INSTALLER_ROOT_PATH . '/system/apps/sitemanager/sql/install_step1.sql', true);
            if (!$install_sql) {
                $_SESSION['db_tables_ok'] = 0;
                echo __HTML::H3('Cannot load installation SQL');
                return false;
            }
            $install_sql_step2 = file(INSTALLER_ROOT_PATH . '/system/apps/sitemanager/sql/install_step2.sql', true);
            if (!$install_sql_step2) {
                $_SESSION['db_tables_ok'] = 0;
                echo __HTML::H3('Cannot load installation SQL Step 2');
                return false;
            }

            $SystemDB = new \SCHLIX\cmsDatabase($_SESSION['db_host'], $_SESSION['db_database'], $_SESSION['db_username'], $_SESSION['db_password'], $_SESSION['db_port'], $_SESSION['db_socket'], $_SESSION['db_use_ssl'], $_SESSION['db_ssl_ca']);
            global $SystemLog;
            $SystemLog = new \SCHLIX\cmsLogger('gk_log_items');

            $this->execsql($install_sql, $SystemDB);
            $this->execsql($install_sql_step2, $SystemDB); // separate it. Install.SQL is run on upgrade
            
            
            //$salt = $this->generateRandomPasswordSalt('blowfish');
            //$hashed_password = crypt($_SESSION['admin_password'], $salt);
            // 2019-05-30
            $salt = '';
            $hashed_password = password_hash($_SESSION['admin_password'], PASSWORD_DEFAULT);
            // admin user - set password
            $SystemDB->query("UPDATE gk_user_items set password = :hashed_password, salt= :salt, email_address = :email_address WHERE id = 1"
            ,['hashed_password' =>$hashed_password, 'salt' => $salt, 'email_address' => $_SESSION['mail_default_email']]); // change password, add salt
            
            if ($_SESSION['admin_username'] != 'admin')
                $SystemDB->query("UPDATE gk_user_items set username = :admin_username  WHERE id = 1", ['admin_username' => $_SESSION['admin_username']]); // change password, add salt
            echo __HTML::H3('Done');
            $_SESSION['db_tables_ok'] = 1;
            $_SESSION['db_blog_ok'] = 0;
        }
        include('step_database.template.php');
    }
    //_________________________________________________________//
    private function getSQLInstallScriptForSampleData($opt_install_sample_data)
    {
        $sample_data = [
            'install_sample_empty.sql', // 0
            'install_sample_magazine.sql', // 1
            'install_sample_company1.sql', // 2
            ];
        $opt_install_sample_data = (int)  $opt_install_sample_data;
        
        $sql_install_script = $sample_data[0];
        if ($opt_install_sample_data < ___c($sample_data))
        {
            $sql_install_script = $sample_data[$opt_install_sample_data];
        } 
        $sqlfname = '/system/apps/sitemanager/sql/'.$sql_install_script ;
        return $sqlfname;
        
    }
    //_________________________________________________________//
    public function stepFinishSetup() {
        global $__html_title;
        
        
        $__html_title = '5. Finish';        
        
        $can_continue =  (fsession_string('agree') == 'yes') && fsession_bool('met_requirement') && (fsession_int('req_database_ok') > 0) &&  
                 $_SESSION['db_init_ok'] && $_SESSION['db_tables_ok'] && $_POST['submit']
                && $_SESSION['db_blog_ok'] == 0;
         

        if (!$can_continue) {
            echo __HTML::DIV_start(array('class'=>'alert alert-danger')).__HTML::P('Please complete the previous steps before attempting to finish the installation').__HTML::DIV_end();
            return false;
        }

        if (empty(fsession_string('admin_password')))
        {
            echo __HTML::DIV_start(array('class'=>'alert alert-danger')).__HTML::P('Admin password cannot be empty').__HTML::DIV_end();
            return false;
            
        }
        $sqlfname = INSTALLER_ROOT_PATH.$this->getSQLInstallScriptForSampleData( (int)  $_POST['installsample']); 
        $_SESSION['db_blog_ok'] = 0;
        echo __HTML::H3('Installing sample data...');

        $install_sql = file($sqlfname, true);
        if (!$install_sql) {
            $_SESSION['db_tables_ok'] = 0;
            echo __HTML::H3('Cannot load sample data SQL');
            return false;
        }

        //$new_images_path = $_SESSION['site_httpbase'] . '/web/' . $_SESSION['short_site_name'] . '/media/images/';
        //$new_images_replacement = 'src="' . $new_images_path;
        //$install_sql = str_replace('src="system/images/', $new_images_replacement, $install_sql);
        //$install_sql = str_replace('src="/src/web/test/media/images/', $new_images_replacement, $install_sql);

        $SystemDB = new SCHLIX\cmsDatabase($_SESSION['db_host'], $_SESSION['db_database'], $_SESSION['db_username'], $_SESSION['db_password'], $_SESSION['db_port'], $_SESSION['db_socket'], $_SESSION['db_use_ssl'], $_SESSION['db_ssl_ca']);
        global $SystemLog;
        $SystemLog = new SCHLIX\cmsLogger('gk_log_items');


        $this->execsql($install_sql, $SystemDB);

        
        $menu_site_httpbase = $_SESSION['site_httpbase'];
        if ($menu_site_httpbase == '/')
            $menu_site_httpbase = ''; // reset it, so just / will be ignored
        if ($menu_site_httpbase != '')
        {
            if (!str_ends_with($menu_site_httpbase, '/'))
                $menu_site_httpbase.='/';
            // March 2, 2017 - in reponse to Softaculous tester. 
            // Menu will be reset after first run again
            $SystemDB->query("update `gk_menu_items` set sefurl = concat(:menu_site_httpbase,sefurl) where `application` NOT IN ('menu_divider','menu_placeholder','external_link')", ['menu_site_httpbase' => $menu_site_httpbase]);
            $SystemDB->query("update `gk_menu_items` set sefurl = :menu_site_httpbase where `application` = 'home'", ['menu_site_httpbase' =>  $_SESSION['site_httpbase']]); 
        }
        
        $SystemDB->query("update `gk_contact_items` set email = :email_address",['email_address' => $_SESSION['mail_default_email']]);
        echo __HTML::H3('Database sample installation is done');

        //$salt = $this->generateRandomPasswordSalt('blowfish');
        //$hashed_password = crypt('testuser', $salt);
        // 2019-05-30
        $salt = '';
//        $hashed_password = password_hash('testuser', PASSWORD_DEFAULT);        
        $hashed_password = password_hash($_SESSION['admin_password'], PASSWORD_DEFAULT);
        $SystemDB->query("UPDATE gk_user_items set password = :hashed_password, salt= :salt WHERE id > 1"
            ,['hashed_password' =>$hashed_password, 'salt' => $salt ]); // change password, add salt

        
        $SystemDB->query("INSERT INTO `gk_config` (`section`, `key`, `value`, `date_modified`, `modified_by_id`) VALUES ('system', 'str_system_id', :uuid, NOW(), 0)", ['uuid' => new_uuid_v4()]); 
        include (realpath('../runtime.inc.php'));
        $SystemDB->query("INSERT INTO `gk_config` (`section`, `key`, `value`, `date_modified`, `modified_by_id`) VALUES ('system', 'str_database_version', :schlix_version, NOW(), 0)",['schlix_version' => SCHLIX_VERSION]);

        $_SESSION['db_blog_ok'] = 1;
        /*copy_recursive(INSTALLER_ROOT_PATH . '/system/themes/html5demo', INSTALLER_ROOT_PATH . '/web/' . $_SESSION['short_site_name'] . '/themes/html5demo');
        copy_recursive(INSTALLER_ROOT_PATH . '/system/images/demo', INSTALLER_ROOT_PATH . '/web/' . $_SESSION['short_site_name'] . '/media/images/demo');
        copy_recursive(INSTALLER_ROOT_PATH . '/system/images/slideshow', INSTALLER_ROOT_PATH . '/web/' . $_SESSION['short_site_name'] . '/media/slideshow');*/
        $this->copySampleData(INSTALLER_ROOT_PATH, $_SESSION['short_site_name'] );
        include('step_finish.template.php');
        $_SESSION['admin_password'] = NULL; 
    }

    //_________________________________________________________//
    /**
     * Test database requirement with ajax
     */
    public function ajaxTestDatabase()
    {
        $_GET['ajax'] = 1;
        $ajax_result = '';
        $error_rep = error_reporting();
        error_reporting(0);
        
        $p_db_host = fpost_string('db_host', 255);
        $p_db_username = fpost_string('db_username', 127);
        $p_db_password = fpost_string('db_password', 255);
        $p_db_database = fpost_string('db_database', 63);
        
        if (!empty($p_db_host) && !empty($p_db_database) && !empty($p_db_username) && !empty($p_db_password)  )
        {
            $db_host = $p_db_host;
            $db_username = $p_db_username;
            $db_password = $p_db_password;
            $db_database = $p_db_database;

            $db_port = fpost_int('db_port') ;            
            $db_socket = fpost_string('db_socket'); 
            $db_use_ssl = fpost_int('db_use_ssl');  
            $db_ssl_ca = fpost_string('db_ssl_ca');  

            $valid_db_port = ($db_port >= 0 && $db_port < 65536);
            if (!$valid_db_port)
            {
                $_SESSION['req_database_ok'] = 0;
                $ajax_result = ajax_reply(200, 'Invalid port - must be between 1 and 65535');
            } else if ($db_use_ssl > 0)
            {
                if (!empty($db_ssl_ca) && !file_exists(realpath($db_ssl_ca)))
                {
                    $_SESSION['req_database_ok'] = 0;
                    $ajax_result = ajax_reply(200, 'SSL CA file cannot be found');
                }
                elseif ($db_host == 'localhost' || $db_host == '127.0.0.1')
                {
                    $_SESSION['req_database_ok'] = 0;
                    $ajax_result = ajax_reply(200, 'Cannot set SSL database connection for localhost/127.0.0.1');    
                }
                
            }
            if (empty($ajax_result))
            {

                if ($db_port == 0 || $db_port == 3306)
                    $db_port = NULL; //ini_get("mysqli.default_port");
                if (empty($db_socket))
                    $db_socket = NULL; // ini_get("mysqli.default_socket");
                if (empty($db_ssl_ca))
                    $db_ssl_ca = NULL;
                        
                
                $link = null;
                try 
                {
                
                if ($db_use_ssl == 1)
                {
                    $link = mysqli_init();
                    if ($db_ssl_ca)
                        mysqli_ssl_set($link,'','', $db_ssl_ca, '', ''); 
                    $opt = MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT;
                    @mysqli_real_connect($link, $db_host, $db_username, $db_password, NULL, $db_port, $db_socket, $opt);                    
                } else
                    $link = @mysqli_connect($db_host, $db_username, $db_password, NULL, $db_port, $db_socket);
                } 
                catch (\mysqli_sql_exception $exc)
                {

                }
                
                if ($link)
                {
                    $_SESSION['db_host'] = $db_host;
                    $_SESSION['db_username'] = $db_username;
                    $_SESSION['db_password'] = $db_password;
                    // new param - Sept 2019 - v2.2.0-8 - for Azure
                    $_SESSION['db_use_ssl'] = $db_use_ssl;
                    $_SESSION['db_port'] = $db_port;
                    $_SESSION['db_socket'] = $db_socket;
                    $_SESSION['db_ssl_ca'] = $db_ssl_ca;
                    // end new params

                    $last_query_result = null;
                    try 
                    {
                        //$select_db_ok = mysqli_select_db($link, $db_database);
                        $last_query_result = mysqli_query($link, "SELECT VERSION()");
                    } 
                    catch (\mysqli_sql_exception $exc)
                    {
                        
                    }
                    if (!$last_query_result)
                    {
                        $error1 = 'Cannot run query: ' . mysqli_error($link);
                        //if (!$select_db_ok) $error1 = "Failed to select database";
                        $ajax_result = ajax_reply(200, $error1);
                    } else
                    {
                        $result = array();
                        while ($row = mysqli_fetch_assoc($last_query_result))
                        {
                            $result[] = $row;
                        }
                        mysqli_free_result($last_query_result);
                        if ($result)
                        {
                            $_SESSION['db_database'] = $db_database;
                            $_SESSION['req_database_ok'] = 1;
                            $ajax_result = ajax_reply(200, 'OK');
                        } else
                        {
                            $_SESSION['req_database_ok'] = 0;
                            $ajax_result = ajax_reply('300', 'Strange error ' . mysqli_error($link));
                        }
                    }
                    if ($link)
                        mysqli_close($link);
                } else
                {
                    $_SESSION['req_database_ok'] = 0;
                    $ajax_result = ajax_reply(200, 'Cannot connect using the entered hostname, username, and password. ' . mysqli_connect_error());
                }
            }
        } else
        {
            $_SESSION['req_database_ok'] = 0;
            $ajax_result = ajax_reply(200, 'Parameters incomplete');
        }
        error_reporting($error_rep);
        ajax_echo($ajax_result);
    }

    //_________________________________________________________//
    private function indicateYesNoWithPicture($req) {
        $active_icon = "fas fa-check";
        $inactive_icon = "fas fa-times";
        $active_color = "green";
        $inactive_color= "red";
        $which_icon = ($req) ? $active_icon : $inactive_icon;
        $which_color = ($req) ? $active_color : $inactive_color;
        return '<i class="' . $which_icon . '" style="color:'.$which_color.'"></i>';
    }

    //_________________________________________________________//
    private function generateRandomPasswordSalt($hash_type, $work_factor = 9) {
        // copy pasted from users.class.php - yes this is messy but the code for install is going to be deleted anyway.

        $work_factor = (int) $work_factor;
        if ($work_factor < 4 || $work_factor > 31)
            $work_factor = 9;
        $rand = function_exists('openssl_random_pseudo_bytes') ? openssl_random_pseudo_bytes(16) : random_bytes($work_factor);
        $silly_string = substr(strtr(base64_encode($rand), '+', '.'), 0, 22);

        $wfstr6 = (string) ($work_factor * 10000);
        $wfstr2 = (string) $work_factor;
        switch ($hash_type) {
            case 'sha256': $salt = '$5$rounds=' . str_pad($wfstr6, 6, '0', STR_PAD_LEFT) . '$' . substr($silly_string, 0, 16);
                break;
            case 'sha512': $salt = '$6$rounds=' . str_pad($wfstr6, 6, '0', STR_PAD_LEFT) . '$' . substr($silly_string, 0, 16);
                break;
            case 'blowfish':
            default: $salt = '$2y$' . str_pad($wfstr2, 2, '0', STR_PAD_LEFT) . '$' . $silly_string;
                break;
        }
        return $salt;
    }

    //_________________________________________________________//
    public function probeHttpBase() {
        $curpath = pathinfo(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), PATHINFO_DIRNAME);
        $pos = strrpos($curpath, '/install');
        $the_httpbase = substr($curpath, 0, $pos);
        if ($the_httpbase == '/')
            $the_httpbase = '';
        return $the_httpbase;
    }

    //_________________________________________________________//		
    public function isUserInstallingSchlixOnDefaultFolder() {
        $my_httpbase = $this->probeHttpBase();
        $pos = strpos($my_httpbase, '/schlix');
        return ($pos !== FALSE);
    }

    //_________________________________________________________//	
    public function step1WeclomePage() {
        global $__html_title;
        
        $__html_title = '1. Welcome';        
        include('step_welcome.template.php');
    }

    /**
     * Check requirement
     * @return array
     */
    protected function checkSystemRequirement()
    {
        $php_extensions = get_loaded_extensions();
        ob_start();
        phpinfo(INFO_MODULES);
        $phpinfostr = ob_get_contents();
        ob_end_clean();
        $result = array();
        $result['session'] = in_array('session', $php_extensions);
        $result['mbstring'] = in_array('mbstring', $php_extensions);
        
        $result['gd'] = function_exists('imagecreatefromjpeg'); // in_array('gd', $php_extensions);
        $result['mysql'] = in_array('mysqli', $php_extensions);
        $result['php50'] = version_compare(PHP_VERSION, '5.6.0');
        $result['zlib'] = function_exists('gzopen');
        $result['json'] = function_exists('json_decode');
        $result['dom'] = class_exists('DOMDocument');
        $result['simplexml'] = function_exists('simplexml_import_dom');        
        $result['bcrypt'] = (CRYPT_BLOWFISH == 1);
        //$result['curl'] = function_exists('curl_init');
        
        $prev_dir = remove_multiple_slashes(realpath(__DIR__.'/..'));
        $config_inc_php = remove_multiple_slashes($prev_dir.'/multisite-config.inc.php');
        $htaccess_real_file = remove_multiple_slashes($prev_dir.'/.htaccess');
        $htaccess_txt_file = remove_multiple_slashes($prev_dir.'/htaccess.txt');
        if (!file_exists($htaccess_real_file))
        {
            @copy($htaccess_txt_file, $htaccess_real_file);
        }
        //$result['configincphp'] = ( is_writable('../') && !(file_exists('../config.inc.php')) ) || ( is_writable('../config.inc.php') && (file_exists('../config.inc.php')) );
        
        $result['configincphp'] = ( is_writable($prev_dir) && !(file_exists($config_inc_php)) ) || ( is_writable($config_inc_php) && (file_exists($config_inc_php)) );
        $result['mod_rewrite'] = (strpos($phpinfostr, 'mod_rewrite') >= 0);
        $result['htaccess'] = file_exists($htaccess_real_file);
        
        $result['met_requirement'] = ($result['htaccess'] && $result['php50'] && $result['mysql'] && $result['mod_rewrite'] && $result['configincphp'] && $result['gd'] && $result['bcrypt']);
        $error_list = array();
        
        if (!$result['htaccess'])
            $error_list[] = '.htaccess file does not exist and the backup file htaccess.txt could not be copied either';        
        if (!$result['zlib'])
            $error_list[] = 'ZLib library is required';
        
        if (!$result['php50'])
            $error_list[] = 'PHP 5.6 or greator is required';
        if (!$result['mysql'])
            $error_list[] = 'MySQLi extension is required';
        if (!$result['mod_rewrite'])
            $error_list[] = 'Mod Rewrite is required for Apache-based installation';
        if (!$result['configincphp'])
            $error_list[] = 'Config file multisite-config.inc.php must be writable. Try chmod 0755 the whole install dir and change the script owner to the same as the website owner';
        if (!$result['gd'])
            $error_list[] = 'GD2 image library is required';
        if (!$result['bcrypt'])
            $error_list[] = 'BCrypt (CRYPT_BLOWFISH) is required';        
        /*if (!$result['curl'])
            $error_list[] = 'CURL extension is required'; */
        if ($error_list)
            $result['error_list'] = $error_list;
        return $result;
    }
    
    /**
     * Returns true if database check is okay, otherwise return an array of error list
     * @param string $db_host
     * @param string $db_database
     * @param string $db_username
     * @param string $db_password
     * @return string|array
     */
    public function testDatabase($db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca)
    {
        $testresult = false;
        $error_msg = null;
        
        if ($db_port == 0 || $db_port == 3306)
            $db_port = NULL; //ini_get("mysqli.default_port");
        if (empty($db_socket))
            $db_socket = NULL; // ini_get("mysqli.default_socket");
        if (empty($db_ssl_ca))
            $db_ssl_ca = NULL;
        $link = null;
        if ($db_use_ssl == 1)
        {
            $link = mysqli_init();
            if ($db_ssl_ca)
                mysqli_ssl_set($link,NULL,NULL, $db_ssl_ca, NULL, NULL); 
            $opt = MYSQLI_CLIENT_SSL | MYSQLI_CLIENT_SSL_DONT_VERIFY_SERVER_CERT;
            @mysqli_real_connect($link, $db_host, $db_username, $db_password, NULL, $db_port, $db_socket, $opt);                    
        } else
            $link = @mysqli_connect($db_host, $db_username, $db_password, NULL, $db_port, $db_socket);
        
        
        if ($link) {
            //  $select_db_ok = mysqli_select_db($db_database, $link);
            //$last_query_result = mysqli_query("select 'abcde' as 'ok'", $link);
            $select_db_ok = mysqli_select_db($link, $db_database);
            $last_query_result = mysqli_query($link, "SELECT VERSION()");
            if ($last_query_result) {
                $result = array();
                while ($row = mysqli_fetch_assoc($last_query_result)) {
                    $result[] = $row;
                }
                mysqli_free_result($last_query_result);
                if ($result) {
                    $testresult = true; 
                } else {
                    $testresult = false;
                }
            } else
                $testresult = false;
            if ($link)
                mysqli_close($link);
        } else {
            $error_msg = mysqli_connect_error();
        }        
        return array('status' => $testresult, 'error' => $error_msg);
    }
    
    private function copySampleData($site_root_file_path, $short_site_name)
    {
        copy_recursive($site_root_file_path . '/system/themes/companyprofile', $site_root_file_path . '/web/' . $short_site_name. '/themes/companyprofile');
        copy_recursive($site_root_file_path . '/system/themes/beritanews', $site_root_file_path . '/web/' . $short_site_name. '/themes/beritanews');
        //copy_recursive($site_root_file_path . '/system/themes/html5demo', $site_root_file_path . '/web/' . $short_site_name. '/themes/html5demo');
        //copy_recursive($site_root_file_path . '/system/themes/samplemagazine', $site_root_file_path . '/web/' . $short_site_name. '/themes/samplemagazine');
        //copy_recursive($site_root_file_path . '/system/images/articles', $site_root_file_path . '/web/' . $short_site_name. '/media/articles');
        //copy_recursive($site_root_file_path . '/system/images/demo', $site_root_file_path . '/web/' . $short_site_name. '/media/images/demo');
        //copy_recursive($site_root_file_path . '/system/images/gallery', $site_root_file_path . '/web/' . $short_site_name. '/media/gallery');
        copy_recursive($site_root_file_path . '/system/images/avatars', $site_root_file_path . '/web/' . $short_site_name. '/data/user/avatars');
        copy($site_root_file_path . '/system/images/schlixlogo.png', $site_root_file_path . '/web/' . $short_site_name. '/media/images/schlixlogo.png');
    }

    /**
     * Perform automatic installation
     * 
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param bool $agree_term
     * @param string $db_host
     * @param string $db_database
     * @param string $db_username
     * @param string $db_password
     * @param string $site_domain_name
     * @param int $site_use_https
     * @param string $site_httpbase
     * @param string $site_name
     * @param string $site_root_file_path
     * @param string $site_id
     * @param string $email_from
     * @param string $email_address
     * @param string $admin_username
     * @param string $admin_password
     * @param int $sampledata
     * @param int $use_wildcard
     * @param string $timezone
     * @param bool $use_www
     * 
     * @param int $db_port
     * @param string $db_socket
     * @param int $db_use_ssl
     * @param string $db_ssl_ca
     * @return boolean
     */
    public function automatedInstall($agree_term, $db_host, $db_database, $db_username, $db_password, $site_domain_name, $site_use_https, $site_httpbase, $site_name, $site_root_file_path, $site_id, $email_from, $email_address, $admin_username, $admin_password, $sampledata = true, $use_wildcard = false, $timezone = 'UTC', $use_www = false, $db_port = NULL, $db_socket  = NULL, $db_use_ssl  = NULL, $db_ssl_ca = NULL) 
    {
            
            $site_httpbase = remove_multiple_slashes($site_httpbase);
            
            $error_list = array();
            ///////////////////////////////////////////////////////////////////////////////
            //1. License Agreement
            ///////////////////////////////////////////////////////////////////////////////
            if (!$agree_term) {
                throw new Exception("You must agree to the GNU GPL v3.0 license before continuing");
            }
            ///////////////////////////////////////////////////////////////////////////////
            //2. Requirement
            ///////////////////////////////////////////////////////////////////////////////
            $requirement = $this->checkSystemRequirement();
            if ($requirement['error_list']) {
                $error_list = $requirement['error_list'];
                $str_error_list = implode(',', $error_list);
                throw new Exception("Requirement not met: ".$str_error_list);
            }
            ///////////////////////////////////////////////////////////////////////////////
            //3. Database
            ///////////////////////////////////////////////////////////////////////////////
            $db_test_result = $this->testDatabase($db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca);
            if ($db_test_result['status'] === false) {
                throw new Exception('Cannot connect to the database :'.$db_test_result['error']);
            } 
            ///////////////////////////////////////////////////////////////////////////////
            // 4. check SiteName, Site ID, Database
            ///////////////////////////////////////////////////////////////////////////////
            $lowercase_site_id = strtolower($site_id);
            if (alpha_numeric_only($lowercase_site_id) != $lowercase_site_id)
                throw new Exception('Site ID must not contain spaces or any special character. Only alphanumeric characters are accepted');
            if (filter_var($email_address, FILTER_VALIDATE_EMAIL) === false) {
                throw new Exception('Invalid email address');
            }
            ///////////////////////////////////////////////////////////////////////////////
            // 5. Write multisite-config.inc.php
            ///////////////////////////////////////////////////////////////////////////////
            $multisite_config_data = $this->generateMultisiteConfigFileContent($lowercase_site_id, $site_domain_name, $use_wildcard, $use_www);
            $multisite_config_write_result = file_put_contents($site_root_file_path . '/multisite-config.inc.php', $multisite_config_data);
            if ($multisite_config_write_result === false) {
                throw new Exception('Cannot write to multisite-config.inc.php');
            }
            ///////////////////////////////////////////////////////////////////////////////
            // 6. create site skeleton
            ///////////////////////////////////////////////////////////////////////////////
            if (!create_site_skeleton($site_root_file_path, $lowercase_site_id)) {
                throw new Exception("Cannot create subsite folder " . $lowercase_site_id);
            }
            ///////////////////////////////////////////////////////////////////////////////
            // 7. Write config.inc.php
            ///////////////////////////////////////////////////////////////////////////////
            $config_data = $this->generateSiteConfigFileContent($site_name, $db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca, 3600 * 6, 86400 * 3, $email_from, $email_address, $timezone, 0, $site_use_https);
            $config_file_location = $site_root_file_path . '/web/' . $lowercase_site_id . '/config.inc.php';

            $config_save_result = file_put_contents($config_file_location, $config_data);
            if (!$config_save_result) {
                throw new Exception("Cannot save subsite configuration file");
            }
            /////////////////////////////////////////////////////////////////////////////////
            // 8. Initialize database
            ///////////////////////////////////////////////////////////////////////////////
            $db_init_result = $this->initializeDatabase($db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca);
            if ($db_init_result['status'] == false) {
                throw new Exception("Error while initializing database: ". implode(',', $db_init_result['messages']));
            }
            ///////////////////////////////////////////////////////////////////////////////
            // 9. Run install_step1.sql
            ///////////////////////////////////////////////////////////////////////////////
            $install_sql = file($site_root_file_path . '/system/apps/sitemanager/sql/install_step1.sql', true);
            if (!$install_sql) {
                throw new Exception('Cannot load installation SQL');
            }
            $install_sql_step2 = file($site_root_file_path . '/system/apps/sitemanager/sql/install_step2.sql', true);
            if (!$install_sql_step2) {
                throw new Exception('Cannot load installation SQL Step 2');
            }

            global $SystemLog;
            
            $SystemDB = new \SCHLIX\cmsDatabase($db_host, $db_database, $db_username, $db_password, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca);
            $SystemLog = new \SCHLIX\cmsLogger('gk_log_items');

            $this->execsql($install_sql, $SystemDB, FALSE);
            $this->execsql($install_sql_step2, $SystemDB, FALSE); // separate it. Install.SQL is run on upgrade
            //$salt = $this->generateRandomPasswordSalt('blowfish');
            //$hashed_password = crypt($admin_password, $salt);
            
            $salt = ''; // 2019
            $hashed_password = password_hash($admin_password, PASSWORD_DEFAULT);
            
          //  $sanitized_email = $SystemDB->sanitizeString($email_address);
//            $sanitized_admin_username = $SystemDB->sanitizeString($admin_username);
            //$sanitized_password = $SystemDB->sanitizeString($hashed_password);
            //$sanitized_salt = $SystemDB->sanitizeString($salt);
            // change password, add salt
            //$SystemDB->query("UPDATE gk_user_items set `username` = {$sanitized_admin_username}, `password` = {$sanitized_password}, `salt`={$sanitized_salt}, email_address = {$sanitized_email} WHERE id = 1"); 
            $SystemDB->query("UPDATE gk_user_items set `username` = :admin_username, `password` = :hashed_password, `salt`= :salt, email_address = :email_address WHERE id = 1",
                    ['admin_username' => $admin_username, 'hashed_password' => $hashed_password, 'salt' => $salt, 'email_address' => $email_address]); 
            ///////////////////////////////////////////////////////////////////////////////
            // 10. Sample Data, yes or no? In the future we should have several sample data
            ///////////////////////////////////////////////////////////////////////////////
            // as of January 2020 (v.2.2.1-x), there will be more than 1 sample data and more will be added
            $sqlfname = $site_root_file_path.$this->getSQLInstallScriptForSampleData($sampledata);

            $install_sql = file($sqlfname, true);
            if (!$install_sql) {
                throw new Exception('Cannot load sample data SQL');
            }

            $new_images_path = remove_multiple_slashes($site_httpbase . '/web/' . $lowercase_site_id. '/media/images/');
            $new_images_replacement = 'src="' . $new_images_path;
            
            $install_sql = str_replace('src="system/images/', $new_images_replacement, $install_sql);
            $install_sql = str_replace('src="/src/web/test/media/images/', $new_images_replacement, $install_sql);


            $this->execsql($install_sql, $SystemDB, FALSE);

            $menu_site_httpbase = $site_httpbase;
            if ($menu_site_httpbase == '/')
                $menu_site_httpbase = ''; // reset it, so just / will be ignored
            if ($menu_site_httpbase != '')
            {
                if (!str_ends_with($menu_site_httpbase, '/'))
                    $menu_site_httpbase.='/'; // add trailing slash for menu only
                // March 2, 2017 - in reponse to Softaculous tester. 
                // Menu will be reset after first run again
                $SystemDB->query("update `gk_menu_items` set sefurl = concat(:menu_site_httpbase,sefurl) where `application` NOT IN ('menu_divider','menu_placeholder','external_link')", ['menu_site_httpbase' => $menu_site_httpbase]);
                $SystemDB->query("update `gk_menu_items` set sefurl = :menu_site_httpbase where `application` = 'home'", ['menu_site_httpbase' => $site_httpbase]);
            }
            $SystemDB->query("update `gk_contact_items` set email = :email_address", ['email_address' => $email_address]);
            
            // use the same password as admin for demo user - sample installation
            $SystemDB->query("UPDATE gk_user_items set `password` = :hashed_password, `salt`= :salt WHERE id = 2",['hashed_password' => $hashed_password, 'salt' => $salt ]); // change password, add salt            
            $SystemDB->query("INSERT INTO `gk_config` (`section`, `key`, `value`, `date_modified`, `modified_by_id`) VALUES ('system', 'str_system_id', :uuid, NOW(), 0)", ['uuid' => new_uuid_v4()]); 
        
            include ( $site_root_file_path.'/runtime.inc.php');
            $schlix_version = SCHLIX_VERSION;            
            $SystemDB->query("INSERT INTO `gk_config` (`section`, `key`, `value`, `date_modified`, `modified_by_id`) VALUES ('system', 'str_database_version', :schlix_version, NOW(), 0)",['schlix_version' => SCHLIX_VERSION]);

            $this->copySampleData($site_root_file_path, $lowercase_site_id);
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
            @__del_tree($site_root_file_path . '/install'); 
            ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        return true;
    }

    //_________________________________________________________//
    public function Run($command) {
        $action = isset($command['action']) ? $command['action'] : null;
        switch ($action) {
            case 'testdatabase': $this->ajaxTestDatabase();
                break;
            case 'requirement': $this->stepRequirementAndDBSetup();
                break;
            case 'variables': $this->stepSiteVariablesSetup();
                break;
            case 'database': $this->stepDBDataSampleAndUsersSetup();
                break;
            case 'finish': $this->stepFinishSetup();
                break;
            case 'welcome':
            default: $this->step1WeclomePage();
        }
    }

}
