<?php
namespace App;

define('UPDATE_FILENAME', 'tmp_schlix_update.zip');
define('BACKUP_DEFAULT_SQL', 'tmp_backup_default_tables.sql');

/**
 * Core: SCHLIX Updater - Admin class
 * 
 * Core - SCHLIX Updater
 *
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @version 1.0
 * @package core
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class Core_SCHLIXUpdater_Admin extends \SCHLIX\cmsAdmin_Basic {


    /**
     * Old version
     * @var string 
     */
    protected $old_version;
    
    /**
     * Updater App
     * @var \App\SCHLIXUpdater 
     */
    protected $app;

    public function __construct() {
        // Data: Item
        $methods = array('standard_main_app' => 'Main Page',);

        parent::__construct('basic', $methods);
        $this->old_version = SCHLIX_VERSION;
    }

    /**
     * Download file with CURL
     * @param string $file_source
     * @param string $file_target
     * @return boolean
     */
    public function downloadWithCurl($file_source, $file_target) {
        
        $data = get_remote_file_content($file_source);
        if ($data)
        {
            file_put_contents($file_target, $data);
            return true;
        }
        return false;
    }

    /**
     * Get latest CMS version
     * @return array
     */
    public function getLatestCMSUniversalVersion() {
        
    
        $update_domain = "https://www.schlix.com";
        $str_content = get_remote_file_content("{$update_domain}/cms-services/action/get-cms-regular-latestversion");
        if ($str_content) {
            $jsdata = json_decode($str_content, true);
            $path = parse_url($jsdata['data']['url'], PHP_URL_PATH);
            if ($path)
            {
                $filename_only = pathinfo($path, PATHINFO_FILENAME);
                $real_version=  str_replace('schlix-cms-v', '', $filename_only);
                $jsdata['data']['version'] = $real_version;
                return $jsdata['data'];
            } else return NULL;
        }
        return null;
    }

    /**
     * Download update, returns JSON response in array form
     * @return array
     */
    public function ajaxDownloadUpdate() {
        //	$_SESSION['download_size'] = 0;
        //	$_SESSION['downloaded'] = 0;
        $errormsg = '';
        $this->app->ensureDataDirectoryExists();
        check_csrf_halt_on_error(true);

        $tmp_dir = $this->app->getDataDirectoryFullPath('tmp');
        if (!is_writable($tmp_dir))
            $errormsg = sprintf(___('Update directory %s is not writable'),$tmp_dir);
        $temp_file = $tmp_dir . '/' . UPDATE_FILENAME;

        $download_data = $this->getLatestCMSUniversalVersion();        
        $updateurl = $download_data['url'];
        if ($errormsg == '') {
            $download_result =  $this->downloadWithCurl($updateurl, $temp_file);
            if ($download_result)
            {
                $this->recordLog('Finished downloading '.$updateurl);
                return ajax_reply(200, array('status' => true, 'message' => 'OK'));
            }
            else
            {
                $this->recordLog('Error while downloading '.$updateurl);
                return ajax_reply(300, ___('Cannot download file'));
            }
        } else
            return ajax_reply(300, $errormsg);
    } 
    /**
     * Run the update. returns JSON response in array form
     * @return array
     */
    public function ajaxRunUpdate() {
        check_csrf_halt_on_error();

        $error_list = [];
        $update_message = [];
        $tmp_dir = $this->app->getDataDirectoryFullPath('tmp');
        $update_file_name = $tmp_dir . '/' .  UPDATE_FILENAME;
        $zip = new \PclZip($update_file_name);
        $targetpath = $tmp_dir . '/';
        @set_time_limit(10 * 60); // 10 minutes

        if (!is_file($update_file_name) || !is_readable($update_file_name))
            $error_list [] = sprintf(___('Update file %s not readable'),$update_file_name);
        if (is_dir($targetpath) && is_writable($targetpath)) {
            $newtargetpath = $targetpath . 'schlix/';
            if ($zip->extract(PCLZIP_OPT_PATH, $newtargetpath, PCLZIP_OPT_REMOVE_PATH, 'schlix') == 0)
                $error_list [] = $zip->errorInfo(true);
            
            $update_message[] = ___('Trying to upgrade').' '.SCHLIX_ROOT_PATH;
            if (is_writable(SCHLIX_SYSTEM_PATH) && is_writable(SCHLIX_ROOT_PATH))
            {
                ///////////////////////////////////////////////////////////////
                // begin update
                ///////////////////////////////////////////////////////////////
                $schlix_target_folder = SCHLIX_ROOT_PATH; 
                $update_message[] = ___('Copying system folder');
                copy_recursive($newtargetpath.'/system', $schlix_target_folder.'/system');
                $iterator = new \DirectoryIterator($newtargetpath);
                foreach ($iterator as $fileinfo) {
                    if ($fileinfo->isFile()) {
                        $filename_only = $fileinfo->getFileName();
                        $filename_fullpath = $fileinfo->getPathname();
                        if ($filename_only != '.htaccess')
                        {
                            $update_message[] = sprintf(___('Copying %s'), $filename_only);
                            copy ($filename_fullpath, $schlix_target_folder.'/'.$filename_only);
                        }
                    }
                }
                ///////////////////////////////////////////////////////////////
                // end update
                ///////////////////////////////////////////////////////////////
            } else
            {
                $error_list[] = sprintf(___('Main directory %s and system directory %s must be writable by the same user as the one running current script'), SCHLIX_ROOT_PATH, SCHLIX_SYSTEM_PATH);
            }
        } else
            $error_list[] = sprintf(___('Temporary directory %s is not writable'), $targetpath);
        // 1) Set Max Execution time
        if ($update_message)
        {
            foreach ($update_message as $msg)
                $this->recordLog('Update process: '.$msg);                
        }
        if ($error_list)
        {
            foreach ($error_list as $msg)
                $this->recordLog('Update error: '.$msg);                
        }
        return ajax_reply(200, ['messages' => $update_message, 'errors' => $error_list]);
    }
    
    /**
     * Post update script
     * @return type
     */
    public function ajaxUpgradeSystemLanguages()
    {
        check_csrf_halt_on_error();
        
        $error_langs = [];
        
        $admin_app = new \App\Admin_Admin();
        $langs = $admin_app->getInstalledLanguages();
        if (is_array($langs))
        {
            foreach ($langs as $lang)
            {
                $result = $admin_app->installSystemLanguagePack($lang, true); // force reinstall
                if ($result !== TRUE)
                {
                    if (is_array($result))
                    {
                        foreach ($result as $err)
                            $error_langs[] = $err;                
                    } else $error_langs[] = $lang;
                }
            }
            // DO NOT TRANSLATE            
            $str_langs = implode(', ',$langs);
            $str_lang_errors = implode (', ', $error_langs);
            if (!$str_lang_errors)
                $str_lang_errors = 'No error';
            // DO NOT TRANSLATE            \
            $this->recordLog("Upgraded languages: {$str_langs}. Error(s): {$str_lang_errors}");
            return ajax_reply(200,  ['messages' => [$str_langs], 'errors' => implode (',', $error_langs)]);
        } else
        return ajax_reply(200, ['messages' => ['OK'], 'errors' => []]);
         
    }    

    /**
     * Post update script
     * @return type
     */
    public function ajaxUpdateScript()
    {
        check_csrf_halt_on_error();
        $SiteManager = new \App\SiteManager();
        $SiteManagerAdmin = new \App\SiteManager_Admin();
        
        $post_upgrade_script = $this->app->getDataFileFullPath('tmp','schlix/install/upgrade.php');
        if (file_exists($post_upgrade_script))
        {
            
            require_once($post_upgrade_script);
            $all_sites = $SiteManager->getAllItems('*');
            if ($all_sites)
            {
                foreach ($all_sites as $site)
                {
                    $configvar = $SiteManagerAdmin->parseSiteConfigFile($site['title']);
                    $upgrade_result = upgrade_schlix($this->old_version, $configvar['SCHLIX_DB_HOST'], $configvar['SCHLIX_DB_DATABASE'], $configvar['SCHLIX_DB_USERNAME'], $configvar['SCHLIX_DB_PASSWORD'], $configvar['SCHLIX_DB_PORT'],
                            $configvar['SCHLIX_DB_SOCKET'], $configvar['SCHLIX_DB_USE_SSL'], $configvar['SCHLIX_DB_SSL_CA']);
                }
            } else
            {
                $db_port = 0;
                $db_socket = '';
                $db_use_ssl = false;
                $db_ssl_ca = '';
                if (defined('SCHLIX_DB_PORT') && (strpos(SCHLIX_DB_HOST,':') === FALSE))
                {

                    $port = (int) defined('SCHLIX_DB_PORT') ? SCHLIX_DB_PORT : 0;
                    if ($port > 0 && $port < 65536)
                        $db_port = $port;
                }
                // socket
                if (defined('SCHLIX_DB_SOCKET') && !empty(SCHLIX_DB_SOCKET) )
                {
                    $db_socket = SCHLIX_DB_SOCKET;
                }
                // use SSL
                if (defined('SCHLIX_DB_USE_SSL')  )
                {
                    $db_use_ssl = (bool) SCHLIX_DB_USE_SSL;
                    if (defined('SCHLIX_DB_SSL_CA')  )
                    {
                        $db_ssl_ca = SCHLIX_DB_SSL_CA;
                    }
                }    
                
                // just upgrade the local (the user has never visited Site Manager yet
                 $upgrade_result = upgrade_schlix($this->old_version, SCHLIX_DB_HOST, SCHLIX_DB_DATABASE, SCHLIX_DB_USERNAME, SCHLIX_DB_PASSWORD, $db_port, $db_socket, $db_use_ssl, $db_ssl_ca);
            }
        } else             
        {
            return ajax_reply(300,"Could not find update file {$post_upgrade_script}");
        }
        $this->recordLog('Finished upgrading');
        __del_tree($this->app->getDataDirectoryFullPath('tmp'));
        return ajax_reply(200,  $upgrade_result);
    }
    

    public function Run() {
        switch (fget_string('action')) {
            case 'downloadupdate':
                return ajax_echo($this->ajaxDownloadUpdate());
                break;            
            case 'runupdate':
                return ajax_echo($this->ajaxRunUpdate());
                break;
            case 'updatescript':
                return ajax_echo($this->ajaxUpdateScript());
                break;
            case 'updatelang':
                return ajax_echo($this->ajaxUpgradeSystemLanguages());
                break;
            
            default: return parent::Run();
        }
    }
}
            