<?php
namespace App;
/**
 * Personal Data Request - Main Class
 * 
 * Plugin for compliance with GDPR
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package personaldatarequest
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class PersonalDataRequest extends \SCHLIX\cmsApplication_List {

    protected $data_directories = ['archives' => '/data/private/gdpr'];
    /**
     * Constructor
     * @global \SCHLIX\cmsDatabase $SystemDB
     */
    public function __construct() {
        global $SystemDB;
        
        parent::__construct("GDPR", 'gk_gdpr_items');
        /* You can modify this  */
        $this->has_versioning = true; // set to false if you don't need versioning capability if this app wants versioning enabled
        $this->disable_frontend_runtime = false; //  set this to true if this is a backend only app         
        $this->has_custom_media_header = true;
        
    }

    /**
     * View Main Page
     * @param int $pg
     * @return boolean
     */
    //_______________________________________________________________________________________________________________//
    public function viewMainPage($pg = 1) {

        // Set Page Title
        $str_page_title = $this->getConfig('str_mainpage_title', true);
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('view.main', $local_variables);
    }
    

    //_______________________________________________________________________________________________________________//
    public function getMainPageMetaOptionKeys() {
        return array(
          array('value' => 'display_error_no_access', 'label' => ___('Display errors for inaccesible item')),
          array('value' => 'display_items', 'label' => ___('Display list of items')),
          array('value' => 'display_item_summary', 'label' => ___('Display item\'s summary')),
          array('value' => 'display_item_created_by', 'label' => ___('Display item\'s created by')),
          array('value' => 'display_item_date_created', 'label' => ___('Display item\'s date created')),
          array('value' => 'display_item_date_modified', 'label' => ___('Display item\'s date modified')),
          array('value' => 'display_item_read_more_link', 'label' => ___('Display item\'s "Read More" link')),
          array('value' => 'display_item_view_count', 'label' => ___('Display item\'s view count')) 
        );
    }
       
    
    /**
     * Validates save item. If there's an error, it will return an array
     * with one or more error string, otherwise it will return a boolean true
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return bool|array String array
     */
    public function getValidationErrorListBeforeSaveItem($datavalues)
    {
        $parent_error_list = parent::getValidationErrorListBeforeSaveItem($datavalues);
        $error_list = array();
        // You can put custom item save validation here
        // e.g. 
        // if (str_contains($datavalues['title'],'a'))
        // $error_list[] = ___('Title may not contain letter a');
        return array_merge($parent_error_list, $error_list);
    }

    /**
     * Before save item
     * @param array $datavalues
     * @return array
     */
    public function onModifyDataBeforeSaveItem($datavalues) {
        $datavalues = parent::onModifyDataBeforeSaveItem($datavalues);
        // You can customize the data values before it's saved here
        // e.g.
        // $datavalues['title'] = real_strip_tags($datavalues['title']);
        //
        return $datavalues;
    }
    

    /**
     * Load default JS and CSS required for frontend
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     */
    public function loadDefaultStaticAssetFiles()
    {
        global $HTMLHeader;
        
        $HTMLHeader->JAVASCRIPT_SCHLIX_UI();
        $HTMLHeader->JAVASCRIPT_SCHLIX_CMS();
        $this->JAVASCRIPT('personaldatarequest.js');
    }
    
    
    
    /**
     * Do something after save item
     * @param array $datavalues
     * @param array $original_datavalues
     * @param array $previous_item
     * @param array $retval
     */
    protected function onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval)
    {
        parent::onAfterSaveItem($datavalues, $original_datavalues, $previous_item, $retval);
    }

    /**
     * 
     * @global \App\Core_EmailQueue $SystemMail
     * @param string $email
     * @param string $template_name
     * @param string $email_vars
     */
    protected function sendEmail($email, $template_name, $email_vars)
    {
        global $SystemMail;
        
        if (is_valid_email_address($email))        
        {
            if (!$SystemMail->sendEmailFromWebMasterWithTemplateName($email, $email, $template_name,  $email_vars)) {
                $error = sprintf(___('SMTP/Local mail send failed for GDPR messaging. Unable to send an email to %s'), $email);
                display_schlix_alert($error);
                $this->logError($error);
            }
            
        } else
        {
            $this->logError("Unable to send GDPR email because the email address {$email} is invalid");
        }
    }
    
    private function getActivityReportByEmailOrUserID($email, $user_id, $request_type, $reason = '')
    {
        global $SystemDB;
        
        $guid = new_uuid_v4();
        $config_valid_for = $this->getConfig('int_validity', 30);
        $expiry = date(DEFAULT_DATE_FORMAT, strtotime("+{$config_valid_for} days"));
        $auth_code = mt_rand(10000, 1000000);
        $data = ['guid' => $guid, 'date_created' => get_current_datetime(), 'user_id' => $user_id, 'email_address' => $email, 'status' => -2, 'request_ip_address' => get_user_real_ip_address(), 'request_user_agent' => get_user_browser_string(), 'request_type' => $request_type, 'date_expiry' => $expiry, 'auth_code' => $auth_code, 'reason' => strip_tags($reason)];
        $this->table_items->quickInsert($data);
        $applications = new \App\Core_ApplicationManager();
        $apps = $applications->getAllItems();
        $user_id = (int) $user_id;
        $all_result = [];
        if ($apps)
        foreach ($apps as $app)
        {
            if (strtolower($app['title']) !== 'personaldatarequest')
            {
                $full_app_name = '\\App\\'.$app['title'];
                if (class_exists($full_app_name))
                {
                    $app_regular = new $full_app_name();
                    $alias = $app_regular->getFullApplicationAlias();
                    $result = ($user_id > 0) ? $app_regular->getPersonalDataByUserID($email) : $app_regular->getPersonalDataByEmail($email);
                    if (___c($result) > 0)
                        //$all_result = (___c($all_result) > 0) ? array_merge($all_result, $result) : $result;
                        $all_result[$alias] = $result;
                    $sub_apps = $applications->getAllSubApplications($app['title']);
                    if ($sub_apps)
                    foreach ($sub_apps as $sub_app)
                    {
                        $sub_app_full_app_name = '\\App\\'.$app['title'].'_'.$sub_app;
                        if (class_exists($sub_app_full_app_name))
                        {
                            $sub_app_regular = new $sub_app_full_app_name();
                            $alias = $app_regular->getFullApplicationAlias();
                            $result = ($user_id > 0) ? $sub_app_regular->getPersonalDataByUserID($email) : $sub_app_regular->getPersonalDataByEmail($email);
                            if (___c($result) > 0)
                                //$all_result = (___c($all_result) > 0) ? array_merge($all_result, $result) : $result;
                                $all_result[$alias] = $result;
                            
                        }  
                    }
                }
            }
        }
        if (___c($all_result) > 0)
        {

            $dirpath = $this->getDataFileFullPath('archives', $guid);
            if (!is_dir($dirpath)) {
                if (!create_directory_if_not_exists($dirpath)) 
                    $this->logError( sprintf( ___('Fatal error: unable to create folder [%s]'), $dirpath));
                else
                    $this->logInfo(sprintf( ___('New folder [%s] has been created'), $dirpath));
                if (!is_writable($dirpath))
                    $this->logError ( sprintf( ___('Fatal error: Folder [%s] for category ID %s is not writable'), $dirpath, $guid) );
                else 
                {
                    foreach ($all_result as $app_name => $data)
                    {
                        $json_result = json_encode($data, JSON_PRETTY_PRINT | JSON_OBJECT_AS_ARRAY);
                        file_put_contents("{$dirpath}/{$app_name}.json", $json_result);
                    }
                    $dt = date('Y-m-d_H-i-s');
                    $archive_name = $email ? "{$email}-{$dt}.zip" : "user-{$user_id}-{$dt}.zip";
                    
                    $archive = new \PclZip($dirpath.'.zip');
                    $new_zip = $archive->create("{$dirpath}",PCLZIP_OPT_REMOVE_PATH, $this->getDataDirectoryFullPath('archives'));
                    
                    if ($new_zip == 0) {
                        $this->logError("Error while creating archive: ".$archive->errorInfo(true));                        
                    } else {
                        $sanitized_guid = sanitize_string($guid);
                        $data = ['status' => 0, 'filename' => $archive_name];
                        $this->table_items->quickUpdate($data, "guid = {$sanitized_guid}");
                        $fsize = number_format(filesize(realpath($dirpath.'.zip')));
                        $this->logInfo("Created GDPR report {$dirpath}.zip successfully. Total Size = {$fsize}");                        
                    }
                    __del_tree($dirpath);
                    $email_template = ($request_type == 1) ? 'gdpr-request-verification' : 'gdpr-removal-verification';
                    $url_confirmation = SCHLIX_SITE_URL.$this->createFriendlyURL("action=confirm&guid={$guid}");
                    $this->sendEmail($email, $email_template,['email_address' => $email, 'ip_address' => get_user_real_ip_address(), 'url' => $url_confirmation, 'auth_code' => $auth_code]);
                    // do not notify the user in case of non-existant data                    
                }
            }
            return true;
        }
        return false;
    }
    
    public function getActivityReportByEmail($email, $request_type, $reason)
    {
        return $this->getActivityReportByEmailOrUserID($email, 0, $request_type, $reason);
    }
    
    public function getActivityReportByUserID($user_id, $request_type, $reason)
    {
        return $this->getActivityReportByEmailOrUserID('', $user_id, $request_type, $reason);
    }
    
    private function removeAllPersonalDataByEmailOrUserID($email, $user_id, $request_guid)
    {
        global $SystemDB;
        
        $applications = new \App\Core_ApplicationManager();
        $apps = $applications->getAllItems();
        $user_id = (int) $user_id;
        $all_result = [];
        if ($apps)
        foreach ($apps as $app)
        {
            if (strtolower($app['title']) !== 'personaldatarequest')
            {
                $full_app_name = '\\App\\'.$app['title'];
                if (class_exists($full_app_name))
                {
                    $app_regular = new $full_app_name();
                    $alias = $app_regular->getFullApplicationAlias();
                    $result = ($user_id > 0) ? $app_regular->removePersonalDataByUserID($email, $request_guid) : $app_regular->removePersonalDataByEmail($email, $request_guid);
                    
                    $sub_apps = $applications->getAllSubApplications($app['title']);
                    if ($sub_apps)
                    foreach ($sub_apps as $sub_app)
                    {
                        $sub_app_full_app_name = '\\App\\'.$app['title'].'_'.$sub_app;
                        if (class_exists($sub_app_full_app_name))
                        {
                            $sub_app_regular = new $sub_app_full_app_name();
                            $alias = $app_regular->getFullApplicationAlias();
                            $result = ($user_id > 0) ? $sub_app_regular->removePersonalDataByUserID($email, $request_guid) : $sub_app_regular->removePersonalDataByEmail($email, $request_guid);
                           
                            
                        }  
                    }
                }
            }
        }
    }
    
    
    /**
     * Returns the error list of request validation
     * @param string $email
     * @param int $request_type
     * @return array
     */
    protected function validateEmailAndRequestType($email, $request_type)
    {
        if (empty($email))
            $error_list[] = ___('Email address cannot be empty');
        else if(!is_valid_email_address($email))
            $error_list[] = ___('Invalid email address');
        if ($request_type > 2 || $request_type < 1)
            $error_list[] = ___('Invalid request type');
        rate_limit_record('gdpr', sprintf( 'GDPR Request for %s', $email));
        $limit = $this->getConfig('int_rate_limit', 1);
        if (rate_limit_exceeded('gdpr', $limit, 3600 * 24)) // only 1 request per day is allowed
        {
            $error_list[] = ___('Rate limit has been exceeded. Only one request per day is allowed');
        }
        return $error_list;
    }
    
    /**
     * AJAX POST handler for action: Request
     * @return array
     */
    public function ajxp_Request()
    {
        $error_list = [];
        check_csrf_halt_on_error();
        $email = fpost_string('email_address');        
        //$email = 'info@schlixcmsdev.com';
        $request_type = fpost_int('request_type');
        $show_captcha = $this->getConfig('bool_enable_captcha_request');        
        $error_list = $this->validateEmailAndRequestType($email, $request_type);
        if ($show_captcha && !is_captcha_verification_valid()) 
           $error_list[] = ___('Invalid verification code');
        $show_removal_reason = $this->getConfig('bool_ask_for_removal_reason');
        $request_to_delete = ($show_removal_reason && $request_type == 2);
        $reason = $request_to_delete ? fpost_string('reason', 1024) : '';
        if ($request_to_delete && empty($reason))
            $error_list[] = ___('Please provide a reason for your request for data removal');
        
        if (empty($error_list))
        {            
            $result = $this->getActivityReportByEmail($email, $request_type, $reason);
            $msg = ___('A report is being generated. If such record exists, a confirmation has been sent to your email. Once you have received it, please click the link to verify the request. If the report contains no data, you will not receive any email');
            return ajax_reply_ok(['messages' => [$msg], 'hide_form' => true]);
        } 
        return ajax_reply_error(['messages' => $error_list]);
    }

    /**
     * Returns a record by GUID and auth code
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $guid
     * @param int $auth_code
     * @return array
     */
    protected function getRequestByGUIDAndAuthCode($guid, $auth_code)
    {
        global $SystemDB;
        
        $cur_dt = get_current_datetime();
        $result = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE status = 0 AND auth_code = :auth_code AND guid = :guid AND TIME_TO_SEC(TIMEDIFF(:current_date_time, date_expiry)) < 0", ['auth_code' => $auth_code, 'guid' => $guid, 'current_date_time' => $cur_dt]);
        return $result;
    }
    
    /**
     * Returns an array of data request if the request type = 1 (data request), not delete/removal
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $guid
     * @return array
     */
    protected function getConfirmedDataRequestOnlyByGUID($guid)
    {
        global $SystemDB;
        
        $cur_dt = get_current_datetime();
        $result = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE status = 2 AND request_type = 1 AND guid = :guid AND TIME_TO_SEC(TIMEDIFF(:current_date_time, date_expiry)) < 0", [ 'guid' => $guid, 'current_date_time' => $cur_dt]);
        return $result;
    }

   
    /**
     * Confirm user request
     * @param int $id
     */
    protected function confirmRequest($id)
    {
        $item = $this->getItemByID($id);
        if ($item)
        {
            $item['status'] = 2;
            unset($item['id']);
            $this->table_items->quickUpdate($item, "id = {$id}");
            $email_vars = [];
            $email_template = '';
            if ($item['request_type'] == 2)
            {
                // if it's a delete request
                $this->removeAllPersonalDataByEmailOrUserID($item['email_address'], $item['user_id'], $item['guid']);
                $email_template = 'gdpr-removal-confirmation';
                $email_vars = ['email_address' => $item['email_address'], 'ip_address' => get_user_real_ip_address()];
            } else 
            {
                // if it's a data request
                $email_template = 'gdpr-request-confirmation';
                $url_download = SCHLIX_SITE_URL.$this->createFriendlyURL("action=download&guid={$item['guid']}");
                $email_vars = ['email_address' => $item['email_address'], 'ip_address' => get_user_real_ip_address(), 'url' => $url_download];
            }
            $this->sendEmail($item['email_address'], $email_template, $email_vars);
            
        }
    }
    
    /**
     * AJAX POST handler for action: Request
     * @return array
     */
    public function ajxp_ConfirmRequest()
    {
        $error_list = [];
        check_csrf_halt_on_error();
        $show_captcha = $this->getConfig('bool_enable_captcha_confirm');
        $guid = fpost_string('guid');
        $auth_code = fpost_int('auth_code');        
        if ($auth_code <= 0)
            $error_list[] = ___('Invalid authentication code');        
        if ($show_captcha && !is_captcha_verification_valid()) 
           $error_list[] = ___('Invalid verification code');
        if (!is_valid_guid($guid))
            $error_list[] = ___('Invalid request code');
        
        if (empty($error_list))
        {
            $result = $this->getRequestByGUIDAndAuthCode($guid, $auth_code);
            if ($result)
            {
                $this->confirmRequest($result['id']);
                $msg = ___('Your request has been confirmed. A notification has been sent to your email address');
                return ajax_reply_ok(['messages' => [$msg], 'hide_form' => true]);                
            } else 
            {
                return ajax_reply_error(['messages' => [___('Unable to find a valid record. The authentication code has been used, or it is invalid or the record has expired')], 'hide_form' => true]);
            }
        } 
        return ajax_reply_error(['messages' => $error_list]);
    }
    

    /**
     * View Confirm Page
     * @param int $pg
     * @return boolean
     */
    //_______________________________________________________________________________________________________________//
    public function viewConfirmPage($command) {

        // Set Page Title
        $this->setPageTitle(___('Confirm'));
        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('view.confirm', $local_variables);
    }
    
    
    public function downloadArchive($command)
    {
        $item = $this->getConfirmedDataRequestOnlyByGUID($command['guid']);
        if ($item)
        {
            $filename = $item['filename'];
            $filename = $item['guid'].'.zip';
            $file = $this->getDataFileFullPath('archives', $filename);
            force_download_file($file, $item['filename']);
        } else die('Unable to find report archive');
    }

    public function viewItem()
    {
        $this->redirectToOtherAction('');
    }
    /**
     * Run Command
     * @param array $command
     * @return boolean
     */
    public function Run($command) {
        $this->loadDefaultStaticAssetFiles();
        switch ($command['action']) {
            case 'download':
                return $this->downloadArchive($command);
            case 'confirm':
                $this->viewConfirmPage($command);return true;
                break;
            case 'main': $this->viewMainPage($command['pg']);
                return true;
                break;
            default: return parent::Run($command);
        }
    }



}
