<?php
namespace App;
/**
 * Core: Editor Manager - Main Class
 * 
 * Core - Editor Manager
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package core
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class Core_EditorManager extends \SCHLIX\cmsApplication_CategorizedList {

    /**
     * The default editor
     * @var \SCHLIX\cmsWysiwygEditor 
     */
    protected $default_editor;
    /**
     * Override the limited profile ID
     * @var int 
     */
    protected $override_limited_profile_id;
    /**
     * Override the full profile ID
     * @var int 
     */    
    protected $override_full_profile_id;
    
    public function __construct() {
        parent::__construct('Editor Manager', 'gk_wysiwyg_profiles', 'gk_wysiwyg_categories');
        $this->schema_org_type_item = '';
        
        /* You can modify this  */
        $this->has_versioning = false; // set to false if you don't need versioning capability if this app wants versioning enabled
        $this->disable_frontend_runtime = false; //  set this to true if this is a backend only app         
        
    }

    //_______________________________________________________________________________________________________________//
    /**
     * Returns duplicate item, given item title
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param array $data
     * @return array
     */
    public function findDuplicateCategories($data) {
        global $SystemDB;

        $current_id = (int) $data[$this->field_id];
        $the_title = sanitize_string($data['title']);
        $sql = "SELECT * from {$this->table_categories} WHERE (title = {$the_title})";

        if (intval($current_id) != 0)
            $sql .= " AND (cid != '{$current_id}')";

        $result = $SystemDB->getQueryResultArray($sql);

        return $result;
    } 
/*
    //_______________________________________________________________________________________________________________//
    public function getDefaultWYSIWYGID($type = 'frontend') {
        $data = $this->getDefaultWYSIWYG();
        return ($data != NULL) ? $data[$this->field_id] : 0;
    }
*/
    /**
     * Return the editor by virtual filename
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $virtual_filename
     * @return array
     */
    public function getWysiwygEditorByName($virtual_filename) {
        global $SystemDB;

        return $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE `title` = :virtual_filename", ['virtual_filename' => $virtual_filename]);
    } 
    
    public function getItemsByVirtualFilename($virtual_filename, $category_id = -1) {

        return $this->getWysiwygEditorByName($virtual_filename);
    } 
    
    /**
     * Loading WYSIWYG configuration script
     * @param int $profile_id
     */
    public function viewInitScript($profile_id) {
        $this->initializeMyWysiwygEditorByProfileID($profile_id);
        $this->default_editor->viewInitScript();
    }

    /**
     * Inits $this->default_editor
     * @param int $profile_id
     */
    protected function initializeMyWysiwygEditorByProfileID($profile_id) {
        $this->default_editor = self::getEditorInstanceByProfileID($profile_id);
    }


    /**
     * Override the default full provilege WYSIWYG editor. Usually called by admin class
     * with specific need for a specific editor
     * @param int $profile_id
     */
    public function overrideFullPrivilegeWysiwygEditor($profile_id)
    {
        $this->override_full_profile_id = $profile_id;
    }
    
    /**
     * Override the default limited WYSIWYG editor. Usually called by admin class
     * with specific need for a specific editor
     * @param int $profile_id
     */
    public function overrideLimitedPrivilegeWysiwygEditor($profile_id)
    {
        $this->override_limited_profile_id = $profile_id;
    }
    
    /**
     * Run the default WYSIWYG profile - backend     
     * @global \SCHLIX\cmsConfigRegistry $SystemConfig
     */
    public function runDefaultLimitedPrivilegeWysiwygEditor() {

        $profile_id = $this->override_limited_profile_id ? $this->override_limited_profile_id : $this->getConfig('int_default_limited_privilege_wysiwyg_profile_id', 2);
        $this->runWysiwygEditorByProfileID($profile_id);
    }
    
    /**
     * Run the default WYSIWYG profile - backend     
     * @global \SCHLIX\cmsConfigRegistry $SystemConfig
     */
    public function runDefaultFullPrivilegeWysiwygEditor() {

        
        $profile_id = $this->override_full_profile_id ? $this->override_full_profile_id : $this->getConfig('int_default_full_privilege_wysiwyg_profile_id', 1);
        $this->runWysiwygEditorByProfileID($profile_id);
    }

    /**
     * Run the WYSIWYG Editor by ID
     * @global \SCHLIX\cmsHTMLPageHeader $HTMLHeader
     * @global \SCHLIX\cmsConfigRegistry $SystemConfig
     * @global \SCHLIX\cmsLogger $SystemLog
     * @param int $profile_id
     */
    public function runWysiwygEditorByProfileID($profile_id) {
        global $HTMLHeader;
        $url = $this->createFriendlyURL("action=init&id={$profile_id}");

        $this->initializeMyWysiwygEditorByProfileID($profile_id);
        if ($this->default_editor) {
            $js = $this->default_editor->getHeaderVariableInitScript($url);
            if ($js)
                $HTMLHeader->JAVASCRIPT_TEXT($js);

            $HTMLHeader->JAVASCRIPT_EXTERNAL($url);
        } else {
            $this->logError('Unable to load WYSIWYG editor ID#' . $profile_id);
        }
    }

    public function spellCheck($editor_name)
    {
        
        $editor_name = alpha_numeric_with_dash_underscore($editor_name);
        $command = [ 'editor' => $editor_name, 'do' => 'spellcheck'];
        $this->callEditorFunction($command);
    }
    
    public function callEditorFunction ($command)
    {
        if ($command['editor'])
        {
            $item = $this->getWysiwygEditorByName($command['editor']);
            if ($item)
            {
                $editor_name ='\\WysiwygEditor\\'. $item['title'];
                try
                {
                    if (class_exists($editor_name))
                    {
                        $editor = new $editor_name();
                        
                        if ($editor) $editor->Run($command);
                    }
                } 
                catch (\Exception $exc) {
                    $this->recordLog("callEditorFunction: Cannot find WYSIWYG editor {$editor_name}");
                }
            } else
            {
                die ("Not a valid WYSIWYG editor class: {$command['editor']}");
            }
        }
    }
    
    /*************************************************************************/

    /**
     * Returns an editor instance
     * @param int $id
     * @return \SCHLIX\cmsWysiwygEditor
     */
    public static function getEditorInstanceByProfileID($id) {
        $editor_name = self::getClassNameByProfileID($id);

        if ($editor_name) {
            $editor_class_name = '\\WysiwygEditor\\' . $editor_name;
            return new $editor_class_name($id);
        }
        return NULL;
    }

    /**
     * Returns the editor name by profile id
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $profile_id
     * @return string
     */
    public static function getProfileNameByID($profile_id) {
        global $SystemDB;

        $row = $SystemDB->getQueryResultSingleRow("SELECT gk_wysiwyg_profiles.*, gk_wysiwyg_categories.title as editor_name FROM gk_wysiwyg_profiles INNER JOIN `gk_wysiwyg_categories` ON category_id = gk_wysiwyg_categories.cid WHERE gk_wysiwyg_profiles.id = :id", ['id' => $profile_id]);
        return $row['virtual_filename'];
    }

    /**
     * Returns the editor name by profile id
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param int $profile_id
     * @return string
     */
    public static function getClassNameByProfileID($profile_id) {
        global $SystemDB;

        if (!$SystemDB->tableExists('gk_wysiwyg_profiles')) return; // workaround for v2.1.1-5
        $row = $SystemDB->getQueryResultSingleRow("SELECT gk_wysiwyg_profiles.*, gk_wysiwyg_categories.title as editor_name FROM gk_wysiwyg_profiles INNER JOIN `gk_wysiwyg_categories` ON category_id = gk_wysiwyg_categories.cid WHERE gk_wysiwyg_profiles.id = :id", ['id' => $profile_id]);
        return $row['editor_name'];
    }

    /**
     * Return the editor by virtual filename
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $editor_name
     * @return array
     */
    public static function getAllWysiwygProfiles() {
        global $SystemDB;

        if (!$SystemDB->tableExists('gk_wysiwyg_profiles')) return; // workaround for v2.1.1-5
        return $SystemDB->getQueryResultArray("SELECT gk_wysiwyg_profiles.*, gk_wysiwyg_categories.title as editor_name FROM gk_wysiwyg_profiles INNER JOIN `gk_wysiwyg_categories` ON category_id = gk_wysiwyg_categories.cid");
    }

    /**
     * Return the editor by virtual filename
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $editor_name
     * @return array
     */
    public function getProfilesByEditorName($editor_name) {
        global $SystemDB;

        return $SystemDB->getQueryResultArray("SELECT {$this->table_items}.*, gk_wysiwyg_categories.title as editor_name FROM {$this->table_items} INNER JOIN `gk_wysiwyg_categories` ON category_id = gk_wysiwyg_categories.id WHERE gk_wysiwyg_categories.`title` = :editor_name", ['editor_name' => $editor_name]);
    }

    /**
     * Create an empty editor profile and populate it with data
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @global type $WYSIWYGEditor
     * @param string $editor_name
     * @param string $profile_code_name
     * @param string $profile_name
     * @param bool $is_system
     * @param array $config
     * @return boolean
     */
    public function createEmptyProfile($editor_name, $profile_code_name, $profile_name, $is_system, $config) {
        global $SystemDB, $WYSIWYGEditor;

        $editor = $WYSIWYGEditor->getWysiwygEditorByName($editor_name);
        if ($editor) {
            $data = ['category_id' => $editor['id'], 'title' => $profile_name, 'virtual_filename' => $profile_code_name, 'is_system' => $is_system];
            $SystemDB->query("INSERT INTO {$this->table_items} (category_id, title, virtual_filename, date_created, date_modified, is_system) VALUES (:category_id, :title, :virtual_filename, NOW(), NOW(), :is_system) ", $data);
            if (is_array($config))
            {
                $wysiwyg_config = new \SCHLIX\cmsConfigRegistry('gk_wysiwyg_config');
                $wysiwyg_config->deleteSection($profile_code_name);
                // handle checkbox save because it won't show up in post
                foreach ($config as $key => $value) {
                    $wysiwyg_config->set($profile_code_name, $key, $value);
                }
                return true;
            }
        }
        return false;
    }
    
    //_______________________________________________________________________________________________________________//
    public function Run($command) {
        switch ($command['action']) {
            case 'init':
                $this->viewInitScript($command['id']);
                return false;
                break;
            case 'spellcheck':
                return $this->spellCheck($command['editor']);
            case 'call':
                $this->callEditorFunction($command);
                return false;
            default:
                redirect_url(SCHLIX_SITE_HTTPBASE . '/');
        }

        return false;
    }
}
            