<?php
namespace App;
/**
 * Tag - Main Class
 * 
 * Tag Manager
 * 
 * @copyright 2016 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @version 1.0
 * @author  SCHLIX Web Inc <info@schlix.com>
 * @link    http://www.schlix.com
 */
class Tag extends \SCHLIX\cmsApplication_CategorizedList {

    public function __construct() {
        parent::__construct('Tag', 'gk_tag_items', 'gk_tag_categories');
        $this->schema_org_type_item = '';
        $this->hook_priority = 9300;
        $this->has_versioning = true;
        $this->has_custom_media_header = true;
    }

    /**
     * None
     * @param int $id
     * @param string
     */
    public function viewItemByID($id = 1, $from_cache = false) {
        $this->redirectToOtherAction("");
    }
    /**
     * Delete all tags with this GUID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string|array $guid
     */
    public function deleteByGUID($guid)
    {
        global $SystemDB;
        
        
        if (is_array($guid))
        {            
            for ($i = 0, $count = ___c($guid); $i < $count; $i++)
            {
                $guid[$i] = sanitize_string($guid[$i]);
            }
            $str_multiple_guids = implode(',', $guid);
            $SystemDB->query("DELETE FROM {$this->table_items} WHERE guid IN ($str_multiple_guids)");
        } else
        {
            $SystemDB->query("DELETE FROM {$this->table_items} WHERE guid = :guid", ['guid' => $guid]);
        }
        
    }
    
    /**
     * Create tags from item with specified id
     * @param \SCHLIX\cmsApplication_List $app_object
     * @param int $id
     */
    public function createTagsFromItem($app_object, $id)
    {
        $id = (int) $id;
        
        if ($app_object && is_a($app_object, '\\SCHLIX\\cmsApplication_List') && ($id > 0))
        {
            
            $item = $app_object->getItemByID($id);
            if ($item && array_key_exists('guid', $item) && array_key_exists('tags', $item))
            {
                $tags_array = $this->getArrayOfTagsFromString($item['tags']);
                if ($tags_array)
                {
                    $this->createTagsFromArray($tags_array);
                    $this->assignExistingTagsToArticleItem($tags_array, $app_object, $item);
                } else
                {
                    $this->removeAllTagsByGUID($item['guid']);
                }
            }
        }
    }
    
    /**
     * Returns all tags associated with this GUID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $guid
     * @return array
     */
    public function getAllExistingTagsByGUID($guid)
    {
        global $SystemDB;
        
        return $SystemDB->getQueryResultArray("SELECT * FROM {$this->table_categories} ".
        "WHERE {$this->field_category_id} IN ".
        "(SELECT category_id FROM {$this->table_items} WHERE `guid` = :guid)", 
            ['guid' => $guid]);
        
    }
    
    /**
     * Remove everything associated with the specified GUID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $guid
     */
    public function removeAllTagsByGUID($guid)
    {
        global $SystemDB;
        
        $SystemDB->query("DELETE FROM {$this->table_items} WHERE `guid` = :guid", 
            ['guid' => $guid]);
    }
    
    /**
     * Returns true if tag is associated with this GUID
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $tag
     * @param string $guid
     * @return array
     */
    public function isTagAssociatedWithGUID($tag, $guid)
    {
        global $SystemDB;
        
        return $SystemDB->getQueryResultSingleRow("SELECT 1 FROM {$this->table_categories} INNER JOIN {$this->table_items} ON category_id = {$this->field_category_id} " .
        "WHERE LOWER({$this->table_categories}.title) = LOWER(:tag) AND {$this->table_items}.`guid` = :guid", 
            ['tag' => $tag, 'guid' => $guid]);
        
    }

    /**
     * Returns all tags with an extra field: total_item_count and percentage.
     * Useful for building tag cloud
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @return array
     */
    public function getAllTagsWithCount()
    {
        global $SystemDB;
        
        $result = $SystemDB->getQueryResultArray( "SELECT  {$this->field_category_id},{$this->table_categories}.title,{$this->table_categories}.virtual_filename, {$this->table_categories}.status, COUNT({$this->field_id}) as total_item_count from {$this->table_categories} LEFT JOIN {$this->table_items} ON {$this->field_category_id} = {$this->field_item_category_id} WHERE {$this->table_categories}.status > 0 GROUP BY {$this->field_category_id}, {$this->table_categories}.title,{$this->table_categories}.virtual_filename,{$this->table_categories}.status ORDER BY total_item_count DESC");
        $total_tag_count = array_sum(array_column($result, 'total_item_count'));
        for ($i = 0, $total_count = count ($result); $i < $total_count;$i++)
        {
            $result[$i]['percentage'] = 1.0 * $result[$i]['total_item_count'] / $total_tag_count;
            $percent_int =  ceil(($result[$i]['percentage'] * 100));
            if ($percent_int > 10)
                $percent_int = ((int) ($percent_int / 10)) * 10;
            $result[$i]['rank'] =$percent_int ;
        }
        return $result;
    }

    /**
     * Assign existing tags
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param array $tags_array
     * @param \SCHLIX\cmsApplication_List $app_object
     * @param array $item
     */
    private function assignExistingTagsToArticleItem(array $tags_array, \SCHLIX\cmsApplication_List $app_object, array $item)
    {
        global $SystemDB;
        
        if (is_a($app_object, '\\SCHLIX\\cmsApplication_List') && is_array($item))
        {
            $all_existing_tags = $this->getAllExistingTagsByGUID($item['guid']);
            $tag_ids_to_delete = [];
            $all_existing_tag_ids = [];
            // delete old tags not specified in the new tags
            if ($all_existing_tags)
            {
                foreach ($all_existing_tags as $existing_tag)
                {
                    $all_existing_tag_ids[] = $existing_tag[$this->field_category_id];
                    $existing_tag_name = $existing_tag['title'];
                    if (!in_array($existing_tag_name, $tags_array))
                    {
                        $tag_ids_to_delete[] = $existing_tag[$this->field_category_id];
                    }
                }
                if (empty($tags_array))
                    $tag_ids_to_delete = $all_existing_tag_ids;
                else
                {
                    if ($tag_ids_to_delete)
                    {
                        $str_tag_ids_to_delete = implode(',', $tag_ids_to_delete);
                        $sanitized_guid = sanitize_string($item['guid']);
                        $sql = "DELETE FROM {$this->table_items} WHERE guid = {$sanitized_guid} AND {$this->field_item_category_id} IN ({$str_tag_ids_to_delete})";
                        $SystemDB->query($sql);
                    }
                }
            }
            // now insert
            foreach ($tags_array as $tag)
            {
                $tag_info = $this->getTagByName($tag);
                if ($tag_info)
                {
                    $tag_info_id = $tag_info[$this->field_category_id];                    
                    if (!$this->isTagAssociatedWithGUID($tag, $item['guid']))
                    {
                        $item_id = $item[$app_object->getFieldID()];
                        $internal_url = "action=viewitem&id={$item_id}";
                        $sef_url = $app_object->createFriendlyURL($internal_url);
                        $date_created = ($item['date_created'] && ($item['date_created'] != NULL_DATE)) ? $item['date_created']  : get_current_datetime();
                        $this->table_items->quickInsert([                      
                          'guid' => $item['guid'],                      
                          'category_id' => $tag_info_id,   
                          'date_created' => $date_created,
                          'internalurl' => $internal_url,
                          'application' => $app_object->getApplicationNameOnly()
                        ]);
                    }
                    
                }
            }
            // now remove any existing tags not associated with this item
           $this->deleteAllEmptyTags();
        }
    }
    /**
     * Create a tag array from a comma separated string
     * @param string $str_tags
     * @return array
     */
    protected function getArrayOfTagsFromString($str_tags)
    {
        $tags = trim($str_tags);
        if ($tags)
        {
            if (strpos($tags,',') !== FALSE)
            {
                $tags_array = array_map('trim', explode(',', $tags));                
                return array_unique($tags_array);
            } else return [$tags];
        }
        return NULL;
    }

    /**
     * Automatically create tags if they don't exist
     * @global \App\Users $CurrentUser
     * @param array $array_of_tags
     */
    protected function createTagsFromArray(array $array_of_tags)
    {
        global $CurrentUser;
        
        $array_of_tags = array_unique($array_of_tags);
        foreach ($array_of_tags as $tag)
        {
            if (!$this->getTagByName($tag))
            {
                $this->table_categories->quickInsert([
                  'title' => $tag, 
                  'virtual_filename' => convert_into_sef_friendly_title($tag),
                  'date_created' => get_current_datetime(),
                  'status' => 1,
                  'guid' => new_uuid_v4(),
                  'created_by_id' => $CurrentUser->getCurrentUserID()
                  ]);
            }
        }
    }
    
    /**
     * Returns a tag by name
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $name
     * @return array
     */
    public function getTagByName($name)
    {
        global $SystemDB;
        
//        $sanitized_name = sanitize_string(strtolower($name));
        return $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE LOWER(`title`) = :title",['title'=>strtolower($name)]);                
//        return $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE LOWER(`title`) = {$sanitized_name}");        
    }

    /**
     * Returns a tag by virtual filename
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $vf_name
     * @return array
     */
    public function getTagByVirtualFileName($vf_name)
    {
        global $SystemDB;
        
        $sanitized_vf_name = sanitize_string(strtolower($vf_name));
        return $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_categories} WHERE LOWER(`virtual_filename`) = {$sanitized_vf_name}");        
    }
        
    public function modifyByGUID($guid, $title)
    {
        
    }
    
    /**
     * Deletes all empty tags
     * @global \SCHLIX\cmsDatabase $SystemDB
     */
    public function deleteAllEmptyTags()
    {
        global $SystemDB;
        
        $sql = "SELECT  {$this->field_category_id}, COUNT({$this->field_id}) as total_item_count from {$this->table_categories} LEFT JOIN {$this->table_items} ON {$this->field_category_id} = {$this->field_item_category_id} GROUP BY {$this->field_category_id} HAVING total_item_count = 0";
        $empty_tags = $SystemDB->getQueryResultArray($sql);
        if ($empty_tags)
        {
            $empty_tags_cids = array_column($empty_tags,$this->field_category_id);
            $str_empty_tags_cids = implode(',', $empty_tags_cids);
            $SystemDB->query("DELETE FROM {$this->table_categories} WHERE {$this->field_category_id} IN ({$str_empty_tags_cids})");
        }
        
        
    }
    /**
     * View Main Page
     * @param int $pg
     */
    public function viewMainPage($pg = 1)
    {
        $str_mainpage_title = $this->getConfig('str_mainpage_title', $this->getApplicationDescription());
        $main_meta_options =  $this->translateMetaOptions($this->getConfig('array_mainpage_meta_options'));        
        $this->setPageTitle($str_mainpage_title );
        $this->setPageMetaDescription($this->getConfig('str_meta_description'));
        $this->setPageMetaKeywords($this->getConfig('str_meta_key'));

        $all_tags = $this->getAllTagsWithCount();
        
        $main_meta_options = convert_array_values_to_keys($this->getConfig('array_mainpage_meta_options'));        

        $local_variables = compact(array_keys(get_defined_vars()));
        $this->loadTemplateFile('view.main', $local_variables);
        
    } 
    
    public function createFriendlyURL($str) {

        $final_url = '';
        if (SCHLIX_SEF_ENABLED) {
            $command_array = [];
            parse_str($str, $command_array);
            if (array_key_exists('action', $command_array))
            {
                switch ($command_array['action'])
                {

                    case 'viewtag':

                        $tag= $this->getTagByName(trim($command_array['tag']));
                        if ($tag)
                        {
                            return parent::createFriendlyURL("action=viewcategory&cid={$tag[$this->field_category_id]}");
                        } 
                        return NULL;
                        break;
                    default: return parent::createFriendlyURL($str);
                }
            }
            $final_url = SCHLIX_SITE_HTTPBASE . '/' . $this->app_name . $final_url;
        } else
            return parent::createFriendlyURL($str);
        return remove_multiple_slashes($final_url);
    }

    /**
     * Interprets command from SEO-friendly URL
     * @param string $urlpath
     * @return array
     */
    public function interpretFriendlyURL($urlpath) {
        if (SCHLIX_SEF_ENABLED && !fget_string('app')) {
            
            $parsedurl = $this->probeFriendlyURLDestination($urlpath);
            $url = $parsedurl['url'];
            $url_array = $parsedurl['url_array'];
            if ($url_array[0] == 'rss')
                $command['action'] = 'rss';
            else
            if ($c = preg_match_all("/(pg)(\d+).*?(html)/is", $url_array[0], $x)) {
                $command = [];
                $folder_requestpage = $x[2][0];
                $command['pg'] = $folder_requestpage;
                $command['action'] = 'main';
            } else
                return parent::interpretFriendlyURL($urlpath);
        } else {
            return parent::interpretFriendlyURL($urlpath);
        }
        return $command;
    }

    /**
     * Returns the default category ID 
     * @return int
     */
    public function getDefaultCategoryID() {
        return 1;
    }  

    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. 
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */    
    
    public function getMainpageMetaOptionKeys() {
        return parent::getMainpageMetaOptionKeys();
    }
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of main page options. In this base
     * class, the key is almost similar to getCategoryMetaOptionKeys
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */        
    public function getCategoryMetaOptionKeys() {
        return parent::getCategoryMetaOptionKeys();
    }
    
    //_______________________________________________________________________________________________________________//
    /**
     * Returns an array containing on array of item options.
     * The values of the options will still be evaluated as a flat list array, 
     * however it is sectioned into array with the following keys:
     * header, value, type, and options.
     * - Label: section title (not used for any evaluation
     * - Type: checkboxgroup, dropdownlist, or none. If none, then it means there 
     *         are suboptions which contain another array of this
     * - Key: the key option. Please note that checkboxgroup doesn't have a key
     *        since the keys are in the options
     * - Options: an array with 2 keys: label and key
     * 
     * @return array
     */
    public function getItemMetaOptionKeys() {
        return parent::getItemMetaOptionKeys();
    }

    
    /**
     * Run command passed by the main router
     * @param array $command
     * @return bool
     */
    public function Run($command) {

        switch ($command['action']) {
            case 'viewitem':
                $this->viewItemByID($command['id'], $this->cache);
                break;
            case 'viewcategory': $this->viewCategoryByID($command['cid'], $command['pg'], 'date_created', 'DESC', $this->cache);
                break;
            case 'main': $this->viewMainPage($command['pg']);
                break;

            default: return parent::Run($command);
        }

        return RETURN_FULLPAGE;
    }
    
    public function regenerateAllTags()
    {
        global $SystemDB;
        
        $applications = new \App\Core_ApplicationManager();
        $apps = $applications->getAllItems();
        //$hooks = $SystemDB->getQueryResultArray("SELECT * FROM gk_hook_functions");
        $SystemDB->query("TRUNCATE TABLE gk_tag_items");        
        $SystemDB->query("TRUNCATE TABLE gk_tag_categories");        
        if ($apps)
        foreach ($apps as $app)
        {
            $full_app_name = '\\App\\'.$app['title'];
            if (class_exists($full_app_name))
            {
                $app_regular = new $full_app_name();
                $items = null;                
                if (!$app_regular->isFrontendRuntimeDisabled() && is_subclass_of($app_regular, '\\SCHLIX\\cmsApplication_List'))
                {
                    
                    $items = $app_regular->getAllItems('*');
                    if ($items)
                    {
                        foreach ($items as $item)
                        $this->createTagsFromItem($app_regular, $item['id']);
                    }
                }
                
                
                $app_regular = null;
                $items = null;
            }
        }
        
        
    }
    /**
     * Do something after save item
     * @param \SCHLIX\cmsApplication_List $obj     
     * @param array $datavalues
     * @param array $original_datavalues
     * @param array $previous_item
     * @param array $retval
     */
    public function hook_onAfterSaveItem($obj, $datavalues, $original_datavalues, $previous_item, $retval)
    {
        
        if ($obj->itemColumnExists('tags') && !empty($datavalues['guid'])) {
            $tag_save_result = $this->createTagsFromItem($obj, $retval['id']);
        }
    }
    
}
