<?php
/**
 * Settings.
 *
 * @package documentor
 */

/**
 * Settings Class
 */
class Documentor_Settings {
    /**
     * Construct
     */
    public function __construct() {
        $this->settings_api = new Documentor_Settings_API();

        add_action( 'admin_init', array( $this, 'admin_init' ) );
        add_action( 'admin_menu', array( $this, 'admin_menu' ) );
    }

    /**
     * Initialize the settings
     */
    public function admin_init() {

        // set the settings.
        $this->settings_api->set_sections( $this->get_settings_sections() );
        $this->settings_api->set_fields( $this->get_settings_fields() );

        // initialize settings.
        $this->settings_api->admin_init();
    }

    /**
     * Register the admin settings menu
     */
    public function admin_menu() {
        add_submenu_page(
            'documentor',
            __( 'Documentor Settings', 'documentor' ),
            __( 'Settings', 'documentor' ),
            'manage_options',
            'documentor-settings',
            array( $this, 'plugin_page' )
        );
    }

    /**
     * Plugin settings sections
     *
     * @return array
     */
    public function get_settings_sections() {
        $sections = array(
            array(
                'id'    => 'documentor_settings',
                'title' => __( 'General', 'documentor' ),
            ),
            array(
                'id'    => 'documentor_single',
                'title' => __( 'Single Document', 'documentor' ),
            ),
            array(
                'id'    => 'documentor_archive',
                'title' => __( 'Archive', 'documentor' ),
            ),
            array(
                'id'    => 'documentor_export',
                'title' => __( 'Export', 'documentor' ),
            ),
            array(
                'id'    => 'documentor_import',
                'title' => __( 'Import', 'documentor' ),
            ),
        );

        return $sections;
    }

    /**
     * Returns all the settings fields
     *
     * @return array settings fields
     */
    public function get_settings_fields() {
        include_once dirname( __FILE__ ) . '/../class-export.php';
        $export_class = new Documentor_Export();

        $settings_fields = array(
            'documentor_settings' => array(
                array(
                    'name'    => 'docs_page_id',
                    'label'   => __( 'Documentation Archive Page', 'documentor' ),
                    'desc'    => __( 'Page to display documentations list. <br> If you see the 404 error, please go to Settings > Permalinks and press "Save Changes" button.', 'documentor' ),
                    'type'    => 'select',
                    'options' => $this->get_pages(),
                ),
            ),
            'documentor_single' => array(
                array(
                    'name'    => 'show_comments',
                    'label'   => __( 'Display Comments', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'show_feedback_buttons',
                    'label'   => __( 'Display Feedback Buttons', 'documentor' ),
                    'desc'    => __( 'Helpful feedback', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'show_feedback_buttons_likes',
                    'desc'    => __( 'Display Likes / Dislikes Count', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'show_feedback_suggestion',
                    'desc'    => __( 'Display Suggestion Form After Like', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'off',
                ),
                array(
                    'name'        => 'show_feedback_suggestion_email',
                    'desc'        => __( 'Suggestion Email', 'documentor' ),
                    'type'        => 'text',
                    'placeholder' => get_option( 'admin_email' ),
                    'default'     => '',
                ),
                array(
                    'name'    => 'show_anchor_links',
                    'label'   => __( 'Display Heading Anchors', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'sidebar',
                    'label'   => __( 'Sidebar', 'documentor' ),
                    'type'    => 'html',
                ),
                array(
                    'name'    => 'sidebar_show_search',
                    'label'   => __( 'Display Search', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'sidebar_show_nav_parents',
                    'label'   => __( 'Display Parent Links', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'off',
                ),
                array(
                    'name'    => 'sidebar_show_nav_childs',
                    'label'   => __( 'Display Child Links', 'documentor' ),
                    'desc'    => __( 'Always display child navigation links (by default displayed only for active parent doc)', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'off',
                ),
                array(
                    'name'    => 'sidebar_show_nav_number_of_childs',
                    'label'   => __( 'Display Number of Childs', 'documentor' ),
                    'desc'    => __( 'Display in the title of parent link the number of childs', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),

                array(
                    'name'    => 'ajax',
                    'label'   => __( 'AJAX loading', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'ajax_custom_js',
                    'label'   => __( 'AJAX Custom JS', 'documentor' ),
                    'desc'    => __( 'Run custom JS after document loaded via AJAX', 'documentor' ),
                    'type'    => 'textarea',
                    'size'    => 'large',
                    'default' => "/*\n * New page content loaded via ajax you can get in variable 'new_page'\n * Example: console.log(new_page);\n */",
                ),
            ),
            'documentor_archive' => array(
                array(
                    'name'    => 'show_articles',
                    'label'   => __( 'Display Articles', 'documentor' ),
                    'desc'    => __( 'Top level articles list', 'documentor' ),
                    'type'    => 'checkbox',
                    'default' => 'on',
                ),
                array(
                    'name'    => 'articles_number',
                    'label'   => __( 'Number of Articles', 'documentor' ),
                    'desc'    => __( 'Type -1 to display all available articles', 'documentor' ),
                    'type'    => 'number',
                    'default' => 3,
                ),
            ),
            'documentor_export' => array(
                array(
                    'name'    => 'custom_css',
                    'label'   => __( 'Custom CSS', 'documentor' ),
                    'desc'    => __( 'Added in exported HTML files', 'documentor' ),
                    'type'    => 'textarea',
                    'size'    => 'large',
                    'default' => $export_class->custom_css,
                ),
                array(
                    'name'    => 'custom_js',
                    'label'   => __( 'Custom JS', 'documentor' ),
                    'desc'    => __( 'Added in exported HTML files', 'documentor' ),
                    'type'    => 'textarea',
                    'size'    => 'large',
                    'default' => $export_class->custom_js,
                ),
                array(
                    'name'    => 'clean_html',
                    'label'   => __( 'Clean HTML RegExp', 'documentor' ),
                    'desc'    => __( 'Each regexp on new line (change it only if you understand what you do)', 'documentor' ),
                    'type'    => 'textarea',
                    'size'    => 'large',
                    'default' => str_replace( '\'', "\\'", $export_class->clean_html_regexp ),
                ),
            ),
            'documentor_import' => array(
                array(
                    'name'    => 'single-template',
                    'label'   => __( 'Single Template', 'documentor' ),
                    'type'    => 'checkbox',
                    'options' => 'off',
                ),
                array(
                    'name'    => 'archive-template',
                    'label'   => __( 'Archive Template', 'documentor' ),
                    'type'    => 'checkbox',
                    'options' => 'off',
                ),
                array(
                    'name'    => 'docs_import',
                    'type'    => 'submit',
                    'placeholder' => 'Import',
                    'class' => 'documentor_button_import',
                ),
            ),
        );

        return $settings_fields;
    }

    /**
     * The plguin page handler
     *
     * @return void
     */
    public function plugin_page() {
        echo '<div class="wrap">';

        $this->settings_api->show_navigation();
        $this->settings_api->show_forms();

        $this->scripts();

        echo '</div>';
    }
    /**
     * Get all the pages
     *
     * @return array page names with key value pairs
     */
    public function get_pages() {
        $pages_options = array( '' => __( '&mdash; Select Page &mdash;', 'documentor' ) );
        $pages         = get_pages(
            array(
                'numberposts' => -1, // phpcs:ignore
            )
        );

        if ( $pages ) {
            foreach ( $pages as $page ) {
                $pages_options[ $page->ID ] = $page->post_title;
            }
        }
        return $pages_options;
    }

    /**
     * Scripts
     *
     * @return void
     */
    public function scripts() {
        ?>
        <script type="text/javascript">
            jQuery(function($) {
                $('input[name="documentor_single[show_feedback_buttons]"]:checkbox').on( 'change', function() {

                    $('tr.show_feedback_buttons_likes')[ $(this).is(':checked' ) ? 'show' : 'hide' ]();

                }).change();

                $('input[name="documentor_single[show_feedback_suggestion]"]:checkbox, input[name="documentor_single[show_feedback_buttons]"]:checkbox').on( 'change', function() {
                    const isCheckedFeedback = $('input[name="documentor_single[show_feedback_buttons]"]').is(':checked' );
                    const isCheckedSuggestion = $('input[name="documentor_single[show_feedback_suggestion]"]').is(':checked' );

                    $('tr.show_feedback_suggestion')[ isCheckedFeedback ? 'show' : 'hide' ]();

                    $('tr.show_feedback_suggestion_email')[ isCheckedFeedback && isCheckedSuggestion ? 'show' : 'hide' ]();

                }).change();

                $('input[name="documentor_single[ajax]"]:checkbox').on( 'change', function() {

                    $('tr.ajax_custom_js')[ $(this).is(':checked' ) ? 'show' : 'hide' ]();

                }).change();

                $('input[name="documentor_archive[show_articles]"]:checkbox').on( 'change', function() {

                    $('tr.articles_number')[ $(this).is(':checked' ) ? 'show' : 'hide' ]();

                }).change();
            });
        </script>
        <?php
    }

}
