<?php
/**
 * Newslist magic space. Allows to output news list with custom layout
 *
 * Optionally magicspace can receive options, they are:
 * 1. 'asc'/'desc' - list order direction
 * 2. N - here N is an integer value which tell the magic space how many news items you want to see in results
 * You can pass option in random order
 */
class MagicSpaces_Newslist_Newslist extends Tools_MagicSpaces_Abstract {

    /**
     * Widget name that will be used in the replacement
     *
     */
    const NEWS_WIDGET_NAME    = 'news';

    /**
     * Exclude events flag
     */
    const NEWS_EXCLUDE_EVENTS = 'exclude-events';

    /**
     * News list order direction. Descending by default.
     *
     * @var string
     */
    protected $_orderDirection = 'DESC';

    /**
     * @var string
     */
    protected $_orderByCol = 'p.publish_at';

    /**
     * How many news items should be in news list. Leave null to get everything.
     *
     * @var mixed integer|null
     */
    protected $_limit         = null;

    /**
     * Display news that have specific tags
     *
     * @var array
     */
    protected $_tags          = array();

    protected $_listName      = 'noname';

    protected $_listAll      =  true;

    protected $_excludeEvents = false;

    /**
     * Enable pagination
     */
    protected $_pagination    = null;

    protected $_allowedOrderParams = array(
        'publish-at-sort',
        'event-sort',
        'created-at-sort'
    );

    protected $_allowedOrderColumnsNames = array(
        'publish-at-sort' => 'p.publish_at',
        'event-sort' => 'n.event_date',
        'created-at-sort' => 'n.created_at'
    );

    /**
     * Main magic space entry point
     *
     * 1. Get the html version of the current page's template
     * 2. Modify news widget by adding the appropriate page id to each of its occurrences and the parse it
     * 3.
     * 4. Output the news list and ..... wait for it ...... profit!
     *
     * @return null|string
     */
    protected function _run() {

        $this->_parseParams();

        $publishAtSortFlag = false;
        if ($this->_orderByCol === 'p.publish_at') {
            $publishAtSortFlag = true;
        }

        $tmpContent     = $this->_content;
        $this->_content = $this->_getCurrentTemplateContent();
        $spaceContent   = $this->_parse();
        $this->_content = $tmpContent;
        $content        = '';
        $translator     = Zend_Registry::get('Zend_Translate');

        if(!$spaceContent) {
            $spaceContent = $this->_parse();
        }

        $request = Zend_Controller_Front::getInstance()->getRequest();
        if(($request->has('list') && ($this->_listName == $request->getParam('list', 'noname'))) || ($request->has('listall') && ($request->getParam('listall') == $this->_listAll))) {
            $this->_tags = $request->getParam('tags', array());
            if(!is_array($this->_tags)) {
                $this->_tags = explode(',', $this->_tags);
            }
        }

        $order[] = $this->_orderByCol . ' ' . $this->_orderDirection;
        if ($publishAtSortFlag) {
            $order[] = 'n.created_at' . ' ' . $this->_orderDirection;
        }

        $newsMapper = Newslog_Models_Mapper_NewsMapper::getInstance();
        $where = $newsMapper->getDbtable()->getAdapter()->quoteInto('n.published = ?', '1');
        if ($this->_excludeEvents === true) {
            $where .= ' AND '. $newsMapper->getDbtable()->getAdapter()->quoteInto('n.event = ?', '0');
        }
        if($this->_limit != null && $this->_pagination) {
            $news = Zend_Paginator::factory($newsMapper->fetchAll($where, $order, null, 0, $this->_tags));
            $news->setDefaultItemCountPerPage($this->_limit);
            $news->setCurrentPageNumber(1);
            $news->setDefaultPageRange(3);

            if($this->_listName == $request->getParam('list') || $request->getParam('listall') == $this->_listAll) {
                $news->setCurrentPageNumber(filter_var($request->getParam('pnum', 1), FILTER_SANITIZE_NUMBER_INT));
            }

            $view  = new Zend_View(array('scriptPath' => __DIR__ . '/views/'));
            $pager = $view->paginationControl($news, 'Sliding', 'pagination.phtml', array(
                'listName' => $this->_listName,
                'tags'     => urlencode(is_array($this->_tags) ? implode(',', $this->_tags) : $this->_tags)
            ));
        } else {
            if (null === $this->_limit) {
                $this->_limit = Newslog::NEWS_DEFAULT_LIMIT;
            }

            $news = $newsMapper->fetchAll($where, $order, $this->_limit, 0, $this->_tags);
        }

        if(empty($news)) {
            return $translator->translate('You don\'t_excludeEvents have news yet');
        }

        $this->_spaceContent = $spaceContent;
        foreach($news as $newsItem) {
            $content .= preg_replace('~{\$' . self::NEWS_WIDGET_NAME . ':(.+)}~uU', '{$' . self::NEWS_WIDGET_NAME . ':' . $newsItem->getPageId() . ':' . $this->_listName . ':$1}', $this->_spaceContent);
        }

        $parser          = new Tools_Content_Parser($content, array());
        $newsListContent = $parser->parseSimple();
        return ($this->_pagination && isset($view)) ? ($newsListContent . $pager) : $newsListContent;
    }

    /**
     * Parse magic space parameters $_params and init appropriate properties
     *
     */
    private function _parseParams() {
        if(!is_array($this->_params)) {
            return false;
        }
        $excludeKeys = array();
        foreach($this->_params as $key => $param) {
            if(is_array($this->_params) && !empty($this->_params)) {
                $param = strtolower($param);
                if (array_key_exists($key, $excludeKeys)) {
                    continue;
                }

                $limit = intval($param);
                if($limit) {
                    $this->_limit = $limit;
                    continue;
                }

                if($param == Widgets_Newslist_Newslist::LOD_ASC || $param == Widgets_Newslist_Newslist::LOD_DESC) {
                    $this->_orderDirection = $param;
                    continue;
                }
                //TODO option 'pag' is deprecated
                if($param == 'pag' || $param == Widgets_Newslist_Newslist::OPT_PAGER) {
                    $this->_pagination = $param;
                    $this->_limit = $this->_limit ? $this->_limit : Newslog::NEWS_DEFAULT_LIMIT;
                    continue;
                }
                if($param == Widgets_Newslist_Newslist::OPT_TAGS) {
                    $tagsSpot    = $key + 1;
                    $this->_tags = isset($this->_params[$tagsSpot]) ? explode(',', $this->_params[$tagsSpot]) : array();
                    unset($this->_params[$tagsSpot]);
                    $excludeKeys[$tagsSpot] = $tagsSpot;
                    continue;
                }
                if(in_array($param, $this->_tags)) {
                    continue;
                }
                //options for order news
                if(in_array($param, $this->_allowedOrderParams)){
                    $sortKey = array_search($param, $this->_params);
                    unset($this->_params[$sortKey]);

                    $this->_orderByCol = $this->_allowedOrderColumnsNames[$param];
                    continue;
                }
                if($param === self::NEWS_EXCLUDE_EVENTS){
                    $this->_excludeEvents = true;
                    continue;
                }
                $this->_listName = $param;
                continue;
            }
        }
    }

    /**
     * Get the html (not parsed) version of the current template
     *
     * @return bool|string
     */
    private function _getCurrentTemplateContent() {
//        $page    = Application_Model_Mappers_PageMapper::getInstance()->find($this->_toasterData['id']);
        $tempate = Application_Model_Mappers_TemplateMapper::getInstance()->find($this->_toasterData['templateId']);
        if(!$tempate instanceof Application_Model_Models_Template) {
            return false;
        }
        return $tempate->getContent();
    }
}
