<?php
/**
 * News rest service api
 *
 * @author: iamne <eugene@seotoaster.com> Seotoaster core team
 * Date: 7/3/12
 * Time: 5:34 PM
 */
class Api_Newslog_News extends Api_Service_Abstract {

    const NEWS_SECURE_TOKEN =  'News';

    protected $_cacheHelper;

    /**
     * @var Helpers_Action_Language Language helper
     * @ignore
     */
    private $_translator = null;

    /**
     * @var Newslog_Models_Mapper_NewsMapper
     */
    protected $_newsMapper = null;

    protected $_accessList = array(
        Tools_Security_Acl::ROLE_GUEST => array('allow' => array('get')),
        Tools_Security_Acl::ROLE_USER => array('allow' => array('get', 'post', 'put', 'delete')),
        Tools_Security_Acl::ROLE_ADMIN => array('allow' => array('get', 'post', 'put', 'delete')),
        Tools_Security_Acl::ROLE_SUPERADMIN => array('allow' => array('get', 'post', 'put', 'delete'))
    );

    public function init() {
        $this->_newsMapper = Newslog_Models_Mapper_NewsMapper::getInstance();
        $this->_cacheHelper = Zend_Controller_Action_HelperBroker::getExistingHelper('cache');
        $this->_translator      = Zend_Controller_Action_HelperBroker::getStaticHelper('language');
    }

    /**
     * Retreive news list or news item if id specified otherwise - 404 not found status returned
     *
     * For the news list additional parameters could be passed:
     * offset - list will start from this number
     * limit - number of the items in the list
     * order - how list will be ordered
     * tags - if this parameter is specified list will contain only news which contain those tags
     *
     * @return array
     */
    public function getAction() {
        $newsId = filter_var($this->_request->getParam('id'), FILTER_SANITIZE_NUMBER_INT);
        if($newsId) {
            $news = $this->_newsMapper->find($newsId);
            if($news instanceof Newslog_Models_Model_News) {
                $pageHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('page');
                $metaData        = $news->getMetaData(true);
                if ($news->getType() === Newslog_Models_Model_News::TYPE_EXTERNAL && !empty($news->getExternalId())) {
                    $websiteConfig = Zend_Controller_Action_HelperBroker::getStaticHelper('config')->getConfig();
                    $websiteHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('website');
                    $sessionHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('session');
                    $currentUser = $sessionHelper->getCurrentUser();
                    $loggedUserEmail = $currentUser->getEmail();
                    $websiteUrl = $websiteHelper->getUrl();
                    if(isset($websiteConfig['sambaToken']) && (isset($websiteConfig['websiteId'])) ) {
                        $url = parse_url($websiteUrl);
                        $data['websiteUrl'] = $url['host'];
                        $data['websiteId'] = $websiteConfig['websiteId'];
                        $data['sambaToken'] = $websiteConfig['sambaToken'];
                        $data['email'] = $loggedUserEmail;
                        $data['nid'] = $news->getExternalId();
                        $seosambaRequest = Tools_Factory_PluginFactory::createPlugin('api',array(), array('websiteUrl' => $websiteUrl));
                        $methodType = 'get';
                        $methodName = 'newsEditLink';
                        $response = $seosambaRequest::request($methodType, $methodName, $data);
                        if (!empty($response) && !empty($response['link'])) {
                            $metaData['newsExternalLinkUrl'] = $response['link'];
                        } else {
                            $metaData['newsExternalLinkUrl'] = '';
                        }
                    }
                }
                $metaData['url'] = $pageHelper->clean($metaData['url']);
                return $news->setMetaData($metaData)
                    ->toArray();
            }

            //news item can not be found
            $this->_error('404 News item can not be found!', self::REST_STATUS_NOT_FOUND);
        }

        //retrieve and validate additional parameters
        $where = null;
        $offset = filter_var($this->_request->getParam('offset'), FILTER_SANITIZE_NUMBER_INT);
        $limit  = filter_var($this->_request->getParam('limit'), FILTER_SANITIZE_NUMBER_INT);
        $order  = filter_var($this->_request->getParam('order'), FILTER_SANITIZE_STRING);
        $search = filter_var($this->_request->getParam('search'), FILTER_SANITIZE_STRING);
        $tags   = (($tags = filter_var($this->_request->getParam('tags', false), FILTER_SANITIZE_STRING)) != '') ? explode(',', $tags) : array();
        $count = (bool)$this->_request->has('count');
        $event = filter_var($this->_request->getParam('event'), FILTER_SANITIZE_STRING);
        $published = filter_var($this->_request->getParam('published'), FILTER_SANITIZE_STRING);
        $searchByPostName = str_replace('*-amp-*', '&',  filter_var($this->_request->getParam('postName'), FILTER_SANITIZE_STRING));
        $publishAtFrom = filter_var($this->_request->getParam('publishAtFrom'), FILTER_SANITIZE_STRING);
        $publishAtTo = filter_var($this->_request->getParam('publishAtTo'), FILTER_SANITIZE_STRING);
        $createdAtFrom = filter_var($this->_request->getParam('createdAtFrom'), FILTER_SANITIZE_STRING);
        $createdAtTo = filter_var($this->_request->getParam('createdAtTo'), FILTER_SANITIZE_STRING);

        if(empty($order)){
            $order = Widgets_Newslist_Newslist::LOD_DESC;
        }

        if($event){
            $where = ($event == 'yes') ? "n.event = '1'" : "n.event = '0'";
        }

        if($published){
            if($where){
                $where .= ($published == 'yes') ? " AND n.published = '1'" : " AND n.published = '0'";
            }else{
                $where = ($published == 'yes') ? "n.published = '1'" : "n.published = '0'";
            }
        }

        if (!empty($searchByPostName)) {
            if (!empty($where)) {
                $where .= ' AND ';
            }

            $attributeValues = explode(' ', trim($searchByPostName));
            $where .= ' (';

            foreach ($attributeValues as $key => $attrVal) {
                $where .= $this->_newsMapper->getDbTable()->getAdapter()->quoteInto('title LIKE ?',
                    '%' . $attrVal . '%');

                if (count($attributeValues) > $key + 1) {
                    $where .= ' AND ';
                }

            }
            $where .= ' )';
        }

        if (!empty($publishAtFrom)) {
            if (!empty($where)) {
                $where .= ' AND ';
            }
            $where .= $this->_newsMapper->getDbTable()->getAdapter()->quoteInto('p.publish_at >=?',  $publishAtFrom);
        }

        if (!empty($publishAtTo)) {
            if (!empty($where)) {
                $where .= ' AND ';
            }
            $where .= $this->_newsMapper->getDbTable()->getAdapter()->quoteInto('p.publish_at <= ?',  $publishAtTo);
        }

        if (!empty($createdAtFrom)) {
            if (!empty($where)) {
                $where .= ' AND ';
            }
            $where .= $this->_newsMapper->getDbTable()->getAdapter()->quoteInto('n.created_at >=?',  $createdAtFrom);
        }

        if (!empty($createdAtTo)) {
            if (!empty($where)) {
                $where .= ' AND ';
            }
            $where .= $this->_newsMapper->getDbTable()->getAdapter()->quoteInto('n.created_at <= ?',  $createdAtTo);
        }

        //fetch all news
        $news   = $this->_newsMapper->getAllNews(
            ($where) ? $where : null,
            ($order)  ? array('p.publish_at ' . strtoupper($order)) : array(),
            ($limit)  ? $limit : null,
            ($offset) ? $offset : null,
            $tags,
            ($search) ? $search : null,
            $count
        );

        $websiteHelper = Zend_Controller_Action_HelperBroker::getStaticHelper('website');
        return ($count) ? $news : array_map(function($newsItem) use ($websiteHelper) {
            $newsItemMeta = $newsItem->getMetaData(true);

            if(!file_exists($websiteHelper->getPath() . $websiteHelper->getPreview() . $newsItemMeta['image'])) {
                $newsItemMeta['image'] =  null;
            }

            return $newsItem->setMetaData($newsItemMeta)
                ->toArray();
        }, $news);
    }

    /**
     * Add new news item to the database
     *
     * @return mixed
     */
    public function postAction() {
        $data = Zend_Json::decode($this->_request->getRawBody());
        $tokenToValidate = $data[Tools_System_Tools::CSRF_SECURE_TOKEN];
        $valid = Tools_System_Tools::validateToken($tokenToValidate, self::NEWS_SECURE_TOKEN);
        if (!$valid) {
            exit;
        }
        unset($data[Tools_System_Tools::CSRF_SECURE_TOKEN]);
        return $this->_saveNewsItem($data);
    }

    /**
     * Update an existing news item
     *
     * @return mixed
     */
    public function putAction() {
        return $this->_saveNewsItem(Zend_Json::decode($this->_request->getRawBody()));
    }

    /**
     * Delete news item by news id
     *
     * Register news garbage collector observer Newslog_Tools_Watchdog_News to clean related news page
     * @return mixed
     */
    public function deleteAction() {
        $id = filter_var($this->_request->getParam('id'), FILTER_SANITIZE_NUMBER_INT);
        if($id) {
            $newsItem = $this->_newsMapper->find($id);
            $newsItem->registerObserver(new Newslog_Tools_Watchdog_News(array(
                'action' => Tools_System_GarbageCollector::CLEAN_ONDELETE
            )));
            $deleteResult = $this->_newsMapper->delete($newsItem);
            Tools_Search_Tools::removeFromIndex($newsItem->getPageId());
            $this->_cacheHelper->clean(false, false,'feeds_news');
            return $deleteResult;
        }
        $this->_error('Unknown news entry', self::REST_STATUS_NOT_FOUND);
    }

    /**
     * Create / Update news item
     *
     * Register news watchdog observer Newslog_Tools_Watchdog_News to create / update relevant news page
     * @param array $newsData
     * @return mixed
     */
    protected function _saveNewsItem(array $newsData) {
        $newsItem = new Newslog_Models_Model_News($newsData);
        if (!empty($newsData['createdAt'])) {
            $newsItem->setCreatedAt(date("Y-m-d", strtotime($newsData['createdAt'])).' '.date("H:i:s", strtotime('now')));
        }

        if (!empty($newsData['eventDate'])) {
            $newsItem->setEventDate(date("Y-m-d", strtotime($newsData['eventDate'])).' '.date("H:i:s", strtotime('now')));
        }
        $userId   = Zend_Controller_Action_HelperBroker::getStaticHelper('session')->getCurrentUser()->getId();


        $newsItem->setUserId($userId)
            ->registerObserver(new Newslog_Tools_Watchdog_News())
            ->registerObserver(new Newslog_Tools_Watchdog_Ping());

        /** Here we check if news with such url already exists */
        $metaData      = Zend_Json::decode($newsData['metaData']);
        $url           = $this->prepareUrl($metaData['url']);
        $oldUrl        = $this->prepareUrl($metaData['oldUrl']);
        $page = Application_Model_Mappers_PageMapper::getInstance()->findByUrl($url . '.html');
        $oldPage = Application_Model_Mappers_PageMapper::getInstance()->findByUrl($oldUrl . '.html');

        if($url !== $oldUrl && $page instanceof Application_Model_Models_Page && $oldPage instanceof Application_Model_Models_Page) {
            $this->_error($this->_translator->translate('The news page with') . " $url.html " . $this->_translator->translate('already exists'));
        }
        if ($page instanceof Application_Model_Models_Page) {
            $isExistNews =  $this->_newsMapper->findByPageId($page->getId());

            if($isExistNews instanceof Newslog_Models_Model_News && is_null($newsItem->getId())) {
                $this->_error($this->_translator->translate('The news page with') . " $url.html " . $this->_translator->translate('already exists'));
            }

            $newsItem->setPageId($page->getId());
        }

        try {
            $newsItem = $this->_newsMapper->save($newsItem);
            if(!$newsItem instanceof Newslog_Models_Model_News) {
                $this->_error('Server encountered an error during the last request');
            }
            $this->_cacheHelper->clean(false, false,'feeds_news');

        } catch (Exception $e) {
            if(Tools_System_Tools::debugMode()) {
                error_log($e->getMessage());
            }
            $this->_error($e->getMessage());
        }

        return $newsItem->toArray();
    }

    protected function prepareUrl($url) {
        $pageHelper    = Zend_Controller_Action_HelperBroker::getStaticHelper('page');
        return str_replace('/', '', $pageHelper->clean($pageHelper->filterUrl($url)));
    }
}
