<?php
/**
 * File Session.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Tomasz Kur <t.kur@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Session;

use App\Exceptions\NotAllowedMethod;
use App\Session;

/**
 * File Session Class.
 */
class File extends Base
{
	/** {@inheritdoc} */
	public static function clean()
	{
		$time = microtime(true);
		$lifeTime = \Config\Security::$maxLifetimeSession;
		$exclusion = ['.htaccess', 'index.html', 'sess_' . session_id()];
		foreach (new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator(Session::SESSION_PATH, \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST) as $item) {
			if ($item->isFile() && !\in_array($item->getBasename(), $exclusion)) {
				$sessionData = static::unserialize(file_get_contents($item->getPathname()));
				if (!empty($sessionData['last_activity']) && ($time - $sessionData['last_activity']) < $lifeTime) {
					continue;
				}
				unlink($item->getPathname());
				if (!empty($sessionData['baseUserId']) || !empty($sessionData['authenticated_user_id'])) {
					$userId = empty($sessionData['baseUserId']) ? $sessionData['authenticated_user_id'] : $sessionData['baseUserId'];
					$userName = \App\User::getUserModel($userId)->getDetail('user_name');
					if (!empty($userName)) {
						yield $userId => $userName;
					}
				}
			}
		}
	}

	/** {@inheritdoc} */
	public static function cleanAll(): int
	{
		$exclusion = ['.htaccess', 'index.html', 'sess_' . session_id()];
		$i = 0;
		foreach (new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator(Session::SESSION_PATH, \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST) as $item) {
			if ($item->isFile() && !\in_array($item->getBasename(), $exclusion)) {
				unlink($item->getPathname());
				++$i;
			}
		}
		return $i;
	}

	/** {@inheritdoc} */
	public function getById(string $sessionId): array
	{
		$sessionFilePath = \App\Session::SESSION_PATH . \DIRECTORY_SEPARATOR . 'sess_' . $sessionId;
		$sessionData = [];
		if (file_exists($sessionFilePath) && ($content = file_get_contents($sessionFilePath))) {
			$sessionData = self::unserialize($content);
		}

		return $sessionData;
	}

	/**
	 * Deserialize session data from string, entry function.
	 *
	 * @param string $session
	 *
	 * @throws NotAllowedMethod
	 *
	 * @return array
	 *
	 * @example http://php.net/manual/en/function.session-decode.php#108037
	 */
	public static function unserialize(string $session)
	{
		$method = \ini_get('session.serialize_handler');
		switch ($method) {
			case 'php':
				return self::unserializePhp($session);
				break;
			case 'php_serialize':
				return self::unserializePhpSerialize($session);
				break;
			default:
				throw new NotAllowedMethod('Unsupported session.serialize_handler: ' . $method . '. Supported: php, php_serialize');
		}
	}

	/**
	 * Deserialize session data from string php handler method.
	 *
	 * @param string $session
	 *
	 * @return array
	 */
	private static function unserializePhp(string $session)
	{
		$return = [];

		preg_match_all('/(^|;|\})([a-zA-Z0-9_-]+)\|/i', $session, $matches, PREG_OFFSET_CAPTURE);
		$matches = $matches[2] ?? [];

		foreach ($matches as $index => [$key, $offset]) {
			$nextIndex = $index + 1;
			$start = $offset + \strlen($key) + 1;

			if (isset($matches[$nextIndex])) {
				[, $nextOffset] = $matches[$nextIndex];
				$valueText = substr($session, $start, $nextOffset - $start);
			} else {
				$valueText = substr($session, $start);
			}

			$return[$key] = unserialize($valueText);
		}

		return $return;
	}

	/**
	 * Deserialize session data from string php_serialize handler method.
	 *
	 * @param string $session
	 *
	 * @return array
	 */
	private static function unserializePhpSerialize(string $session)
	{
		return unserialize($session);
	}
}
