<?php

/**
 * Basic class for alternate auth methods.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

namespace App;

/**
 * Basic class for alternate auth methods.
 */
class UserAuth extends \App\Base
{
	/** @var string Table name where configuration is set */
	public const CONFIG_TABLE = 'i_#__auth';
	/**
	 * Providers objects.
	 *
	 * @var UserAuth\AbstractProvider[]
	 */
	private static $providers;

	public static function getProvidersByType(string $type, bool $active = true): array
	{
		$providers = [];
		foreach (static::getProviders() as $provider) {
			if ($type === $provider->getType()) {
				$providers[$provider->getName()] = $provider;
			}
		}

		return $providers;
	}

	/**
	 * Get providers.
	 *
	 * @return UserAuth\AbstractProvider[]
	 */
	public static function getProviders(): array
	{
		if (null === static::$providers) {
			$iterator = new \DirectoryIterator(__DIR__ . '/Integrations/UserAuth');
			foreach ($iterator as $item) {
				if ($item->isFile() && 'php' === $item->getExtension() && $provider = self::getProviderByName($item->getBasename('.php'))) {
					static::$providers[$provider->getName()] = $provider;
				}
			}
		}

		return static::$providers;
	}

	/**
	 * Get provider by name.
	 *
	 * @param string $name
	 *
	 * @return Authenticator\AbstractProvider|null
	 */
	public static function getProviderByName(string $name): ?Authenticator\AbstractProvider
	{
		$className = "\\App\\Integrations\\UserAuth\\{$name}";
		return class_exists($className) ? new $className() : null;
	}

	public static function getSSOProviders(): array
	{
		$providers = [];
		foreach (static::getProviders() as $provider) {
			if ($provider instanceof \App\Authenticator\SSOProvider && $provider->isActive()) {
				$providers[$provider->getName()] = $provider;
			}
		}

		return $providers;
	}

	/**
	 * Get active providers authorization host.
	 * Required for SSO | CSP: form-action and redirects.
	 *
	 * @return array
	 */
	public static function getActiveProvidersAuthorizationHost(): array
	{
		$urls = [];
		foreach (static::getSSOProviders() as $provider) {
			$urls[] = $provider->getAuthorizationHost();
		}

		return $urls;
	}

	/**
	 * Get provider configurations.
	 *
	 * @param string $providerName
	 *
	 * @return array
	 */
	public static function getConfig(string $providerName): array
	{
		if (Cache::has('UserAuthConfig', $providerName)) {
			return \App\Cache::get('UserAuthConfig', $providerName);
		}
		$data = (new Db\Query())->from(self::CONFIG_TABLE)->where(['providertype' => $providerName])
			->one(Db::getInstance('admin')) ?: [];
		$data['config'] ??= '';
		$data['config'] = Json::isEmpty($data['config']) ? [] : Json::decode($data['config']);

		return Cache::save('UserAuthConfig', $providerName, $data, Cache::LONG);
	}

	/**
	 * A hook to be called when the encryption key is changed, during the encryption process.
	 *
	 * @param array                     $values
	 * @param \App\Encryptions\Settings $encryptInstance
	 */
	public static function onEncrypt(array $values, Encryptions\Settings $encryptInstance)
	{
		$forUpdate = [];
		foreach (static::getProviders() as $provider) {
			$config = $provider->get('config');

			foreach ($provider->getEncryptedFields() as $fieldName) {
				if (!empty($values[$provider->getName()][$fieldName])) {
					$config[$fieldName] = $encryptInstance->encrypt($values[$provider->getName()][$fieldName]);
					if (empty($config[$fieldName])) {
						throw new \App\Exceptions\AppException('ERR_IMPOSSIBLE_ENCRYPT');
					}
				}
			}
			if ($config !== $provider->get('config')) {
				$forUpdate[self::CONFIG_TABLE][] = ['update' => ['config' => Json::encode($config)], 'where' => ['providertype' => $provider->getName()]];
			}
		}

		return $forUpdate;
	}

	/**
	 * A hook to be called when the encryption key is changed, during the decryption process.
	 *
	 * @return string The decrypted secret
	 */
	public static function onDecrypt(): array
	{
		$values = [];
		foreach (static::getProviders() as $provider) {
			foreach ($provider->getEncryptedFields() as $fieldName) {
				$values[$provider->getName()][$fieldName] = $provider->get($fieldName);
			}
		}

		return $values;
	}
}
